// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
package software.amazon.encryption.s3.internal;

import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

import javax.crypto.Cipher;
import javax.crypto.SecretKey;
import javax.crypto.spec.GCMParameterSpec;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;

import software.amazon.encryption.s3.S3EncryptionClientException;
import software.amazon.encryption.s3.S3EncryptionClientSecurityException;
import software.amazon.encryption.s3.algorithms.AlgorithmSuite;
import software.amazon.encryption.s3.materials.CryptographicMaterials;
import software.amazon.encryption.s3.materials.DecryptionMaterials;

/**
 * Composes a CMM to provide S3 specific functionality
 */
public class CipherProvider {

    private static byte[] DERIVE_KEY_LABEL_TEMPLATE = "__DERIVEKEY".getBytes(StandardCharsets.UTF_8);
    private static byte[] COMMITKEY_LABEL_TEMPLATE = "__COMMITKEY".getBytes(StandardCharsets.UTF_8);
    // Empty arrays created once when class loads
    // Reused for all zero-check comparisons
    private static final byte[] EMPTY_IV = new byte[12];
    private static final byte[] EMPTY_MESSAGE_ID = new byte[28];
    private static final byte[] FIXED_IV_FOR_COMMIT_ALG = new byte[]{
            0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01
    };

    public static SecretKey generateDerivedEncryptionKey(final DecryptionMaterials materials, byte[] messageId) {
        //= specification/s3-encryption/key-derivation.md#hkdf-operation
        //= type=implication
        //# - The hash function MUST be specified by the algorithm suite commitment settings.
        String macAlgorithm = materials.algorithmSuite().kdfHashAlgorithm();
        HmacKeyDerivationFunction kdf;
        try {
            kdf = HmacKeyDerivationFunction.getInstance(macAlgorithm, materials.cryptoProvider());
        } catch (NoSuchAlgorithmException e) {
            throw new IllegalStateException(e);
        }
        //= specification/s3-encryption/key-derivation.md#hkdf-operation
        //= type=implication
        //# - The input keying material MUST be the plaintext data key (PDK) generated by the key provider.
        kdf.init(materials.dataKey().getEncoded(), messageId);

        //= specification/s3-encryption/key-derivation.md#hkdf-operation
        //# - The length of the input keying material MUST equal the key derivation input length specified by the
        //# algorithm suite commit key derivation setting.
        if (materials.dataKey().getEncoded().length != materials.algorithmSuite().dataKeyLengthBytes()) {
            throw new S3EncryptionClientException("Length of Input key material does not match the expected value!");
        }

        //= specification/s3-encryption/key-derivation.md#hkdf-operation
        //# - The salt MUST be the Message ID with the length defined in the algorithm suite.
        if (messageId.length != materials.algorithmSuite().commitmentNonceLengthBytes()) {
            throw new S3EncryptionClientException("Length of Input Message ID does not match the expected value!");
        }

        //= specification/s3-encryption/key-derivation.md#hkdf-operation
        //= type=implication
        //# - The input info MUST be a concatenation of the algorithm suite ID as bytes followed by the string COMMITKEY as UTF8 encoded bytes.
        // Clone to prevent modification of the master copy
        final byte[] commitKeyLabel = COMMITKEY_LABEL_TEMPLATE.clone();
        short algId;
        if (materials.algorithmSuite().id() == AlgorithmSuite.ALG_AES_256_CTR_HKDF_SHA512_COMMIT_KEY.id()) {
            algId = (short) AlgorithmSuite.ALG_AES_256_GCM_HKDF_SHA512_COMMIT_KEY.id();
        } else {
            algId = (short) materials.algorithmSuite().id();
        }
        commitKeyLabel[0] = (byte) ((algId >> 8) & 0xFF);
        commitKeyLabel[1] = (byte) (algId & 0xFF);

        //= specification/s3-encryption/key-derivation.md#hkdf-operation
        //# - The length of the output keying material MUST equal the commit key length specified by the supported algorithm suites.
        //= specification/s3-encryption/encryption.md#alg-aes-256-gcm-hkdf-sha512-commit-key
        //= type=exception
        //# The derived key commitment value MUST be set or returned from the encryption process such that it can be included in the content metadata.
        final byte[] commitment = kdf.deriveKey(commitKeyLabel, materials.algorithmSuite().commitmentLengthBytes());
        //= specification/s3-encryption/decryption.md#decrypting-with-commitment
        //# When using an algorithm suite which supports key commitment, the client MUST verify the key commitment values match
        //# before deriving the [derived encryption key](./key-derivation.md#hkdf-operation).
        //= specification/s3-encryption/decryption.md#decrypting-with-commitment
        //# When using an algorithm suite which supports key commitment, the client MUST verify that the
        //# [derived key commitment](./key-derivation.md#hkdf-operation) contains the same bytes as the stored key
        //# commitment retrieved from the stored object's metadata.
        //= specification/s3-encryption/decryption.md#decrypting-with-commitment
        //= type=implication
        //# When using an algorithm suite which supports key commitment, the verification of the derived key commitment value
        //# MUST be done in constant time.
        if (!MessageDigest.isEqual(commitment, materials.getKeyCommitment())) {
            //= specification/s3-encryption/decryption.md#decrypting-with-commitment
            //# When using an algorithm suite which supports key commitment, the client MUST throw an exception when the
            //# derived key commitment value and stored key commitment value do not match.
            throw new S3EncryptionClientSecurityException("Key commitment validation failed. " +
                    "The derived key commitment does not match the stored key commitment value. " +
                    "This indicates potential data tampering or corruption.");
        }

        //= specification/s3-encryption/key-derivation.md#hkdf-operation
        //= type=implication
        //# - The input info MUST be a concatenation of the algorithm suite ID as bytes followed by the string DERIVEKEY as UTF8 encoded bytes.
        // Clone to prevent modification of the master copy
        final byte[] deriveKeyLabel = DERIVE_KEY_LABEL_TEMPLATE.clone();
        deriveKeyLabel[0] = (byte) ((algId >> 8) & 0xFF);
        deriveKeyLabel[1] = (byte) (algId & 0xFF);

        //= specification/s3-encryption/key-derivation.md#hkdf-operation
        //# - The length of the output keying material MUST equal the encryption key length specified by the algorithm suite encryption settings.
        return new SecretKeySpec(kdf.deriveKey(deriveKeyLabel, materials.algorithmSuite().dataKeyLengthBytes()), materials.algorithmSuite().dataKeyAlgorithm());
    }

    /**
     * Given some materials and an IV, create and init a Cipher object.
     *
     * @param materials the materials which dictate e.g. algorithm suite
     * @param iv
     * @param messageId
     * @return a Cipher object, initialized and ready to use
     */
    public static Cipher createAndInitCipher(final CryptographicMaterials materials, byte[] iv, byte[] messageId) {
        //= specification/s3-encryption/key-derivation.md#hkdf-operation
        //= type=implication
        //# The IV's total length MUST match the IV length defined by the algorithm suite.
        if (iv.length != materials.algorithmSuite().iVLengthBytes()) {
            throw new S3EncryptionClientSecurityException("IV has not been initialized!");
        }
        //= specification/s3-encryption/encryption.md#cipher-initialization
        //# The client SHOULD validate that the generated IV or Message ID is not zeros.
        if (materials.algorithmSuite().isCommitting()) {
            if (MessageDigest.isEqual(messageId, EMPTY_MESSAGE_ID)) {
                throw new S3EncryptionClientSecurityException("MessageId has not been initialized!");
            }
        } else {
            if (MessageDigest.isEqual(iv, EMPTY_IV)) {
                throw new S3EncryptionClientSecurityException("IV has not been initialized!");
            }
        }

        try {
            Cipher cipher = CryptoFactory.createCipher(materials.algorithmSuite().cipherName(), materials.cryptoProvider());
            SecretKey actualKey;
            switch (materials.algorithmSuite()) {
                case ALG_AES_256_GCM_HKDF_SHA512_COMMIT_KEY:
                    // TODO: Better Error Logging
                    // Sanity Check, Should have been already initialized to all 1's
                    if (!MessageDigest.isEqual(iv, FIXED_IV_FOR_COMMIT_ALG)) {
                        throw new S3EncryptionClientSecurityException("IV has not been initialized!");
                    }
                    //= specification/s3-encryption/encryption.md#alg-aes-256-gcm-hkdf-sha512-commit-key
                    //# The client MUST use HKDF to derive the key commitment value and the derived encrypting key
                    //# as described in [Key Derivation](key-derivation.md).
                    actualKey = generateDerivedEncryptionKey((DecryptionMaterials)materials, messageId);
                    //= specification/s3-encryption/key-derivation.md#hkdf-operation
                    //# The client MUST initialize the cipher, or call an AES-GCM encryption API, with the derived encryption key, an IV containing only bytes with the value 0x01,
                    //# and the tag length defined in the Algorithm Suite when encrypting or decrypting with ALG_AES_256_GCM_HKDF_SHA512_COMMIT_KEY.
                    cipher.init(materials.cipherMode().opMode(), actualKey, new GCMParameterSpec(materials.algorithmSuite().cipherTagLengthBits(), iv));
                    //= specification/s3-encryption/key-derivation.md#hkdf-operation
                    //= type=implication
                    //# The client MUST set the AAD to the Algorithm Suite ID represented as bytes.
                    cipher.updateAAD(materials.algorithmSuite().idAsBytes());
                    break;
                case ALG_AES_256_GCM_IV12_TAG16_NO_KDF:
                    //= specification/s3-encryption/encryption.md#alg-aes-256-gcm-iv12-tag16-no-kdf
                    //# The client MUST initialize the cipher, or call an AES-GCM encryption API, with the plaintext data key, the generated IV,
                    //# and the tag length defined in the Algorithm Suite when encrypting with ALG_AES_256_GCM_IV12_TAG16_NO_KDF.
                    cipher.init(materials.cipherMode().opMode(), materials.dataKey(), new GCMParameterSpec(materials.algorithmSuite().cipherTagLengthBits(), iv));
                    //= specification/s3-encryption/encryption.md#alg-aes-256-gcm-iv12-tag16-no-kdf
                    //= type=implication
                    //# The client MUST NOT provide any AAD when encrypting with ALG_AES_256_GCM_IV12_TAG16_NO_KDF.
                    break;
                case ALG_AES_256_CTR_IV16_TAG16_NO_KDF:
                    // WARNING: The IV may be adjusted for Ranged Gets.
                case ALG_AES_256_CBC_IV16_NO_KDF:
                    //= specification/s3-encryption/encryption.md#alg-aes-256-ctr-iv16-tag16-no-kdf
                    //# Attempts to encrypt using AES-CTR MUST fail.
                    if (materials.cipherMode().opMode() == Cipher.ENCRYPT_MODE) {
                        throw new S3EncryptionClientException("Encryption is not supported for algorithm: " + materials.algorithmSuite().cipherName());
                    }
                    cipher.init(materials.cipherMode().opMode(), materials.dataKey(), new IvParameterSpec(iv));
                    break;
                case ALG_AES_256_CTR_HKDF_SHA512_COMMIT_KEY:
                    // WARNING: The IV may be adjusted for Ranged Gets.
                    //= specification/s3-encryption/encryption.md#alg-aes-256-ctr-hkdf-sha512-commit-key
                    //# Attempts to encrypt using key committing AES-CTR MUST fail.
                    if (materials.cipherMode().opMode() == Cipher.ENCRYPT_MODE) {
                        throw new S3EncryptionClientException("Encryption is not supported for algorithm: " + materials.algorithmSuite().cipherName());
                    }
                    //= specification/s3-encryption/encryption.md#alg-aes-256-gcm-hkdf-sha512-commit-key
                    //# The client MUST use HKDF to derive the key commitment value and the derived encrypting key
                    //# as described in [Key Derivation](key-derivation.md).
                    actualKey = generateDerivedEncryptionKey((DecryptionMaterials) materials, messageId);
                    cipher.init(materials.cipherMode().opMode(), actualKey, new IvParameterSpec(iv));
                    break;

                default:
                    throw new S3EncryptionClientException("Unknown algorithm: " + materials.algorithmSuite().cipherName());
            }
            return cipher;
        } catch (Exception exception) {
            throw new S3EncryptionClientException(exception.getMessage(), exception);
        }
    }

}
