/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sqs.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p/>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateQueueRequest extends SqsRequest implements
        ToCopyableBuilder<CreateQueueRequest.Builder, CreateQueueRequest> {
    private static final SdkField<String> QUEUE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("QueueName").getter(getter(CreateQueueRequest::queueName)).setter(setter(Builder::queueName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("QueueName").build()).build();

    private static final SdkField<Map<String, String>> ATTRIBUTES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Attributes")
            .getter(getter(CreateQueueRequest::attributesAsStrings))
            .setter(setter(Builder::attributesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Attributes").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).isFlattened(true).build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(CreateQueueRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).isFlattened(true).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(QUEUE_NAME_FIELD,
            ATTRIBUTES_FIELD, TAGS_FIELD));

    private final String queueName;

    private final Map<String, String> attributes;

    private final Map<String, String> tags;

    private CreateQueueRequest(BuilderImpl builder) {
        super(builder);
        this.queueName = builder.queueName;
        this.attributes = builder.attributes;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The name of the new queue. The following limits apply to this name:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A queue name can have up to 80 characters.
     * </p>
     * </li>
     * <li>
     * <p>
     * Valid values: alphanumeric characters, hyphens (<code>-</code>), and underscores (<code>_</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * A FIFO queue name must end with the <code>.fifo</code> suffix.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Queue URLs and names are case-sensitive.
     * </p>
     * 
     * @return The name of the new queue. The following limits apply to this name:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         A queue name can have up to 80 characters.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Valid values: alphanumeric characters, hyphens (<code>-</code>), and underscores (<code>_</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A FIFO queue name must end with the <code>.fifo</code> suffix.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Queue URLs and names are case-sensitive.
     */
    public final String queueName() {
        return queueName;
    }

    /**
     * <p>
     * A map of attributes with their corresponding values.
     * </p>
     * <p>
     * The following lists the names, descriptions, and values of the special request parameters that the
     * <code>CreateQueue</code> action uses:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DelaySeconds</code> – The length of time, in seconds, for which the delivery of all messages in the queue
     * is delayed. Valid values: An integer from 0 to 900 seconds (15 minutes). Default: 0.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MaximumMessageSize</code> – The limit of how many bytes a message can contain before Amazon SQS rejects it.
     * Valid values: An integer from 1,024 bytes (1 KiB) to 262,144 bytes (256 KiB). Default: 262,144 (256 KiB).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MessageRetentionPeriod</code> – The length of time, in seconds, for which Amazon SQS retains a message.
     * Valid values: An integer from 60 seconds (1 minute) to 1,209,600 seconds (14 days). Default: 345,600 (4 days).
     * When you change a queue's attributes, the change can take up to 60 seconds for most of the attributes to
     * propagate throughout the Amazon SQS system. Changes made to the <code>MessageRetentionPeriod</code> attribute can
     * take up to 15 minutes and will impact existing messages in the queue potentially causing them to be expired and
     * deleted if the <code>MessageRetentionPeriod</code> is reduced below the age of existing messages.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Policy</code> – The queue's policy. A valid Amazon Web Services policy. For more information about policy
     * structure, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/PoliciesOverview.html">Overview of
     * Amazon Web Services IAM Policies</a> in the <i>IAM User Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ReceiveMessageWaitTimeSeconds</code> – The length of time, in seconds, for which a
     * <code> <a>ReceiveMessage</a> </code> action waits for a message to arrive. Valid values: An integer from 0 to 20
     * (seconds). Default: 0.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>VisibilityTimeout</code> – The visibility timeout for the queue, in seconds. Valid values: An integer from
     * 0 to 43,200 (12 hours). Default: 30. For more information about the visibility timeout, see <a
     * href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-visibility-timeout.html"
     * >Visibility Timeout</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * The following attributes apply only to <a
     * href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-dead-letter-queues.html"
     * >dead-letter queues:</a>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>RedrivePolicy</code> – The string that includes the parameters for the dead-letter queue functionality of
     * the source queue as a JSON object. The parameters are as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>deadLetterTargetArn</code> – The Amazon Resource Name (ARN) of the dead-letter queue to which Amazon SQS
     * moves messages after the value of <code>maxReceiveCount</code> is exceeded.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>maxReceiveCount</code> – The number of times a message is delivered to the source queue before being moved
     * to the dead-letter queue. Default: 10. When the <code>ReceiveCount</code> for a message exceeds the
     * <code>maxReceiveCount</code> for a queue, Amazon SQS moves the message to the dead-letter-queue.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>RedriveAllowPolicy</code> – The string that includes the parameters for the permissions for the dead-letter
     * queue redrive permission and which source queues can specify dead-letter queues as a JSON object. The parameters
     * are as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>redrivePermission</code> – The permission type that defines which source queues can specify the current
     * queue as the dead-letter queue. Valid values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>allowAll</code> – (Default) Any source queues in this Amazon Web Services account in the same Region can
     * specify this queue as the dead-letter queue.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>denyAll</code> – No source queues can specify this queue as the dead-letter queue.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>byQueue</code> – Only queues specified by the <code>sourceQueueArns</code> parameter can specify this queue
     * as the dead-letter queue.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>sourceQueueArns</code> – The Amazon Resource Names (ARN)s of the source queues that can specify this queue
     * as the dead-letter queue and redrive messages. You can specify this parameter only when the
     * <code>redrivePermission</code> parameter is set to <code>byQueue</code>. You can specify up to 10 source queue
     * ARNs. To allow more than 10 source queues to specify dead-letter queues, set the <code>redrivePermission</code>
     * parameter to <code>allowAll</code>.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The dead-letter queue of a FIFO queue must also be a FIFO queue. Similarly, the dead-letter queue of a standard
     * queue must also be a standard queue.
     * </p>
     * </note>
     * <p>
     * The following attributes apply only to <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-server-side-encryption.html"
     * >server-side-encryption</a>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>KmsMasterKeyId</code> – The ID of an Amazon Web Services managed customer master key (CMK) for Amazon SQS
     * or a custom CMK. For more information, see <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-server-side-encryption.html#sqs-sse-key-terms"
     * >Key Terms</a>. While the alias of the Amazon Web Services managed CMK for Amazon SQS is always
     * <code>alias/aws/sqs</code>, the alias of a custom CMK can, for example, be <code>alias/<i>MyAlias</i> </code>.
     * For more examples, see <a href=
     * "https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeKey.html#API_DescribeKey_RequestParameters"
     * >KeyId</a> in the <i>Key Management Service API Reference</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KmsDataKeyReusePeriodSeconds</code> – The length of time, in seconds, for which Amazon SQS can reuse a <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#data-keys">data key</a> to encrypt or
     * decrypt messages before calling KMS again. An integer representing seconds, between 60 seconds (1 minute) and
     * 86,400 seconds (24 hours). Default: 300 (5 minutes). A shorter time period provides better security but results
     * in more calls to KMS which might incur charges after Free Tier. For more information, see <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-server-side-encryption.html#sqs-how-does-the-data-key-reuse-period-work"
     * >How Does the Data Key Reuse Period Work?</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SqsManagedSseEnabled</code> – Enables server-side queue encryption using SQS owned encryption keys. Only
     * one server-side encryption option is supported per queue (for example, <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-configure-sse-existing-queue.html"
     * >SSE-KMS</a> or <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-configure-sqs-sse-queue.html"
     * >SSE-SQS</a>).
     * </p>
     * </li>
     * </ul>
     * <p>
     * The following attributes apply only to <a
     * href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues.html">FIFO
     * (first-in-first-out) queues</a>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>FifoQueue</code> – Designates a queue as FIFO. Valid values are <code>true</code> and <code>false</code>.
     * If you don't specify the <code>FifoQueue</code> attribute, Amazon SQS creates a standard queue. You can provide
     * this attribute only during queue creation. You can't change it for an existing queue. When you set this
     * attribute, you must also provide the <code>MessageGroupId</code> for your messages explicitly.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues-understanding-logic.html"
     * >FIFO queue logic</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ContentBasedDeduplication</code> – Enables content-based deduplication. Valid values are <code>true</code>
     * and <code>false</code>. For more information, see <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues-exactly-once-processing.html"
     * >Exactly-once processing</a> in the <i>Amazon SQS Developer Guide</i>. Note the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Every message must have a unique <code>MessageDeduplicationId</code>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * You may provide a <code>MessageDeduplicationId</code> explicitly.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you aren't able to provide a <code>MessageDeduplicationId</code> and you enable
     * <code>ContentBasedDeduplication</code> for your queue, Amazon SQS uses a SHA-256 hash to generate the
     * <code>MessageDeduplicationId</code> using the body of the message (but not the attributes of the message).
     * </p>
     * </li>
     * <li>
     * <p>
     * If you don't provide a <code>MessageDeduplicationId</code> and the queue doesn't have
     * <code>ContentBasedDeduplication</code> set, the action fails with an error.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the queue has <code>ContentBasedDeduplication</code> set, your <code>MessageDeduplicationId</code> overrides
     * the generated one.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * When <code>ContentBasedDeduplication</code> is in effect, messages with identical content sent within the
     * deduplication interval are treated as duplicates and only one copy of the message is delivered.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you send one message with <code>ContentBasedDeduplication</code> enabled and then another message with a
     * <code>MessageDeduplicationId</code> that is the same as the one generated for the first
     * <code>MessageDeduplicationId</code>, the two messages are treated as duplicates and only one copy of the message
     * is delivered.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * The following attributes apply only to <a
     * href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/high-throughput-fifo.html">high
     * throughput for FIFO queues</a>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DeduplicationScope</code> – Specifies whether message deduplication occurs at the message group or queue
     * level. Valid values are <code>messageGroup</code> and <code>queue</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FifoThroughputLimit</code> – Specifies whether the FIFO queue throughput quota applies to the entire queue
     * or per message group. Valid values are <code>perQueue</code> and <code>perMessageGroupId</code>. The
     * <code>perMessageGroupId</code> value is allowed only when the value for <code>DeduplicationScope</code> is
     * <code>messageGroup</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * To enable high throughput for FIFO queues, do the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Set <code>DeduplicationScope</code> to <code>messageGroup</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Set <code>FifoThroughputLimit</code> to <code>perMessageGroupId</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you set these attributes to anything other than the values shown for enabling high throughput, normal
     * throughput is in effect and deduplication occurs as specified.
     * </p>
     * <p>
     * For information on throughput quotas, see <a
     * href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/quotas-messages.html">Quotas
     * related to messages</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAttributes} method.
     * </p>
     * 
     * @return A map of attributes with their corresponding values.</p>
     *         <p>
     *         The following lists the names, descriptions, and values of the special request parameters that the
     *         <code>CreateQueue</code> action uses:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DelaySeconds</code> – The length of time, in seconds, for which the delivery of all messages in the
     *         queue is delayed. Valid values: An integer from 0 to 900 seconds (15 minutes). Default: 0.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MaximumMessageSize</code> – The limit of how many bytes a message can contain before Amazon SQS
     *         rejects it. Valid values: An integer from 1,024 bytes (1 KiB) to 262,144 bytes (256 KiB). Default:
     *         262,144 (256 KiB).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MessageRetentionPeriod</code> – The length of time, in seconds, for which Amazon SQS retains a
     *         message. Valid values: An integer from 60 seconds (1 minute) to 1,209,600 seconds (14 days). Default:
     *         345,600 (4 days). When you change a queue's attributes, the change can take up to 60 seconds for most of
     *         the attributes to propagate throughout the Amazon SQS system. Changes made to the
     *         <code>MessageRetentionPeriod</code> attribute can take up to 15 minutes and will impact existing messages
     *         in the queue potentially causing them to be expired and deleted if the
     *         <code>MessageRetentionPeriod</code> is reduced below the age of existing messages.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Policy</code> – The queue's policy. A valid Amazon Web Services policy. For more information about
     *         policy structure, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/PoliciesOverview.html">Overview of Amazon Web
     *         Services IAM Policies</a> in the <i>IAM User Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ReceiveMessageWaitTimeSeconds</code> – The length of time, in seconds, for which a
     *         <code> <a>ReceiveMessage</a> </code> action waits for a message to arrive. Valid values: An integer from
     *         0 to 20 (seconds). Default: 0.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>VisibilityTimeout</code> – The visibility timeout for the queue, in seconds. Valid values: An
     *         integer from 0 to 43,200 (12 hours). Default: 30. For more information about the visibility timeout, see
     *         <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-visibility-timeout.html"
     *         >Visibility Timeout</a> in the <i>Amazon SQS Developer Guide</i>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The following attributes apply only to <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-dead-letter-queues.html"
     *         >dead-letter queues:</a>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>RedrivePolicy</code> – The string that includes the parameters for the dead-letter queue
     *         functionality of the source queue as a JSON object. The parameters are as follows:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>deadLetterTargetArn</code> – The Amazon Resource Name (ARN) of the dead-letter queue to which
     *         Amazon SQS moves messages after the value of <code>maxReceiveCount</code> is exceeded.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>maxReceiveCount</code> – The number of times a message is delivered to the source queue before
     *         being moved to the dead-letter queue. Default: 10. When the <code>ReceiveCount</code> for a message
     *         exceeds the <code>maxReceiveCount</code> for a queue, Amazon SQS moves the message to the
     *         dead-letter-queue.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RedriveAllowPolicy</code> – The string that includes the parameters for the permissions for the
     *         dead-letter queue redrive permission and which source queues can specify dead-letter queues as a JSON
     *         object. The parameters are as follows:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>redrivePermission</code> – The permission type that defines which source queues can specify the
     *         current queue as the dead-letter queue. Valid values are:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>allowAll</code> – (Default) Any source queues in this Amazon Web Services account in the same
     *         Region can specify this queue as the dead-letter queue.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>denyAll</code> – No source queues can specify this queue as the dead-letter queue.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>byQueue</code> – Only queues specified by the <code>sourceQueueArns</code> parameter can specify
     *         this queue as the dead-letter queue.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>sourceQueueArns</code> – The Amazon Resource Names (ARN)s of the source queues that can specify
     *         this queue as the dead-letter queue and redrive messages. You can specify this parameter only when the
     *         <code>redrivePermission</code> parameter is set to <code>byQueue</code>. You can specify up to 10 source
     *         queue ARNs. To allow more than 10 source queues to specify dead-letter queues, set the
     *         <code>redrivePermission</code> parameter to <code>allowAll</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         The dead-letter queue of a FIFO queue must also be a FIFO queue. Similarly, the dead-letter queue of a
     *         standard queue must also be a standard queue.
     *         </p>
     *         </note>
     *         <p>
     *         The following attributes apply only to <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-server-side-encryption.html"
     *         >server-side-encryption</a>:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>KmsMasterKeyId</code> – The ID of an Amazon Web Services managed customer master key (CMK) for
     *         Amazon SQS or a custom CMK. For more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-server-side-encryption.html#sqs-sse-key-terms"
     *         >Key Terms</a>. While the alias of the Amazon Web Services managed CMK for Amazon SQS is always
     *         <code>alias/aws/sqs</code>, the alias of a custom CMK can, for example, be
     *         <code>alias/<i>MyAlias</i> </code>. For more examples, see <a href=
     *         "https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeKey.html#API_DescribeKey_RequestParameters"
     *         >KeyId</a> in the <i>Key Management Service API Reference</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>KmsDataKeyReusePeriodSeconds</code> – The length of time, in seconds, for which Amazon SQS can
     *         reuse a <a href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#data-keys">data
     *         key</a> to encrypt or decrypt messages before calling KMS again. An integer representing seconds, between
     *         60 seconds (1 minute) and 86,400 seconds (24 hours). Default: 300 (5 minutes). A shorter time period
     *         provides better security but results in more calls to KMS which might incur charges after Free Tier. For
     *         more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-server-side-encryption.html#sqs-how-does-the-data-key-reuse-period-work"
     *         >How Does the Data Key Reuse Period Work?</a>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SqsManagedSseEnabled</code> – Enables server-side queue encryption using SQS owned encryption keys.
     *         Only one server-side encryption option is supported per queue (for example, <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-configure-sse-existing-queue.html"
     *         >SSE-KMS</a> or <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-configure-sqs-sse-queue.html"
     *         >SSE-SQS</a>).
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The following attributes apply only to <a
     *         href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues.html">FIFO
     *         (first-in-first-out) queues</a>:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>FifoQueue</code> – Designates a queue as FIFO. Valid values are <code>true</code> and
     *         <code>false</code>. If you don't specify the <code>FifoQueue</code> attribute, Amazon SQS creates a
     *         standard queue. You can provide this attribute only during queue creation. You can't change it for an
     *         existing queue. When you set this attribute, you must also provide the <code>MessageGroupId</code> for
     *         your messages explicitly.
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues-understanding-logic.html"
     *         >FIFO queue logic</a> in the <i>Amazon SQS Developer Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ContentBasedDeduplication</code> – Enables content-based deduplication. Valid values are
     *         <code>true</code> and <code>false</code>. For more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues-exactly-once-processing.html"
     *         >Exactly-once processing</a> in the <i>Amazon SQS Developer Guide</i>. Note the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Every message must have a unique <code>MessageDeduplicationId</code>.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You may provide a <code>MessageDeduplicationId</code> explicitly.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you aren't able to provide a <code>MessageDeduplicationId</code> and you enable
     *         <code>ContentBasedDeduplication</code> for your queue, Amazon SQS uses a SHA-256 hash to generate the
     *         <code>MessageDeduplicationId</code> using the body of the message (but not the attributes of the
     *         message).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you don't provide a <code>MessageDeduplicationId</code> and the queue doesn't have
     *         <code>ContentBasedDeduplication</code> set, the action fails with an error.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the queue has <code>ContentBasedDeduplication</code> set, your <code>MessageDeduplicationId</code>
     *         overrides the generated one.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         When <code>ContentBasedDeduplication</code> is in effect, messages with identical content sent within the
     *         deduplication interval are treated as duplicates and only one copy of the message is delivered.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you send one message with <code>ContentBasedDeduplication</code> enabled and then another message with
     *         a <code>MessageDeduplicationId</code> that is the same as the one generated for the first
     *         <code>MessageDeduplicationId</code>, the two messages are treated as duplicates and only one copy of the
     *         message is delivered.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         </ul>
     *         <p>
     *         The following attributes apply only to <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/high-throughput-fifo.html"
     *         >high throughput for FIFO queues</a>:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DeduplicationScope</code> – Specifies whether message deduplication occurs at the message group or
     *         queue level. Valid values are <code>messageGroup</code> and <code>queue</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FifoThroughputLimit</code> – Specifies whether the FIFO queue throughput quota applies to the
     *         entire queue or per message group. Valid values are <code>perQueue</code> and
     *         <code>perMessageGroupId</code>. The <code>perMessageGroupId</code> value is allowed only when the value
     *         for <code>DeduplicationScope</code> is <code>messageGroup</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         To enable high throughput for FIFO queues, do the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Set <code>DeduplicationScope</code> to <code>messageGroup</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Set <code>FifoThroughputLimit</code> to <code>perMessageGroupId</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you set these attributes to anything other than the values shown for enabling high throughput, normal
     *         throughput is in effect and deduplication occurs as specified.
     *         </p>
     *         <p>
     *         For information on throughput quotas, see <a
     *         href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/quotas-messages.html"
     *         >Quotas related to messages</a> in the <i>Amazon SQS Developer Guide</i>.
     */
    public final Map<QueueAttributeName, String> attributes() {
        return QueueAttributeMapCopier.copyStringToEnum(attributes);
    }

    /**
     * For responses, this returns true if the service returned a value for the Attributes property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAttributes() {
        return attributes != null && !(attributes instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map of attributes with their corresponding values.
     * </p>
     * <p>
     * The following lists the names, descriptions, and values of the special request parameters that the
     * <code>CreateQueue</code> action uses:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DelaySeconds</code> – The length of time, in seconds, for which the delivery of all messages in the queue
     * is delayed. Valid values: An integer from 0 to 900 seconds (15 minutes). Default: 0.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MaximumMessageSize</code> – The limit of how many bytes a message can contain before Amazon SQS rejects it.
     * Valid values: An integer from 1,024 bytes (1 KiB) to 262,144 bytes (256 KiB). Default: 262,144 (256 KiB).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MessageRetentionPeriod</code> – The length of time, in seconds, for which Amazon SQS retains a message.
     * Valid values: An integer from 60 seconds (1 minute) to 1,209,600 seconds (14 days). Default: 345,600 (4 days).
     * When you change a queue's attributes, the change can take up to 60 seconds for most of the attributes to
     * propagate throughout the Amazon SQS system. Changes made to the <code>MessageRetentionPeriod</code> attribute can
     * take up to 15 minutes and will impact existing messages in the queue potentially causing them to be expired and
     * deleted if the <code>MessageRetentionPeriod</code> is reduced below the age of existing messages.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Policy</code> – The queue's policy. A valid Amazon Web Services policy. For more information about policy
     * structure, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/PoliciesOverview.html">Overview of
     * Amazon Web Services IAM Policies</a> in the <i>IAM User Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ReceiveMessageWaitTimeSeconds</code> – The length of time, in seconds, for which a
     * <code> <a>ReceiveMessage</a> </code> action waits for a message to arrive. Valid values: An integer from 0 to 20
     * (seconds). Default: 0.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>VisibilityTimeout</code> – The visibility timeout for the queue, in seconds. Valid values: An integer from
     * 0 to 43,200 (12 hours). Default: 30. For more information about the visibility timeout, see <a
     * href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-visibility-timeout.html"
     * >Visibility Timeout</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * The following attributes apply only to <a
     * href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-dead-letter-queues.html"
     * >dead-letter queues:</a>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>RedrivePolicy</code> – The string that includes the parameters for the dead-letter queue functionality of
     * the source queue as a JSON object. The parameters are as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>deadLetterTargetArn</code> – The Amazon Resource Name (ARN) of the dead-letter queue to which Amazon SQS
     * moves messages after the value of <code>maxReceiveCount</code> is exceeded.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>maxReceiveCount</code> – The number of times a message is delivered to the source queue before being moved
     * to the dead-letter queue. Default: 10. When the <code>ReceiveCount</code> for a message exceeds the
     * <code>maxReceiveCount</code> for a queue, Amazon SQS moves the message to the dead-letter-queue.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>RedriveAllowPolicy</code> – The string that includes the parameters for the permissions for the dead-letter
     * queue redrive permission and which source queues can specify dead-letter queues as a JSON object. The parameters
     * are as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>redrivePermission</code> – The permission type that defines which source queues can specify the current
     * queue as the dead-letter queue. Valid values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>allowAll</code> – (Default) Any source queues in this Amazon Web Services account in the same Region can
     * specify this queue as the dead-letter queue.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>denyAll</code> – No source queues can specify this queue as the dead-letter queue.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>byQueue</code> – Only queues specified by the <code>sourceQueueArns</code> parameter can specify this queue
     * as the dead-letter queue.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>sourceQueueArns</code> – The Amazon Resource Names (ARN)s of the source queues that can specify this queue
     * as the dead-letter queue and redrive messages. You can specify this parameter only when the
     * <code>redrivePermission</code> parameter is set to <code>byQueue</code>. You can specify up to 10 source queue
     * ARNs. To allow more than 10 source queues to specify dead-letter queues, set the <code>redrivePermission</code>
     * parameter to <code>allowAll</code>.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The dead-letter queue of a FIFO queue must also be a FIFO queue. Similarly, the dead-letter queue of a standard
     * queue must also be a standard queue.
     * </p>
     * </note>
     * <p>
     * The following attributes apply only to <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-server-side-encryption.html"
     * >server-side-encryption</a>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>KmsMasterKeyId</code> – The ID of an Amazon Web Services managed customer master key (CMK) for Amazon SQS
     * or a custom CMK. For more information, see <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-server-side-encryption.html#sqs-sse-key-terms"
     * >Key Terms</a>. While the alias of the Amazon Web Services managed CMK for Amazon SQS is always
     * <code>alias/aws/sqs</code>, the alias of a custom CMK can, for example, be <code>alias/<i>MyAlias</i> </code>.
     * For more examples, see <a href=
     * "https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeKey.html#API_DescribeKey_RequestParameters"
     * >KeyId</a> in the <i>Key Management Service API Reference</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KmsDataKeyReusePeriodSeconds</code> – The length of time, in seconds, for which Amazon SQS can reuse a <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#data-keys">data key</a> to encrypt or
     * decrypt messages before calling KMS again. An integer representing seconds, between 60 seconds (1 minute) and
     * 86,400 seconds (24 hours). Default: 300 (5 minutes). A shorter time period provides better security but results
     * in more calls to KMS which might incur charges after Free Tier. For more information, see <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-server-side-encryption.html#sqs-how-does-the-data-key-reuse-period-work"
     * >How Does the Data Key Reuse Period Work?</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SqsManagedSseEnabled</code> – Enables server-side queue encryption using SQS owned encryption keys. Only
     * one server-side encryption option is supported per queue (for example, <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-configure-sse-existing-queue.html"
     * >SSE-KMS</a> or <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-configure-sqs-sse-queue.html"
     * >SSE-SQS</a>).
     * </p>
     * </li>
     * </ul>
     * <p>
     * The following attributes apply only to <a
     * href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues.html">FIFO
     * (first-in-first-out) queues</a>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>FifoQueue</code> – Designates a queue as FIFO. Valid values are <code>true</code> and <code>false</code>.
     * If you don't specify the <code>FifoQueue</code> attribute, Amazon SQS creates a standard queue. You can provide
     * this attribute only during queue creation. You can't change it for an existing queue. When you set this
     * attribute, you must also provide the <code>MessageGroupId</code> for your messages explicitly.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues-understanding-logic.html"
     * >FIFO queue logic</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ContentBasedDeduplication</code> – Enables content-based deduplication. Valid values are <code>true</code>
     * and <code>false</code>. For more information, see <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues-exactly-once-processing.html"
     * >Exactly-once processing</a> in the <i>Amazon SQS Developer Guide</i>. Note the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Every message must have a unique <code>MessageDeduplicationId</code>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * You may provide a <code>MessageDeduplicationId</code> explicitly.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you aren't able to provide a <code>MessageDeduplicationId</code> and you enable
     * <code>ContentBasedDeduplication</code> for your queue, Amazon SQS uses a SHA-256 hash to generate the
     * <code>MessageDeduplicationId</code> using the body of the message (but not the attributes of the message).
     * </p>
     * </li>
     * <li>
     * <p>
     * If you don't provide a <code>MessageDeduplicationId</code> and the queue doesn't have
     * <code>ContentBasedDeduplication</code> set, the action fails with an error.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the queue has <code>ContentBasedDeduplication</code> set, your <code>MessageDeduplicationId</code> overrides
     * the generated one.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * When <code>ContentBasedDeduplication</code> is in effect, messages with identical content sent within the
     * deduplication interval are treated as duplicates and only one copy of the message is delivered.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you send one message with <code>ContentBasedDeduplication</code> enabled and then another message with a
     * <code>MessageDeduplicationId</code> that is the same as the one generated for the first
     * <code>MessageDeduplicationId</code>, the two messages are treated as duplicates and only one copy of the message
     * is delivered.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * The following attributes apply only to <a
     * href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/high-throughput-fifo.html">high
     * throughput for FIFO queues</a>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DeduplicationScope</code> – Specifies whether message deduplication occurs at the message group or queue
     * level. Valid values are <code>messageGroup</code> and <code>queue</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FifoThroughputLimit</code> – Specifies whether the FIFO queue throughput quota applies to the entire queue
     * or per message group. Valid values are <code>perQueue</code> and <code>perMessageGroupId</code>. The
     * <code>perMessageGroupId</code> value is allowed only when the value for <code>DeduplicationScope</code> is
     * <code>messageGroup</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * To enable high throughput for FIFO queues, do the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Set <code>DeduplicationScope</code> to <code>messageGroup</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Set <code>FifoThroughputLimit</code> to <code>perMessageGroupId</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you set these attributes to anything other than the values shown for enabling high throughput, normal
     * throughput is in effect and deduplication occurs as specified.
     * </p>
     * <p>
     * For information on throughput quotas, see <a
     * href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/quotas-messages.html">Quotas
     * related to messages</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAttributes} method.
     * </p>
     * 
     * @return A map of attributes with their corresponding values.</p>
     *         <p>
     *         The following lists the names, descriptions, and values of the special request parameters that the
     *         <code>CreateQueue</code> action uses:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DelaySeconds</code> – The length of time, in seconds, for which the delivery of all messages in the
     *         queue is delayed. Valid values: An integer from 0 to 900 seconds (15 minutes). Default: 0.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MaximumMessageSize</code> – The limit of how many bytes a message can contain before Amazon SQS
     *         rejects it. Valid values: An integer from 1,024 bytes (1 KiB) to 262,144 bytes (256 KiB). Default:
     *         262,144 (256 KiB).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MessageRetentionPeriod</code> – The length of time, in seconds, for which Amazon SQS retains a
     *         message. Valid values: An integer from 60 seconds (1 minute) to 1,209,600 seconds (14 days). Default:
     *         345,600 (4 days). When you change a queue's attributes, the change can take up to 60 seconds for most of
     *         the attributes to propagate throughout the Amazon SQS system. Changes made to the
     *         <code>MessageRetentionPeriod</code> attribute can take up to 15 minutes and will impact existing messages
     *         in the queue potentially causing them to be expired and deleted if the
     *         <code>MessageRetentionPeriod</code> is reduced below the age of existing messages.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Policy</code> – The queue's policy. A valid Amazon Web Services policy. For more information about
     *         policy structure, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/PoliciesOverview.html">Overview of Amazon Web
     *         Services IAM Policies</a> in the <i>IAM User Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ReceiveMessageWaitTimeSeconds</code> – The length of time, in seconds, for which a
     *         <code> <a>ReceiveMessage</a> </code> action waits for a message to arrive. Valid values: An integer from
     *         0 to 20 (seconds). Default: 0.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>VisibilityTimeout</code> – The visibility timeout for the queue, in seconds. Valid values: An
     *         integer from 0 to 43,200 (12 hours). Default: 30. For more information about the visibility timeout, see
     *         <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-visibility-timeout.html"
     *         >Visibility Timeout</a> in the <i>Amazon SQS Developer Guide</i>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The following attributes apply only to <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-dead-letter-queues.html"
     *         >dead-letter queues:</a>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>RedrivePolicy</code> – The string that includes the parameters for the dead-letter queue
     *         functionality of the source queue as a JSON object. The parameters are as follows:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>deadLetterTargetArn</code> – The Amazon Resource Name (ARN) of the dead-letter queue to which
     *         Amazon SQS moves messages after the value of <code>maxReceiveCount</code> is exceeded.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>maxReceiveCount</code> – The number of times a message is delivered to the source queue before
     *         being moved to the dead-letter queue. Default: 10. When the <code>ReceiveCount</code> for a message
     *         exceeds the <code>maxReceiveCount</code> for a queue, Amazon SQS moves the message to the
     *         dead-letter-queue.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RedriveAllowPolicy</code> – The string that includes the parameters for the permissions for the
     *         dead-letter queue redrive permission and which source queues can specify dead-letter queues as a JSON
     *         object. The parameters are as follows:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>redrivePermission</code> – The permission type that defines which source queues can specify the
     *         current queue as the dead-letter queue. Valid values are:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>allowAll</code> – (Default) Any source queues in this Amazon Web Services account in the same
     *         Region can specify this queue as the dead-letter queue.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>denyAll</code> – No source queues can specify this queue as the dead-letter queue.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>byQueue</code> – Only queues specified by the <code>sourceQueueArns</code> parameter can specify
     *         this queue as the dead-letter queue.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>sourceQueueArns</code> – The Amazon Resource Names (ARN)s of the source queues that can specify
     *         this queue as the dead-letter queue and redrive messages. You can specify this parameter only when the
     *         <code>redrivePermission</code> parameter is set to <code>byQueue</code>. You can specify up to 10 source
     *         queue ARNs. To allow more than 10 source queues to specify dead-letter queues, set the
     *         <code>redrivePermission</code> parameter to <code>allowAll</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         The dead-letter queue of a FIFO queue must also be a FIFO queue. Similarly, the dead-letter queue of a
     *         standard queue must also be a standard queue.
     *         </p>
     *         </note>
     *         <p>
     *         The following attributes apply only to <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-server-side-encryption.html"
     *         >server-side-encryption</a>:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>KmsMasterKeyId</code> – The ID of an Amazon Web Services managed customer master key (CMK) for
     *         Amazon SQS or a custom CMK. For more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-server-side-encryption.html#sqs-sse-key-terms"
     *         >Key Terms</a>. While the alias of the Amazon Web Services managed CMK for Amazon SQS is always
     *         <code>alias/aws/sqs</code>, the alias of a custom CMK can, for example, be
     *         <code>alias/<i>MyAlias</i> </code>. For more examples, see <a href=
     *         "https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeKey.html#API_DescribeKey_RequestParameters"
     *         >KeyId</a> in the <i>Key Management Service API Reference</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>KmsDataKeyReusePeriodSeconds</code> – The length of time, in seconds, for which Amazon SQS can
     *         reuse a <a href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#data-keys">data
     *         key</a> to encrypt or decrypt messages before calling KMS again. An integer representing seconds, between
     *         60 seconds (1 minute) and 86,400 seconds (24 hours). Default: 300 (5 minutes). A shorter time period
     *         provides better security but results in more calls to KMS which might incur charges after Free Tier. For
     *         more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-server-side-encryption.html#sqs-how-does-the-data-key-reuse-period-work"
     *         >How Does the Data Key Reuse Period Work?</a>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SqsManagedSseEnabled</code> – Enables server-side queue encryption using SQS owned encryption keys.
     *         Only one server-side encryption option is supported per queue (for example, <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-configure-sse-existing-queue.html"
     *         >SSE-KMS</a> or <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-configure-sqs-sse-queue.html"
     *         >SSE-SQS</a>).
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The following attributes apply only to <a
     *         href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues.html">FIFO
     *         (first-in-first-out) queues</a>:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>FifoQueue</code> – Designates a queue as FIFO. Valid values are <code>true</code> and
     *         <code>false</code>. If you don't specify the <code>FifoQueue</code> attribute, Amazon SQS creates a
     *         standard queue. You can provide this attribute only during queue creation. You can't change it for an
     *         existing queue. When you set this attribute, you must also provide the <code>MessageGroupId</code> for
     *         your messages explicitly.
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues-understanding-logic.html"
     *         >FIFO queue logic</a> in the <i>Amazon SQS Developer Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ContentBasedDeduplication</code> – Enables content-based deduplication. Valid values are
     *         <code>true</code> and <code>false</code>. For more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues-exactly-once-processing.html"
     *         >Exactly-once processing</a> in the <i>Amazon SQS Developer Guide</i>. Note the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Every message must have a unique <code>MessageDeduplicationId</code>.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You may provide a <code>MessageDeduplicationId</code> explicitly.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you aren't able to provide a <code>MessageDeduplicationId</code> and you enable
     *         <code>ContentBasedDeduplication</code> for your queue, Amazon SQS uses a SHA-256 hash to generate the
     *         <code>MessageDeduplicationId</code> using the body of the message (but not the attributes of the
     *         message).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you don't provide a <code>MessageDeduplicationId</code> and the queue doesn't have
     *         <code>ContentBasedDeduplication</code> set, the action fails with an error.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the queue has <code>ContentBasedDeduplication</code> set, your <code>MessageDeduplicationId</code>
     *         overrides the generated one.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         When <code>ContentBasedDeduplication</code> is in effect, messages with identical content sent within the
     *         deduplication interval are treated as duplicates and only one copy of the message is delivered.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you send one message with <code>ContentBasedDeduplication</code> enabled and then another message with
     *         a <code>MessageDeduplicationId</code> that is the same as the one generated for the first
     *         <code>MessageDeduplicationId</code>, the two messages are treated as duplicates and only one copy of the
     *         message is delivered.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         </ul>
     *         <p>
     *         The following attributes apply only to <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/high-throughput-fifo.html"
     *         >high throughput for FIFO queues</a>:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DeduplicationScope</code> – Specifies whether message deduplication occurs at the message group or
     *         queue level. Valid values are <code>messageGroup</code> and <code>queue</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FifoThroughputLimit</code> – Specifies whether the FIFO queue throughput quota applies to the
     *         entire queue or per message group. Valid values are <code>perQueue</code> and
     *         <code>perMessageGroupId</code>. The <code>perMessageGroupId</code> value is allowed only when the value
     *         for <code>DeduplicationScope</code> is <code>messageGroup</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         To enable high throughput for FIFO queues, do the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Set <code>DeduplicationScope</code> to <code>messageGroup</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Set <code>FifoThroughputLimit</code> to <code>perMessageGroupId</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you set these attributes to anything other than the values shown for enabling high throughput, normal
     *         throughput is in effect and deduplication occurs as specified.
     *         </p>
     *         <p>
     *         For information on throughput quotas, see <a
     *         href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/quotas-messages.html"
     *         >Quotas related to messages</a> in the <i>Amazon SQS Developer Guide</i>.
     */
    public final Map<String, String> attributesAsStrings() {
        return attributes;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Add cost allocation tags to the specified Amazon SQS queue. For an overview, see <a
     * href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-queue-tags.html">Tagging
     * Your Amazon SQS Queues</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * <p>
     * When you use queue tags, keep the following guidelines in mind:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Adding more than 50 tags to a queue isn't recommended.
     * </p>
     * </li>
     * <li>
     * <p>
     * Tags don't have any semantic meaning. Amazon SQS interprets tags as character strings.
     * </p>
     * </li>
     * <li>
     * <p>
     * Tags are case-sensitive.
     * </p>
     * </li>
     * <li>
     * <p>
     * A new tag with a key identical to that of an existing tag overwrites the existing tag.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For a full list of tag restrictions, see <a
     * href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-limits.html#limits-queues"
     * >Quotas related to queues</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * <note>
     * <p>
     * To be able to tag a queue on creation, you must have the <code>sqs:CreateQueue</code> and
     * <code>sqs:TagQueue</code> permissions.
     * </p>
     * <p>
     * Cross-account permissions don't apply to this action. For more information, see <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-customer-managed-policy-examples.html#grant-cross-account-permissions-to-role-and-user-name"
     * >Grant cross-account permissions to a role and a username</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Add cost allocation tags to the specified Amazon SQS queue. For an overview, see <a
     *         href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-queue-tags.html"
     *         >Tagging Your Amazon SQS Queues</a> in the <i>Amazon SQS Developer Guide</i>.</p>
     *         <p>
     *         When you use queue tags, keep the following guidelines in mind:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Adding more than 50 tags to a queue isn't recommended.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Tags don't have any semantic meaning. Amazon SQS interprets tags as character strings.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Tags are case-sensitive.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A new tag with a key identical to that of an existing tag overwrites the existing tag.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For a full list of tag restrictions, see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-limits.html#limits-queues"
     *         >Quotas related to queues</a> in the <i>Amazon SQS Developer Guide</i>.
     *         </p>
     *         <note>
     *         <p>
     *         To be able to tag a queue on creation, you must have the <code>sqs:CreateQueue</code> and
     *         <code>sqs:TagQueue</code> permissions.
     *         </p>
     *         <p>
     *         Cross-account permissions don't apply to this action. For more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-customer-managed-policy-examples.html#grant-cross-account-permissions-to-role-and-user-name"
     *         >Grant cross-account permissions to a role and a username</a> in the <i>Amazon SQS Developer Guide</i>.
     *         </p>
     */
    public final Map<String, String> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(queueName());
        hashCode = 31 * hashCode + Objects.hashCode(hasAttributes() ? attributesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateQueueRequest)) {
            return false;
        }
        CreateQueueRequest other = (CreateQueueRequest) obj;
        return Objects.equals(queueName(), other.queueName()) && hasAttributes() == other.hasAttributes()
                && Objects.equals(attributesAsStrings(), other.attributesAsStrings()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateQueueRequest").add("QueueName", queueName())
                .add("Attributes", hasAttributes() ? attributesAsStrings() : null).add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "QueueName":
            return Optional.ofNullable(clazz.cast(queueName()));
        case "Attributes":
            return Optional.ofNullable(clazz.cast(attributesAsStrings()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateQueueRequest, T> g) {
        return obj -> g.apply((CreateQueueRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SqsRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateQueueRequest> {
        /**
         * <p>
         * The name of the new queue. The following limits apply to this name:
         * </p>
         * <ul>
         * <li>
         * <p>
         * A queue name can have up to 80 characters.
         * </p>
         * </li>
         * <li>
         * <p>
         * Valid values: alphanumeric characters, hyphens (<code>-</code>), and underscores (<code>_</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * A FIFO queue name must end with the <code>.fifo</code> suffix.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Queue URLs and names are case-sensitive.
         * </p>
         * 
         * @param queueName
         *        The name of the new queue. The following limits apply to this name:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        A queue name can have up to 80 characters.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Valid values: alphanumeric characters, hyphens (<code>-</code>), and underscores (<code>_</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A FIFO queue name must end with the <code>.fifo</code> suffix.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Queue URLs and names are case-sensitive.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder queueName(String queueName);

        /**
         * <p>
         * A map of attributes with their corresponding values.
         * </p>
         * <p>
         * The following lists the names, descriptions, and values of the special request parameters that the
         * <code>CreateQueue</code> action uses:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DelaySeconds</code> – The length of time, in seconds, for which the delivery of all messages in the
         * queue is delayed. Valid values: An integer from 0 to 900 seconds (15 minutes). Default: 0.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MaximumMessageSize</code> – The limit of how many bytes a message can contain before Amazon SQS rejects
         * it. Valid values: An integer from 1,024 bytes (1 KiB) to 262,144 bytes (256 KiB). Default: 262,144 (256 KiB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MessageRetentionPeriod</code> – The length of time, in seconds, for which Amazon SQS retains a message.
         * Valid values: An integer from 60 seconds (1 minute) to 1,209,600 seconds (14 days). Default: 345,600 (4
         * days). When you change a queue's attributes, the change can take up to 60 seconds for most of the attributes
         * to propagate throughout the Amazon SQS system. Changes made to the <code>MessageRetentionPeriod</code>
         * attribute can take up to 15 minutes and will impact existing messages in the queue potentially causing them
         * to be expired and deleted if the <code>MessageRetentionPeriod</code> is reduced below the age of existing
         * messages.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Policy</code> – The queue's policy. A valid Amazon Web Services policy. For more information about
         * policy structure, see <a
         * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/PoliciesOverview.html">Overview of Amazon Web Services
         * IAM Policies</a> in the <i>IAM User Guide</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ReceiveMessageWaitTimeSeconds</code> – The length of time, in seconds, for which a
         * <code> <a>ReceiveMessage</a> </code> action waits for a message to arrive. Valid values: An integer from 0 to
         * 20 (seconds). Default: 0.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>VisibilityTimeout</code> – The visibility timeout for the queue, in seconds. Valid values: An integer
         * from 0 to 43,200 (12 hours). Default: 30. For more information about the visibility timeout, see <a href=
         * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-visibility-timeout.html"
         * >Visibility Timeout</a> in the <i>Amazon SQS Developer Guide</i>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * The following attributes apply only to <a href=
         * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-dead-letter-queues.html"
         * >dead-letter queues:</a>
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>RedrivePolicy</code> – The string that includes the parameters for the dead-letter queue functionality
         * of the source queue as a JSON object. The parameters are as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>deadLetterTargetArn</code> – The Amazon Resource Name (ARN) of the dead-letter queue to which Amazon
         * SQS moves messages after the value of <code>maxReceiveCount</code> is exceeded.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>maxReceiveCount</code> – The number of times a message is delivered to the source queue before being
         * moved to the dead-letter queue. Default: 10. When the <code>ReceiveCount</code> for a message exceeds the
         * <code>maxReceiveCount</code> for a queue, Amazon SQS moves the message to the dead-letter-queue.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>RedriveAllowPolicy</code> – The string that includes the parameters for the permissions for the
         * dead-letter queue redrive permission and which source queues can specify dead-letter queues as a JSON object.
         * The parameters are as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>redrivePermission</code> – The permission type that defines which source queues can specify the current
         * queue as the dead-letter queue. Valid values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>allowAll</code> – (Default) Any source queues in this Amazon Web Services account in the same Region
         * can specify this queue as the dead-letter queue.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>denyAll</code> – No source queues can specify this queue as the dead-letter queue.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>byQueue</code> – Only queues specified by the <code>sourceQueueArns</code> parameter can specify this
         * queue as the dead-letter queue.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>sourceQueueArns</code> – The Amazon Resource Names (ARN)s of the source queues that can specify this
         * queue as the dead-letter queue and redrive messages. You can specify this parameter only when the
         * <code>redrivePermission</code> parameter is set to <code>byQueue</code>. You can specify up to 10 source
         * queue ARNs. To allow more than 10 source queues to specify dead-letter queues, set the
         * <code>redrivePermission</code> parameter to <code>allowAll</code>.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The dead-letter queue of a FIFO queue must also be a FIFO queue. Similarly, the dead-letter queue of a
         * standard queue must also be a standard queue.
         * </p>
         * </note>
         * <p>
         * The following attributes apply only to <a href=
         * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-server-side-encryption.html"
         * >server-side-encryption</a>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>KmsMasterKeyId</code> – The ID of an Amazon Web Services managed customer master key (CMK) for Amazon
         * SQS or a custom CMK. For more information, see <a href=
         * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-server-side-encryption.html#sqs-sse-key-terms"
         * >Key Terms</a>. While the alias of the Amazon Web Services managed CMK for Amazon SQS is always
         * <code>alias/aws/sqs</code>, the alias of a custom CMK can, for example, be <code>alias/<i>MyAlias</i> </code>
         * . For more examples, see <a href=
         * "https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeKey.html#API_DescribeKey_RequestParameters"
         * >KeyId</a> in the <i>Key Management Service API Reference</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KmsDataKeyReusePeriodSeconds</code> – The length of time, in seconds, for which Amazon SQS can reuse a
         * <a href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#data-keys">data key</a> to
         * encrypt or decrypt messages before calling KMS again. An integer representing seconds, between 60 seconds (1
         * minute) and 86,400 seconds (24 hours). Default: 300 (5 minutes). A shorter time period provides better
         * security but results in more calls to KMS which might incur charges after Free Tier. For more information,
         * see <a href=
         * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-server-side-encryption.html#sqs-how-does-the-data-key-reuse-period-work"
         * >How Does the Data Key Reuse Period Work?</a>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SqsManagedSseEnabled</code> – Enables server-side queue encryption using SQS owned encryption keys.
         * Only one server-side encryption option is supported per queue (for example, <a href=
         * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-configure-sse-existing-queue.html"
         * >SSE-KMS</a> or <a href=
         * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-configure-sqs-sse-queue.html"
         * >SSE-SQS</a>).
         * </p>
         * </li>
         * </ul>
         * <p>
         * The following attributes apply only to <a
         * href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues.html">FIFO
         * (first-in-first-out) queues</a>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>FifoQueue</code> – Designates a queue as FIFO. Valid values are <code>true</code> and
         * <code>false</code>. If you don't specify the <code>FifoQueue</code> attribute, Amazon SQS creates a standard
         * queue. You can provide this attribute only during queue creation. You can't change it for an existing queue.
         * When you set this attribute, you must also provide the <code>MessageGroupId</code> for your messages
         * explicitly.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues-understanding-logic.html"
         * >FIFO queue logic</a> in the <i>Amazon SQS Developer Guide</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ContentBasedDeduplication</code> – Enables content-based deduplication. Valid values are
         * <code>true</code> and <code>false</code>. For more information, see <a href=
         * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues-exactly-once-processing.html"
         * >Exactly-once processing</a> in the <i>Amazon SQS Developer Guide</i>. Note the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Every message must have a unique <code>MessageDeduplicationId</code>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * You may provide a <code>MessageDeduplicationId</code> explicitly.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you aren't able to provide a <code>MessageDeduplicationId</code> and you enable
         * <code>ContentBasedDeduplication</code> for your queue, Amazon SQS uses a SHA-256 hash to generate the
         * <code>MessageDeduplicationId</code> using the body of the message (but not the attributes of the message).
         * </p>
         * </li>
         * <li>
         * <p>
         * If you don't provide a <code>MessageDeduplicationId</code> and the queue doesn't have
         * <code>ContentBasedDeduplication</code> set, the action fails with an error.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the queue has <code>ContentBasedDeduplication</code> set, your <code>MessageDeduplicationId</code>
         * overrides the generated one.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * When <code>ContentBasedDeduplication</code> is in effect, messages with identical content sent within the
         * deduplication interval are treated as duplicates and only one copy of the message is delivered.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you send one message with <code>ContentBasedDeduplication</code> enabled and then another message with a
         * <code>MessageDeduplicationId</code> that is the same as the one generated for the first
         * <code>MessageDeduplicationId</code>, the two messages are treated as duplicates and only one copy of the
         * message is delivered.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * <p>
         * The following attributes apply only to <a
         * href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/high-throughput-fifo.html"
         * >high throughput for FIFO queues</a>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DeduplicationScope</code> – Specifies whether message deduplication occurs at the message group or
         * queue level. Valid values are <code>messageGroup</code> and <code>queue</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FifoThroughputLimit</code> – Specifies whether the FIFO queue throughput quota applies to the entire
         * queue or per message group. Valid values are <code>perQueue</code> and <code>perMessageGroupId</code>. The
         * <code>perMessageGroupId</code> value is allowed only when the value for <code>DeduplicationScope</code> is
         * <code>messageGroup</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * To enable high throughput for FIFO queues, do the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Set <code>DeduplicationScope</code> to <code>messageGroup</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Set <code>FifoThroughputLimit</code> to <code>perMessageGroupId</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you set these attributes to anything other than the values shown for enabling high throughput, normal
         * throughput is in effect and deduplication occurs as specified.
         * </p>
         * <p>
         * For information on throughput quotas, see <a
         * href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/quotas-messages.html">Quotas
         * related to messages</a> in the <i>Amazon SQS Developer Guide</i>.
         * </p>
         * 
         * @param attributes
         *        A map of attributes with their corresponding values.</p>
         *        <p>
         *        The following lists the names, descriptions, and values of the special request parameters that the
         *        <code>CreateQueue</code> action uses:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DelaySeconds</code> – The length of time, in seconds, for which the delivery of all messages in
         *        the queue is delayed. Valid values: An integer from 0 to 900 seconds (15 minutes). Default: 0.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MaximumMessageSize</code> – The limit of how many bytes a message can contain before Amazon SQS
         *        rejects it. Valid values: An integer from 1,024 bytes (1 KiB) to 262,144 bytes (256 KiB). Default:
         *        262,144 (256 KiB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MessageRetentionPeriod</code> – The length of time, in seconds, for which Amazon SQS retains a
         *        message. Valid values: An integer from 60 seconds (1 minute) to 1,209,600 seconds (14 days). Default:
         *        345,600 (4 days). When you change a queue's attributes, the change can take up to 60 seconds for most
         *        of the attributes to propagate throughout the Amazon SQS system. Changes made to the
         *        <code>MessageRetentionPeriod</code> attribute can take up to 15 minutes and will impact existing
         *        messages in the queue potentially causing them to be expired and deleted if the
         *        <code>MessageRetentionPeriod</code> is reduced below the age of existing messages.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Policy</code> – The queue's policy. A valid Amazon Web Services policy. For more information
         *        about policy structure, see <a
         *        href="https://docs.aws.amazon.com/IAM/latest/UserGuide/PoliciesOverview.html">Overview of Amazon Web
         *        Services IAM Policies</a> in the <i>IAM User Guide</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ReceiveMessageWaitTimeSeconds</code> – The length of time, in seconds, for which a
         *        <code> <a>ReceiveMessage</a> </code> action waits for a message to arrive. Valid values: An integer
         *        from 0 to 20 (seconds). Default: 0.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>VisibilityTimeout</code> – The visibility timeout for the queue, in seconds. Valid values: An
         *        integer from 0 to 43,200 (12 hours). Default: 30. For more information about the visibility timeout,
         *        see <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-visibility-timeout.html"
         *        >Visibility Timeout</a> in the <i>Amazon SQS Developer Guide</i>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        The following attributes apply only to <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-dead-letter-queues.html"
         *        >dead-letter queues:</a>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>RedrivePolicy</code> – The string that includes the parameters for the dead-letter queue
         *        functionality of the source queue as a JSON object. The parameters are as follows:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>deadLetterTargetArn</code> – The Amazon Resource Name (ARN) of the dead-letter queue to which
         *        Amazon SQS moves messages after the value of <code>maxReceiveCount</code> is exceeded.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>maxReceiveCount</code> – The number of times a message is delivered to the source queue before
         *        being moved to the dead-letter queue. Default: 10. When the <code>ReceiveCount</code> for a message
         *        exceeds the <code>maxReceiveCount</code> for a queue, Amazon SQS moves the message to the
         *        dead-letter-queue.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RedriveAllowPolicy</code> – The string that includes the parameters for the permissions for the
         *        dead-letter queue redrive permission and which source queues can specify dead-letter queues as a JSON
         *        object. The parameters are as follows:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>redrivePermission</code> – The permission type that defines which source queues can specify the
         *        current queue as the dead-letter queue. Valid values are:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>allowAll</code> – (Default) Any source queues in this Amazon Web Services account in the same
         *        Region can specify this queue as the dead-letter queue.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>denyAll</code> – No source queues can specify this queue as the dead-letter queue.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>byQueue</code> – Only queues specified by the <code>sourceQueueArns</code> parameter can specify
         *        this queue as the dead-letter queue.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>sourceQueueArns</code> – The Amazon Resource Names (ARN)s of the source queues that can specify
         *        this queue as the dead-letter queue and redrive messages. You can specify this parameter only when the
         *        <code>redrivePermission</code> parameter is set to <code>byQueue</code>. You can specify up to 10
         *        source queue ARNs. To allow more than 10 source queues to specify dead-letter queues, set the
         *        <code>redrivePermission</code> parameter to <code>allowAll</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        The dead-letter queue of a FIFO queue must also be a FIFO queue. Similarly, the dead-letter queue of a
         *        standard queue must also be a standard queue.
         *        </p>
         *        </note>
         *        <p>
         *        The following attributes apply only to <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-server-side-encryption.html"
         *        >server-side-encryption</a>:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>KmsMasterKeyId</code> – The ID of an Amazon Web Services managed customer master key (CMK) for
         *        Amazon SQS or a custom CMK. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-server-side-encryption.html#sqs-sse-key-terms"
         *        >Key Terms</a>. While the alias of the Amazon Web Services managed CMK for Amazon SQS is always
         *        <code>alias/aws/sqs</code>, the alias of a custom CMK can, for example, be
         *        <code>alias/<i>MyAlias</i> </code>. For more examples, see <a href=
         *        "https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeKey.html#API_DescribeKey_RequestParameters"
         *        >KeyId</a> in the <i>Key Management Service API Reference</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KmsDataKeyReusePeriodSeconds</code> – The length of time, in seconds, for which Amazon SQS can
         *        reuse a <a href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#data-keys">data
         *        key</a> to encrypt or decrypt messages before calling KMS again. An integer representing seconds,
         *        between 60 seconds (1 minute) and 86,400 seconds (24 hours). Default: 300 (5 minutes). A shorter time
         *        period provides better security but results in more calls to KMS which might incur charges after Free
         *        Tier. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-server-side-encryption.html#sqs-how-does-the-data-key-reuse-period-work"
         *        >How Does the Data Key Reuse Period Work?</a>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SqsManagedSseEnabled</code> – Enables server-side queue encryption using SQS owned encryption
         *        keys. Only one server-side encryption option is supported per queue (for example, <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-configure-sse-existing-queue.html"
         *        >SSE-KMS</a> or <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-configure-sqs-sse-queue.html"
         *        >SSE-SQS</a>).
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        The following attributes apply only to <a
         *        href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues.html"
         *        >FIFO (first-in-first-out) queues</a>:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>FifoQueue</code> – Designates a queue as FIFO. Valid values are <code>true</code> and
         *        <code>false</code>. If you don't specify the <code>FifoQueue</code> attribute, Amazon SQS creates a
         *        standard queue. You can provide this attribute only during queue creation. You can't change it for an
         *        existing queue. When you set this attribute, you must also provide the <code>MessageGroupId</code> for
         *        your messages explicitly.
         *        </p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues-understanding-logic.html"
         *        >FIFO queue logic</a> in the <i>Amazon SQS Developer Guide</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ContentBasedDeduplication</code> – Enables content-based deduplication. Valid values are
         *        <code>true</code> and <code>false</code>. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues-exactly-once-processing.html"
         *        >Exactly-once processing</a> in the <i>Amazon SQS Developer Guide</i>. Note the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Every message must have a unique <code>MessageDeduplicationId</code>.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        You may provide a <code>MessageDeduplicationId</code> explicitly.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you aren't able to provide a <code>MessageDeduplicationId</code> and you enable
         *        <code>ContentBasedDeduplication</code> for your queue, Amazon SQS uses a SHA-256 hash to generate the
         *        <code>MessageDeduplicationId</code> using the body of the message (but not the attributes of the
         *        message).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you don't provide a <code>MessageDeduplicationId</code> and the queue doesn't have
         *        <code>ContentBasedDeduplication</code> set, the action fails with an error.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the queue has <code>ContentBasedDeduplication</code> set, your <code>MessageDeduplicationId</code>
         *        overrides the generated one.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        When <code>ContentBasedDeduplication</code> is in effect, messages with identical content sent within
         *        the deduplication interval are treated as duplicates and only one copy of the message is delivered.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you send one message with <code>ContentBasedDeduplication</code> enabled and then another message
         *        with a <code>MessageDeduplicationId</code> that is the same as the one generated for the first
         *        <code>MessageDeduplicationId</code>, the two messages are treated as duplicates and only one copy of
         *        the message is delivered.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        </ul>
         *        <p>
         *        The following attributes apply only to <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/high-throughput-fifo.html"
         *        >high throughput for FIFO queues</a>:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DeduplicationScope</code> – Specifies whether message deduplication occurs at the message group
         *        or queue level. Valid values are <code>messageGroup</code> and <code>queue</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FifoThroughputLimit</code> – Specifies whether the FIFO queue throughput quota applies to the
         *        entire queue or per message group. Valid values are <code>perQueue</code> and
         *        <code>perMessageGroupId</code>. The <code>perMessageGroupId</code> value is allowed only when the
         *        value for <code>DeduplicationScope</code> is <code>messageGroup</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        To enable high throughput for FIFO queues, do the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Set <code>DeduplicationScope</code> to <code>messageGroup</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Set <code>FifoThroughputLimit</code> to <code>perMessageGroupId</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you set these attributes to anything other than the values shown for enabling high throughput,
         *        normal throughput is in effect and deduplication occurs as specified.
         *        </p>
         *        <p>
         *        For information on throughput quotas, see <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/quotas-messages.html"
         *        >Quotas related to messages</a> in the <i>Amazon SQS Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributesWithStrings(Map<String, String> attributes);

        /**
         * <p>
         * A map of attributes with their corresponding values.
         * </p>
         * <p>
         * The following lists the names, descriptions, and values of the special request parameters that the
         * <code>CreateQueue</code> action uses:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DelaySeconds</code> – The length of time, in seconds, for which the delivery of all messages in the
         * queue is delayed. Valid values: An integer from 0 to 900 seconds (15 minutes). Default: 0.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MaximumMessageSize</code> – The limit of how many bytes a message can contain before Amazon SQS rejects
         * it. Valid values: An integer from 1,024 bytes (1 KiB) to 262,144 bytes (256 KiB). Default: 262,144 (256 KiB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MessageRetentionPeriod</code> – The length of time, in seconds, for which Amazon SQS retains a message.
         * Valid values: An integer from 60 seconds (1 minute) to 1,209,600 seconds (14 days). Default: 345,600 (4
         * days). When you change a queue's attributes, the change can take up to 60 seconds for most of the attributes
         * to propagate throughout the Amazon SQS system. Changes made to the <code>MessageRetentionPeriod</code>
         * attribute can take up to 15 minutes and will impact existing messages in the queue potentially causing them
         * to be expired and deleted if the <code>MessageRetentionPeriod</code> is reduced below the age of existing
         * messages.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Policy</code> – The queue's policy. A valid Amazon Web Services policy. For more information about
         * policy structure, see <a
         * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/PoliciesOverview.html">Overview of Amazon Web Services
         * IAM Policies</a> in the <i>IAM User Guide</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ReceiveMessageWaitTimeSeconds</code> – The length of time, in seconds, for which a
         * <code> <a>ReceiveMessage</a> </code> action waits for a message to arrive. Valid values: An integer from 0 to
         * 20 (seconds). Default: 0.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>VisibilityTimeout</code> – The visibility timeout for the queue, in seconds. Valid values: An integer
         * from 0 to 43,200 (12 hours). Default: 30. For more information about the visibility timeout, see <a href=
         * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-visibility-timeout.html"
         * >Visibility Timeout</a> in the <i>Amazon SQS Developer Guide</i>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * The following attributes apply only to <a href=
         * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-dead-letter-queues.html"
         * >dead-letter queues:</a>
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>RedrivePolicy</code> – The string that includes the parameters for the dead-letter queue functionality
         * of the source queue as a JSON object. The parameters are as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>deadLetterTargetArn</code> – The Amazon Resource Name (ARN) of the dead-letter queue to which Amazon
         * SQS moves messages after the value of <code>maxReceiveCount</code> is exceeded.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>maxReceiveCount</code> – The number of times a message is delivered to the source queue before being
         * moved to the dead-letter queue. Default: 10. When the <code>ReceiveCount</code> for a message exceeds the
         * <code>maxReceiveCount</code> for a queue, Amazon SQS moves the message to the dead-letter-queue.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>RedriveAllowPolicy</code> – The string that includes the parameters for the permissions for the
         * dead-letter queue redrive permission and which source queues can specify dead-letter queues as a JSON object.
         * The parameters are as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>redrivePermission</code> – The permission type that defines which source queues can specify the current
         * queue as the dead-letter queue. Valid values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>allowAll</code> – (Default) Any source queues in this Amazon Web Services account in the same Region
         * can specify this queue as the dead-letter queue.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>denyAll</code> – No source queues can specify this queue as the dead-letter queue.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>byQueue</code> – Only queues specified by the <code>sourceQueueArns</code> parameter can specify this
         * queue as the dead-letter queue.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>sourceQueueArns</code> – The Amazon Resource Names (ARN)s of the source queues that can specify this
         * queue as the dead-letter queue and redrive messages. You can specify this parameter only when the
         * <code>redrivePermission</code> parameter is set to <code>byQueue</code>. You can specify up to 10 source
         * queue ARNs. To allow more than 10 source queues to specify dead-letter queues, set the
         * <code>redrivePermission</code> parameter to <code>allowAll</code>.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The dead-letter queue of a FIFO queue must also be a FIFO queue. Similarly, the dead-letter queue of a
         * standard queue must also be a standard queue.
         * </p>
         * </note>
         * <p>
         * The following attributes apply only to <a href=
         * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-server-side-encryption.html"
         * >server-side-encryption</a>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>KmsMasterKeyId</code> – The ID of an Amazon Web Services managed customer master key (CMK) for Amazon
         * SQS or a custom CMK. For more information, see <a href=
         * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-server-side-encryption.html#sqs-sse-key-terms"
         * >Key Terms</a>. While the alias of the Amazon Web Services managed CMK for Amazon SQS is always
         * <code>alias/aws/sqs</code>, the alias of a custom CMK can, for example, be <code>alias/<i>MyAlias</i> </code>
         * . For more examples, see <a href=
         * "https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeKey.html#API_DescribeKey_RequestParameters"
         * >KeyId</a> in the <i>Key Management Service API Reference</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KmsDataKeyReusePeriodSeconds</code> – The length of time, in seconds, for which Amazon SQS can reuse a
         * <a href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#data-keys">data key</a> to
         * encrypt or decrypt messages before calling KMS again. An integer representing seconds, between 60 seconds (1
         * minute) and 86,400 seconds (24 hours). Default: 300 (5 minutes). A shorter time period provides better
         * security but results in more calls to KMS which might incur charges after Free Tier. For more information,
         * see <a href=
         * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-server-side-encryption.html#sqs-how-does-the-data-key-reuse-period-work"
         * >How Does the Data Key Reuse Period Work?</a>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SqsManagedSseEnabled</code> – Enables server-side queue encryption using SQS owned encryption keys.
         * Only one server-side encryption option is supported per queue (for example, <a href=
         * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-configure-sse-existing-queue.html"
         * >SSE-KMS</a> or <a href=
         * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-configure-sqs-sse-queue.html"
         * >SSE-SQS</a>).
         * </p>
         * </li>
         * </ul>
         * <p>
         * The following attributes apply only to <a
         * href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues.html">FIFO
         * (first-in-first-out) queues</a>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>FifoQueue</code> – Designates a queue as FIFO. Valid values are <code>true</code> and
         * <code>false</code>. If you don't specify the <code>FifoQueue</code> attribute, Amazon SQS creates a standard
         * queue. You can provide this attribute only during queue creation. You can't change it for an existing queue.
         * When you set this attribute, you must also provide the <code>MessageGroupId</code> for your messages
         * explicitly.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues-understanding-logic.html"
         * >FIFO queue logic</a> in the <i>Amazon SQS Developer Guide</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ContentBasedDeduplication</code> – Enables content-based deduplication. Valid values are
         * <code>true</code> and <code>false</code>. For more information, see <a href=
         * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues-exactly-once-processing.html"
         * >Exactly-once processing</a> in the <i>Amazon SQS Developer Guide</i>. Note the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Every message must have a unique <code>MessageDeduplicationId</code>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * You may provide a <code>MessageDeduplicationId</code> explicitly.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you aren't able to provide a <code>MessageDeduplicationId</code> and you enable
         * <code>ContentBasedDeduplication</code> for your queue, Amazon SQS uses a SHA-256 hash to generate the
         * <code>MessageDeduplicationId</code> using the body of the message (but not the attributes of the message).
         * </p>
         * </li>
         * <li>
         * <p>
         * If you don't provide a <code>MessageDeduplicationId</code> and the queue doesn't have
         * <code>ContentBasedDeduplication</code> set, the action fails with an error.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the queue has <code>ContentBasedDeduplication</code> set, your <code>MessageDeduplicationId</code>
         * overrides the generated one.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * When <code>ContentBasedDeduplication</code> is in effect, messages with identical content sent within the
         * deduplication interval are treated as duplicates and only one copy of the message is delivered.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you send one message with <code>ContentBasedDeduplication</code> enabled and then another message with a
         * <code>MessageDeduplicationId</code> that is the same as the one generated for the first
         * <code>MessageDeduplicationId</code>, the two messages are treated as duplicates and only one copy of the
         * message is delivered.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * <p>
         * The following attributes apply only to <a
         * href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/high-throughput-fifo.html"
         * >high throughput for FIFO queues</a>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DeduplicationScope</code> – Specifies whether message deduplication occurs at the message group or
         * queue level. Valid values are <code>messageGroup</code> and <code>queue</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FifoThroughputLimit</code> – Specifies whether the FIFO queue throughput quota applies to the entire
         * queue or per message group. Valid values are <code>perQueue</code> and <code>perMessageGroupId</code>. The
         * <code>perMessageGroupId</code> value is allowed only when the value for <code>DeduplicationScope</code> is
         * <code>messageGroup</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * To enable high throughput for FIFO queues, do the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Set <code>DeduplicationScope</code> to <code>messageGroup</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Set <code>FifoThroughputLimit</code> to <code>perMessageGroupId</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you set these attributes to anything other than the values shown for enabling high throughput, normal
         * throughput is in effect and deduplication occurs as specified.
         * </p>
         * <p>
         * For information on throughput quotas, see <a
         * href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/quotas-messages.html">Quotas
         * related to messages</a> in the <i>Amazon SQS Developer Guide</i>.
         * </p>
         * 
         * @param attributes
         *        A map of attributes with their corresponding values.</p>
         *        <p>
         *        The following lists the names, descriptions, and values of the special request parameters that the
         *        <code>CreateQueue</code> action uses:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DelaySeconds</code> – The length of time, in seconds, for which the delivery of all messages in
         *        the queue is delayed. Valid values: An integer from 0 to 900 seconds (15 minutes). Default: 0.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MaximumMessageSize</code> – The limit of how many bytes a message can contain before Amazon SQS
         *        rejects it. Valid values: An integer from 1,024 bytes (1 KiB) to 262,144 bytes (256 KiB). Default:
         *        262,144 (256 KiB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MessageRetentionPeriod</code> – The length of time, in seconds, for which Amazon SQS retains a
         *        message. Valid values: An integer from 60 seconds (1 minute) to 1,209,600 seconds (14 days). Default:
         *        345,600 (4 days). When you change a queue's attributes, the change can take up to 60 seconds for most
         *        of the attributes to propagate throughout the Amazon SQS system. Changes made to the
         *        <code>MessageRetentionPeriod</code> attribute can take up to 15 minutes and will impact existing
         *        messages in the queue potentially causing them to be expired and deleted if the
         *        <code>MessageRetentionPeriod</code> is reduced below the age of existing messages.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Policy</code> – The queue's policy. A valid Amazon Web Services policy. For more information
         *        about policy structure, see <a
         *        href="https://docs.aws.amazon.com/IAM/latest/UserGuide/PoliciesOverview.html">Overview of Amazon Web
         *        Services IAM Policies</a> in the <i>IAM User Guide</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ReceiveMessageWaitTimeSeconds</code> – The length of time, in seconds, for which a
         *        <code> <a>ReceiveMessage</a> </code> action waits for a message to arrive. Valid values: An integer
         *        from 0 to 20 (seconds). Default: 0.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>VisibilityTimeout</code> – The visibility timeout for the queue, in seconds. Valid values: An
         *        integer from 0 to 43,200 (12 hours). Default: 30. For more information about the visibility timeout,
         *        see <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-visibility-timeout.html"
         *        >Visibility Timeout</a> in the <i>Amazon SQS Developer Guide</i>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        The following attributes apply only to <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-dead-letter-queues.html"
         *        >dead-letter queues:</a>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>RedrivePolicy</code> – The string that includes the parameters for the dead-letter queue
         *        functionality of the source queue as a JSON object. The parameters are as follows:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>deadLetterTargetArn</code> – The Amazon Resource Name (ARN) of the dead-letter queue to which
         *        Amazon SQS moves messages after the value of <code>maxReceiveCount</code> is exceeded.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>maxReceiveCount</code> – The number of times a message is delivered to the source queue before
         *        being moved to the dead-letter queue. Default: 10. When the <code>ReceiveCount</code> for a message
         *        exceeds the <code>maxReceiveCount</code> for a queue, Amazon SQS moves the message to the
         *        dead-letter-queue.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RedriveAllowPolicy</code> – The string that includes the parameters for the permissions for the
         *        dead-letter queue redrive permission and which source queues can specify dead-letter queues as a JSON
         *        object. The parameters are as follows:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>redrivePermission</code> – The permission type that defines which source queues can specify the
         *        current queue as the dead-letter queue. Valid values are:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>allowAll</code> – (Default) Any source queues in this Amazon Web Services account in the same
         *        Region can specify this queue as the dead-letter queue.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>denyAll</code> – No source queues can specify this queue as the dead-letter queue.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>byQueue</code> – Only queues specified by the <code>sourceQueueArns</code> parameter can specify
         *        this queue as the dead-letter queue.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>sourceQueueArns</code> – The Amazon Resource Names (ARN)s of the source queues that can specify
         *        this queue as the dead-letter queue and redrive messages. You can specify this parameter only when the
         *        <code>redrivePermission</code> parameter is set to <code>byQueue</code>. You can specify up to 10
         *        source queue ARNs. To allow more than 10 source queues to specify dead-letter queues, set the
         *        <code>redrivePermission</code> parameter to <code>allowAll</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        The dead-letter queue of a FIFO queue must also be a FIFO queue. Similarly, the dead-letter queue of a
         *        standard queue must also be a standard queue.
         *        </p>
         *        </note>
         *        <p>
         *        The following attributes apply only to <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-server-side-encryption.html"
         *        >server-side-encryption</a>:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>KmsMasterKeyId</code> – The ID of an Amazon Web Services managed customer master key (CMK) for
         *        Amazon SQS or a custom CMK. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-server-side-encryption.html#sqs-sse-key-terms"
         *        >Key Terms</a>. While the alias of the Amazon Web Services managed CMK for Amazon SQS is always
         *        <code>alias/aws/sqs</code>, the alias of a custom CMK can, for example, be
         *        <code>alias/<i>MyAlias</i> </code>. For more examples, see <a href=
         *        "https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeKey.html#API_DescribeKey_RequestParameters"
         *        >KeyId</a> in the <i>Key Management Service API Reference</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KmsDataKeyReusePeriodSeconds</code> – The length of time, in seconds, for which Amazon SQS can
         *        reuse a <a href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#data-keys">data
         *        key</a> to encrypt or decrypt messages before calling KMS again. An integer representing seconds,
         *        between 60 seconds (1 minute) and 86,400 seconds (24 hours). Default: 300 (5 minutes). A shorter time
         *        period provides better security but results in more calls to KMS which might incur charges after Free
         *        Tier. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-server-side-encryption.html#sqs-how-does-the-data-key-reuse-period-work"
         *        >How Does the Data Key Reuse Period Work?</a>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SqsManagedSseEnabled</code> – Enables server-side queue encryption using SQS owned encryption
         *        keys. Only one server-side encryption option is supported per queue (for example, <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-configure-sse-existing-queue.html"
         *        >SSE-KMS</a> or <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-configure-sqs-sse-queue.html"
         *        >SSE-SQS</a>).
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        The following attributes apply only to <a
         *        href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues.html"
         *        >FIFO (first-in-first-out) queues</a>:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>FifoQueue</code> – Designates a queue as FIFO. Valid values are <code>true</code> and
         *        <code>false</code>. If you don't specify the <code>FifoQueue</code> attribute, Amazon SQS creates a
         *        standard queue. You can provide this attribute only during queue creation. You can't change it for an
         *        existing queue. When you set this attribute, you must also provide the <code>MessageGroupId</code> for
         *        your messages explicitly.
         *        </p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues-understanding-logic.html"
         *        >FIFO queue logic</a> in the <i>Amazon SQS Developer Guide</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ContentBasedDeduplication</code> – Enables content-based deduplication. Valid values are
         *        <code>true</code> and <code>false</code>. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues-exactly-once-processing.html"
         *        >Exactly-once processing</a> in the <i>Amazon SQS Developer Guide</i>. Note the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Every message must have a unique <code>MessageDeduplicationId</code>.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        You may provide a <code>MessageDeduplicationId</code> explicitly.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you aren't able to provide a <code>MessageDeduplicationId</code> and you enable
         *        <code>ContentBasedDeduplication</code> for your queue, Amazon SQS uses a SHA-256 hash to generate the
         *        <code>MessageDeduplicationId</code> using the body of the message (but not the attributes of the
         *        message).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you don't provide a <code>MessageDeduplicationId</code> and the queue doesn't have
         *        <code>ContentBasedDeduplication</code> set, the action fails with an error.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the queue has <code>ContentBasedDeduplication</code> set, your <code>MessageDeduplicationId</code>
         *        overrides the generated one.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        When <code>ContentBasedDeduplication</code> is in effect, messages with identical content sent within
         *        the deduplication interval are treated as duplicates and only one copy of the message is delivered.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you send one message with <code>ContentBasedDeduplication</code> enabled and then another message
         *        with a <code>MessageDeduplicationId</code> that is the same as the one generated for the first
         *        <code>MessageDeduplicationId</code>, the two messages are treated as duplicates and only one copy of
         *        the message is delivered.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        </ul>
         *        <p>
         *        The following attributes apply only to <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/high-throughput-fifo.html"
         *        >high throughput for FIFO queues</a>:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DeduplicationScope</code> – Specifies whether message deduplication occurs at the message group
         *        or queue level. Valid values are <code>messageGroup</code> and <code>queue</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FifoThroughputLimit</code> – Specifies whether the FIFO queue throughput quota applies to the
         *        entire queue or per message group. Valid values are <code>perQueue</code> and
         *        <code>perMessageGroupId</code>. The <code>perMessageGroupId</code> value is allowed only when the
         *        value for <code>DeduplicationScope</code> is <code>messageGroup</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        To enable high throughput for FIFO queues, do the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Set <code>DeduplicationScope</code> to <code>messageGroup</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Set <code>FifoThroughputLimit</code> to <code>perMessageGroupId</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you set these attributes to anything other than the values shown for enabling high throughput,
         *        normal throughput is in effect and deduplication occurs as specified.
         *        </p>
         *        <p>
         *        For information on throughput quotas, see <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/quotas-messages.html"
         *        >Quotas related to messages</a> in the <i>Amazon SQS Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributes(Map<QueueAttributeName, String> attributes);

        /**
         * <p>
         * Add cost allocation tags to the specified Amazon SQS queue. For an overview, see <a
         * href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-queue-tags.html">Tagging
         * Your Amazon SQS Queues</a> in the <i>Amazon SQS Developer Guide</i>.
         * </p>
         * <p>
         * When you use queue tags, keep the following guidelines in mind:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Adding more than 50 tags to a queue isn't recommended.
         * </p>
         * </li>
         * <li>
         * <p>
         * Tags don't have any semantic meaning. Amazon SQS interprets tags as character strings.
         * </p>
         * </li>
         * <li>
         * <p>
         * Tags are case-sensitive.
         * </p>
         * </li>
         * <li>
         * <p>
         * A new tag with a key identical to that of an existing tag overwrites the existing tag.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For a full list of tag restrictions, see <a href=
         * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-limits.html#limits-queues"
         * >Quotas related to queues</a> in the <i>Amazon SQS Developer Guide</i>.
         * </p>
         * <note>
         * <p>
         * To be able to tag a queue on creation, you must have the <code>sqs:CreateQueue</code> and
         * <code>sqs:TagQueue</code> permissions.
         * </p>
         * <p>
         * Cross-account permissions don't apply to this action. For more information, see <a href=
         * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-customer-managed-policy-examples.html#grant-cross-account-permissions-to-role-and-user-name"
         * >Grant cross-account permissions to a role and a username</a> in the <i>Amazon SQS Developer Guide</i>.
         * </p>
         * </note>
         * 
         * @param tags
         *        Add cost allocation tags to the specified Amazon SQS queue. For an overview, see <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-queue-tags.html"
         *        >Tagging Your Amazon SQS Queues</a> in the <i>Amazon SQS Developer Guide</i>.</p>
         *        <p>
         *        When you use queue tags, keep the following guidelines in mind:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Adding more than 50 tags to a queue isn't recommended.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Tags don't have any semantic meaning. Amazon SQS interprets tags as character strings.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Tags are case-sensitive.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A new tag with a key identical to that of an existing tag overwrites the existing tag.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For a full list of tag restrictions, see <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-limits.html#limits-queues"
         *        >Quotas related to queues</a> in the <i>Amazon SQS Developer Guide</i>.
         *        </p>
         *        <note>
         *        <p>
         *        To be able to tag a queue on creation, you must have the <code>sqs:CreateQueue</code> and
         *        <code>sqs:TagQueue</code> permissions.
         *        </p>
         *        <p>
         *        Cross-account permissions don't apply to this action. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-customer-managed-policy-examples.html#grant-cross-account-permissions-to-role-and-user-name"
         *        >Grant cross-account permissions to a role and a username</a> in the <i>Amazon SQS Developer
         *        Guide</i>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SqsRequest.BuilderImpl implements Builder {
        private String queueName;

        private Map<String, String> attributes = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateQueueRequest model) {
            super(model);
            queueName(model.queueName);
            attributesWithStrings(model.attributes);
            tags(model.tags);
        }

        public final String getQueueName() {
            return queueName;
        }

        public final void setQueueName(String queueName) {
            this.queueName = queueName;
        }

        @Override
        public final Builder queueName(String queueName) {
            this.queueName = queueName;
            return this;
        }

        public final Map<String, String> getAttributes() {
            if (attributes instanceof SdkAutoConstructMap) {
                return null;
            }
            return attributes;
        }

        public final void setAttributes(Map<String, String> attributes) {
            this.attributes = QueueAttributeMapCopier.copy(attributes);
        }

        @Override
        public final Builder attributesWithStrings(Map<String, String> attributes) {
            this.attributes = QueueAttributeMapCopier.copy(attributes);
            return this;
        }

        @Override
        public final Builder attributes(Map<QueueAttributeName, String> attributes) {
            this.attributes = QueueAttributeMapCopier.copyEnumToString(attributes);
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateQueueRequest build() {
            return new CreateQueueRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
