/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.shield.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A grouping of protected resources that you and Shield Advanced can monitor as a collective. This resource grouping
 * improves the accuracy of detection and reduces false positives.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ProtectionGroup implements SdkPojo, Serializable, ToCopyableBuilder<ProtectionGroup.Builder, ProtectionGroup> {
    private static final SdkField<String> PROTECTION_GROUP_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ProtectionGroupId").getter(getter(ProtectionGroup::protectionGroupId))
            .setter(setter(Builder::protectionGroupId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProtectionGroupId").build()).build();

    private static final SdkField<String> AGGREGATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Aggregation").getter(getter(ProtectionGroup::aggregationAsString)).setter(setter(Builder::aggregation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Aggregation").build()).build();

    private static final SdkField<String> PATTERN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Pattern")
            .getter(getter(ProtectionGroup::patternAsString)).setter(setter(Builder::pattern))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Pattern").build()).build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceType").getter(getter(ProtectionGroup::resourceTypeAsString))
            .setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceType").build()).build();

    private static final SdkField<List<String>> MEMBERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Members")
            .getter(getter(ProtectionGroup::members))
            .setter(setter(Builder::members))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Members").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> PROTECTION_GROUP_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ProtectionGroupArn").getter(getter(ProtectionGroup::protectionGroupArn))
            .setter(setter(Builder::protectionGroupArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProtectionGroupArn").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PROTECTION_GROUP_ID_FIELD,
            AGGREGATION_FIELD, PATTERN_FIELD, RESOURCE_TYPE_FIELD, MEMBERS_FIELD, PROTECTION_GROUP_ARN_FIELD));

    private static final long serialVersionUID = 1L;

    private final String protectionGroupId;

    private final String aggregation;

    private final String pattern;

    private final String resourceType;

    private final List<String> members;

    private final String protectionGroupArn;

    private ProtectionGroup(BuilderImpl builder) {
        this.protectionGroupId = builder.protectionGroupId;
        this.aggregation = builder.aggregation;
        this.pattern = builder.pattern;
        this.resourceType = builder.resourceType;
        this.members = builder.members;
        this.protectionGroupArn = builder.protectionGroupArn;
    }

    /**
     * <p>
     * The name of the protection group. You use this to identify the protection group in lists and to manage the
     * protection group, for example to update, delete, or describe it.
     * </p>
     * 
     * @return The name of the protection group. You use this to identify the protection group in lists and to manage
     *         the protection group, for example to update, delete, or describe it.
     */
    public final String protectionGroupId() {
        return protectionGroupId;
    }

    /**
     * <p>
     * Defines how Shield combines resource data for the group in order to detect, mitigate, and report events.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Sum - Use the total traffic across the group. This is a good choice for most cases. Examples include Elastic IP
     * addresses for EC2 instances that scale manually or automatically.
     * </p>
     * </li>
     * <li>
     * <p>
     * Mean - Use the average of the traffic across the group. This is a good choice for resources that share traffic
     * uniformly. Examples include accelerators and load balancers.
     * </p>
     * </li>
     * <li>
     * <p>
     * Max - Use the highest traffic from each resource. This is useful for resources that don't share traffic and for
     * resources that share that traffic in a non-uniform way. Examples include Amazon CloudFront distributions and
     * origin resources for CloudFront distributions.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #aggregation} will
     * return {@link ProtectionGroupAggregation#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #aggregationAsString}.
     * </p>
     * 
     * @return Defines how Shield combines resource data for the group in order to detect, mitigate, and report
     *         events.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Sum - Use the total traffic across the group. This is a good choice for most cases. Examples include
     *         Elastic IP addresses for EC2 instances that scale manually or automatically.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Mean - Use the average of the traffic across the group. This is a good choice for resources that share
     *         traffic uniformly. Examples include accelerators and load balancers.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Max - Use the highest traffic from each resource. This is useful for resources that don't share traffic
     *         and for resources that share that traffic in a non-uniform way. Examples include Amazon CloudFront
     *         distributions and origin resources for CloudFront distributions.
     *         </p>
     *         </li>
     * @see ProtectionGroupAggregation
     */
    public final ProtectionGroupAggregation aggregation() {
        return ProtectionGroupAggregation.fromValue(aggregation);
    }

    /**
     * <p>
     * Defines how Shield combines resource data for the group in order to detect, mitigate, and report events.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Sum - Use the total traffic across the group. This is a good choice for most cases. Examples include Elastic IP
     * addresses for EC2 instances that scale manually or automatically.
     * </p>
     * </li>
     * <li>
     * <p>
     * Mean - Use the average of the traffic across the group. This is a good choice for resources that share traffic
     * uniformly. Examples include accelerators and load balancers.
     * </p>
     * </li>
     * <li>
     * <p>
     * Max - Use the highest traffic from each resource. This is useful for resources that don't share traffic and for
     * resources that share that traffic in a non-uniform way. Examples include Amazon CloudFront distributions and
     * origin resources for CloudFront distributions.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #aggregation} will
     * return {@link ProtectionGroupAggregation#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #aggregationAsString}.
     * </p>
     * 
     * @return Defines how Shield combines resource data for the group in order to detect, mitigate, and report
     *         events.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Sum - Use the total traffic across the group. This is a good choice for most cases. Examples include
     *         Elastic IP addresses for EC2 instances that scale manually or automatically.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Mean - Use the average of the traffic across the group. This is a good choice for resources that share
     *         traffic uniformly. Examples include accelerators and load balancers.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Max - Use the highest traffic from each resource. This is useful for resources that don't share traffic
     *         and for resources that share that traffic in a non-uniform way. Examples include Amazon CloudFront
     *         distributions and origin resources for CloudFront distributions.
     *         </p>
     *         </li>
     * @see ProtectionGroupAggregation
     */
    public final String aggregationAsString() {
        return aggregation;
    }

    /**
     * <p>
     * The criteria to use to choose the protected resources for inclusion in the group. You can include all resources
     * that have protections, provide a list of resource Amazon Resource Names (ARNs), or include all resources of a
     * specified resource type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #pattern} will
     * return {@link ProtectionGroupPattern#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #patternAsString}.
     * </p>
     * 
     * @return The criteria to use to choose the protected resources for inclusion in the group. You can include all
     *         resources that have protections, provide a list of resource Amazon Resource Names (ARNs), or include all
     *         resources of a specified resource type.
     * @see ProtectionGroupPattern
     */
    public final ProtectionGroupPattern pattern() {
        return ProtectionGroupPattern.fromValue(pattern);
    }

    /**
     * <p>
     * The criteria to use to choose the protected resources for inclusion in the group. You can include all resources
     * that have protections, provide a list of resource Amazon Resource Names (ARNs), or include all resources of a
     * specified resource type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #pattern} will
     * return {@link ProtectionGroupPattern#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #patternAsString}.
     * </p>
     * 
     * @return The criteria to use to choose the protected resources for inclusion in the group. You can include all
     *         resources that have protections, provide a list of resource Amazon Resource Names (ARNs), or include all
     *         resources of a specified resource type.
     * @see ProtectionGroupPattern
     */
    public final String patternAsString() {
        return pattern;
    }

    /**
     * <p>
     * The resource type to include in the protection group. All protected resources of this type are included in the
     * protection group. You must set this when you set <code>Pattern</code> to <code>BY_RESOURCE_TYPE</code> and you
     * must not set it for any other <code>Pattern</code> setting.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ProtectedResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The resource type to include in the protection group. All protected resources of this type are included
     *         in the protection group. You must set this when you set <code>Pattern</code> to
     *         <code>BY_RESOURCE_TYPE</code> and you must not set it for any other <code>Pattern</code> setting.
     * @see ProtectedResourceType
     */
    public final ProtectedResourceType resourceType() {
        return ProtectedResourceType.fromValue(resourceType);
    }

    /**
     * <p>
     * The resource type to include in the protection group. All protected resources of this type are included in the
     * protection group. You must set this when you set <code>Pattern</code> to <code>BY_RESOURCE_TYPE</code> and you
     * must not set it for any other <code>Pattern</code> setting.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ProtectedResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The resource type to include in the protection group. All protected resources of this type are included
     *         in the protection group. You must set this when you set <code>Pattern</code> to
     *         <code>BY_RESOURCE_TYPE</code> and you must not set it for any other <code>Pattern</code> setting.
     * @see ProtectedResourceType
     */
    public final String resourceTypeAsString() {
        return resourceType;
    }

    /**
     * For responses, this returns true if the service returned a value for the Members property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasMembers() {
        return members != null && !(members instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon Resource Names (ARNs) of the resources to include in the protection group. You must set this when you
     * set <code>Pattern</code> to <code>ARBITRARY</code> and you must not set it for any other <code>Pattern</code>
     * setting.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMembers} method.
     * </p>
     * 
     * @return The Amazon Resource Names (ARNs) of the resources to include in the protection group. You must set this
     *         when you set <code>Pattern</code> to <code>ARBITRARY</code> and you must not set it for any other
     *         <code>Pattern</code> setting.
     */
    public final List<String> members() {
        return members;
    }

    /**
     * <p>
     * The ARN (Amazon Resource Name) of the protection group.
     * </p>
     * 
     * @return The ARN (Amazon Resource Name) of the protection group.
     */
    public final String protectionGroupArn() {
        return protectionGroupArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(protectionGroupId());
        hashCode = 31 * hashCode + Objects.hashCode(aggregationAsString());
        hashCode = 31 * hashCode + Objects.hashCode(patternAsString());
        hashCode = 31 * hashCode + Objects.hashCode(resourceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasMembers() ? members() : null);
        hashCode = 31 * hashCode + Objects.hashCode(protectionGroupArn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ProtectionGroup)) {
            return false;
        }
        ProtectionGroup other = (ProtectionGroup) obj;
        return Objects.equals(protectionGroupId(), other.protectionGroupId())
                && Objects.equals(aggregationAsString(), other.aggregationAsString())
                && Objects.equals(patternAsString(), other.patternAsString())
                && Objects.equals(resourceTypeAsString(), other.resourceTypeAsString()) && hasMembers() == other.hasMembers()
                && Objects.equals(members(), other.members()) && Objects.equals(protectionGroupArn(), other.protectionGroupArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ProtectionGroup").add("ProtectionGroupId", protectionGroupId())
                .add("Aggregation", aggregationAsString()).add("Pattern", patternAsString())
                .add("ResourceType", resourceTypeAsString()).add("Members", hasMembers() ? members() : null)
                .add("ProtectionGroupArn", protectionGroupArn()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ProtectionGroupId":
            return Optional.ofNullable(clazz.cast(protectionGroupId()));
        case "Aggregation":
            return Optional.ofNullable(clazz.cast(aggregationAsString()));
        case "Pattern":
            return Optional.ofNullable(clazz.cast(patternAsString()));
        case "ResourceType":
            return Optional.ofNullable(clazz.cast(resourceTypeAsString()));
        case "Members":
            return Optional.ofNullable(clazz.cast(members()));
        case "ProtectionGroupArn":
            return Optional.ofNullable(clazz.cast(protectionGroupArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ProtectionGroup, T> g) {
        return obj -> g.apply((ProtectionGroup) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ProtectionGroup> {
        /**
         * <p>
         * The name of the protection group. You use this to identify the protection group in lists and to manage the
         * protection group, for example to update, delete, or describe it.
         * </p>
         * 
         * @param protectionGroupId
         *        The name of the protection group. You use this to identify the protection group in lists and to manage
         *        the protection group, for example to update, delete, or describe it.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protectionGroupId(String protectionGroupId);

        /**
         * <p>
         * Defines how Shield combines resource data for the group in order to detect, mitigate, and report events.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Sum - Use the total traffic across the group. This is a good choice for most cases. Examples include Elastic
         * IP addresses for EC2 instances that scale manually or automatically.
         * </p>
         * </li>
         * <li>
         * <p>
         * Mean - Use the average of the traffic across the group. This is a good choice for resources that share
         * traffic uniformly. Examples include accelerators and load balancers.
         * </p>
         * </li>
         * <li>
         * <p>
         * Max - Use the highest traffic from each resource. This is useful for resources that don't share traffic and
         * for resources that share that traffic in a non-uniform way. Examples include Amazon CloudFront distributions
         * and origin resources for CloudFront distributions.
         * </p>
         * </li>
         * </ul>
         * 
         * @param aggregation
         *        Defines how Shield combines resource data for the group in order to detect, mitigate, and report
         *        events.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Sum - Use the total traffic across the group. This is a good choice for most cases. Examples include
         *        Elastic IP addresses for EC2 instances that scale manually or automatically.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Mean - Use the average of the traffic across the group. This is a good choice for resources that share
         *        traffic uniformly. Examples include accelerators and load balancers.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Max - Use the highest traffic from each resource. This is useful for resources that don't share
         *        traffic and for resources that share that traffic in a non-uniform way. Examples include Amazon
         *        CloudFront distributions and origin resources for CloudFront distributions.
         *        </p>
         *        </li>
         * @see ProtectionGroupAggregation
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProtectionGroupAggregation
         */
        Builder aggregation(String aggregation);

        /**
         * <p>
         * Defines how Shield combines resource data for the group in order to detect, mitigate, and report events.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Sum - Use the total traffic across the group. This is a good choice for most cases. Examples include Elastic
         * IP addresses for EC2 instances that scale manually or automatically.
         * </p>
         * </li>
         * <li>
         * <p>
         * Mean - Use the average of the traffic across the group. This is a good choice for resources that share
         * traffic uniformly. Examples include accelerators and load balancers.
         * </p>
         * </li>
         * <li>
         * <p>
         * Max - Use the highest traffic from each resource. This is useful for resources that don't share traffic and
         * for resources that share that traffic in a non-uniform way. Examples include Amazon CloudFront distributions
         * and origin resources for CloudFront distributions.
         * </p>
         * </li>
         * </ul>
         * 
         * @param aggregation
         *        Defines how Shield combines resource data for the group in order to detect, mitigate, and report
         *        events.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Sum - Use the total traffic across the group. This is a good choice for most cases. Examples include
         *        Elastic IP addresses for EC2 instances that scale manually or automatically.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Mean - Use the average of the traffic across the group. This is a good choice for resources that share
         *        traffic uniformly. Examples include accelerators and load balancers.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Max - Use the highest traffic from each resource. This is useful for resources that don't share
         *        traffic and for resources that share that traffic in a non-uniform way. Examples include Amazon
         *        CloudFront distributions and origin resources for CloudFront distributions.
         *        </p>
         *        </li>
         * @see ProtectionGroupAggregation
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProtectionGroupAggregation
         */
        Builder aggregation(ProtectionGroupAggregation aggregation);

        /**
         * <p>
         * The criteria to use to choose the protected resources for inclusion in the group. You can include all
         * resources that have protections, provide a list of resource Amazon Resource Names (ARNs), or include all
         * resources of a specified resource type.
         * </p>
         * 
         * @param pattern
         *        The criteria to use to choose the protected resources for inclusion in the group. You can include all
         *        resources that have protections, provide a list of resource Amazon Resource Names (ARNs), or include
         *        all resources of a specified resource type.
         * @see ProtectionGroupPattern
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProtectionGroupPattern
         */
        Builder pattern(String pattern);

        /**
         * <p>
         * The criteria to use to choose the protected resources for inclusion in the group. You can include all
         * resources that have protections, provide a list of resource Amazon Resource Names (ARNs), or include all
         * resources of a specified resource type.
         * </p>
         * 
         * @param pattern
         *        The criteria to use to choose the protected resources for inclusion in the group. You can include all
         *        resources that have protections, provide a list of resource Amazon Resource Names (ARNs), or include
         *        all resources of a specified resource type.
         * @see ProtectionGroupPattern
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProtectionGroupPattern
         */
        Builder pattern(ProtectionGroupPattern pattern);

        /**
         * <p>
         * The resource type to include in the protection group. All protected resources of this type are included in
         * the protection group. You must set this when you set <code>Pattern</code> to <code>BY_RESOURCE_TYPE</code>
         * and you must not set it for any other <code>Pattern</code> setting.
         * </p>
         * 
         * @param resourceType
         *        The resource type to include in the protection group. All protected resources of this type are
         *        included in the protection group. You must set this when you set <code>Pattern</code> to
         *        <code>BY_RESOURCE_TYPE</code> and you must not set it for any other <code>Pattern</code> setting.
         * @see ProtectedResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProtectedResourceType
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The resource type to include in the protection group. All protected resources of this type are included in
         * the protection group. You must set this when you set <code>Pattern</code> to <code>BY_RESOURCE_TYPE</code>
         * and you must not set it for any other <code>Pattern</code> setting.
         * </p>
         * 
         * @param resourceType
         *        The resource type to include in the protection group. All protected resources of this type are
         *        included in the protection group. You must set this when you set <code>Pattern</code> to
         *        <code>BY_RESOURCE_TYPE</code> and you must not set it for any other <code>Pattern</code> setting.
         * @see ProtectedResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProtectedResourceType
         */
        Builder resourceType(ProtectedResourceType resourceType);

        /**
         * <p>
         * The Amazon Resource Names (ARNs) of the resources to include in the protection group. You must set this when
         * you set <code>Pattern</code> to <code>ARBITRARY</code> and you must not set it for any other
         * <code>Pattern</code> setting.
         * </p>
         * 
         * @param members
         *        The Amazon Resource Names (ARNs) of the resources to include in the protection group. You must set
         *        this when you set <code>Pattern</code> to <code>ARBITRARY</code> and you must not set it for any other
         *        <code>Pattern</code> setting.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder members(Collection<String> members);

        /**
         * <p>
         * The Amazon Resource Names (ARNs) of the resources to include in the protection group. You must set this when
         * you set <code>Pattern</code> to <code>ARBITRARY</code> and you must not set it for any other
         * <code>Pattern</code> setting.
         * </p>
         * 
         * @param members
         *        The Amazon Resource Names (ARNs) of the resources to include in the protection group. You must set
         *        this when you set <code>Pattern</code> to <code>ARBITRARY</code> and you must not set it for any other
         *        <code>Pattern</code> setting.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder members(String... members);

        /**
         * <p>
         * The ARN (Amazon Resource Name) of the protection group.
         * </p>
         * 
         * @param protectionGroupArn
         *        The ARN (Amazon Resource Name) of the protection group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protectionGroupArn(String protectionGroupArn);
    }

    static final class BuilderImpl implements Builder {
        private String protectionGroupId;

        private String aggregation;

        private String pattern;

        private String resourceType;

        private List<String> members = DefaultSdkAutoConstructList.getInstance();

        private String protectionGroupArn;

        private BuilderImpl() {
        }

        private BuilderImpl(ProtectionGroup model) {
            protectionGroupId(model.protectionGroupId);
            aggregation(model.aggregation);
            pattern(model.pattern);
            resourceType(model.resourceType);
            members(model.members);
            protectionGroupArn(model.protectionGroupArn);
        }

        public final String getProtectionGroupId() {
            return protectionGroupId;
        }

        public final void setProtectionGroupId(String protectionGroupId) {
            this.protectionGroupId = protectionGroupId;
        }

        @Override
        @Transient
        public final Builder protectionGroupId(String protectionGroupId) {
            this.protectionGroupId = protectionGroupId;
            return this;
        }

        public final String getAggregation() {
            return aggregation;
        }

        public final void setAggregation(String aggregation) {
            this.aggregation = aggregation;
        }

        @Override
        @Transient
        public final Builder aggregation(String aggregation) {
            this.aggregation = aggregation;
            return this;
        }

        @Override
        @Transient
        public final Builder aggregation(ProtectionGroupAggregation aggregation) {
            this.aggregation(aggregation == null ? null : aggregation.toString());
            return this;
        }

        public final String getPattern() {
            return pattern;
        }

        public final void setPattern(String pattern) {
            this.pattern = pattern;
        }

        @Override
        @Transient
        public final Builder pattern(String pattern) {
            this.pattern = pattern;
            return this;
        }

        @Override
        @Transient
        public final Builder pattern(ProtectionGroupPattern pattern) {
            this.pattern(pattern == null ? null : pattern.toString());
            return this;
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        @Transient
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        @Override
        @Transient
        public final Builder resourceType(ProtectedResourceType resourceType) {
            this.resourceType(resourceType == null ? null : resourceType.toString());
            return this;
        }

        public final Collection<String> getMembers() {
            if (members instanceof SdkAutoConstructList) {
                return null;
            }
            return members;
        }

        public final void setMembers(Collection<String> members) {
            this.members = ProtectionGroupMembersCopier.copy(members);
        }

        @Override
        @Transient
        public final Builder members(Collection<String> members) {
            this.members = ProtectionGroupMembersCopier.copy(members);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder members(String... members) {
            members(Arrays.asList(members));
            return this;
        }

        public final String getProtectionGroupArn() {
            return protectionGroupArn;
        }

        public final void setProtectionGroupArn(String protectionGroupArn) {
            this.protectionGroupArn = protectionGroupArn;
        }

        @Override
        @Transient
        public final Builder protectionGroupArn(String protectionGroupArn) {
            this.protectionGroupArn = protectionGroupArn;
            return this;
        }

        @Override
        public ProtectionGroup build() {
            return new ProtectionGroup(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
