/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3tables.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains details about a table.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TableSummary implements SdkPojo, Serializable, ToCopyableBuilder<TableSummary.Builder, TableSummary> {
    private static final SdkField<List<String>> NAMESPACE_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("namespace")
            .getter(getter(TableSummary::namespace))
            .setter(setter(Builder::namespace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("namespace").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(TableSummary::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(TableSummary::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<String> TABLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("tableARN").getter(getter(TableSummary::tableARN)).setter(setter(Builder::tableARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tableARN").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt")
            .getter(getter(TableSummary::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Instant> MODIFIED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("modifiedAt")
            .getter(getter(TableSummary::modifiedAt))
            .setter(setter(Builder::modifiedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("modifiedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAMESPACE_FIELD, NAME_FIELD,
            TYPE_FIELD, TABLE_ARN_FIELD, CREATED_AT_FIELD, MODIFIED_AT_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("namespace", NAMESPACE_FIELD);
                    put("name", NAME_FIELD);
                    put("type", TYPE_FIELD);
                    put("tableARN", TABLE_ARN_FIELD);
                    put("createdAt", CREATED_AT_FIELD);
                    put("modifiedAt", MODIFIED_AT_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final List<String> namespace;

    private final String name;

    private final String type;

    private final String tableARN;

    private final Instant createdAt;

    private final Instant modifiedAt;

    private TableSummary(BuilderImpl builder) {
        this.namespace = builder.namespace;
        this.name = builder.name;
        this.type = builder.type;
        this.tableARN = builder.tableARN;
        this.createdAt = builder.createdAt;
        this.modifiedAt = builder.modifiedAt;
    }

    /**
     * For responses, this returns true if the service returned a value for the Namespace property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasNamespace() {
        return namespace != null && !(namespace instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The name of the namespace.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNamespace} method.
     * </p>
     * 
     * @return The name of the namespace.
     */
    public final List<String> namespace() {
        return namespace;
    }

    /**
     * <p>
     * The name of the table.
     * </p>
     * 
     * @return The name of the table.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The type of the table.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link TableType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of the table.
     * @see TableType
     */
    public final TableType type() {
        return TableType.fromValue(type);
    }

    /**
     * <p>
     * The type of the table.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link TableType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of the table.
     * @see TableType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The Amazon Resource Number (ARN) of the table.
     * </p>
     * 
     * @return The Amazon Resource Number (ARN) of the table.
     */
    public final String tableARN() {
        return tableARN;
    }

    /**
     * <p>
     * The date and time the table was created at.
     * </p>
     * 
     * @return The date and time the table was created at.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The date and time the table was last modified at.
     * </p>
     * 
     * @return The date and time the table was last modified at.
     */
    public final Instant modifiedAt() {
        return modifiedAt;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasNamespace() ? namespace() : null);
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(tableARN());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(modifiedAt());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TableSummary)) {
            return false;
        }
        TableSummary other = (TableSummary) obj;
        return hasNamespace() == other.hasNamespace() && Objects.equals(namespace(), other.namespace())
                && Objects.equals(name(), other.name()) && Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(tableARN(), other.tableARN()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(modifiedAt(), other.modifiedAt());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TableSummary").add("Namespace", hasNamespace() ? namespace() : null).add("Name", name())
                .add("Type", typeAsString()).add("TableARN", tableARN()).add("CreatedAt", createdAt())
                .add("ModifiedAt", modifiedAt()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "namespace":
            return Optional.ofNullable(clazz.cast(namespace()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "tableARN":
            return Optional.ofNullable(clazz.cast(tableARN()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "modifiedAt":
            return Optional.ofNullable(clazz.cast(modifiedAt()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<TableSummary, T> g) {
        return obj -> g.apply((TableSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TableSummary> {
        /**
         * <p>
         * The name of the namespace.
         * </p>
         * 
         * @param namespace
         *        The name of the namespace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namespace(Collection<String> namespace);

        /**
         * <p>
         * The name of the namespace.
         * </p>
         * 
         * @param namespace
         *        The name of the namespace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namespace(String... namespace);

        /**
         * <p>
         * The name of the table.
         * </p>
         * 
         * @param name
         *        The name of the table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The type of the table.
         * </p>
         * 
         * @param type
         *        The type of the table.
         * @see TableType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TableType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of the table.
         * </p>
         * 
         * @param type
         *        The type of the table.
         * @see TableType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TableType
         */
        Builder type(TableType type);

        /**
         * <p>
         * The Amazon Resource Number (ARN) of the table.
         * </p>
         * 
         * @param tableARN
         *        The Amazon Resource Number (ARN) of the table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tableARN(String tableARN);

        /**
         * <p>
         * The date and time the table was created at.
         * </p>
         * 
         * @param createdAt
         *        The date and time the table was created at.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The date and time the table was last modified at.
         * </p>
         * 
         * @param modifiedAt
         *        The date and time the table was last modified at.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modifiedAt(Instant modifiedAt);
    }

    static final class BuilderImpl implements Builder {
        private List<String> namespace = DefaultSdkAutoConstructList.getInstance();

        private String name;

        private String type;

        private String tableARN;

        private Instant createdAt;

        private Instant modifiedAt;

        private BuilderImpl() {
        }

        private BuilderImpl(TableSummary model) {
            namespace(model.namespace);
            name(model.name);
            type(model.type);
            tableARN(model.tableARN);
            createdAt(model.createdAt);
            modifiedAt(model.modifiedAt);
        }

        public final Collection<String> getNamespace() {
            if (namespace instanceof SdkAutoConstructList) {
                return null;
            }
            return namespace;
        }

        public final void setNamespace(Collection<String> namespace) {
            this.namespace = NamespaceListCopier.copy(namespace);
        }

        @Override
        public final Builder namespace(Collection<String> namespace) {
            this.namespace = NamespaceListCopier.copy(namespace);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder namespace(String... namespace) {
            namespace(Arrays.asList(namespace));
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(TableType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getTableARN() {
            return tableARN;
        }

        public final void setTableARN(String tableARN) {
            this.tableARN = tableARN;
        }

        @Override
        public final Builder tableARN(String tableARN) {
            this.tableARN = tableARN;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final Instant getModifiedAt() {
            return modifiedAt;
        }

        public final void setModifiedAt(Instant modifiedAt) {
            this.modifiedAt = modifiedAt;
        }

        @Override
        public final Builder modifiedAt(Instant modifiedAt) {
            this.modifiedAt = modifiedAt;
            return this;
        }

        @Override
        public TableSummary build() {
            return new TableSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
