/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.RequiredTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The journal table record expiration settings for a journal table in an S3 Metadata configuration.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RecordExpiration implements SdkPojo, Serializable,
        ToCopyableBuilder<RecordExpiration.Builder, RecordExpiration> {
    private static final SdkField<String> EXPIRATION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Expiration")
            .getter(getter(RecordExpiration::expirationAsString))
            .setter(setter(Builder::expiration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Expiration")
                    .unmarshallLocationName("Expiration").build(), RequiredTrait.create()).build();

    private static final SdkField<Integer> DAYS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("Days")
            .getter(getter(RecordExpiration::days))
            .setter(setter(Builder::days))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Days")
                    .unmarshallLocationName("Days").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EXPIRATION_FIELD, DAYS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String expiration;

    private final Integer days;

    private RecordExpiration(BuilderImpl builder) {
        this.expiration = builder.expiration;
        this.days = builder.days;
    }

    /**
     * <p>
     * Specifies whether journal table record expiration is enabled or disabled.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #expiration} will
     * return {@link ExpirationState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #expirationAsString}.
     * </p>
     * 
     * @return Specifies whether journal table record expiration is enabled or disabled.
     * @see ExpirationState
     */
    public final ExpirationState expiration() {
        return ExpirationState.fromValue(expiration);
    }

    /**
     * <p>
     * Specifies whether journal table record expiration is enabled or disabled.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #expiration} will
     * return {@link ExpirationState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #expirationAsString}.
     * </p>
     * 
     * @return Specifies whether journal table record expiration is enabled or disabled.
     * @see ExpirationState
     */
    public final String expirationAsString() {
        return expiration;
    }

    /**
     * <p>
     * If you enable journal table record expiration, you can set the number of days to retain your journal table
     * records. Journal table records must be retained for a minimum of 7 days. To set this value, specify any whole
     * number from <code>7</code> to <code>2147483647</code>. For example, to retain your journal table records for one
     * year, set this value to <code>365</code>.
     * </p>
     * 
     * @return If you enable journal table record expiration, you can set the number of days to retain your journal
     *         table records. Journal table records must be retained for a minimum of 7 days. To set this value, specify
     *         any whole number from <code>7</code> to <code>2147483647</code>. For example, to retain your journal
     *         table records for one year, set this value to <code>365</code>.
     */
    public final Integer days() {
        return days;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(expirationAsString());
        hashCode = 31 * hashCode + Objects.hashCode(days());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RecordExpiration)) {
            return false;
        }
        RecordExpiration other = (RecordExpiration) obj;
        return Objects.equals(expirationAsString(), other.expirationAsString()) && Objects.equals(days(), other.days());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RecordExpiration").add("Expiration", expirationAsString()).add("Days", days()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Expiration":
            return Optional.ofNullable(clazz.cast(expirationAsString()));
        case "Days":
            return Optional.ofNullable(clazz.cast(days()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Expiration", EXPIRATION_FIELD);
        map.put("Days", DAYS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<RecordExpiration, T> g) {
        return obj -> g.apply((RecordExpiration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RecordExpiration> {
        /**
         * <p>
         * Specifies whether journal table record expiration is enabled or disabled.
         * </p>
         * 
         * @param expiration
         *        Specifies whether journal table record expiration is enabled or disabled.
         * @see ExpirationState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExpirationState
         */
        Builder expiration(String expiration);

        /**
         * <p>
         * Specifies whether journal table record expiration is enabled or disabled.
         * </p>
         * 
         * @param expiration
         *        Specifies whether journal table record expiration is enabled or disabled.
         * @see ExpirationState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExpirationState
         */
        Builder expiration(ExpirationState expiration);

        /**
         * <p>
         * If you enable journal table record expiration, you can set the number of days to retain your journal table
         * records. Journal table records must be retained for a minimum of 7 days. To set this value, specify any whole
         * number from <code>7</code> to <code>2147483647</code>. For example, to retain your journal table records for
         * one year, set this value to <code>365</code>.
         * </p>
         * 
         * @param days
         *        If you enable journal table record expiration, you can set the number of days to retain your journal
         *        table records. Journal table records must be retained for a minimum of 7 days. To set this value,
         *        specify any whole number from <code>7</code> to <code>2147483647</code>. For example, to retain your
         *        journal table records for one year, set this value to <code>365</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder days(Integer days);
    }

    static final class BuilderImpl implements Builder {
        private String expiration;

        private Integer days;

        private BuilderImpl() {
        }

        private BuilderImpl(RecordExpiration model) {
            expiration(model.expiration);
            days(model.days);
        }

        public final String getExpiration() {
            return expiration;
        }

        public final void setExpiration(String expiration) {
            this.expiration = expiration;
        }

        @Override
        public final Builder expiration(String expiration) {
            this.expiration = expiration;
            return this;
        }

        @Override
        public final Builder expiration(ExpirationState expiration) {
            this.expiration(expiration == null ? null : expiration.toString());
            return this;
        }

        public final Integer getDays() {
            return days;
        }

        public final void setDays(Integer days) {
            this.days = days;
        }

        @Override
        public final Builder days(Integer days) {
            this.days = days;
            return this;
        }

        @Override
        public RecordExpiration build() {
            return new RecordExpiration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
