/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.RequiredTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateSessionRequest extends S3Request implements
        ToCopyableBuilder<CreateSessionRequest.Builder, CreateSessionRequest> {
    private static final SdkField<String> SESSION_MODE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SessionMode")
            .getter(getter(CreateSessionRequest::sessionModeAsString))
            .setter(setter(Builder::sessionMode))
            .traits(LocationTrait.builder().location(MarshallLocation.HEADER).locationName("x-amz-create-session-mode")
                    .unmarshallLocationName("x-amz-create-session-mode").build()).build();

    private static final SdkField<String> BUCKET_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Bucket")
            .getter(getter(CreateSessionRequest::bucket))
            .setter(setter(Builder::bucket))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("Bucket")
                    .unmarshallLocationName("Bucket").build(), RequiredTrait.create()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SESSION_MODE_FIELD,
            BUCKET_FIELD));

    private final String sessionMode;

    private final String bucket;

    private CreateSessionRequest(BuilderImpl builder) {
        super(builder);
        this.sessionMode = builder.sessionMode;
        this.bucket = builder.bucket;
    }

    /**
     * <p>
     * Specifies the mode of the session that will be created, either <code>ReadWrite</code> or <code>ReadOnly</code>.
     * By default, a <code>ReadWrite</code> session is created. A <code>ReadWrite</code> session is capable of executing
     * all the Zonal endpoint APIs on a directory bucket. A <code>ReadOnly</code> session is constrained to execute the
     * following Zonal endpoint APIs: <code>GetObject</code>, <code>HeadObject</code>, <code>ListObjectsV2</code>,
     * <code>GetObjectAttributes</code>, <code>ListParts</code>, and <code>ListMultipartUploads</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #sessionMode} will
     * return {@link SessionMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #sessionModeAsString}.
     * </p>
     * 
     * @return Specifies the mode of the session that will be created, either <code>ReadWrite</code> or
     *         <code>ReadOnly</code>. By default, a <code>ReadWrite</code> session is created. A <code>ReadWrite</code>
     *         session is capable of executing all the Zonal endpoint APIs on a directory bucket. A
     *         <code>ReadOnly</code> session is constrained to execute the following Zonal endpoint APIs:
     *         <code>GetObject</code>, <code>HeadObject</code>, <code>ListObjectsV2</code>,
     *         <code>GetObjectAttributes</code>, <code>ListParts</code>, and <code>ListMultipartUploads</code>.
     * @see SessionMode
     */
    public final SessionMode sessionMode() {
        return SessionMode.fromValue(sessionMode);
    }

    /**
     * <p>
     * Specifies the mode of the session that will be created, either <code>ReadWrite</code> or <code>ReadOnly</code>.
     * By default, a <code>ReadWrite</code> session is created. A <code>ReadWrite</code> session is capable of executing
     * all the Zonal endpoint APIs on a directory bucket. A <code>ReadOnly</code> session is constrained to execute the
     * following Zonal endpoint APIs: <code>GetObject</code>, <code>HeadObject</code>, <code>ListObjectsV2</code>,
     * <code>GetObjectAttributes</code>, <code>ListParts</code>, and <code>ListMultipartUploads</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #sessionMode} will
     * return {@link SessionMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #sessionModeAsString}.
     * </p>
     * 
     * @return Specifies the mode of the session that will be created, either <code>ReadWrite</code> or
     *         <code>ReadOnly</code>. By default, a <code>ReadWrite</code> session is created. A <code>ReadWrite</code>
     *         session is capable of executing all the Zonal endpoint APIs on a directory bucket. A
     *         <code>ReadOnly</code> session is constrained to execute the following Zonal endpoint APIs:
     *         <code>GetObject</code>, <code>HeadObject</code>, <code>ListObjectsV2</code>,
     *         <code>GetObjectAttributes</code>, <code>ListParts</code>, and <code>ListMultipartUploads</code>.
     * @see SessionMode
     */
    public final String sessionModeAsString() {
        return sessionMode;
    }

    /**
     * <p>
     * The name of the bucket that you create a session for.
     * </p>
     * 
     * @return The name of the bucket that you create a session for.
     */
    public final String bucket() {
        return bucket;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(sessionModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(bucket());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateSessionRequest)) {
            return false;
        }
        CreateSessionRequest other = (CreateSessionRequest) obj;
        return Objects.equals(sessionModeAsString(), other.sessionModeAsString()) && Objects.equals(bucket(), other.bucket());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateSessionRequest").add("SessionMode", sessionModeAsString()).add("Bucket", bucket()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SessionMode":
            return Optional.ofNullable(clazz.cast(sessionModeAsString()));
        case "Bucket":
            return Optional.ofNullable(clazz.cast(bucket()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateSessionRequest, T> g) {
        return obj -> g.apply((CreateSessionRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends S3Request.Builder, SdkPojo, CopyableBuilder<Builder, CreateSessionRequest> {
        /**
         * <p>
         * Specifies the mode of the session that will be created, either <code>ReadWrite</code> or
         * <code>ReadOnly</code>. By default, a <code>ReadWrite</code> session is created. A <code>ReadWrite</code>
         * session is capable of executing all the Zonal endpoint APIs on a directory bucket. A <code>ReadOnly</code>
         * session is constrained to execute the following Zonal endpoint APIs: <code>GetObject</code>,
         * <code>HeadObject</code>, <code>ListObjectsV2</code>, <code>GetObjectAttributes</code>, <code>ListParts</code>
         * , and <code>ListMultipartUploads</code>.
         * </p>
         * 
         * @param sessionMode
         *        Specifies the mode of the session that will be created, either <code>ReadWrite</code> or
         *        <code>ReadOnly</code>. By default, a <code>ReadWrite</code> session is created. A
         *        <code>ReadWrite</code> session is capable of executing all the Zonal endpoint APIs on a directory
         *        bucket. A <code>ReadOnly</code> session is constrained to execute the following Zonal endpoint APIs:
         *        <code>GetObject</code>, <code>HeadObject</code>, <code>ListObjectsV2</code>,
         *        <code>GetObjectAttributes</code>, <code>ListParts</code>, and <code>ListMultipartUploads</code>.
         * @see SessionMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SessionMode
         */
        Builder sessionMode(String sessionMode);

        /**
         * <p>
         * Specifies the mode of the session that will be created, either <code>ReadWrite</code> or
         * <code>ReadOnly</code>. By default, a <code>ReadWrite</code> session is created. A <code>ReadWrite</code>
         * session is capable of executing all the Zonal endpoint APIs on a directory bucket. A <code>ReadOnly</code>
         * session is constrained to execute the following Zonal endpoint APIs: <code>GetObject</code>,
         * <code>HeadObject</code>, <code>ListObjectsV2</code>, <code>GetObjectAttributes</code>, <code>ListParts</code>
         * , and <code>ListMultipartUploads</code>.
         * </p>
         * 
         * @param sessionMode
         *        Specifies the mode of the session that will be created, either <code>ReadWrite</code> or
         *        <code>ReadOnly</code>. By default, a <code>ReadWrite</code> session is created. A
         *        <code>ReadWrite</code> session is capable of executing all the Zonal endpoint APIs on a directory
         *        bucket. A <code>ReadOnly</code> session is constrained to execute the following Zonal endpoint APIs:
         *        <code>GetObject</code>, <code>HeadObject</code>, <code>ListObjectsV2</code>,
         *        <code>GetObjectAttributes</code>, <code>ListParts</code>, and <code>ListMultipartUploads</code>.
         * @see SessionMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SessionMode
         */
        Builder sessionMode(SessionMode sessionMode);

        /**
         * <p>
         * The name of the bucket that you create a session for.
         * </p>
         * 
         * @param bucket
         *        The name of the bucket that you create a session for.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucket(String bucket);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends S3Request.BuilderImpl implements Builder {
        private String sessionMode;

        private String bucket;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateSessionRequest model) {
            super(model);
            sessionMode(model.sessionMode);
            bucket(model.bucket);
        }

        public final String getSessionMode() {
            return sessionMode;
        }

        public final void setSessionMode(String sessionMode) {
            this.sessionMode = sessionMode;
        }

        @Override
        public final Builder sessionMode(String sessionMode) {
            this.sessionMode = sessionMode;
            return this;
        }

        @Override
        public final Builder sessionMode(SessionMode sessionMode) {
            this.sessionMode(sessionMode == null ? null : sessionMode.toString());
            return this;
        }

        public final String getBucket() {
            return bucket;
        }

        public final void setBucket(String bucket) {
            this.bucket = bucket;
        }

        @Override
        public final Builder bucket(String bucket) {
            this.bucket = bucket;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateSessionRequest build() {
            return new CreateSessionRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
