/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.redshift.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a reserved node. You can call the <a>DescribeReservedNodeOfferings</a> API to obtain the available reserved
 * node offerings.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ReservedNode implements SdkPojo, Serializable, ToCopyableBuilder<ReservedNode.Builder, ReservedNode> {
    private static final SdkField<String> RESERVED_NODE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ReservedNodeId").getter(getter(ReservedNode::reservedNodeId)).setter(setter(Builder::reservedNodeId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReservedNodeId").build()).build();

    private static final SdkField<String> RESERVED_NODE_OFFERING_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ReservedNodeOfferingId").getter(getter(ReservedNode::reservedNodeOfferingId))
            .setter(setter(Builder::reservedNodeOfferingId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReservedNodeOfferingId").build())
            .build();

    private static final SdkField<String> NODE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NodeType").getter(getter(ReservedNode::nodeType)).setter(setter(Builder::nodeType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NodeType").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartTime").getter(getter(ReservedNode::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartTime").build()).build();

    private static final SdkField<Integer> DURATION_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Duration").getter(getter(ReservedNode::duration)).setter(setter(Builder::duration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Duration").build()).build();

    private static final SdkField<Double> FIXED_PRICE_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("FixedPrice").getter(getter(ReservedNode::fixedPrice)).setter(setter(Builder::fixedPrice))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FixedPrice").build()).build();

    private static final SdkField<Double> USAGE_PRICE_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("UsagePrice").getter(getter(ReservedNode::usagePrice)).setter(setter(Builder::usagePrice))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UsagePrice").build()).build();

    private static final SdkField<String> CURRENCY_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CurrencyCode").getter(getter(ReservedNode::currencyCode)).setter(setter(Builder::currencyCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CurrencyCode").build()).build();

    private static final SdkField<Integer> NODE_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("NodeCount").getter(getter(ReservedNode::nodeCount)).setter(setter(Builder::nodeCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NodeCount").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("State")
            .getter(getter(ReservedNode::state)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State").build()).build();

    private static final SdkField<String> OFFERING_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OfferingType").getter(getter(ReservedNode::offeringType)).setter(setter(Builder::offeringType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OfferingType").build()).build();

    private static final SdkField<List<RecurringCharge>> RECURRING_CHARGES_FIELD = SdkField
            .<List<RecurringCharge>> builder(MarshallingType.LIST)
            .memberName("RecurringCharges")
            .getter(getter(ReservedNode::recurringCharges))
            .setter(setter(Builder::recurringCharges))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RecurringCharges").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("RecurringCharge")
                            .memberFieldInfo(
                                    SdkField.<RecurringCharge> builder(MarshallingType.SDK_POJO)
                                            .constructor(RecurringCharge::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("RecurringCharge").build()).build()).build()).build();

    private static final SdkField<String> RESERVED_NODE_OFFERING_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ReservedNodeOfferingType").getter(getter(ReservedNode::reservedNodeOfferingTypeAsString))
            .setter(setter(Builder::reservedNodeOfferingType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReservedNodeOfferingType").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RESERVED_NODE_ID_FIELD,
            RESERVED_NODE_OFFERING_ID_FIELD, NODE_TYPE_FIELD, START_TIME_FIELD, DURATION_FIELD, FIXED_PRICE_FIELD,
            USAGE_PRICE_FIELD, CURRENCY_CODE_FIELD, NODE_COUNT_FIELD, STATE_FIELD, OFFERING_TYPE_FIELD, RECURRING_CHARGES_FIELD,
            RESERVED_NODE_OFFERING_TYPE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String reservedNodeId;

    private final String reservedNodeOfferingId;

    private final String nodeType;

    private final Instant startTime;

    private final Integer duration;

    private final Double fixedPrice;

    private final Double usagePrice;

    private final String currencyCode;

    private final Integer nodeCount;

    private final String state;

    private final String offeringType;

    private final List<RecurringCharge> recurringCharges;

    private final String reservedNodeOfferingType;

    private ReservedNode(BuilderImpl builder) {
        this.reservedNodeId = builder.reservedNodeId;
        this.reservedNodeOfferingId = builder.reservedNodeOfferingId;
        this.nodeType = builder.nodeType;
        this.startTime = builder.startTime;
        this.duration = builder.duration;
        this.fixedPrice = builder.fixedPrice;
        this.usagePrice = builder.usagePrice;
        this.currencyCode = builder.currencyCode;
        this.nodeCount = builder.nodeCount;
        this.state = builder.state;
        this.offeringType = builder.offeringType;
        this.recurringCharges = builder.recurringCharges;
        this.reservedNodeOfferingType = builder.reservedNodeOfferingType;
    }

    /**
     * <p>
     * The unique identifier for the reservation.
     * </p>
     * 
     * @return The unique identifier for the reservation.
     */
    public final String reservedNodeId() {
        return reservedNodeId;
    }

    /**
     * <p>
     * The identifier for the reserved node offering.
     * </p>
     * 
     * @return The identifier for the reserved node offering.
     */
    public final String reservedNodeOfferingId() {
        return reservedNodeOfferingId;
    }

    /**
     * <p>
     * The node type of the reserved node.
     * </p>
     * 
     * @return The node type of the reserved node.
     */
    public final String nodeType() {
        return nodeType;
    }

    /**
     * <p>
     * The time the reservation started. You purchase a reserved node offering for a duration. This is the start time of
     * that duration.
     * </p>
     * 
     * @return The time the reservation started. You purchase a reserved node offering for a duration. This is the start
     *         time of that duration.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The duration of the node reservation in seconds.
     * </p>
     * 
     * @return The duration of the node reservation in seconds.
     */
    public final Integer duration() {
        return duration;
    }

    /**
     * <p>
     * The fixed cost Amazon Redshift charges you for this reserved node.
     * </p>
     * 
     * @return The fixed cost Amazon Redshift charges you for this reserved node.
     */
    public final Double fixedPrice() {
        return fixedPrice;
    }

    /**
     * <p>
     * The hourly rate Amazon Redshift charges you for this reserved node.
     * </p>
     * 
     * @return The hourly rate Amazon Redshift charges you for this reserved node.
     */
    public final Double usagePrice() {
        return usagePrice;
    }

    /**
     * <p>
     * The currency code for the reserved cluster.
     * </p>
     * 
     * @return The currency code for the reserved cluster.
     */
    public final String currencyCode() {
        return currencyCode;
    }

    /**
     * <p>
     * The number of reserved compute nodes.
     * </p>
     * 
     * @return The number of reserved compute nodes.
     */
    public final Integer nodeCount() {
        return nodeCount;
    }

    /**
     * <p>
     * The state of the reserved compute node.
     * </p>
     * <p>
     * Possible Values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * pending-payment-This reserved node has recently been purchased, and the sale has been approved, but payment has
     * not yet been confirmed.
     * </p>
     * </li>
     * <li>
     * <p>
     * active-This reserved node is owned by the caller and is available for use.
     * </p>
     * </li>
     * <li>
     * <p>
     * payment-failed-Payment failed for the purchase attempt.
     * </p>
     * </li>
     * <li>
     * <p>
     * retired-The reserved node is no longer available.
     * </p>
     * </li>
     * <li>
     * <p>
     * exchanging-The owner is exchanging the reserved node for another reserved node.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The state of the reserved compute node.</p>
     *         <p>
     *         Possible Values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         pending-payment-This reserved node has recently been purchased, and the sale has been approved, but
     *         payment has not yet been confirmed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         active-This reserved node is owned by the caller and is available for use.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         payment-failed-Payment failed for the purchase attempt.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         retired-The reserved node is no longer available.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         exchanging-The owner is exchanging the reserved node for another reserved node.
     *         </p>
     *         </li>
     */
    public final String state() {
        return state;
    }

    /**
     * <p>
     * The anticipated utilization of the reserved node, as defined in the reserved node offering.
     * </p>
     * 
     * @return The anticipated utilization of the reserved node, as defined in the reserved node offering.
     */
    public final String offeringType() {
        return offeringType;
    }

    /**
     * For responses, this returns true if the service returned a value for the RecurringCharges property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRecurringCharges() {
        return recurringCharges != null && !(recurringCharges instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The recurring charges for the reserved node.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRecurringCharges} method.
     * </p>
     * 
     * @return The recurring charges for the reserved node.
     */
    public final List<RecurringCharge> recurringCharges() {
        return recurringCharges;
    }

    /**
     * <p/>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #reservedNodeOfferingType} will return {@link ReservedNodeOfferingType#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #reservedNodeOfferingTypeAsString}.
     * </p>
     * 
     * @return
     * @see ReservedNodeOfferingType
     */
    public final ReservedNodeOfferingType reservedNodeOfferingType() {
        return ReservedNodeOfferingType.fromValue(reservedNodeOfferingType);
    }

    /**
     * <p/>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #reservedNodeOfferingType} will return {@link ReservedNodeOfferingType#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #reservedNodeOfferingTypeAsString}.
     * </p>
     * 
     * @return
     * @see ReservedNodeOfferingType
     */
    public final String reservedNodeOfferingTypeAsString() {
        return reservedNodeOfferingType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(reservedNodeId());
        hashCode = 31 * hashCode + Objects.hashCode(reservedNodeOfferingId());
        hashCode = 31 * hashCode + Objects.hashCode(nodeType());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(duration());
        hashCode = 31 * hashCode + Objects.hashCode(fixedPrice());
        hashCode = 31 * hashCode + Objects.hashCode(usagePrice());
        hashCode = 31 * hashCode + Objects.hashCode(currencyCode());
        hashCode = 31 * hashCode + Objects.hashCode(nodeCount());
        hashCode = 31 * hashCode + Objects.hashCode(state());
        hashCode = 31 * hashCode + Objects.hashCode(offeringType());
        hashCode = 31 * hashCode + Objects.hashCode(hasRecurringCharges() ? recurringCharges() : null);
        hashCode = 31 * hashCode + Objects.hashCode(reservedNodeOfferingTypeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ReservedNode)) {
            return false;
        }
        ReservedNode other = (ReservedNode) obj;
        return Objects.equals(reservedNodeId(), other.reservedNodeId())
                && Objects.equals(reservedNodeOfferingId(), other.reservedNodeOfferingId())
                && Objects.equals(nodeType(), other.nodeType()) && Objects.equals(startTime(), other.startTime())
                && Objects.equals(duration(), other.duration()) && Objects.equals(fixedPrice(), other.fixedPrice())
                && Objects.equals(usagePrice(), other.usagePrice()) && Objects.equals(currencyCode(), other.currencyCode())
                && Objects.equals(nodeCount(), other.nodeCount()) && Objects.equals(state(), other.state())
                && Objects.equals(offeringType(), other.offeringType()) && hasRecurringCharges() == other.hasRecurringCharges()
                && Objects.equals(recurringCharges(), other.recurringCharges())
                && Objects.equals(reservedNodeOfferingTypeAsString(), other.reservedNodeOfferingTypeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ReservedNode").add("ReservedNodeId", reservedNodeId())
                .add("ReservedNodeOfferingId", reservedNodeOfferingId()).add("NodeType", nodeType())
                .add("StartTime", startTime()).add("Duration", duration()).add("FixedPrice", fixedPrice())
                .add("UsagePrice", usagePrice()).add("CurrencyCode", currencyCode()).add("NodeCount", nodeCount())
                .add("State", state()).add("OfferingType", offeringType())
                .add("RecurringCharges", hasRecurringCharges() ? recurringCharges() : null)
                .add("ReservedNodeOfferingType", reservedNodeOfferingTypeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ReservedNodeId":
            return Optional.ofNullable(clazz.cast(reservedNodeId()));
        case "ReservedNodeOfferingId":
            return Optional.ofNullable(clazz.cast(reservedNodeOfferingId()));
        case "NodeType":
            return Optional.ofNullable(clazz.cast(nodeType()));
        case "StartTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "Duration":
            return Optional.ofNullable(clazz.cast(duration()));
        case "FixedPrice":
            return Optional.ofNullable(clazz.cast(fixedPrice()));
        case "UsagePrice":
            return Optional.ofNullable(clazz.cast(usagePrice()));
        case "CurrencyCode":
            return Optional.ofNullable(clazz.cast(currencyCode()));
        case "NodeCount":
            return Optional.ofNullable(clazz.cast(nodeCount()));
        case "State":
            return Optional.ofNullable(clazz.cast(state()));
        case "OfferingType":
            return Optional.ofNullable(clazz.cast(offeringType()));
        case "RecurringCharges":
            return Optional.ofNullable(clazz.cast(recurringCharges()));
        case "ReservedNodeOfferingType":
            return Optional.ofNullable(clazz.cast(reservedNodeOfferingTypeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ReservedNodeId", RESERVED_NODE_ID_FIELD);
        map.put("ReservedNodeOfferingId", RESERVED_NODE_OFFERING_ID_FIELD);
        map.put("NodeType", NODE_TYPE_FIELD);
        map.put("StartTime", START_TIME_FIELD);
        map.put("Duration", DURATION_FIELD);
        map.put("FixedPrice", FIXED_PRICE_FIELD);
        map.put("UsagePrice", USAGE_PRICE_FIELD);
        map.put("CurrencyCode", CURRENCY_CODE_FIELD);
        map.put("NodeCount", NODE_COUNT_FIELD);
        map.put("State", STATE_FIELD);
        map.put("OfferingType", OFFERING_TYPE_FIELD);
        map.put("RecurringCharges", RECURRING_CHARGES_FIELD);
        map.put("ReservedNodeOfferingType", RESERVED_NODE_OFFERING_TYPE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ReservedNode, T> g) {
        return obj -> g.apply((ReservedNode) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ReservedNode> {
        /**
         * <p>
         * The unique identifier for the reservation.
         * </p>
         * 
         * @param reservedNodeId
         *        The unique identifier for the reservation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reservedNodeId(String reservedNodeId);

        /**
         * <p>
         * The identifier for the reserved node offering.
         * </p>
         * 
         * @param reservedNodeOfferingId
         *        The identifier for the reserved node offering.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reservedNodeOfferingId(String reservedNodeOfferingId);

        /**
         * <p>
         * The node type of the reserved node.
         * </p>
         * 
         * @param nodeType
         *        The node type of the reserved node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodeType(String nodeType);

        /**
         * <p>
         * The time the reservation started. You purchase a reserved node offering for a duration. This is the start
         * time of that duration.
         * </p>
         * 
         * @param startTime
         *        The time the reservation started. You purchase a reserved node offering for a duration. This is the
         *        start time of that duration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The duration of the node reservation in seconds.
         * </p>
         * 
         * @param duration
         *        The duration of the node reservation in seconds.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder duration(Integer duration);

        /**
         * <p>
         * The fixed cost Amazon Redshift charges you for this reserved node.
         * </p>
         * 
         * @param fixedPrice
         *        The fixed cost Amazon Redshift charges you for this reserved node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fixedPrice(Double fixedPrice);

        /**
         * <p>
         * The hourly rate Amazon Redshift charges you for this reserved node.
         * </p>
         * 
         * @param usagePrice
         *        The hourly rate Amazon Redshift charges you for this reserved node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder usagePrice(Double usagePrice);

        /**
         * <p>
         * The currency code for the reserved cluster.
         * </p>
         * 
         * @param currencyCode
         *        The currency code for the reserved cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder currencyCode(String currencyCode);

        /**
         * <p>
         * The number of reserved compute nodes.
         * </p>
         * 
         * @param nodeCount
         *        The number of reserved compute nodes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodeCount(Integer nodeCount);

        /**
         * <p>
         * The state of the reserved compute node.
         * </p>
         * <p>
         * Possible Values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * pending-payment-This reserved node has recently been purchased, and the sale has been approved, but payment
         * has not yet been confirmed.
         * </p>
         * </li>
         * <li>
         * <p>
         * active-This reserved node is owned by the caller and is available for use.
         * </p>
         * </li>
         * <li>
         * <p>
         * payment-failed-Payment failed for the purchase attempt.
         * </p>
         * </li>
         * <li>
         * <p>
         * retired-The reserved node is no longer available.
         * </p>
         * </li>
         * <li>
         * <p>
         * exchanging-The owner is exchanging the reserved node for another reserved node.
         * </p>
         * </li>
         * </ul>
         * 
         * @param state
         *        The state of the reserved compute node.</p>
         *        <p>
         *        Possible Values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        pending-payment-This reserved node has recently been purchased, and the sale has been approved, but
         *        payment has not yet been confirmed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        active-This reserved node is owned by the caller and is available for use.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        payment-failed-Payment failed for the purchase attempt.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        retired-The reserved node is no longer available.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        exchanging-The owner is exchanging the reserved node for another reserved node.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder state(String state);

        /**
         * <p>
         * The anticipated utilization of the reserved node, as defined in the reserved node offering.
         * </p>
         * 
         * @param offeringType
         *        The anticipated utilization of the reserved node, as defined in the reserved node offering.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder offeringType(String offeringType);

        /**
         * <p>
         * The recurring charges for the reserved node.
         * </p>
         * 
         * @param recurringCharges
         *        The recurring charges for the reserved node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recurringCharges(Collection<RecurringCharge> recurringCharges);

        /**
         * <p>
         * The recurring charges for the reserved node.
         * </p>
         * 
         * @param recurringCharges
         *        The recurring charges for the reserved node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recurringCharges(RecurringCharge... recurringCharges);

        /**
         * <p>
         * The recurring charges for the reserved node.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.redshift.model.RecurringCharge.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.redshift.model.RecurringCharge#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.redshift.model.RecurringCharge.Builder#build()} is called immediately
         * and its result is passed to {@link #recurringCharges(List<RecurringCharge>)}.
         * 
         * @param recurringCharges
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.redshift.model.RecurringCharge.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #recurringCharges(java.util.Collection<RecurringCharge>)
         */
        Builder recurringCharges(Consumer<RecurringCharge.Builder>... recurringCharges);

        /**
         * <p/>
         * 
         * @param reservedNodeOfferingType
         * @see ReservedNodeOfferingType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReservedNodeOfferingType
         */
        Builder reservedNodeOfferingType(String reservedNodeOfferingType);

        /**
         * <p/>
         * 
         * @param reservedNodeOfferingType
         * @see ReservedNodeOfferingType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReservedNodeOfferingType
         */
        Builder reservedNodeOfferingType(ReservedNodeOfferingType reservedNodeOfferingType);
    }

    static final class BuilderImpl implements Builder {
        private String reservedNodeId;

        private String reservedNodeOfferingId;

        private String nodeType;

        private Instant startTime;

        private Integer duration;

        private Double fixedPrice;

        private Double usagePrice;

        private String currencyCode;

        private Integer nodeCount;

        private String state;

        private String offeringType;

        private List<RecurringCharge> recurringCharges = DefaultSdkAutoConstructList.getInstance();

        private String reservedNodeOfferingType;

        private BuilderImpl() {
        }

        private BuilderImpl(ReservedNode model) {
            reservedNodeId(model.reservedNodeId);
            reservedNodeOfferingId(model.reservedNodeOfferingId);
            nodeType(model.nodeType);
            startTime(model.startTime);
            duration(model.duration);
            fixedPrice(model.fixedPrice);
            usagePrice(model.usagePrice);
            currencyCode(model.currencyCode);
            nodeCount(model.nodeCount);
            state(model.state);
            offeringType(model.offeringType);
            recurringCharges(model.recurringCharges);
            reservedNodeOfferingType(model.reservedNodeOfferingType);
        }

        public final String getReservedNodeId() {
            return reservedNodeId;
        }

        public final void setReservedNodeId(String reservedNodeId) {
            this.reservedNodeId = reservedNodeId;
        }

        @Override
        public final Builder reservedNodeId(String reservedNodeId) {
            this.reservedNodeId = reservedNodeId;
            return this;
        }

        public final String getReservedNodeOfferingId() {
            return reservedNodeOfferingId;
        }

        public final void setReservedNodeOfferingId(String reservedNodeOfferingId) {
            this.reservedNodeOfferingId = reservedNodeOfferingId;
        }

        @Override
        public final Builder reservedNodeOfferingId(String reservedNodeOfferingId) {
            this.reservedNodeOfferingId = reservedNodeOfferingId;
            return this;
        }

        public final String getNodeType() {
            return nodeType;
        }

        public final void setNodeType(String nodeType) {
            this.nodeType = nodeType;
        }

        @Override
        public final Builder nodeType(String nodeType) {
            this.nodeType = nodeType;
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final Integer getDuration() {
            return duration;
        }

        public final void setDuration(Integer duration) {
            this.duration = duration;
        }

        @Override
        public final Builder duration(Integer duration) {
            this.duration = duration;
            return this;
        }

        public final Double getFixedPrice() {
            return fixedPrice;
        }

        public final void setFixedPrice(Double fixedPrice) {
            this.fixedPrice = fixedPrice;
        }

        @Override
        public final Builder fixedPrice(Double fixedPrice) {
            this.fixedPrice = fixedPrice;
            return this;
        }

        public final Double getUsagePrice() {
            return usagePrice;
        }

        public final void setUsagePrice(Double usagePrice) {
            this.usagePrice = usagePrice;
        }

        @Override
        public final Builder usagePrice(Double usagePrice) {
            this.usagePrice = usagePrice;
            return this;
        }

        public final String getCurrencyCode() {
            return currencyCode;
        }

        public final void setCurrencyCode(String currencyCode) {
            this.currencyCode = currencyCode;
        }

        @Override
        public final Builder currencyCode(String currencyCode) {
            this.currencyCode = currencyCode;
            return this;
        }

        public final Integer getNodeCount() {
            return nodeCount;
        }

        public final void setNodeCount(Integer nodeCount) {
            this.nodeCount = nodeCount;
        }

        @Override
        public final Builder nodeCount(Integer nodeCount) {
            this.nodeCount = nodeCount;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        public final String getOfferingType() {
            return offeringType;
        }

        public final void setOfferingType(String offeringType) {
            this.offeringType = offeringType;
        }

        @Override
        public final Builder offeringType(String offeringType) {
            this.offeringType = offeringType;
            return this;
        }

        public final List<RecurringCharge.Builder> getRecurringCharges() {
            List<RecurringCharge.Builder> result = RecurringChargeListCopier.copyToBuilder(this.recurringCharges);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRecurringCharges(Collection<RecurringCharge.BuilderImpl> recurringCharges) {
            this.recurringCharges = RecurringChargeListCopier.copyFromBuilder(recurringCharges);
        }

        @Override
        public final Builder recurringCharges(Collection<RecurringCharge> recurringCharges) {
            this.recurringCharges = RecurringChargeListCopier.copy(recurringCharges);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recurringCharges(RecurringCharge... recurringCharges) {
            recurringCharges(Arrays.asList(recurringCharges));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recurringCharges(Consumer<RecurringCharge.Builder>... recurringCharges) {
            recurringCharges(Stream.of(recurringCharges).map(c -> RecurringCharge.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getReservedNodeOfferingType() {
            return reservedNodeOfferingType;
        }

        public final void setReservedNodeOfferingType(String reservedNodeOfferingType) {
            this.reservedNodeOfferingType = reservedNodeOfferingType;
        }

        @Override
        public final Builder reservedNodeOfferingType(String reservedNodeOfferingType) {
            this.reservedNodeOfferingType = reservedNodeOfferingType;
            return this;
        }

        @Override
        public final Builder reservedNodeOfferingType(ReservedNodeOfferingType reservedNodeOfferingType) {
            this.reservedNodeOfferingType(reservedNodeOfferingType == null ? null : reservedNodeOfferingType.toString());
            return this;
        }

        @Override
        public ReservedNode build() {
            return new ReservedNode(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
