/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.redshift.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Returns information about an HSM client certificate. The certificate is stored in a secure Hardware Storage Module
 * (HSM), and used by the Amazon Redshift cluster to encrypt data files.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class HsmClientCertificate implements SdkPojo, Serializable,
        ToCopyableBuilder<HsmClientCertificate.Builder, HsmClientCertificate> {
    private static final SdkField<String> HSM_CLIENT_CERTIFICATE_IDENTIFIER_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("HsmClientCertificateIdentifier")
            .getter(getter(HsmClientCertificate::hsmClientCertificateIdentifier))
            .setter(setter(Builder::hsmClientCertificateIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HsmClientCertificateIdentifier")
                    .build()).build();

    private static final SdkField<String> HSM_CLIENT_CERTIFICATE_PUBLIC_KEY_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("HsmClientCertificatePublicKey")
            .getter(getter(HsmClientCertificate::hsmClientCertificatePublicKey))
            .setter(setter(Builder::hsmClientCertificatePublicKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HsmClientCertificatePublicKey")
                    .build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(HsmClientCertificate::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("Tag")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Tag").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            HSM_CLIENT_CERTIFICATE_IDENTIFIER_FIELD, HSM_CLIENT_CERTIFICATE_PUBLIC_KEY_FIELD, TAGS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String hsmClientCertificateIdentifier;

    private final String hsmClientCertificatePublicKey;

    private final List<Tag> tags;

    private HsmClientCertificate(BuilderImpl builder) {
        this.hsmClientCertificateIdentifier = builder.hsmClientCertificateIdentifier;
        this.hsmClientCertificatePublicKey = builder.hsmClientCertificatePublicKey;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The identifier of the HSM client certificate.
     * </p>
     * 
     * @return The identifier of the HSM client certificate.
     */
    public final String hsmClientCertificateIdentifier() {
        return hsmClientCertificateIdentifier;
    }

    /**
     * <p>
     * The public key that the Amazon Redshift cluster will use to connect to the HSM. You must register the public key
     * in the HSM.
     * </p>
     * 
     * @return The public key that the Amazon Redshift cluster will use to connect to the HSM. You must register the
     *         public key in the HSM.
     */
    public final String hsmClientCertificatePublicKey() {
        return hsmClientCertificatePublicKey;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of tags for the HSM client certificate.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The list of tags for the HSM client certificate.
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hsmClientCertificateIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(hsmClientCertificatePublicKey());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof HsmClientCertificate)) {
            return false;
        }
        HsmClientCertificate other = (HsmClientCertificate) obj;
        return Objects.equals(hsmClientCertificateIdentifier(), other.hsmClientCertificateIdentifier())
                && Objects.equals(hsmClientCertificatePublicKey(), other.hsmClientCertificatePublicKey())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("HsmClientCertificate").add("HsmClientCertificateIdentifier", hsmClientCertificateIdentifier())
                .add("HsmClientCertificatePublicKey", hsmClientCertificatePublicKey()).add("Tags", hasTags() ? tags() : null)
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "HsmClientCertificateIdentifier":
            return Optional.ofNullable(clazz.cast(hsmClientCertificateIdentifier()));
        case "HsmClientCertificatePublicKey":
            return Optional.ofNullable(clazz.cast(hsmClientCertificatePublicKey()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("HsmClientCertificateIdentifier", HSM_CLIENT_CERTIFICATE_IDENTIFIER_FIELD);
        map.put("HsmClientCertificatePublicKey", HSM_CLIENT_CERTIFICATE_PUBLIC_KEY_FIELD);
        map.put("Tags", TAGS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<HsmClientCertificate, T> g) {
        return obj -> g.apply((HsmClientCertificate) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, HsmClientCertificate> {
        /**
         * <p>
         * The identifier of the HSM client certificate.
         * </p>
         * 
         * @param hsmClientCertificateIdentifier
         *        The identifier of the HSM client certificate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hsmClientCertificateIdentifier(String hsmClientCertificateIdentifier);

        /**
         * <p>
         * The public key that the Amazon Redshift cluster will use to connect to the HSM. You must register the public
         * key in the HSM.
         * </p>
         * 
         * @param hsmClientCertificatePublicKey
         *        The public key that the Amazon Redshift cluster will use to connect to the HSM. You must register the
         *        public key in the HSM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hsmClientCertificatePublicKey(String hsmClientCertificatePublicKey);

        /**
         * <p>
         * The list of tags for the HSM client certificate.
         * </p>
         * 
         * @param tags
         *        The list of tags for the HSM client certificate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The list of tags for the HSM client certificate.
         * </p>
         * 
         * @param tags
         *        The list of tags for the HSM client certificate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The list of tags for the HSM client certificate.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.redshift.model.Tag.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.redshift.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.redshift.model.Tag.Builder#build()} is called immediately and its
         * result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.redshift.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String hsmClientCertificateIdentifier;

        private String hsmClientCertificatePublicKey;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(HsmClientCertificate model) {
            hsmClientCertificateIdentifier(model.hsmClientCertificateIdentifier);
            hsmClientCertificatePublicKey(model.hsmClientCertificatePublicKey);
            tags(model.tags);
        }

        public final String getHsmClientCertificateIdentifier() {
            return hsmClientCertificateIdentifier;
        }

        public final void setHsmClientCertificateIdentifier(String hsmClientCertificateIdentifier) {
            this.hsmClientCertificateIdentifier = hsmClientCertificateIdentifier;
        }

        @Override
        public final Builder hsmClientCertificateIdentifier(String hsmClientCertificateIdentifier) {
            this.hsmClientCertificateIdentifier = hsmClientCertificateIdentifier;
            return this;
        }

        public final String getHsmClientCertificatePublicKey() {
            return hsmClientCertificatePublicKey;
        }

        public final void setHsmClientCertificatePublicKey(String hsmClientCertificatePublicKey) {
            this.hsmClientCertificatePublicKey = hsmClientCertificatePublicKey;
        }

        @Override
        public final Builder hsmClientCertificatePublicKey(String hsmClientCertificatePublicKey) {
            this.hsmClientCertificatePublicKey = hsmClientCertificatePublicKey;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public HsmClientCertificate build() {
            return new HsmClientCertificate(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
