/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.redshift.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an endpoint authorization for authorizing Redshift-managed VPC endpoint access to a cluster across Amazon
 * Web Services accounts.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AuthorizeEndpointAccessResponse extends RedshiftResponse implements
        ToCopyableBuilder<AuthorizeEndpointAccessResponse.Builder, AuthorizeEndpointAccessResponse> {
    private static final SdkField<String> GRANTOR_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Grantor")
            .getter(getter(AuthorizeEndpointAccessResponse::grantor)).setter(setter(Builder::grantor))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Grantor").build()).build();

    private static final SdkField<String> GRANTEE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Grantee")
            .getter(getter(AuthorizeEndpointAccessResponse::grantee)).setter(setter(Builder::grantee))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Grantee").build()).build();

    private static final SdkField<String> CLUSTER_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ClusterIdentifier").getter(getter(AuthorizeEndpointAccessResponse::clusterIdentifier))
            .setter(setter(Builder::clusterIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClusterIdentifier").build()).build();

    private static final SdkField<Instant> AUTHORIZE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("AuthorizeTime").getter(getter(AuthorizeEndpointAccessResponse::authorizeTime))
            .setter(setter(Builder::authorizeTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AuthorizeTime").build()).build();

    private static final SdkField<String> CLUSTER_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ClusterStatus").getter(getter(AuthorizeEndpointAccessResponse::clusterStatus))
            .setter(setter(Builder::clusterStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClusterStatus").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(AuthorizeEndpointAccessResponse::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Boolean> ALLOWED_ALL_VP_CS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("AllowedAllVPCs").getter(getter(AuthorizeEndpointAccessResponse::allowedAllVPCs))
            .setter(setter(Builder::allowedAllVPCs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllowedAllVPCs").build()).build();

    private static final SdkField<List<String>> ALLOWED_VP_CS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AllowedVPCs")
            .getter(getter(AuthorizeEndpointAccessResponse::allowedVPCs))
            .setter(setter(Builder::allowedVPCs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllowedVPCs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("VpcIdentifier")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("VpcIdentifier").build()).build()).build()).build();

    private static final SdkField<Integer> ENDPOINT_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("EndpointCount").getter(getter(AuthorizeEndpointAccessResponse::endpointCount))
            .setter(setter(Builder::endpointCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointCount").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(GRANTOR_FIELD, GRANTEE_FIELD,
            CLUSTER_IDENTIFIER_FIELD, AUTHORIZE_TIME_FIELD, CLUSTER_STATUS_FIELD, STATUS_FIELD, ALLOWED_ALL_VP_CS_FIELD,
            ALLOWED_VP_CS_FIELD, ENDPOINT_COUNT_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String grantor;

    private final String grantee;

    private final String clusterIdentifier;

    private final Instant authorizeTime;

    private final String clusterStatus;

    private final String status;

    private final Boolean allowedAllVPCs;

    private final List<String> allowedVPCs;

    private final Integer endpointCount;

    private AuthorizeEndpointAccessResponse(BuilderImpl builder) {
        super(builder);
        this.grantor = builder.grantor;
        this.grantee = builder.grantee;
        this.clusterIdentifier = builder.clusterIdentifier;
        this.authorizeTime = builder.authorizeTime;
        this.clusterStatus = builder.clusterStatus;
        this.status = builder.status;
        this.allowedAllVPCs = builder.allowedAllVPCs;
        this.allowedVPCs = builder.allowedVPCs;
        this.endpointCount = builder.endpointCount;
    }

    /**
     * <p>
     * The Amazon Web Services account ID of the cluster owner.
     * </p>
     * 
     * @return The Amazon Web Services account ID of the cluster owner.
     */
    public final String grantor() {
        return grantor;
    }

    /**
     * <p>
     * The Amazon Web Services account ID of the grantee of the cluster.
     * </p>
     * 
     * @return The Amazon Web Services account ID of the grantee of the cluster.
     */
    public final String grantee() {
        return grantee;
    }

    /**
     * <p>
     * The cluster identifier.
     * </p>
     * 
     * @return The cluster identifier.
     */
    public final String clusterIdentifier() {
        return clusterIdentifier;
    }

    /**
     * <p>
     * The time (UTC) when the authorization was created.
     * </p>
     * 
     * @return The time (UTC) when the authorization was created.
     */
    public final Instant authorizeTime() {
        return authorizeTime;
    }

    /**
     * <p>
     * The status of the cluster.
     * </p>
     * 
     * @return The status of the cluster.
     */
    public final String clusterStatus() {
        return clusterStatus;
    }

    /**
     * <p>
     * The status of the authorization action.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AuthorizationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the authorization action.
     * @see AuthorizationStatus
     */
    public final AuthorizationStatus status() {
        return AuthorizationStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the authorization action.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AuthorizationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the authorization action.
     * @see AuthorizationStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * Indicates whether all VPCs in the grantee account are allowed access to the cluster.
     * </p>
     * 
     * @return Indicates whether all VPCs in the grantee account are allowed access to the cluster.
     */
    public final Boolean allowedAllVPCs() {
        return allowedAllVPCs;
    }

    /**
     * For responses, this returns true if the service returned a value for the AllowedVPCs property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAllowedVPCs() {
        return allowedVPCs != null && !(allowedVPCs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The VPCs allowed access to the cluster.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAllowedVPCs} method.
     * </p>
     * 
     * @return The VPCs allowed access to the cluster.
     */
    public final List<String> allowedVPCs() {
        return allowedVPCs;
    }

    /**
     * <p>
     * The number of Redshift-managed VPC endpoints created for the authorization.
     * </p>
     * 
     * @return The number of Redshift-managed VPC endpoints created for the authorization.
     */
    public final Integer endpointCount() {
        return endpointCount;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(grantor());
        hashCode = 31 * hashCode + Objects.hashCode(grantee());
        hashCode = 31 * hashCode + Objects.hashCode(clusterIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(authorizeTime());
        hashCode = 31 * hashCode + Objects.hashCode(clusterStatus());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(allowedAllVPCs());
        hashCode = 31 * hashCode + Objects.hashCode(hasAllowedVPCs() ? allowedVPCs() : null);
        hashCode = 31 * hashCode + Objects.hashCode(endpointCount());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AuthorizeEndpointAccessResponse)) {
            return false;
        }
        AuthorizeEndpointAccessResponse other = (AuthorizeEndpointAccessResponse) obj;
        return Objects.equals(grantor(), other.grantor()) && Objects.equals(grantee(), other.grantee())
                && Objects.equals(clusterIdentifier(), other.clusterIdentifier())
                && Objects.equals(authorizeTime(), other.authorizeTime())
                && Objects.equals(clusterStatus(), other.clusterStatus())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(allowedAllVPCs(), other.allowedAllVPCs()) && hasAllowedVPCs() == other.hasAllowedVPCs()
                && Objects.equals(allowedVPCs(), other.allowedVPCs()) && Objects.equals(endpointCount(), other.endpointCount());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AuthorizeEndpointAccessResponse").add("Grantor", grantor()).add("Grantee", grantee())
                .add("ClusterIdentifier", clusterIdentifier()).add("AuthorizeTime", authorizeTime())
                .add("ClusterStatus", clusterStatus()).add("Status", statusAsString()).add("AllowedAllVPCs", allowedAllVPCs())
                .add("AllowedVPCs", hasAllowedVPCs() ? allowedVPCs() : null).add("EndpointCount", endpointCount()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Grantor":
            return Optional.ofNullable(clazz.cast(grantor()));
        case "Grantee":
            return Optional.ofNullable(clazz.cast(grantee()));
        case "ClusterIdentifier":
            return Optional.ofNullable(clazz.cast(clusterIdentifier()));
        case "AuthorizeTime":
            return Optional.ofNullable(clazz.cast(authorizeTime()));
        case "ClusterStatus":
            return Optional.ofNullable(clazz.cast(clusterStatus()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "AllowedAllVPCs":
            return Optional.ofNullable(clazz.cast(allowedAllVPCs()));
        case "AllowedVPCs":
            return Optional.ofNullable(clazz.cast(allowedVPCs()));
        case "EndpointCount":
            return Optional.ofNullable(clazz.cast(endpointCount()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Grantor", GRANTOR_FIELD);
        map.put("Grantee", GRANTEE_FIELD);
        map.put("ClusterIdentifier", CLUSTER_IDENTIFIER_FIELD);
        map.put("AuthorizeTime", AUTHORIZE_TIME_FIELD);
        map.put("ClusterStatus", CLUSTER_STATUS_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("AllowedAllVPCs", ALLOWED_ALL_VP_CS_FIELD);
        map.put("AllowedVPCs", ALLOWED_VP_CS_FIELD);
        map.put("EndpointCount", ENDPOINT_COUNT_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AuthorizeEndpointAccessResponse, T> g) {
        return obj -> g.apply((AuthorizeEndpointAccessResponse) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends RedshiftResponse.Builder, SdkPojo, CopyableBuilder<Builder, AuthorizeEndpointAccessResponse> {
        /**
         * <p>
         * The Amazon Web Services account ID of the cluster owner.
         * </p>
         * 
         * @param grantor
         *        The Amazon Web Services account ID of the cluster owner.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder grantor(String grantor);

        /**
         * <p>
         * The Amazon Web Services account ID of the grantee of the cluster.
         * </p>
         * 
         * @param grantee
         *        The Amazon Web Services account ID of the grantee of the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder grantee(String grantee);

        /**
         * <p>
         * The cluster identifier.
         * </p>
         * 
         * @param clusterIdentifier
         *        The cluster identifier.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clusterIdentifier(String clusterIdentifier);

        /**
         * <p>
         * The time (UTC) when the authorization was created.
         * </p>
         * 
         * @param authorizeTime
         *        The time (UTC) when the authorization was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authorizeTime(Instant authorizeTime);

        /**
         * <p>
         * The status of the cluster.
         * </p>
         * 
         * @param clusterStatus
         *        The status of the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clusterStatus(String clusterStatus);

        /**
         * <p>
         * The status of the authorization action.
         * </p>
         * 
         * @param status
         *        The status of the authorization action.
         * @see AuthorizationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AuthorizationStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the authorization action.
         * </p>
         * 
         * @param status
         *        The status of the authorization action.
         * @see AuthorizationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AuthorizationStatus
         */
        Builder status(AuthorizationStatus status);

        /**
         * <p>
         * Indicates whether all VPCs in the grantee account are allowed access to the cluster.
         * </p>
         * 
         * @param allowedAllVPCs
         *        Indicates whether all VPCs in the grantee account are allowed access to the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedAllVPCs(Boolean allowedAllVPCs);

        /**
         * <p>
         * The VPCs allowed access to the cluster.
         * </p>
         * 
         * @param allowedVPCs
         *        The VPCs allowed access to the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedVPCs(Collection<String> allowedVPCs);

        /**
         * <p>
         * The VPCs allowed access to the cluster.
         * </p>
         * 
         * @param allowedVPCs
         *        The VPCs allowed access to the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedVPCs(String... allowedVPCs);

        /**
         * <p>
         * The number of Redshift-managed VPC endpoints created for the authorization.
         * </p>
         * 
         * @param endpointCount
         *        The number of Redshift-managed VPC endpoints created for the authorization.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointCount(Integer endpointCount);
    }

    static final class BuilderImpl extends RedshiftResponse.BuilderImpl implements Builder {
        private String grantor;

        private String grantee;

        private String clusterIdentifier;

        private Instant authorizeTime;

        private String clusterStatus;

        private String status;

        private Boolean allowedAllVPCs;

        private List<String> allowedVPCs = DefaultSdkAutoConstructList.getInstance();

        private Integer endpointCount;

        private BuilderImpl() {
        }

        private BuilderImpl(AuthorizeEndpointAccessResponse model) {
            super(model);
            grantor(model.grantor);
            grantee(model.grantee);
            clusterIdentifier(model.clusterIdentifier);
            authorizeTime(model.authorizeTime);
            clusterStatus(model.clusterStatus);
            status(model.status);
            allowedAllVPCs(model.allowedAllVPCs);
            allowedVPCs(model.allowedVPCs);
            endpointCount(model.endpointCount);
        }

        public final String getGrantor() {
            return grantor;
        }

        public final void setGrantor(String grantor) {
            this.grantor = grantor;
        }

        @Override
        public final Builder grantor(String grantor) {
            this.grantor = grantor;
            return this;
        }

        public final String getGrantee() {
            return grantee;
        }

        public final void setGrantee(String grantee) {
            this.grantee = grantee;
        }

        @Override
        public final Builder grantee(String grantee) {
            this.grantee = grantee;
            return this;
        }

        public final String getClusterIdentifier() {
            return clusterIdentifier;
        }

        public final void setClusterIdentifier(String clusterIdentifier) {
            this.clusterIdentifier = clusterIdentifier;
        }

        @Override
        public final Builder clusterIdentifier(String clusterIdentifier) {
            this.clusterIdentifier = clusterIdentifier;
            return this;
        }

        public final Instant getAuthorizeTime() {
            return authorizeTime;
        }

        public final void setAuthorizeTime(Instant authorizeTime) {
            this.authorizeTime = authorizeTime;
        }

        @Override
        public final Builder authorizeTime(Instant authorizeTime) {
            this.authorizeTime = authorizeTime;
            return this;
        }

        public final String getClusterStatus() {
            return clusterStatus;
        }

        public final void setClusterStatus(String clusterStatus) {
            this.clusterStatus = clusterStatus;
        }

        @Override
        public final Builder clusterStatus(String clusterStatus) {
            this.clusterStatus = clusterStatus;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(AuthorizationStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Boolean getAllowedAllVPCs() {
            return allowedAllVPCs;
        }

        public final void setAllowedAllVPCs(Boolean allowedAllVPCs) {
            this.allowedAllVPCs = allowedAllVPCs;
        }

        @Override
        public final Builder allowedAllVPCs(Boolean allowedAllVPCs) {
            this.allowedAllVPCs = allowedAllVPCs;
            return this;
        }

        public final Collection<String> getAllowedVPCs() {
            if (allowedVPCs instanceof SdkAutoConstructList) {
                return null;
            }
            return allowedVPCs;
        }

        public final void setAllowedVPCs(Collection<String> allowedVPCs) {
            this.allowedVPCs = VpcIdentifierListCopier.copy(allowedVPCs);
        }

        @Override
        public final Builder allowedVPCs(Collection<String> allowedVPCs) {
            this.allowedVPCs = VpcIdentifierListCopier.copy(allowedVPCs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder allowedVPCs(String... allowedVPCs) {
            allowedVPCs(Arrays.asList(allowedVPCs));
            return this;
        }

        public final Integer getEndpointCount() {
            return endpointCount;
        }

        public final void setEndpointCount(Integer endpointCount) {
            this.endpointCount = endpointCount;
        }

        @Override
        public final Builder endpointCount(Integer endpointCount) {
            this.endpointCount = endpointCount;
            return this;
        }

        @Override
        public AuthorizeEndpointAccessResponse build() {
            return new AuthorizeEndpointAccessResponse(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
