/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.location.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the position update details for a device.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DevicePositionUpdate implements SdkPojo, Serializable,
        ToCopyableBuilder<DevicePositionUpdate.Builder, DevicePositionUpdate> {
    private static final SdkField<PositionalAccuracy> ACCURACY_FIELD = SdkField
            .<PositionalAccuracy> builder(MarshallingType.SDK_POJO).memberName("Accuracy")
            .getter(getter(DevicePositionUpdate::accuracy)).setter(setter(Builder::accuracy))
            .constructor(PositionalAccuracy::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Accuracy").build()).build();

    private static final SdkField<String> DEVICE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DeviceId").getter(getter(DevicePositionUpdate::deviceId)).setter(setter(Builder::deviceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeviceId").build()).build();

    private static final SdkField<List<Double>> POSITION_FIELD = SdkField
            .<List<Double>> builder(MarshallingType.LIST)
            .memberName("Position")
            .getter(getter(DevicePositionUpdate::position))
            .setter(setter(Builder::position))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Position").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Double> builder(MarshallingType.DOUBLE)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> POSITION_PROPERTIES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("PositionProperties")
            .getter(getter(DevicePositionUpdate::positionProperties))
            .setter(setter(Builder::positionProperties))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PositionProperties").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Instant> SAMPLE_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("SampleTime")
            .getter(getter(DevicePositionUpdate::sampleTime))
            .setter(setter(Builder::sampleTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SampleTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCURACY_FIELD,
            DEVICE_ID_FIELD, POSITION_FIELD, POSITION_PROPERTIES_FIELD, SAMPLE_TIME_FIELD));

    private static final long serialVersionUID = 1L;

    private final PositionalAccuracy accuracy;

    private final String deviceId;

    private final List<Double> position;

    private final Map<String, String> positionProperties;

    private final Instant sampleTime;

    private DevicePositionUpdate(BuilderImpl builder) {
        this.accuracy = builder.accuracy;
        this.deviceId = builder.deviceId;
        this.position = builder.position;
        this.positionProperties = builder.positionProperties;
        this.sampleTime = builder.sampleTime;
    }

    /**
     * <p>
     * The accuracy of the device position.
     * </p>
     * 
     * @return The accuracy of the device position.
     */
    public final PositionalAccuracy accuracy() {
        return accuracy;
    }

    /**
     * <p>
     * The device associated to the position update.
     * </p>
     * 
     * @return The device associated to the position update.
     */
    public final String deviceId() {
        return deviceId;
    }

    /**
     * For responses, this returns true if the service returned a value for the Position property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasPosition() {
        return position != null && !(position instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The latest device position defined in <a href="https://earth-info.nga.mil/GandG/wgs84/index.html">WGS 84</a>
     * format: <code>[X or longitude, Y or latitude]</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPosition} method.
     * </p>
     * 
     * @return The latest device position defined in <a href="https://earth-info.nga.mil/GandG/wgs84/index.html">WGS
     *         84</a> format: <code>[X or longitude, Y or latitude]</code>.
     */
    public final List<Double> position() {
        return position;
    }

    /**
     * For responses, this returns true if the service returned a value for the PositionProperties property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasPositionProperties() {
        return positionProperties != null && !(positionProperties instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Associates one of more properties with the position update. A property is a key-value pair stored with the
     * position update and added to any geofence event the update may trigger.
     * </p>
     * <p>
     * Format: <code>"key" : "value"</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPositionProperties} method.
     * </p>
     * 
     * @return Associates one of more properties with the position update. A property is a key-value pair stored with
     *         the position update and added to any geofence event the update may trigger.</p>
     *         <p>
     *         Format: <code>"key" : "value"</code>
     */
    public final Map<String, String> positionProperties() {
        return positionProperties;
    }

    /**
     * <p>
     * The timestamp at which the device's position was determined. Uses <a
     * href="https://www.iso.org/iso-8601-date-and-time-format.html">ISO 8601</a> format:
     * <code>YYYY-MM-DDThh:mm:ss.sssZ</code>
     * </p>
     * 
     * @return The timestamp at which the device's position was determined. Uses <a
     *         href="https://www.iso.org/iso-8601-date-and-time-format.html">ISO 8601</a> format:
     *         <code>YYYY-MM-DDThh:mm:ss.sssZ</code>
     */
    public final Instant sampleTime() {
        return sampleTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(accuracy());
        hashCode = 31 * hashCode + Objects.hashCode(deviceId());
        hashCode = 31 * hashCode + Objects.hashCode(hasPosition() ? position() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasPositionProperties() ? positionProperties() : null);
        hashCode = 31 * hashCode + Objects.hashCode(sampleTime());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DevicePositionUpdate)) {
            return false;
        }
        DevicePositionUpdate other = (DevicePositionUpdate) obj;
        return Objects.equals(accuracy(), other.accuracy()) && Objects.equals(deviceId(), other.deviceId())
                && hasPosition() == other.hasPosition() && Objects.equals(position(), other.position())
                && hasPositionProperties() == other.hasPositionProperties()
                && Objects.equals(positionProperties(), other.positionProperties())
                && Objects.equals(sampleTime(), other.sampleTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DevicePositionUpdate").add("Accuracy", accuracy()).add("DeviceId", deviceId())
                .add("Position", position() == null ? null : "*** Sensitive Data Redacted ***")
                .add("PositionProperties", positionProperties() == null ? null : "*** Sensitive Data Redacted ***")
                .add("SampleTime", sampleTime()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Accuracy":
            return Optional.ofNullable(clazz.cast(accuracy()));
        case "DeviceId":
            return Optional.ofNullable(clazz.cast(deviceId()));
        case "Position":
            return Optional.ofNullable(clazz.cast(position()));
        case "PositionProperties":
            return Optional.ofNullable(clazz.cast(positionProperties()));
        case "SampleTime":
            return Optional.ofNullable(clazz.cast(sampleTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DevicePositionUpdate, T> g) {
        return obj -> g.apply((DevicePositionUpdate) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DevicePositionUpdate> {
        /**
         * <p>
         * The accuracy of the device position.
         * </p>
         * 
         * @param accuracy
         *        The accuracy of the device position.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accuracy(PositionalAccuracy accuracy);

        /**
         * <p>
         * The accuracy of the device position.
         * </p>
         * This is a convenience that creates an instance of the {@link PositionalAccuracy.Builder} avoiding the need to
         * create one manually via {@link PositionalAccuracy#builder()}.
         *
         * When the {@link Consumer} completes, {@link PositionalAccuracy.Builder#build()} is called immediately and its
         * result is passed to {@link #accuracy(PositionalAccuracy)}.
         * 
         * @param accuracy
         *        a consumer that will call methods on {@link PositionalAccuracy.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #accuracy(PositionalAccuracy)
         */
        default Builder accuracy(Consumer<PositionalAccuracy.Builder> accuracy) {
            return accuracy(PositionalAccuracy.builder().applyMutation(accuracy).build());
        }

        /**
         * <p>
         * The device associated to the position update.
         * </p>
         * 
         * @param deviceId
         *        The device associated to the position update.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deviceId(String deviceId);

        /**
         * <p>
         * The latest device position defined in <a href="https://earth-info.nga.mil/GandG/wgs84/index.html">WGS 84</a>
         * format: <code>[X or longitude, Y or latitude]</code>.
         * </p>
         * 
         * @param position
         *        The latest device position defined in <a href="https://earth-info.nga.mil/GandG/wgs84/index.html">WGS
         *        84</a> format: <code>[X or longitude, Y or latitude]</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder position(Collection<Double> position);

        /**
         * <p>
         * The latest device position defined in <a href="https://earth-info.nga.mil/GandG/wgs84/index.html">WGS 84</a>
         * format: <code>[X or longitude, Y or latitude]</code>.
         * </p>
         * 
         * @param position
         *        The latest device position defined in <a href="https://earth-info.nga.mil/GandG/wgs84/index.html">WGS
         *        84</a> format: <code>[X or longitude, Y or latitude]</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder position(Double... position);

        /**
         * <p>
         * Associates one of more properties with the position update. A property is a key-value pair stored with the
         * position update and added to any geofence event the update may trigger.
         * </p>
         * <p>
         * Format: <code>"key" : "value"</code>
         * </p>
         * 
         * @param positionProperties
         *        Associates one of more properties with the position update. A property is a key-value pair stored with
         *        the position update and added to any geofence event the update may trigger.</p>
         *        <p>
         *        Format: <code>"key" : "value"</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder positionProperties(Map<String, String> positionProperties);

        /**
         * <p>
         * The timestamp at which the device's position was determined. Uses <a
         * href="https://www.iso.org/iso-8601-date-and-time-format.html">ISO 8601</a> format:
         * <code>YYYY-MM-DDThh:mm:ss.sssZ</code>
         * </p>
         * 
         * @param sampleTime
         *        The timestamp at which the device's position was determined. Uses <a
         *        href="https://www.iso.org/iso-8601-date-and-time-format.html">ISO 8601</a> format:
         *        <code>YYYY-MM-DDThh:mm:ss.sssZ</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sampleTime(Instant sampleTime);
    }

    static final class BuilderImpl implements Builder {
        private PositionalAccuracy accuracy;

        private String deviceId;

        private List<Double> position = DefaultSdkAutoConstructList.getInstance();

        private Map<String, String> positionProperties = DefaultSdkAutoConstructMap.getInstance();

        private Instant sampleTime;

        private BuilderImpl() {
        }

        private BuilderImpl(DevicePositionUpdate model) {
            accuracy(model.accuracy);
            deviceId(model.deviceId);
            position(model.position);
            positionProperties(model.positionProperties);
            sampleTime(model.sampleTime);
        }

        public final PositionalAccuracy.Builder getAccuracy() {
            return accuracy != null ? accuracy.toBuilder() : null;
        }

        public final void setAccuracy(PositionalAccuracy.BuilderImpl accuracy) {
            this.accuracy = accuracy != null ? accuracy.build() : null;
        }

        @Override
        @Transient
        public final Builder accuracy(PositionalAccuracy accuracy) {
            this.accuracy = accuracy;
            return this;
        }

        public final String getDeviceId() {
            return deviceId;
        }

        public final void setDeviceId(String deviceId) {
            this.deviceId = deviceId;
        }

        @Override
        @Transient
        public final Builder deviceId(String deviceId) {
            this.deviceId = deviceId;
            return this;
        }

        public final Collection<Double> getPosition() {
            if (position instanceof SdkAutoConstructList) {
                return null;
            }
            return position;
        }

        public final void setPosition(Collection<Double> position) {
            this.position = PositionCopier.copy(position);
        }

        @Override
        @Transient
        public final Builder position(Collection<Double> position) {
            this.position = PositionCopier.copy(position);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder position(Double... position) {
            position(Arrays.asList(position));
            return this;
        }

        public final Map<String, String> getPositionProperties() {
            if (positionProperties instanceof SdkAutoConstructMap) {
                return null;
            }
            return positionProperties;
        }

        public final void setPositionProperties(Map<String, String> positionProperties) {
            this.positionProperties = PropertyMapCopier.copy(positionProperties);
        }

        @Override
        @Transient
        public final Builder positionProperties(Map<String, String> positionProperties) {
            this.positionProperties = PropertyMapCopier.copy(positionProperties);
            return this;
        }

        public final Instant getSampleTime() {
            return sampleTime;
        }

        public final void setSampleTime(Instant sampleTime) {
            this.sampleTime = sampleTime;
        }

        @Override
        @Transient
        public final Builder sampleTime(Instant sampleTime) {
            this.sampleTime = sampleTime;
            return this;
        }

        @Override
        public DevicePositionUpdate build() {
            return new DevicePositionUpdate(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
