/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lexruntimev2.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The value of a slot.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Value implements SdkPojo, Serializable, ToCopyableBuilder<Value.Builder, Value> {
    private static final SdkField<String> ORIGINAL_VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("originalValue").getter(getter(Value::originalValue)).setter(setter(Builder::originalValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("originalValue").build()).build();

    private static final SdkField<String> INTERPRETED_VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("interpretedValue").getter(getter(Value::interpretedValue)).setter(setter(Builder::interpretedValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("interpretedValue").build()).build();

    private static final SdkField<List<String>> RESOLVED_VALUES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("resolvedValues")
            .getter(getter(Value::resolvedValues))
            .setter(setter(Builder::resolvedValues))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resolvedValues").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ORIGINAL_VALUE_FIELD,
            INTERPRETED_VALUE_FIELD, RESOLVED_VALUES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String originalValue;

    private final String interpretedValue;

    private final List<String> resolvedValues;

    private Value(BuilderImpl builder) {
        this.originalValue = builder.originalValue;
        this.interpretedValue = builder.interpretedValue;
        this.resolvedValues = builder.resolvedValues;
    }

    /**
     * <p>
     * The text of the utterance from the user that was entered for the slot.
     * </p>
     * 
     * @return The text of the utterance from the user that was entered for the slot.
     */
    public final String originalValue() {
        return originalValue;
    }

    /**
     * <p>
     * The value that Amazon Lex V2 determines for the slot. The actual value depends on the setting of the value
     * selection strategy for the bot. You can choose to use the value entered by the user, or you can have Amazon Lex
     * V2 choose the first value in the <code>resolvedValues</code> list.
     * </p>
     * 
     * @return The value that Amazon Lex V2 determines for the slot. The actual value depends on the setting of the
     *         value selection strategy for the bot. You can choose to use the value entered by the user, or you can
     *         have Amazon Lex V2 choose the first value in the <code>resolvedValues</code> list.
     */
    public final String interpretedValue() {
        return interpretedValue;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResolvedValues property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResolvedValues() {
        return resolvedValues != null && !(resolvedValues instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of additional values that have been recognized for the slot.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResolvedValues} method.
     * </p>
     * 
     * @return A list of additional values that have been recognized for the slot.
     */
    public final List<String> resolvedValues() {
        return resolvedValues;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(originalValue());
        hashCode = 31 * hashCode + Objects.hashCode(interpretedValue());
        hashCode = 31 * hashCode + Objects.hashCode(hasResolvedValues() ? resolvedValues() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Value)) {
            return false;
        }
        Value other = (Value) obj;
        return Objects.equals(originalValue(), other.originalValue())
                && Objects.equals(interpretedValue(), other.interpretedValue())
                && hasResolvedValues() == other.hasResolvedValues() && Objects.equals(resolvedValues(), other.resolvedValues());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Value").add("OriginalValue", originalValue()).add("InterpretedValue", interpretedValue())
                .add("ResolvedValues", hasResolvedValues() ? resolvedValues() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "originalValue":
            return Optional.ofNullable(clazz.cast(originalValue()));
        case "interpretedValue":
            return Optional.ofNullable(clazz.cast(interpretedValue()));
        case "resolvedValues":
            return Optional.ofNullable(clazz.cast(resolvedValues()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Value, T> g) {
        return obj -> g.apply((Value) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Value> {
        /**
         * <p>
         * The text of the utterance from the user that was entered for the slot.
         * </p>
         * 
         * @param originalValue
         *        The text of the utterance from the user that was entered for the slot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder originalValue(String originalValue);

        /**
         * <p>
         * The value that Amazon Lex V2 determines for the slot. The actual value depends on the setting of the value
         * selection strategy for the bot. You can choose to use the value entered by the user, or you can have Amazon
         * Lex V2 choose the first value in the <code>resolvedValues</code> list.
         * </p>
         * 
         * @param interpretedValue
         *        The value that Amazon Lex V2 determines for the slot. The actual value depends on the setting of the
         *        value selection strategy for the bot. You can choose to use the value entered by the user, or you can
         *        have Amazon Lex V2 choose the first value in the <code>resolvedValues</code> list.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder interpretedValue(String interpretedValue);

        /**
         * <p>
         * A list of additional values that have been recognized for the slot.
         * </p>
         * 
         * @param resolvedValues
         *        A list of additional values that have been recognized for the slot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resolvedValues(Collection<String> resolvedValues);

        /**
         * <p>
         * A list of additional values that have been recognized for the slot.
         * </p>
         * 
         * @param resolvedValues
         *        A list of additional values that have been recognized for the slot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resolvedValues(String... resolvedValues);
    }

    static final class BuilderImpl implements Builder {
        private String originalValue;

        private String interpretedValue;

        private List<String> resolvedValues = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Value model) {
            originalValue(model.originalValue);
            interpretedValue(model.interpretedValue);
            resolvedValues(model.resolvedValues);
        }

        public final String getOriginalValue() {
            return originalValue;
        }

        public final void setOriginalValue(String originalValue) {
            this.originalValue = originalValue;
        }

        @Override
        @Transient
        public final Builder originalValue(String originalValue) {
            this.originalValue = originalValue;
            return this;
        }

        public final String getInterpretedValue() {
            return interpretedValue;
        }

        public final void setInterpretedValue(String interpretedValue) {
            this.interpretedValue = interpretedValue;
        }

        @Override
        @Transient
        public final Builder interpretedValue(String interpretedValue) {
            this.interpretedValue = interpretedValue;
            return this;
        }

        public final Collection<String> getResolvedValues() {
            if (resolvedValues instanceof SdkAutoConstructList) {
                return null;
            }
            return resolvedValues;
        }

        public final void setResolvedValues(Collection<String> resolvedValues) {
            this.resolvedValues = StringListCopier.copy(resolvedValues);
        }

        @Override
        @Transient
        public final Builder resolvedValues(Collection<String> resolvedValues) {
            this.resolvedValues = StringListCopier.copy(resolvedValues);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder resolvedValues(String... resolvedValues) {
            resolvedValues(Arrays.asList(resolvedValues));
            return this;
        }

        @Override
        public Value build() {
            return new Value(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
