/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotevents.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Defines an action to write to the Amazon DynamoDB table that you created. The standard action payload contains all
 * the information about the detector model instance and the event that triggered the action. You can customize the <a
 * href="https://docs.aws.amazon.com/iotevents/latest/apireference/API_Payload.html">payload</a>. One column of the
 * DynamoDB table receives all attribute-value pairs in the payload that you specify.
 * </p>
 * <p>
 * You must use expressions for all parameters in <code>DynamoDBAction</code>. The expressions accept literals,
 * operators, functions, references, and substitution templates.
 * </p>
 * <p class="title">
 * <b>Examples</b>
 * </p>
 * <ul>
 * <li>
 * <p>
 * For literal values, the expressions must contain single quotes. For example, the value for the
 * <code>hashKeyType</code> parameter can be <code>'STRING'</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * For references, you must specify either variables or input values. For example, the value for the
 * <code>hashKeyField</code> parameter can be <code>$input.GreenhouseInput.name</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * For a substitution template, you must use <code>${}</code>, and the template must be in single quotes. A substitution
 * template can also contain a combination of literals, operators, functions, references, and substitution templates.
 * </p>
 * <p>
 * In the following example, the value for the <code>hashKeyValue</code> parameter uses a substitution template.
 * </p>
 * <p>
 * <code>'${$input.GreenhouseInput.temperature * 6 / 5 + 32} in Fahrenheit'</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * For a string concatenation, you must use <code>+</code>. A string concatenation can also contain a combination of
 * literals, operators, functions, references, and substitution templates.
 * </p>
 * <p>
 * In the following example, the value for the <code>tableName</code> parameter uses a string concatenation.
 * </p>
 * <p>
 * <code>'GreenhouseTemperatureTable ' + $input.GreenhouseInput.date</code>
 * </p>
 * </li>
 * </ul>
 * <p>
 * For more information, see <a
 * href="https://docs.aws.amazon.com/iotevents/latest/developerguide/iotevents-expressions.html">Expressions</a> in the
 * <i>AWS IoT Events Developer Guide</i>.
 * </p>
 * <p>
 * If the defined payload type is a string, <code>DynamoDBAction</code> writes non-JSON data to the DynamoDB table as
 * binary data. The DynamoDB console displays the data as Base64-encoded text. The value for the
 * <code>payloadField</code> parameter is <code>&lt;payload-field&gt;_raw</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DynamoDBAction implements SdkPojo, Serializable, ToCopyableBuilder<DynamoDBAction.Builder, DynamoDBAction> {
    private static final SdkField<String> HASH_KEY_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("hashKeyType").getter(getter(DynamoDBAction::hashKeyType)).setter(setter(Builder::hashKeyType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("hashKeyType").build()).build();

    private static final SdkField<String> HASH_KEY_FIELD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("hashKeyField").getter(getter(DynamoDBAction::hashKeyField)).setter(setter(Builder::hashKeyField))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("hashKeyField").build()).build();

    private static final SdkField<String> HASH_KEY_VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("hashKeyValue").getter(getter(DynamoDBAction::hashKeyValue)).setter(setter(Builder::hashKeyValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("hashKeyValue").build()).build();

    private static final SdkField<String> RANGE_KEY_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("rangeKeyType").getter(getter(DynamoDBAction::rangeKeyType)).setter(setter(Builder::rangeKeyType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("rangeKeyType").build()).build();

    private static final SdkField<String> RANGE_KEY_FIELD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("rangeKeyField").getter(getter(DynamoDBAction::rangeKeyField)).setter(setter(Builder::rangeKeyField))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("rangeKeyField").build()).build();

    private static final SdkField<String> RANGE_KEY_VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("rangeKeyValue").getter(getter(DynamoDBAction::rangeKeyValue)).setter(setter(Builder::rangeKeyValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("rangeKeyValue").build()).build();

    private static final SdkField<String> OPERATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("operation").getter(getter(DynamoDBAction::operation)).setter(setter(Builder::operation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("operation").build()).build();

    private static final SdkField<String> PAYLOAD_FIELD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("payloadField").getter(getter(DynamoDBAction::payloadField)).setter(setter(Builder::payloadField))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("payloadField").build()).build();

    private static final SdkField<String> TABLE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("tableName").getter(getter(DynamoDBAction::tableName)).setter(setter(Builder::tableName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tableName").build()).build();

    private static final SdkField<Payload> PAYLOAD_FIELD = SdkField.<Payload> builder(MarshallingType.SDK_POJO)
            .memberName("payload").getter(getter(DynamoDBAction::payload)).setter(setter(Builder::payload))
            .constructor(Payload::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("payload").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(HASH_KEY_TYPE_FIELD,
            HASH_KEY_FIELD_FIELD, HASH_KEY_VALUE_FIELD, RANGE_KEY_TYPE_FIELD, RANGE_KEY_FIELD_FIELD, RANGE_KEY_VALUE_FIELD,
            OPERATION_FIELD, PAYLOAD_FIELD_FIELD, TABLE_NAME_FIELD, PAYLOAD_FIELD));

    private static final long serialVersionUID = 1L;

    private final String hashKeyType;

    private final String hashKeyField;

    private final String hashKeyValue;

    private final String rangeKeyType;

    private final String rangeKeyField;

    private final String rangeKeyValue;

    private final String operation;

    private final String payloadField;

    private final String tableName;

    private final Payload payload;

    private DynamoDBAction(BuilderImpl builder) {
        this.hashKeyType = builder.hashKeyType;
        this.hashKeyField = builder.hashKeyField;
        this.hashKeyValue = builder.hashKeyValue;
        this.rangeKeyType = builder.rangeKeyType;
        this.rangeKeyField = builder.rangeKeyField;
        this.rangeKeyValue = builder.rangeKeyValue;
        this.operation = builder.operation;
        this.payloadField = builder.payloadField;
        this.tableName = builder.tableName;
        this.payload = builder.payload;
    }

    /**
     * <p>
     * The data type for the hash key (also called the partition key). You can specify the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>'STRING'</code> - The hash key is a string.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>'NUMBER'</code> - The hash key is a number.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you don't specify <code>hashKeyType</code>, the default value is <code>'STRING'</code>.
     * </p>
     * 
     * @return The data type for the hash key (also called the partition key). You can specify the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>'STRING'</code> - The hash key is a string.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>'NUMBER'</code> - The hash key is a number.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you don't specify <code>hashKeyType</code>, the default value is <code>'STRING'</code>.
     */
    public final String hashKeyType() {
        return hashKeyType;
    }

    /**
     * <p>
     * The name of the hash key (also called the partition key). The <code>hashKeyField</code> value must match the
     * partition key of the target DynamoDB table.
     * </p>
     * 
     * @return The name of the hash key (also called the partition key). The <code>hashKeyField</code> value must match
     *         the partition key of the target DynamoDB table.
     */
    public final String hashKeyField() {
        return hashKeyField;
    }

    /**
     * <p>
     * The value of the hash key (also called the partition key).
     * </p>
     * 
     * @return The value of the hash key (also called the partition key).
     */
    public final String hashKeyValue() {
        return hashKeyValue;
    }

    /**
     * <p>
     * The data type for the range key (also called the sort key), You can specify the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>'STRING'</code> - The range key is a string.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>'NUMBER'</code> - The range key is number.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you don't specify <code>rangeKeyField</code>, the default value is <code>'STRING'</code>.
     * </p>
     * 
     * @return The data type for the range key (also called the sort key), You can specify the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>'STRING'</code> - The range key is a string.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>'NUMBER'</code> - The range key is number.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you don't specify <code>rangeKeyField</code>, the default value is <code>'STRING'</code>.
     */
    public final String rangeKeyType() {
        return rangeKeyType;
    }

    /**
     * <p>
     * The name of the range key (also called the sort key). The <code>rangeKeyField</code> value must match the sort
     * key of the target DynamoDB table.
     * </p>
     * 
     * @return The name of the range key (also called the sort key). The <code>rangeKeyField</code> value must match the
     *         sort key of the target DynamoDB table.
     */
    public final String rangeKeyField() {
        return rangeKeyField;
    }

    /**
     * <p>
     * The value of the range key (also called the sort key).
     * </p>
     * 
     * @return The value of the range key (also called the sort key).
     */
    public final String rangeKeyValue() {
        return rangeKeyValue;
    }

    /**
     * <p>
     * The type of operation to perform. You can specify the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>'INSERT'</code> - Insert data as a new item into the DynamoDB table. This item uses the specified hash key
     * as a partition key. If you specified a range key, the item uses the range key as a sort key.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>'UPDATE'</code> - Update an existing item of the DynamoDB table with new data. This item's partition key
     * must match the specified hash key. If you specified a range key, the range key must match the item's sort key.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>'DELETE'</code> - Delete an existing item of the DynamoDB table. This item's partition key must match the
     * specified hash key. If you specified a range key, the range key must match the item's sort key.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you don't specify this parameter, AWS IoT Events triggers the <code>'INSERT'</code> operation.
     * </p>
     * 
     * @return The type of operation to perform. You can specify the following values: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>'INSERT'</code> - Insert data as a new item into the DynamoDB table. This item uses the specified
     *         hash key as a partition key. If you specified a range key, the item uses the range key as a sort key.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>'UPDATE'</code> - Update an existing item of the DynamoDB table with new data. This item's
     *         partition key must match the specified hash key. If you specified a range key, the range key must match
     *         the item's sort key.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>'DELETE'</code> - Delete an existing item of the DynamoDB table. This item's partition key must
     *         match the specified hash key. If you specified a range key, the range key must match the item's sort key.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you don't specify this parameter, AWS IoT Events triggers the <code>'INSERT'</code> operation.
     */
    public final String operation() {
        return operation;
    }

    /**
     * <p>
     * The name of the DynamoDB column that receives the action payload.
     * </p>
     * <p>
     * If you don't specify this parameter, the name of the DynamoDB column is <code>payload</code>.
     * </p>
     * 
     * @return The name of the DynamoDB column that receives the action payload.</p>
     *         <p>
     *         If you don't specify this parameter, the name of the DynamoDB column is <code>payload</code>.
     */
    public final String payloadField() {
        return payloadField;
    }

    /**
     * <p>
     * The name of the DynamoDB table. The <code>tableName</code> value must match the table name of the target DynamoDB
     * table.
     * </p>
     * 
     * @return The name of the DynamoDB table. The <code>tableName</code> value must match the table name of the target
     *         DynamoDB table.
     */
    public final String tableName() {
        return tableName;
    }

    /**
     * Returns the value of the Payload property for this object.
     * 
     * @return The value of the Payload property for this object.
     */
    public final Payload payload() {
        return payload;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hashKeyType());
        hashCode = 31 * hashCode + Objects.hashCode(hashKeyField());
        hashCode = 31 * hashCode + Objects.hashCode(hashKeyValue());
        hashCode = 31 * hashCode + Objects.hashCode(rangeKeyType());
        hashCode = 31 * hashCode + Objects.hashCode(rangeKeyField());
        hashCode = 31 * hashCode + Objects.hashCode(rangeKeyValue());
        hashCode = 31 * hashCode + Objects.hashCode(operation());
        hashCode = 31 * hashCode + Objects.hashCode(payloadField());
        hashCode = 31 * hashCode + Objects.hashCode(tableName());
        hashCode = 31 * hashCode + Objects.hashCode(payload());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DynamoDBAction)) {
            return false;
        }
        DynamoDBAction other = (DynamoDBAction) obj;
        return Objects.equals(hashKeyType(), other.hashKeyType()) && Objects.equals(hashKeyField(), other.hashKeyField())
                && Objects.equals(hashKeyValue(), other.hashKeyValue()) && Objects.equals(rangeKeyType(), other.rangeKeyType())
                && Objects.equals(rangeKeyField(), other.rangeKeyField())
                && Objects.equals(rangeKeyValue(), other.rangeKeyValue()) && Objects.equals(operation(), other.operation())
                && Objects.equals(payloadField(), other.payloadField()) && Objects.equals(tableName(), other.tableName())
                && Objects.equals(payload(), other.payload());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DynamoDBAction").add("HashKeyType", hashKeyType()).add("HashKeyField", hashKeyField())
                .add("HashKeyValue", hashKeyValue()).add("RangeKeyType", rangeKeyType()).add("RangeKeyField", rangeKeyField())
                .add("RangeKeyValue", rangeKeyValue()).add("Operation", operation()).add("PayloadField", payloadField())
                .add("TableName", tableName()).add("Payload", payload()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "hashKeyType":
            return Optional.ofNullable(clazz.cast(hashKeyType()));
        case "hashKeyField":
            return Optional.ofNullable(clazz.cast(hashKeyField()));
        case "hashKeyValue":
            return Optional.ofNullable(clazz.cast(hashKeyValue()));
        case "rangeKeyType":
            return Optional.ofNullable(clazz.cast(rangeKeyType()));
        case "rangeKeyField":
            return Optional.ofNullable(clazz.cast(rangeKeyField()));
        case "rangeKeyValue":
            return Optional.ofNullable(clazz.cast(rangeKeyValue()));
        case "operation":
            return Optional.ofNullable(clazz.cast(operation()));
        case "payloadField":
            return Optional.ofNullable(clazz.cast(payloadField()));
        case "tableName":
            return Optional.ofNullable(clazz.cast(tableName()));
        case "payload":
            return Optional.ofNullable(clazz.cast(payload()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DynamoDBAction, T> g) {
        return obj -> g.apply((DynamoDBAction) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DynamoDBAction> {
        /**
         * <p>
         * The data type for the hash key (also called the partition key). You can specify the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>'STRING'</code> - The hash key is a string.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>'NUMBER'</code> - The hash key is a number.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you don't specify <code>hashKeyType</code>, the default value is <code>'STRING'</code>.
         * </p>
         * 
         * @param hashKeyType
         *        The data type for the hash key (also called the partition key). You can specify the following
         *        values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>'STRING'</code> - The hash key is a string.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>'NUMBER'</code> - The hash key is a number.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you don't specify <code>hashKeyType</code>, the default value is <code>'STRING'</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hashKeyType(String hashKeyType);

        /**
         * <p>
         * The name of the hash key (also called the partition key). The <code>hashKeyField</code> value must match the
         * partition key of the target DynamoDB table.
         * </p>
         * 
         * @param hashKeyField
         *        The name of the hash key (also called the partition key). The <code>hashKeyField</code> value must
         *        match the partition key of the target DynamoDB table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hashKeyField(String hashKeyField);

        /**
         * <p>
         * The value of the hash key (also called the partition key).
         * </p>
         * 
         * @param hashKeyValue
         *        The value of the hash key (also called the partition key).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hashKeyValue(String hashKeyValue);

        /**
         * <p>
         * The data type for the range key (also called the sort key), You can specify the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>'STRING'</code> - The range key is a string.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>'NUMBER'</code> - The range key is number.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you don't specify <code>rangeKeyField</code>, the default value is <code>'STRING'</code>.
         * </p>
         * 
         * @param rangeKeyType
         *        The data type for the range key (also called the sort key), You can specify the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>'STRING'</code> - The range key is a string.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>'NUMBER'</code> - The range key is number.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you don't specify <code>rangeKeyField</code>, the default value is <code>'STRING'</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rangeKeyType(String rangeKeyType);

        /**
         * <p>
         * The name of the range key (also called the sort key). The <code>rangeKeyField</code> value must match the
         * sort key of the target DynamoDB table.
         * </p>
         * 
         * @param rangeKeyField
         *        The name of the range key (also called the sort key). The <code>rangeKeyField</code> value must match
         *        the sort key of the target DynamoDB table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rangeKeyField(String rangeKeyField);

        /**
         * <p>
         * The value of the range key (also called the sort key).
         * </p>
         * 
         * @param rangeKeyValue
         *        The value of the range key (also called the sort key).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rangeKeyValue(String rangeKeyValue);

        /**
         * <p>
         * The type of operation to perform. You can specify the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>'INSERT'</code> - Insert data as a new item into the DynamoDB table. This item uses the specified hash
         * key as a partition key. If you specified a range key, the item uses the range key as a sort key.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>'UPDATE'</code> - Update an existing item of the DynamoDB table with new data. This item's partition
         * key must match the specified hash key. If you specified a range key, the range key must match the item's sort
         * key.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>'DELETE'</code> - Delete an existing item of the DynamoDB table. This item's partition key must match
         * the specified hash key. If you specified a range key, the range key must match the item's sort key.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you don't specify this parameter, AWS IoT Events triggers the <code>'INSERT'</code> operation.
         * </p>
         * 
         * @param operation
         *        The type of operation to perform. You can specify the following values: </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>'INSERT'</code> - Insert data as a new item into the DynamoDB table. This item uses the
         *        specified hash key as a partition key. If you specified a range key, the item uses the range key as a
         *        sort key.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>'UPDATE'</code> - Update an existing item of the DynamoDB table with new data. This item's
         *        partition key must match the specified hash key. If you specified a range key, the range key must
         *        match the item's sort key.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>'DELETE'</code> - Delete an existing item of the DynamoDB table. This item's partition key must
         *        match the specified hash key. If you specified a range key, the range key must match the item's sort
         *        key.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you don't specify this parameter, AWS IoT Events triggers the <code>'INSERT'</code> operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operation(String operation);

        /**
         * <p>
         * The name of the DynamoDB column that receives the action payload.
         * </p>
         * <p>
         * If you don't specify this parameter, the name of the DynamoDB column is <code>payload</code>.
         * </p>
         * 
         * @param payloadField
         *        The name of the DynamoDB column that receives the action payload.</p>
         *        <p>
         *        If you don't specify this parameter, the name of the DynamoDB column is <code>payload</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder payloadField(String payloadField);

        /**
         * <p>
         * The name of the DynamoDB table. The <code>tableName</code> value must match the table name of the target
         * DynamoDB table.
         * </p>
         * 
         * @param tableName
         *        The name of the DynamoDB table. The <code>tableName</code> value must match the table name of the
         *        target DynamoDB table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tableName(String tableName);

        /**
         * Sets the value of the Payload property for this object.
         *
         * @param payload
         *        The new value for the Payload property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder payload(Payload payload);

        /**
         * Sets the value of the Payload property for this object.
         *
         * This is a convenience that creates an instance of the {@link Payload.Builder} avoiding the need to create one
         * manually via {@link Payload#builder()}.
         *
         * When the {@link Consumer} completes, {@link Payload.Builder#build()} is called immediately and its result is
         * passed to {@link #payload(Payload)}.
         * 
         * @param payload
         *        a consumer that will call methods on {@link Payload.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #payload(Payload)
         */
        default Builder payload(Consumer<Payload.Builder> payload) {
            return payload(Payload.builder().applyMutation(payload).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String hashKeyType;

        private String hashKeyField;

        private String hashKeyValue;

        private String rangeKeyType;

        private String rangeKeyField;

        private String rangeKeyValue;

        private String operation;

        private String payloadField;

        private String tableName;

        private Payload payload;

        private BuilderImpl() {
        }

        private BuilderImpl(DynamoDBAction model) {
            hashKeyType(model.hashKeyType);
            hashKeyField(model.hashKeyField);
            hashKeyValue(model.hashKeyValue);
            rangeKeyType(model.rangeKeyType);
            rangeKeyField(model.rangeKeyField);
            rangeKeyValue(model.rangeKeyValue);
            operation(model.operation);
            payloadField(model.payloadField);
            tableName(model.tableName);
            payload(model.payload);
        }

        public final String getHashKeyType() {
            return hashKeyType;
        }

        public final void setHashKeyType(String hashKeyType) {
            this.hashKeyType = hashKeyType;
        }

        @Override
        @Transient
        public final Builder hashKeyType(String hashKeyType) {
            this.hashKeyType = hashKeyType;
            return this;
        }

        public final String getHashKeyField() {
            return hashKeyField;
        }

        public final void setHashKeyField(String hashKeyField) {
            this.hashKeyField = hashKeyField;
        }

        @Override
        @Transient
        public final Builder hashKeyField(String hashKeyField) {
            this.hashKeyField = hashKeyField;
            return this;
        }

        public final String getHashKeyValue() {
            return hashKeyValue;
        }

        public final void setHashKeyValue(String hashKeyValue) {
            this.hashKeyValue = hashKeyValue;
        }

        @Override
        @Transient
        public final Builder hashKeyValue(String hashKeyValue) {
            this.hashKeyValue = hashKeyValue;
            return this;
        }

        public final String getRangeKeyType() {
            return rangeKeyType;
        }

        public final void setRangeKeyType(String rangeKeyType) {
            this.rangeKeyType = rangeKeyType;
        }

        @Override
        @Transient
        public final Builder rangeKeyType(String rangeKeyType) {
            this.rangeKeyType = rangeKeyType;
            return this;
        }

        public final String getRangeKeyField() {
            return rangeKeyField;
        }

        public final void setRangeKeyField(String rangeKeyField) {
            this.rangeKeyField = rangeKeyField;
        }

        @Override
        @Transient
        public final Builder rangeKeyField(String rangeKeyField) {
            this.rangeKeyField = rangeKeyField;
            return this;
        }

        public final String getRangeKeyValue() {
            return rangeKeyValue;
        }

        public final void setRangeKeyValue(String rangeKeyValue) {
            this.rangeKeyValue = rangeKeyValue;
        }

        @Override
        @Transient
        public final Builder rangeKeyValue(String rangeKeyValue) {
            this.rangeKeyValue = rangeKeyValue;
            return this;
        }

        public final String getOperation() {
            return operation;
        }

        public final void setOperation(String operation) {
            this.operation = operation;
        }

        @Override
        @Transient
        public final Builder operation(String operation) {
            this.operation = operation;
            return this;
        }

        public final String getPayloadField() {
            return payloadField;
        }

        public final void setPayloadField(String payloadField) {
            this.payloadField = payloadField;
        }

        @Override
        @Transient
        public final Builder payloadField(String payloadField) {
            this.payloadField = payloadField;
            return this;
        }

        public final String getTableName() {
            return tableName;
        }

        public final void setTableName(String tableName) {
            this.tableName = tableName;
        }

        @Override
        @Transient
        public final Builder tableName(String tableName) {
            this.tableName = tableName;
            return this;
        }

        public final Payload.Builder getPayload() {
            return payload != null ? payload.toBuilder() : null;
        }

        public final void setPayload(Payload.BuilderImpl payload) {
            this.payload = payload != null ? payload.build() : null;
        }

        @Override
        @Transient
        public final Builder payload(Payload payload) {
            this.payload = payload;
            return this;
        }

        @Override
        public DynamoDBAction build() {
            return new DynamoDBAction(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
