/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iot.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Send data to an HTTPS endpoint.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class HttpAction implements SdkPojo, Serializable, ToCopyableBuilder<HttpAction.Builder, HttpAction> {
    private static final SdkField<String> URL_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("url")
            .getter(getter(HttpAction::url)).setter(setter(Builder::url))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("url").build()).build();

    private static final SdkField<String> CONFIRMATION_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("confirmationUrl").getter(getter(HttpAction::confirmationUrl)).setter(setter(Builder::confirmationUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("confirmationUrl").build()).build();

    private static final SdkField<List<HttpActionHeader>> HEADERS_FIELD = SdkField
            .<List<HttpActionHeader>> builder(MarshallingType.LIST)
            .memberName("headers")
            .getter(getter(HttpAction::headers))
            .setter(setter(Builder::headers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("headers").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<HttpActionHeader> builder(MarshallingType.SDK_POJO)
                                            .constructor(HttpActionHeader::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<HttpAuthorization> AUTH_FIELD = SdkField.<HttpAuthorization> builder(MarshallingType.SDK_POJO)
            .memberName("auth").getter(getter(HttpAction::auth)).setter(setter(Builder::auth))
            .constructor(HttpAuthorization::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("auth").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(URL_FIELD,
            CONFIRMATION_URL_FIELD, HEADERS_FIELD, AUTH_FIELD));

    private static final long serialVersionUID = 1L;

    private final String url;

    private final String confirmationUrl;

    private final List<HttpActionHeader> headers;

    private final HttpAuthorization auth;

    private HttpAction(BuilderImpl builder) {
        this.url = builder.url;
        this.confirmationUrl = builder.confirmationUrl;
        this.headers = builder.headers;
        this.auth = builder.auth;
    }

    /**
     * <p>
     * The endpoint URL. If substitution templates are used in the URL, you must also specify a
     * <code>confirmationUrl</code>. If this is a new destination, a new <code>TopicRuleDestination</code> is created if
     * possible.
     * </p>
     * 
     * @return The endpoint URL. If substitution templates are used in the URL, you must also specify a
     *         <code>confirmationUrl</code>. If this is a new destination, a new <code>TopicRuleDestination</code> is
     *         created if possible.
     */
    public final String url() {
        return url;
    }

    /**
     * <p>
     * The URL to which IoT sends a confirmation message. The value of the confirmation URL must be a prefix of the
     * endpoint URL. If you do not specify a confirmation URL IoT uses the endpoint URL as the confirmation URL. If you
     * use substitution templates in the confirmationUrl, you must create and enable topic rule destinations that match
     * each possible value of the substitution template before traffic is allowed to your endpoint URL.
     * </p>
     * 
     * @return The URL to which IoT sends a confirmation message. The value of the confirmation URL must be a prefix of
     *         the endpoint URL. If you do not specify a confirmation URL IoT uses the endpoint URL as the confirmation
     *         URL. If you use substitution templates in the confirmationUrl, you must create and enable topic rule
     *         destinations that match each possible value of the substitution template before traffic is allowed to
     *         your endpoint URL.
     */
    public final String confirmationUrl() {
        return confirmationUrl;
    }

    /**
     * For responses, this returns true if the service returned a value for the Headers property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasHeaders() {
        return headers != null && !(headers instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The HTTP headers to send with the message data.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasHeaders} method.
     * </p>
     * 
     * @return The HTTP headers to send with the message data.
     */
    public final List<HttpActionHeader> headers() {
        return headers;
    }

    /**
     * <p>
     * The authentication method to use when sending data to an HTTPS endpoint.
     * </p>
     * 
     * @return The authentication method to use when sending data to an HTTPS endpoint.
     */
    public final HttpAuthorization auth() {
        return auth;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(url());
        hashCode = 31 * hashCode + Objects.hashCode(confirmationUrl());
        hashCode = 31 * hashCode + Objects.hashCode(hasHeaders() ? headers() : null);
        hashCode = 31 * hashCode + Objects.hashCode(auth());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof HttpAction)) {
            return false;
        }
        HttpAction other = (HttpAction) obj;
        return Objects.equals(url(), other.url()) && Objects.equals(confirmationUrl(), other.confirmationUrl())
                && hasHeaders() == other.hasHeaders() && Objects.equals(headers(), other.headers())
                && Objects.equals(auth(), other.auth());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("HttpAction").add("Url", url()).add("ConfirmationUrl", confirmationUrl())
                .add("Headers", hasHeaders() ? headers() : null).add("Auth", auth()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "url":
            return Optional.ofNullable(clazz.cast(url()));
        case "confirmationUrl":
            return Optional.ofNullable(clazz.cast(confirmationUrl()));
        case "headers":
            return Optional.ofNullable(clazz.cast(headers()));
        case "auth":
            return Optional.ofNullable(clazz.cast(auth()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<HttpAction, T> g) {
        return obj -> g.apply((HttpAction) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, HttpAction> {
        /**
         * <p>
         * The endpoint URL. If substitution templates are used in the URL, you must also specify a
         * <code>confirmationUrl</code>. If this is a new destination, a new <code>TopicRuleDestination</code> is
         * created if possible.
         * </p>
         * 
         * @param url
         *        The endpoint URL. If substitution templates are used in the URL, you must also specify a
         *        <code>confirmationUrl</code>. If this is a new destination, a new <code>TopicRuleDestination</code> is
         *        created if possible.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder url(String url);

        /**
         * <p>
         * The URL to which IoT sends a confirmation message. The value of the confirmation URL must be a prefix of the
         * endpoint URL. If you do not specify a confirmation URL IoT uses the endpoint URL as the confirmation URL. If
         * you use substitution templates in the confirmationUrl, you must create and enable topic rule destinations
         * that match each possible value of the substitution template before traffic is allowed to your endpoint URL.
         * </p>
         * 
         * @param confirmationUrl
         *        The URL to which IoT sends a confirmation message. The value of the confirmation URL must be a prefix
         *        of the endpoint URL. If you do not specify a confirmation URL IoT uses the endpoint URL as the
         *        confirmation URL. If you use substitution templates in the confirmationUrl, you must create and enable
         *        topic rule destinations that match each possible value of the substitution template before traffic is
         *        allowed to your endpoint URL.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder confirmationUrl(String confirmationUrl);

        /**
         * <p>
         * The HTTP headers to send with the message data.
         * </p>
         * 
         * @param headers
         *        The HTTP headers to send with the message data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder headers(Collection<HttpActionHeader> headers);

        /**
         * <p>
         * The HTTP headers to send with the message data.
         * </p>
         * 
         * @param headers
         *        The HTTP headers to send with the message data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder headers(HttpActionHeader... headers);

        /**
         * <p>
         * The HTTP headers to send with the message data.
         * </p>
         * This is a convenience that creates an instance of the {@link List<HttpActionHeader>.Builder} avoiding the
         * need to create one manually via {@link List<HttpActionHeader>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<HttpActionHeader>.Builder#build()} is called immediately and
         * its result is passed to {@link #headers(List<HttpActionHeader>)}.
         * 
         * @param headers
         *        a consumer that will call methods on {@link List<HttpActionHeader>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #headers(List<HttpActionHeader>)
         */
        Builder headers(Consumer<HttpActionHeader.Builder>... headers);

        /**
         * <p>
         * The authentication method to use when sending data to an HTTPS endpoint.
         * </p>
         * 
         * @param auth
         *        The authentication method to use when sending data to an HTTPS endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder auth(HttpAuthorization auth);

        /**
         * <p>
         * The authentication method to use when sending data to an HTTPS endpoint.
         * </p>
         * This is a convenience that creates an instance of the {@link HttpAuthorization.Builder} avoiding the need to
         * create one manually via {@link HttpAuthorization#builder()}.
         *
         * When the {@link Consumer} completes, {@link HttpAuthorization.Builder#build()} is called immediately and its
         * result is passed to {@link #auth(HttpAuthorization)}.
         * 
         * @param auth
         *        a consumer that will call methods on {@link HttpAuthorization.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #auth(HttpAuthorization)
         */
        default Builder auth(Consumer<HttpAuthorization.Builder> auth) {
            return auth(HttpAuthorization.builder().applyMutation(auth).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String url;

        private String confirmationUrl;

        private List<HttpActionHeader> headers = DefaultSdkAutoConstructList.getInstance();

        private HttpAuthorization auth;

        private BuilderImpl() {
        }

        private BuilderImpl(HttpAction model) {
            url(model.url);
            confirmationUrl(model.confirmationUrl);
            headers(model.headers);
            auth(model.auth);
        }

        public final String getUrl() {
            return url;
        }

        public final void setUrl(String url) {
            this.url = url;
        }

        @Override
        @Transient
        public final Builder url(String url) {
            this.url = url;
            return this;
        }

        public final String getConfirmationUrl() {
            return confirmationUrl;
        }

        public final void setConfirmationUrl(String confirmationUrl) {
            this.confirmationUrl = confirmationUrl;
        }

        @Override
        @Transient
        public final Builder confirmationUrl(String confirmationUrl) {
            this.confirmationUrl = confirmationUrl;
            return this;
        }

        public final List<HttpActionHeader.Builder> getHeaders() {
            List<HttpActionHeader.Builder> result = HeaderListCopier.copyToBuilder(this.headers);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setHeaders(Collection<HttpActionHeader.BuilderImpl> headers) {
            this.headers = HeaderListCopier.copyFromBuilder(headers);
        }

        @Override
        @Transient
        public final Builder headers(Collection<HttpActionHeader> headers) {
            this.headers = HeaderListCopier.copy(headers);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder headers(HttpActionHeader... headers) {
            headers(Arrays.asList(headers));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder headers(Consumer<HttpActionHeader.Builder>... headers) {
            headers(Stream.of(headers).map(c -> HttpActionHeader.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final HttpAuthorization.Builder getAuth() {
            return auth != null ? auth.toBuilder() : null;
        }

        public final void setAuth(HttpAuthorization.BuilderImpl auth) {
            this.auth = auth != null ? auth.build() : null;
        }

        @Override
        @Transient
        public final Builder auth(HttpAuthorization auth) {
            this.auth = auth;
            return this;
        }

        @Override
        public HttpAction build() {
            return new HttpAction(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
