/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iam.endpoints.internal;

import java.net.URI;
import java.util.Arrays;
import java.util.concurrent.CompletableFuture;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.endpoints.AwsEndpointAttribute;
import software.amazon.awssdk.awscore.endpoints.authscheme.SigV4AuthScheme;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.endpoints.Endpoint;
import software.amazon.awssdk.regions.Region;
import software.amazon.awssdk.services.iam.endpoints.IamEndpointParams;
import software.amazon.awssdk.services.iam.endpoints.IamEndpointProvider;
import software.amazon.awssdk.utils.CompletableFutureUtils;
import software.amazon.awssdk.utils.Validate;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
public final class DefaultIamEndpointProvider implements IamEndpointProvider {
    @Override
    public CompletableFuture<Endpoint> resolveEndpoint(IamEndpointParams params) {
        Validate.notNull(params.useDualStack(), "Parameter 'UseDualStack' must not be null");
        Validate.notNull(params.useFips(), "Parameter 'UseFIPS' must not be null");
        try {
            RuleResult result = endpointRule0(params, new LocalState(params.region()));
            if (result.canContinue()) {
                throw SdkClientException.create("Rule engine did not reach an error or endpoint result");
            }
            if (result.isError()) {
                String errorMsg = result.error();
                if (errorMsg.contains("Invalid ARN") && errorMsg.contains(":s3:::")) {
                    errorMsg += ". Use the bucket name instead of simple bucket ARNs in GetBucketLocationRequest.";
                }
                throw SdkClientException.create(errorMsg);
            }
            return CompletableFuture.completedFuture(result.endpoint());
        } catch (Exception error) {
            return CompletableFutureUtils.failedFuture(error);
        }
    }

    private static RuleResult endpointRule0(IamEndpointParams params, LocalState locals) {
        RuleResult result = endpointRule1(params, locals);
        if (result.isResolved()) {
            return result;
        }
        result = endpointRule5(params, locals);
        if (result.isResolved()) {
            return result;
        }
        return endpointRule29(params, locals);
    }

    private static RuleResult endpointRule1(IamEndpointParams params, LocalState locals) {
        if (params.endpoint() != null) {
            RuleResult result = endpointRule2(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule3(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule4(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule2(IamEndpointParams params, LocalState locals) {
        if (params.useFips()) {
            return RuleResult.error("Invalid Configuration: FIPS and custom endpoint are not supported");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule3(IamEndpointParams params, LocalState locals) {
        if (params.useDualStack()) {
            return RuleResult.error("Invalid Configuration: Dualstack and custom endpoint are not supported");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule4(IamEndpointParams params, LocalState locals) {
        return RuleResult.endpoint(Endpoint.builder().url(URI.create(params.endpoint())).build());
    }

    private static RuleResult endpointRule5(IamEndpointParams params, LocalState locals) {
        if (locals.region() != null) {
            return endpointRule6(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule6(IamEndpointParams params, LocalState locals) {
        RulePartition partitionResult = null;
        if ((partitionResult = RulesFunctions.awsPartition(locals.region())) != null) {
            locals = locals.toBuilder().partitionResult(partitionResult).build();
            RuleResult result = endpointRule7(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule8(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule9(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule10(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule11(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule12(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule13(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule14(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule15(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule16(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule20(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule24(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule28(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule7(IamEndpointParams params, LocalState locals) {
        if ("aws".equals(locals.partitionResult().name()) && !params.useFips() && !params.useDualStack()) {
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://iam.amazonaws.com"))
                    .putAttribute(AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().signingName("iam").signingRegion("us-east-1").build()))
                    .build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule8(IamEndpointParams params, LocalState locals) {
        if ("aws".equals(locals.partitionResult().name()) && params.useFips() && !params.useDualStack()) {
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://iam-fips.amazonaws.com"))
                    .putAttribute(AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().signingName("iam").signingRegion("us-east-1").build()))
                    .build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule9(IamEndpointParams params, LocalState locals) {
        if ("aws-cn".equals(locals.partitionResult().name()) && !params.useFips() && !params.useDualStack()) {
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://iam.cn-north-1.amazonaws.com.cn"))
                    .putAttribute(AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().signingName("iam").signingRegion("cn-north-1").build()))
                    .build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule10(IamEndpointParams params, LocalState locals) {
        if ("aws-us-gov".equals(locals.partitionResult().name()) && !params.useFips() && !params.useDualStack()) {
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://iam.us-gov.amazonaws.com"))
                    .putAttribute(AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().signingName("iam").signingRegion("us-gov-west-1").build()))
                    .build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule11(IamEndpointParams params, LocalState locals) {
        if ("aws-us-gov".equals(locals.partitionResult().name()) && params.useFips() && !params.useDualStack()) {
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://iam.us-gov.amazonaws.com"))
                    .putAttribute(AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().signingName("iam").signingRegion("us-gov-west-1").build()))
                    .build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule12(IamEndpointParams params, LocalState locals) {
        if ("aws-iso".equals(locals.partitionResult().name()) && !params.useFips() && !params.useDualStack()) {
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://iam.us-iso-east-1.c2s.ic.gov"))
                    .putAttribute(AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().signingName("iam").signingRegion("us-iso-east-1").build()))
                    .build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule13(IamEndpointParams params, LocalState locals) {
        if ("aws-iso-b".equals(locals.partitionResult().name()) && !params.useFips() && !params.useDualStack()) {
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://iam.us-isob-east-1.sc2s.sgov.gov"))
                    .putAttribute(AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().signingName("iam").signingRegion("us-isob-east-1").build()))
                    .build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule14(IamEndpointParams params, LocalState locals) {
        if ("aws-iso-e".equals(locals.partitionResult().name()) && !params.useFips() && !params.useDualStack()) {
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://iam.eu-isoe-west-1.cloud.adc-e.uk"))
                    .putAttribute(AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().signingName("iam").signingRegion("eu-isoe-west-1").build()))
                    .build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule15(IamEndpointParams params, LocalState locals) {
        if ("aws-iso-f".equals(locals.partitionResult().name()) && !params.useFips() && !params.useDualStack()) {
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://iam.us-isof-south-1.csp.hci.ic.gov"))
                    .putAttribute(AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().signingName("iam").signingRegion("us-isof-south-1").build()))
                    .build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule16(IamEndpointParams params, LocalState locals) {
        if (params.useFips() && params.useDualStack()) {
            RuleResult result = endpointRule17(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule19(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule17(IamEndpointParams params, LocalState locals) {
        if (locals.partitionResult().supportsFIPS() && locals.partitionResult().supportsDualStack()) {
            return endpointRule18(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule18(IamEndpointParams params, LocalState locals) {
        return RuleResult.endpoint(Endpoint.builder()
                .url(URI.create("https://iam-fips." + locals.region() + "." + locals.partitionResult().dualStackDnsSuffix()))
                .build());
    }

    private static RuleResult endpointRule19(IamEndpointParams params, LocalState locals) {
        return RuleResult.error("FIPS and DualStack are enabled, but this partition does not support one or both");
    }

    private static RuleResult endpointRule20(IamEndpointParams params, LocalState locals) {
        if (params.useFips()) {
            RuleResult result = endpointRule21(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule23(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule21(IamEndpointParams params, LocalState locals) {
        if (locals.partitionResult().supportsFIPS()) {
            return endpointRule22(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule22(IamEndpointParams params, LocalState locals) {
        return RuleResult.endpoint(Endpoint.builder()
                .url(URI.create("https://iam-fips." + locals.region() + "." + locals.partitionResult().dnsSuffix())).build());
    }

    private static RuleResult endpointRule23(IamEndpointParams params, LocalState locals) {
        return RuleResult.error("FIPS is enabled but this partition does not support FIPS");
    }

    private static RuleResult endpointRule24(IamEndpointParams params, LocalState locals) {
        if (params.useDualStack()) {
            RuleResult result = endpointRule25(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule27(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule25(IamEndpointParams params, LocalState locals) {
        if (locals.partitionResult().supportsDualStack()) {
            return endpointRule26(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule26(IamEndpointParams params, LocalState locals) {
        return RuleResult.endpoint(Endpoint.builder()
                .url(URI.create("https://iam." + locals.region() + "." + locals.partitionResult().dualStackDnsSuffix())).build());
    }

    private static RuleResult endpointRule27(IamEndpointParams params, LocalState locals) {
        return RuleResult.error("DualStack is enabled but this partition does not support DualStack");
    }

    private static RuleResult endpointRule28(IamEndpointParams params, LocalState locals) {
        return RuleResult.endpoint(Endpoint.builder()
                .url(URI.create("https://iam." + locals.region() + "." + locals.partitionResult().dnsSuffix())).build());
    }

    private static RuleResult endpointRule29(IamEndpointParams params, LocalState locals) {
        return RuleResult.error("Invalid Configuration: Missing Region");
    }

    @Override
    public boolean equals(Object rhs) {
        return rhs != null && getClass().equals(rhs.getClass());
    }

    @Override
    public int hashCode() {
        return getClass().hashCode();
    }

    private static final class LocalState {
        private final String region;

        private final RulePartition partitionResult;

        LocalState() {
            this.region = null;
            this.partitionResult = null;
        }

        LocalState(Region region) {
            if (region != null) {
                this.region = region.id();
            } else {
                this.region = null;
            }
            this.partitionResult = null;
        }

        LocalState(LocalStateBuilder builder) {
            this.region = builder.region;
            this.partitionResult = builder.partitionResult;
        }

        public String region() {
            return this.region;
        }

        public RulePartition partitionResult() {
            return this.partitionResult;
        }

        public LocalStateBuilder toBuilder() {
            return new LocalStateBuilder(this);
        }
    }

    private static final class LocalStateBuilder {
        private String region;

        private RulePartition partitionResult;

        LocalStateBuilder() {
            this.region = null;
            this.partitionResult = null;
        }

        LocalStateBuilder(LocalState locals) {
            this.region = locals.region;
            this.partitionResult = locals.partitionResult;
        }

        public LocalStateBuilder region(String value) {
            this.region = value;
            return this;
        }

        public LocalStateBuilder partitionResult(RulePartition value) {
            this.partitionResult = value;
            return this;
        }

        LocalState build() {
            return new LocalState(this);
        }
    }
}
