/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glue.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A structure that is used to specify testing a connection to a service.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TestConnectionInput implements SdkPojo, Serializable,
        ToCopyableBuilder<TestConnectionInput.Builder, TestConnectionInput> {
    private static final SdkField<String> CONNECTION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ConnectionType").getter(getter(TestConnectionInput::connectionTypeAsString))
            .setter(setter(Builder::connectionType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConnectionType").build()).build();

    private static final SdkField<Map<String, String>> CONNECTION_PROPERTIES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("ConnectionProperties")
            .getter(getter(TestConnectionInput::connectionPropertiesAsStrings))
            .setter(setter(Builder::connectionPropertiesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConnectionProperties").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<AuthenticationConfigurationInput> AUTHENTICATION_CONFIGURATION_FIELD = SdkField
            .<AuthenticationConfigurationInput> builder(MarshallingType.SDK_POJO)
            .memberName("AuthenticationConfiguration")
            .getter(getter(TestConnectionInput::authenticationConfiguration))
            .setter(setter(Builder::authenticationConfiguration))
            .constructor(AuthenticationConfigurationInput::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AuthenticationConfiguration")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CONNECTION_TYPE_FIELD,
            CONNECTION_PROPERTIES_FIELD, AUTHENTICATION_CONFIGURATION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("ConnectionType", CONNECTION_TYPE_FIELD);
                    put("ConnectionProperties", CONNECTION_PROPERTIES_FIELD);
                    put("AuthenticationConfiguration", AUTHENTICATION_CONFIGURATION_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String connectionType;

    private final Map<String, String> connectionProperties;

    private final AuthenticationConfigurationInput authenticationConfiguration;

    private TestConnectionInput(BuilderImpl builder) {
        this.connectionType = builder.connectionType;
        this.connectionProperties = builder.connectionProperties;
        this.authenticationConfiguration = builder.authenticationConfiguration;
    }

    /**
     * <p>
     * The type of connection to test. This operation is only available for the <code>JDBC</code> or
     * <code>SALESFORCE</code> connection types.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #connectionType}
     * will return {@link ConnectionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #connectionTypeAsString}.
     * </p>
     * 
     * @return The type of connection to test. This operation is only available for the <code>JDBC</code> or
     *         <code>SALESFORCE</code> connection types.
     * @see ConnectionType
     */
    public final ConnectionType connectionType() {
        return ConnectionType.fromValue(connectionType);
    }

    /**
     * <p>
     * The type of connection to test. This operation is only available for the <code>JDBC</code> or
     * <code>SALESFORCE</code> connection types.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #connectionType}
     * will return {@link ConnectionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #connectionTypeAsString}.
     * </p>
     * 
     * @return The type of connection to test. This operation is only available for the <code>JDBC</code> or
     *         <code>SALESFORCE</code> connection types.
     * @see ConnectionType
     */
    public final String connectionTypeAsString() {
        return connectionType;
    }

    /**
     * <p>
     * The key-value pairs that define parameters for the connection.
     * </p>
     * <p>
     * JDBC connections use the following connection properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Required: All of (<code>HOST</code>, <code>PORT</code>, <code>JDBC_ENGINE</code>) or
     * <code>JDBC_CONNECTION_URL</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Required: All of (<code>USERNAME</code>, <code>PASSWORD</code>) or <code>SECRET_ID</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Optional: <code>JDBC_ENFORCE_SSL</code>, <code>CUSTOM_JDBC_CERT</code>, <code>CUSTOM_JDBC_CERT_STRING</code>,
     * <code>SKIP_CUSTOM_JDBC_CERT_VALIDATION</code>. These parameters are used to configure SSL with JDBC.
     * </p>
     * </li>
     * </ul>
     * <p>
     * SALESFORCE connections require the <code>AuthenticationConfiguration</code> member to be configured.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasConnectionProperties} method.
     * </p>
     * 
     * @return The key-value pairs that define parameters for the connection.</p>
     *         <p>
     *         JDBC connections use the following connection properties:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Required: All of (<code>HOST</code>, <code>PORT</code>, <code>JDBC_ENGINE</code>) or
     *         <code>JDBC_CONNECTION_URL</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Required: All of (<code>USERNAME</code>, <code>PASSWORD</code>) or <code>SECRET_ID</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Optional: <code>JDBC_ENFORCE_SSL</code>, <code>CUSTOM_JDBC_CERT</code>,
     *         <code>CUSTOM_JDBC_CERT_STRING</code>, <code>SKIP_CUSTOM_JDBC_CERT_VALIDATION</code>. These parameters are
     *         used to configure SSL with JDBC.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         SALESFORCE connections require the <code>AuthenticationConfiguration</code> member to be configured.
     */
    public final Map<ConnectionPropertyKey, String> connectionProperties() {
        return ConnectionPropertiesCopier.copyStringToEnum(connectionProperties);
    }

    /**
     * For responses, this returns true if the service returned a value for the ConnectionProperties property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasConnectionProperties() {
        return connectionProperties != null && !(connectionProperties instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The key-value pairs that define parameters for the connection.
     * </p>
     * <p>
     * JDBC connections use the following connection properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Required: All of (<code>HOST</code>, <code>PORT</code>, <code>JDBC_ENGINE</code>) or
     * <code>JDBC_CONNECTION_URL</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Required: All of (<code>USERNAME</code>, <code>PASSWORD</code>) or <code>SECRET_ID</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Optional: <code>JDBC_ENFORCE_SSL</code>, <code>CUSTOM_JDBC_CERT</code>, <code>CUSTOM_JDBC_CERT_STRING</code>,
     * <code>SKIP_CUSTOM_JDBC_CERT_VALIDATION</code>. These parameters are used to configure SSL with JDBC.
     * </p>
     * </li>
     * </ul>
     * <p>
     * SALESFORCE connections require the <code>AuthenticationConfiguration</code> member to be configured.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasConnectionProperties} method.
     * </p>
     * 
     * @return The key-value pairs that define parameters for the connection.</p>
     *         <p>
     *         JDBC connections use the following connection properties:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Required: All of (<code>HOST</code>, <code>PORT</code>, <code>JDBC_ENGINE</code>) or
     *         <code>JDBC_CONNECTION_URL</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Required: All of (<code>USERNAME</code>, <code>PASSWORD</code>) or <code>SECRET_ID</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Optional: <code>JDBC_ENFORCE_SSL</code>, <code>CUSTOM_JDBC_CERT</code>,
     *         <code>CUSTOM_JDBC_CERT_STRING</code>, <code>SKIP_CUSTOM_JDBC_CERT_VALIDATION</code>. These parameters are
     *         used to configure SSL with JDBC.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         SALESFORCE connections require the <code>AuthenticationConfiguration</code> member to be configured.
     */
    public final Map<String, String> connectionPropertiesAsStrings() {
        return connectionProperties;
    }

    /**
     * <p>
     * A structure containing the authentication configuration in the TestConnection request. Required for a connection
     * to Salesforce using OAuth authentication.
     * </p>
     * 
     * @return A structure containing the authentication configuration in the TestConnection request. Required for a
     *         connection to Salesforce using OAuth authentication.
     */
    public final AuthenticationConfigurationInput authenticationConfiguration() {
        return authenticationConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(connectionTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasConnectionProperties() ? connectionPropertiesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(authenticationConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TestConnectionInput)) {
            return false;
        }
        TestConnectionInput other = (TestConnectionInput) obj;
        return Objects.equals(connectionTypeAsString(), other.connectionTypeAsString())
                && hasConnectionProperties() == other.hasConnectionProperties()
                && Objects.equals(connectionPropertiesAsStrings(), other.connectionPropertiesAsStrings())
                && Objects.equals(authenticationConfiguration(), other.authenticationConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TestConnectionInput").add("ConnectionType", connectionTypeAsString())
                .add("ConnectionProperties", hasConnectionProperties() ? connectionPropertiesAsStrings() : null)
                .add("AuthenticationConfiguration", authenticationConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ConnectionType":
            return Optional.ofNullable(clazz.cast(connectionTypeAsString()));
        case "ConnectionProperties":
            return Optional.ofNullable(clazz.cast(connectionPropertiesAsStrings()));
        case "AuthenticationConfiguration":
            return Optional.ofNullable(clazz.cast(authenticationConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<TestConnectionInput, T> g) {
        return obj -> g.apply((TestConnectionInput) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TestConnectionInput> {
        /**
         * <p>
         * The type of connection to test. This operation is only available for the <code>JDBC</code> or
         * <code>SALESFORCE</code> connection types.
         * </p>
         * 
         * @param connectionType
         *        The type of connection to test. This operation is only available for the <code>JDBC</code> or
         *        <code>SALESFORCE</code> connection types.
         * @see ConnectionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConnectionType
         */
        Builder connectionType(String connectionType);

        /**
         * <p>
         * The type of connection to test. This operation is only available for the <code>JDBC</code> or
         * <code>SALESFORCE</code> connection types.
         * </p>
         * 
         * @param connectionType
         *        The type of connection to test. This operation is only available for the <code>JDBC</code> or
         *        <code>SALESFORCE</code> connection types.
         * @see ConnectionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConnectionType
         */
        Builder connectionType(ConnectionType connectionType);

        /**
         * <p>
         * The key-value pairs that define parameters for the connection.
         * </p>
         * <p>
         * JDBC connections use the following connection properties:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Required: All of (<code>HOST</code>, <code>PORT</code>, <code>JDBC_ENGINE</code>) or
         * <code>JDBC_CONNECTION_URL</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Required: All of (<code>USERNAME</code>, <code>PASSWORD</code>) or <code>SECRET_ID</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Optional: <code>JDBC_ENFORCE_SSL</code>, <code>CUSTOM_JDBC_CERT</code>, <code>CUSTOM_JDBC_CERT_STRING</code>,
         * <code>SKIP_CUSTOM_JDBC_CERT_VALIDATION</code>. These parameters are used to configure SSL with JDBC.
         * </p>
         * </li>
         * </ul>
         * <p>
         * SALESFORCE connections require the <code>AuthenticationConfiguration</code> member to be configured.
         * </p>
         * 
         * @param connectionProperties
         *        The key-value pairs that define parameters for the connection.</p>
         *        <p>
         *        JDBC connections use the following connection properties:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Required: All of (<code>HOST</code>, <code>PORT</code>, <code>JDBC_ENGINE</code>) or
         *        <code>JDBC_CONNECTION_URL</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Required: All of (<code>USERNAME</code>, <code>PASSWORD</code>) or <code>SECRET_ID</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Optional: <code>JDBC_ENFORCE_SSL</code>, <code>CUSTOM_JDBC_CERT</code>,
         *        <code>CUSTOM_JDBC_CERT_STRING</code>, <code>SKIP_CUSTOM_JDBC_CERT_VALIDATION</code>. These parameters
         *        are used to configure SSL with JDBC.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        SALESFORCE connections require the <code>AuthenticationConfiguration</code> member to be configured.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connectionPropertiesWithStrings(Map<String, String> connectionProperties);

        /**
         * <p>
         * The key-value pairs that define parameters for the connection.
         * </p>
         * <p>
         * JDBC connections use the following connection properties:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Required: All of (<code>HOST</code>, <code>PORT</code>, <code>JDBC_ENGINE</code>) or
         * <code>JDBC_CONNECTION_URL</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Required: All of (<code>USERNAME</code>, <code>PASSWORD</code>) or <code>SECRET_ID</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Optional: <code>JDBC_ENFORCE_SSL</code>, <code>CUSTOM_JDBC_CERT</code>, <code>CUSTOM_JDBC_CERT_STRING</code>,
         * <code>SKIP_CUSTOM_JDBC_CERT_VALIDATION</code>. These parameters are used to configure SSL with JDBC.
         * </p>
         * </li>
         * </ul>
         * <p>
         * SALESFORCE connections require the <code>AuthenticationConfiguration</code> member to be configured.
         * </p>
         * 
         * @param connectionProperties
         *        The key-value pairs that define parameters for the connection.</p>
         *        <p>
         *        JDBC connections use the following connection properties:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Required: All of (<code>HOST</code>, <code>PORT</code>, <code>JDBC_ENGINE</code>) or
         *        <code>JDBC_CONNECTION_URL</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Required: All of (<code>USERNAME</code>, <code>PASSWORD</code>) or <code>SECRET_ID</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Optional: <code>JDBC_ENFORCE_SSL</code>, <code>CUSTOM_JDBC_CERT</code>,
         *        <code>CUSTOM_JDBC_CERT_STRING</code>, <code>SKIP_CUSTOM_JDBC_CERT_VALIDATION</code>. These parameters
         *        are used to configure SSL with JDBC.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        SALESFORCE connections require the <code>AuthenticationConfiguration</code> member to be configured.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connectionProperties(Map<ConnectionPropertyKey, String> connectionProperties);

        /**
         * <p>
         * A structure containing the authentication configuration in the TestConnection request. Required for a
         * connection to Salesforce using OAuth authentication.
         * </p>
         * 
         * @param authenticationConfiguration
         *        A structure containing the authentication configuration in the TestConnection request. Required for a
         *        connection to Salesforce using OAuth authentication.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authenticationConfiguration(AuthenticationConfigurationInput authenticationConfiguration);

        /**
         * <p>
         * A structure containing the authentication configuration in the TestConnection request. Required for a
         * connection to Salesforce using OAuth authentication.
         * </p>
         * This is a convenience method that creates an instance of the {@link AuthenticationConfigurationInput.Builder}
         * avoiding the need to create one manually via {@link AuthenticationConfigurationInput#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AuthenticationConfigurationInput.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #authenticationConfiguration(AuthenticationConfigurationInput)}.
         * 
         * @param authenticationConfiguration
         *        a consumer that will call methods on {@link AuthenticationConfigurationInput.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #authenticationConfiguration(AuthenticationConfigurationInput)
         */
        default Builder authenticationConfiguration(Consumer<AuthenticationConfigurationInput.Builder> authenticationConfiguration) {
            return authenticationConfiguration(AuthenticationConfigurationInput.builder()
                    .applyMutation(authenticationConfiguration).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String connectionType;

        private Map<String, String> connectionProperties = DefaultSdkAutoConstructMap.getInstance();

        private AuthenticationConfigurationInput authenticationConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(TestConnectionInput model) {
            connectionType(model.connectionType);
            connectionPropertiesWithStrings(model.connectionProperties);
            authenticationConfiguration(model.authenticationConfiguration);
        }

        public final String getConnectionType() {
            return connectionType;
        }

        public final void setConnectionType(String connectionType) {
            this.connectionType = connectionType;
        }

        @Override
        public final Builder connectionType(String connectionType) {
            this.connectionType = connectionType;
            return this;
        }

        @Override
        public final Builder connectionType(ConnectionType connectionType) {
            this.connectionType(connectionType == null ? null : connectionType.toString());
            return this;
        }

        public final Map<String, String> getConnectionProperties() {
            if (connectionProperties instanceof SdkAutoConstructMap) {
                return null;
            }
            return connectionProperties;
        }

        public final void setConnectionProperties(Map<String, String> connectionProperties) {
            this.connectionProperties = ConnectionPropertiesCopier.copy(connectionProperties);
        }

        @Override
        public final Builder connectionPropertiesWithStrings(Map<String, String> connectionProperties) {
            this.connectionProperties = ConnectionPropertiesCopier.copy(connectionProperties);
            return this;
        }

        @Override
        public final Builder connectionProperties(Map<ConnectionPropertyKey, String> connectionProperties) {
            this.connectionProperties = ConnectionPropertiesCopier.copyEnumToString(connectionProperties);
            return this;
        }

        public final AuthenticationConfigurationInput.Builder getAuthenticationConfiguration() {
            return authenticationConfiguration != null ? authenticationConfiguration.toBuilder() : null;
        }

        public final void setAuthenticationConfiguration(AuthenticationConfigurationInput.BuilderImpl authenticationConfiguration) {
            this.authenticationConfiguration = authenticationConfiguration != null ? authenticationConfiguration.build() : null;
        }

        @Override
        public final Builder authenticationConfiguration(AuthenticationConfigurationInput authenticationConfiguration) {
            this.authenticationConfiguration = authenticationConfiguration;
            return this;
        }

        @Override
        public TestConnectionInput build() {
            return new TestConnectionInput(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
