/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.firehose.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The OpenX SerDe. Used by Firehose for deserializing data, which means converting it from the JSON format in
 * preparation for serializing it to the Parquet or ORC format. This is one of two deserializers you can choose,
 * depending on which one offers the functionality you need. The other option is the native Hive / HCatalog JsonSerDe.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class OpenXJsonSerDe implements SdkPojo, Serializable, ToCopyableBuilder<OpenXJsonSerDe.Builder, OpenXJsonSerDe> {
    private static final SdkField<Boolean> CONVERT_DOTS_IN_JSON_KEYS_TO_UNDERSCORES_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("ConvertDotsInJsonKeysToUnderscores")
            .getter(getter(OpenXJsonSerDe::convertDotsInJsonKeysToUnderscores))
            .setter(setter(Builder::convertDotsInJsonKeysToUnderscores))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConvertDotsInJsonKeysToUnderscores")
                    .build()).build();

    private static final SdkField<Boolean> CASE_INSENSITIVE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("CaseInsensitive").getter(getter(OpenXJsonSerDe::caseInsensitive))
            .setter(setter(Builder::caseInsensitive))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CaseInsensitive").build()).build();

    private static final SdkField<Map<String, String>> COLUMN_TO_JSON_KEY_MAPPINGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("ColumnToJsonKeyMappings")
            .getter(getter(OpenXJsonSerDe::columnToJsonKeyMappings))
            .setter(setter(Builder::columnToJsonKeyMappings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ColumnToJsonKeyMappings").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            CONVERT_DOTS_IN_JSON_KEYS_TO_UNDERSCORES_FIELD, CASE_INSENSITIVE_FIELD, COLUMN_TO_JSON_KEY_MAPPINGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final Boolean convertDotsInJsonKeysToUnderscores;

    private final Boolean caseInsensitive;

    private final Map<String, String> columnToJsonKeyMappings;

    private OpenXJsonSerDe(BuilderImpl builder) {
        this.convertDotsInJsonKeysToUnderscores = builder.convertDotsInJsonKeysToUnderscores;
        this.caseInsensitive = builder.caseInsensitive;
        this.columnToJsonKeyMappings = builder.columnToJsonKeyMappings;
    }

    /**
     * <p>
     * When set to <code>true</code>, specifies that the names of the keys include dots and that you want Firehose to
     * replace them with underscores. This is useful because Apache Hive does not allow dots in column names. For
     * example, if the JSON contains a key whose name is "a.b", you can define the column name to be "a_b" when using
     * this option.
     * </p>
     * <p>
     * The default is <code>false</code>.
     * </p>
     * 
     * @return When set to <code>true</code>, specifies that the names of the keys include dots and that you want
     *         Firehose to replace them with underscores. This is useful because Apache Hive does not allow dots in
     *         column names. For example, if the JSON contains a key whose name is "a.b", you can define the column name
     *         to be "a_b" when using this option.</p>
     *         <p>
     *         The default is <code>false</code>.
     */
    public final Boolean convertDotsInJsonKeysToUnderscores() {
        return convertDotsInJsonKeysToUnderscores;
    }

    /**
     * <p>
     * When set to <code>true</code>, which is the default, Firehose converts JSON keys to lowercase before
     * deserializing them.
     * </p>
     * 
     * @return When set to <code>true</code>, which is the default, Firehose converts JSON keys to lowercase before
     *         deserializing them.
     */
    public final Boolean caseInsensitive() {
        return caseInsensitive;
    }

    /**
     * For responses, this returns true if the service returned a value for the ColumnToJsonKeyMappings property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasColumnToJsonKeyMappings() {
        return columnToJsonKeyMappings != null && !(columnToJsonKeyMappings instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Maps column names to JSON keys that aren't identical to the column names. This is useful when the JSON contains
     * keys that are Hive keywords. For example, <code>timestamp</code> is a Hive keyword. If you have a JSON key named
     * <code>timestamp</code>, set this parameter to <code>{"ts": "timestamp"}</code> to map this key to a column named
     * <code>ts</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasColumnToJsonKeyMappings} method.
     * </p>
     * 
     * @return Maps column names to JSON keys that aren't identical to the column names. This is useful when the JSON
     *         contains keys that are Hive keywords. For example, <code>timestamp</code> is a Hive keyword. If you have
     *         a JSON key named <code>timestamp</code>, set this parameter to <code>{"ts": "timestamp"}</code> to map
     *         this key to a column named <code>ts</code>.
     */
    public final Map<String, String> columnToJsonKeyMappings() {
        return columnToJsonKeyMappings;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(convertDotsInJsonKeysToUnderscores());
        hashCode = 31 * hashCode + Objects.hashCode(caseInsensitive());
        hashCode = 31 * hashCode + Objects.hashCode(hasColumnToJsonKeyMappings() ? columnToJsonKeyMappings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof OpenXJsonSerDe)) {
            return false;
        }
        OpenXJsonSerDe other = (OpenXJsonSerDe) obj;
        return Objects.equals(convertDotsInJsonKeysToUnderscores(), other.convertDotsInJsonKeysToUnderscores())
                && Objects.equals(caseInsensitive(), other.caseInsensitive())
                && hasColumnToJsonKeyMappings() == other.hasColumnToJsonKeyMappings()
                && Objects.equals(columnToJsonKeyMappings(), other.columnToJsonKeyMappings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("OpenXJsonSerDe").add("ConvertDotsInJsonKeysToUnderscores", convertDotsInJsonKeysToUnderscores())
                .add("CaseInsensitive", caseInsensitive())
                .add("ColumnToJsonKeyMappings", hasColumnToJsonKeyMappings() ? columnToJsonKeyMappings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ConvertDotsInJsonKeysToUnderscores":
            return Optional.ofNullable(clazz.cast(convertDotsInJsonKeysToUnderscores()));
        case "CaseInsensitive":
            return Optional.ofNullable(clazz.cast(caseInsensitive()));
        case "ColumnToJsonKeyMappings":
            return Optional.ofNullable(clazz.cast(columnToJsonKeyMappings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<OpenXJsonSerDe, T> g) {
        return obj -> g.apply((OpenXJsonSerDe) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, OpenXJsonSerDe> {
        /**
         * <p>
         * When set to <code>true</code>, specifies that the names of the keys include dots and that you want Firehose
         * to replace them with underscores. This is useful because Apache Hive does not allow dots in column names. For
         * example, if the JSON contains a key whose name is "a.b", you can define the column name to be "a_b" when
         * using this option.
         * </p>
         * <p>
         * The default is <code>false</code>.
         * </p>
         * 
         * @param convertDotsInJsonKeysToUnderscores
         *        When set to <code>true</code>, specifies that the names of the keys include dots and that you want
         *        Firehose to replace them with underscores. This is useful because Apache Hive does not allow dots in
         *        column names. For example, if the JSON contains a key whose name is "a.b", you can define the column
         *        name to be "a_b" when using this option.</p>
         *        <p>
         *        The default is <code>false</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder convertDotsInJsonKeysToUnderscores(Boolean convertDotsInJsonKeysToUnderscores);

        /**
         * <p>
         * When set to <code>true</code>, which is the default, Firehose converts JSON keys to lowercase before
         * deserializing them.
         * </p>
         * 
         * @param caseInsensitive
         *        When set to <code>true</code>, which is the default, Firehose converts JSON keys to lowercase before
         *        deserializing them.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder caseInsensitive(Boolean caseInsensitive);

        /**
         * <p>
         * Maps column names to JSON keys that aren't identical to the column names. This is useful when the JSON
         * contains keys that are Hive keywords. For example, <code>timestamp</code> is a Hive keyword. If you have a
         * JSON key named <code>timestamp</code>, set this parameter to <code>{"ts": "timestamp"}</code> to map this key
         * to a column named <code>ts</code>.
         * </p>
         * 
         * @param columnToJsonKeyMappings
         *        Maps column names to JSON keys that aren't identical to the column names. This is useful when the JSON
         *        contains keys that are Hive keywords. For example, <code>timestamp</code> is a Hive keyword. If you
         *        have a JSON key named <code>timestamp</code>, set this parameter to <code>{"ts": "timestamp"}</code>
         *        to map this key to a column named <code>ts</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder columnToJsonKeyMappings(Map<String, String> columnToJsonKeyMappings);
    }

    static final class BuilderImpl implements Builder {
        private Boolean convertDotsInJsonKeysToUnderscores;

        private Boolean caseInsensitive;

        private Map<String, String> columnToJsonKeyMappings = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(OpenXJsonSerDe model) {
            convertDotsInJsonKeysToUnderscores(model.convertDotsInJsonKeysToUnderscores);
            caseInsensitive(model.caseInsensitive);
            columnToJsonKeyMappings(model.columnToJsonKeyMappings);
        }

        public final Boolean getConvertDotsInJsonKeysToUnderscores() {
            return convertDotsInJsonKeysToUnderscores;
        }

        public final void setConvertDotsInJsonKeysToUnderscores(Boolean convertDotsInJsonKeysToUnderscores) {
            this.convertDotsInJsonKeysToUnderscores = convertDotsInJsonKeysToUnderscores;
        }

        @Override
        public final Builder convertDotsInJsonKeysToUnderscores(Boolean convertDotsInJsonKeysToUnderscores) {
            this.convertDotsInJsonKeysToUnderscores = convertDotsInJsonKeysToUnderscores;
            return this;
        }

        public final Boolean getCaseInsensitive() {
            return caseInsensitive;
        }

        public final void setCaseInsensitive(Boolean caseInsensitive) {
            this.caseInsensitive = caseInsensitive;
        }

        @Override
        public final Builder caseInsensitive(Boolean caseInsensitive) {
            this.caseInsensitive = caseInsensitive;
            return this;
        }

        public final Map<String, String> getColumnToJsonKeyMappings() {
            if (columnToJsonKeyMappings instanceof SdkAutoConstructMap) {
                return null;
            }
            return columnToJsonKeyMappings;
        }

        public final void setColumnToJsonKeyMappings(Map<String, String> columnToJsonKeyMappings) {
            this.columnToJsonKeyMappings = ColumnToJsonKeyMappingsCopier.copy(columnToJsonKeyMappings);
        }

        @Override
        public final Builder columnToJsonKeyMappings(Map<String, String> columnToJsonKeyMappings) {
            this.columnToJsonKeyMappings = ColumnToJsonKeyMappingsCopier.copy(columnToJsonKeyMappings);
            return this;
        }

        @Override
        public OpenXJsonSerDe build() {
            return new OpenXJsonSerDe(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
