/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.awssdk.metrics.publishers.emf.internal;

import java.nio.charset.StandardCharsets;
import java.time.Clock;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.SdkTestInternalApi;
import software.amazon.awssdk.metrics.MetricCategory;
import software.amazon.awssdk.metrics.MetricCollection;
import software.amazon.awssdk.metrics.MetricRecord;
import software.amazon.awssdk.metrics.SdkMetric;
import software.amazon.awssdk.metrics.publishers.emf.internal.EmfMetricConfiguration;
import software.amazon.awssdk.protocols.jsoncore.JsonWriter;
import software.amazon.awssdk.utils.Logger;
import software.amazon.awssdk.utils.MetricValueNormalizer;
import software.amazon.awssdk.utils.Validate;

@SdkInternalApi
public class MetricEmfConverter {
    private static final int MAX_RECORD_SIZE = 100;
    private static final int MAX_METRIC_NUM = 100;
    private static final Logger logger = Logger.loggerFor(MetricEmfConverter.class);
    private final List<String> dimensions = new ArrayList<String>();
    private final EmfMetricConfiguration config;
    private final boolean metricCategoriesContainsAll;
    private final Clock clock;

    @SdkTestInternalApi
    public MetricEmfConverter(EmfMetricConfiguration config, Clock clock) {
        this.config = config;
        this.clock = clock;
        this.metricCategoriesContainsAll = config.metricCategories().contains(MetricCategory.ALL);
    }

    public MetricEmfConverter(EmfMetricConfiguration config) {
        this(config, Clock.systemUTC());
    }

    public List<String> convertMetricCollectionToEmf(MetricCollection metricCollection) {
        HashMap aggregatedMetrics = new HashMap();
        LinkedList<MetricCollection> queue = new LinkedList<MetricCollection>();
        if (!queue.offer(metricCollection)) {
            logger.warn(() -> "failed to add metricCollection to the queue");
        }
        while (!queue.isEmpty()) {
            MetricCollection current = (MetricCollection)queue.poll();
            current.stream().forEach(metricRecord -> {
                SdkMetric metric = metricRecord.metric();
                String metricName = metric.name();
                if (this.isDimension(metric) && !this.dimensions.contains(metricName)) {
                    this.dimensions.add(metricName);
                }
                if (this.shouldReport((MetricRecord<?>)metricRecord) || this.isDimension(metric)) {
                    aggregatedMetrics.computeIfAbsent(metric, k -> new ArrayList()).add(metricRecord);
                }
            });
            if (current.children() == null) continue;
            queue.addAll(current.children());
        }
        return this.createEmfStrings(aggregatedMetrics);
    }

    private void processAndWriteValue(JsonWriter jsonWriter, MetricRecord<?> mRecord) {
        Object value = mRecord.value();
        Class valueClass = mRecord.metric().valueClass();
        if (value == null) {
            return;
        }
        if (Boolean.class.isAssignableFrom(valueClass)) {
            jsonWriter.writeValue(value.equals(true) ? 1.0 : 0.0);
            return;
        }
        if (Duration.class.isAssignableFrom(valueClass)) {
            Duration duration = (Duration)value;
            double millisValue = duration.toMillis();
            jsonWriter.writeValue(millisValue);
            return;
        }
        if (Double.class.isAssignableFrom(valueClass)) {
            jsonWriter.writeValue(MetricValueNormalizer.normalize((double)((Double)value)));
            return;
        }
        if (Integer.class.isAssignableFrom(valueClass) || Long.class.isAssignableFrom(valueClass)) {
            jsonWriter.writeValue(((Integer)value).intValue());
        }
    }

    private List<String> createEmfStrings(Map<SdkMetric<?>, List<MetricRecord<?>>> aggregatedMetrics) {
        ArrayList<String> emfStrings = new ArrayList<String>();
        HashMap<Object, List<Object>> currentMetricBatch = new HashMap();
        for (Map.Entry<SdkMetric<?>, List<MetricRecord<?>>> entry : aggregatedMetrics.entrySet()) {
            SdkMetric<?> metric = entry.getKey();
            List<MetricRecord<?>> records = entry.getValue();
            int size = records.size();
            if (records.size() > 100) {
                records = records.subList(0, 100);
                logger.warn(() -> String.format("Some AWS SDK client-side metric data have been dropped because it exceeds the cloudwatch requirements. There are %d values for metric %s", size, metric.name()));
            }
            if (currentMetricBatch.size() == 100) {
                emfStrings.add(this.createEmfString(currentMetricBatch));
                currentMetricBatch = new HashMap();
            }
            currentMetricBatch.put(metric, records);
        }
        emfStrings.add(this.createEmfString(currentMetricBatch));
        return emfStrings;
    }

    private String createEmfString(Map<SdkMetric<?>, List<MetricRecord<?>>> metrics) {
        JsonWriter jsonWriter = JsonWriter.create();
        jsonWriter.writeStartObject();
        this.writeAwsObject(jsonWriter, metrics.keySet());
        this.writeMetricValues(jsonWriter, metrics);
        jsonWriter.writeEndObject();
        return new String(jsonWriter.getBytes(), StandardCharsets.UTF_8);
    }

    private void writeAwsObject(JsonWriter jsonWriter, Set<SdkMetric<?>> metricNames) {
        jsonWriter.writeFieldName("_aws");
        jsonWriter.writeStartObject();
        jsonWriter.writeFieldName("Timestamp");
        jsonWriter.writeValue(this.clock.instant().toEpochMilli());
        jsonWriter.writeFieldName("LogGroupName");
        jsonWriter.writeValue(this.config.logGroupName());
        this.writeCloudWatchMetricsArray(jsonWriter, metricNames);
        jsonWriter.writeEndObject();
    }

    private void writeCloudWatchMetricsArray(JsonWriter jsonWriter, Set<SdkMetric<?>> metricNames) {
        jsonWriter.writeFieldName("CloudWatchMetrics");
        jsonWriter.writeStartArray();
        this.writeCloudWatchMetricsObjects(jsonWriter, metricNames);
        jsonWriter.writeEndArray();
    }

    private void writeCloudWatchMetricsObjects(JsonWriter jsonWriter, Set<SdkMetric<?>> metricNames) {
        jsonWriter.writeStartObject();
        jsonWriter.writeFieldName("Namespace");
        jsonWriter.writeValue(this.config.namespace());
        this.writeDimensionSetArray(jsonWriter);
        this.writeMetricDefinitionArray(jsonWriter, metricNames);
        jsonWriter.writeEndObject();
    }

    private void writeDimensionSetArray(JsonWriter jsonWriter) {
        jsonWriter.writeFieldName("Dimensions");
        jsonWriter.writeStartArray();
        jsonWriter.writeStartArray();
        for (String dimension : this.dimensions) {
            jsonWriter.writeValue(dimension);
        }
        jsonWriter.writeEndArray();
        jsonWriter.writeEndArray();
    }

    private void writeMetricDefinitionArray(JsonWriter jsonWriter, Set<SdkMetric<?>> metricNames) {
        jsonWriter.writeFieldName("Metrics");
        jsonWriter.writeStartArray();
        metricNames.forEach(sdkMetric -> this.writeMetricDefinition(jsonWriter, (SdkMetric<?>)sdkMetric));
        jsonWriter.writeEndArray();
    }

    private void writeMetricDefinition(JsonWriter jsonWriter, SdkMetric<?> sdkMetric) {
        if (!this.isNumericMetric(sdkMetric)) {
            return;
        }
        jsonWriter.writeStartObject();
        jsonWriter.writeFieldName("Name");
        jsonWriter.writeValue(sdkMetric.name());
        String unit = this.getMetricUnit(sdkMetric.valueClass());
        if (unit != null) {
            jsonWriter.writeFieldName("Unit");
            jsonWriter.writeValue(unit);
        }
        jsonWriter.writeEndObject();
    }

    private void writeMetricValues(JsonWriter jsonWriter, Map<SdkMetric<?>, List<MetricRecord<?>>> metrics) {
        metrics.forEach((metric, records) -> {
            if (this.isDimension((SdkMetric<?>)metric)) {
                this.writeDimensionValue(jsonWriter, (SdkMetric<?>)metric, (List<MetricRecord<?>>)records);
            } else {
                this.writeMetricRecord(jsonWriter, (SdkMetric<?>)metric, (List<MetricRecord<?>>)records);
            }
        });
    }

    private void writeDimensionValue(JsonWriter jsonWriter, SdkMetric<?> metric, List<MetricRecord<?>> records) {
        Validate.validState((records.size() == 1 && String.class.isAssignableFrom(metric.valueClass()) ? 1 : 0) != 0, (String)"Metric (%s) is configured as a dimension, and the value must be a single string", (Object[])new Object[]{metric.name()});
        jsonWriter.writeFieldName(metric.name());
        jsonWriter.writeValue((String)records.get(0).value());
    }

    private void writeMetricRecord(JsonWriter jsonWriter, SdkMetric<?> metric, List<MetricRecord<?>> records) {
        if (!this.isNumericMetric(metric)) {
            return;
        }
        jsonWriter.writeFieldName(metric.name());
        if (records.size() == 1) {
            this.processAndWriteValue(jsonWriter, records.get(0));
        } else {
            this.writeMetricArray(jsonWriter, records);
        }
    }

    private boolean isNumericMetric(SdkMetric<?> metric) {
        return Integer.class.isAssignableFrom(metric.valueClass()) || Boolean.class.isAssignableFrom(metric.valueClass()) || Long.class.isAssignableFrom(metric.valueClass()) || Duration.class.isAssignableFrom(metric.valueClass()) || Double.class.isAssignableFrom(metric.valueClass());
    }

    private void writeMetricArray(JsonWriter jsonWriter, List<MetricRecord<?>> records) {
        jsonWriter.writeStartArray();
        for (MetricRecord<?> mRecord : records) {
            this.processAndWriteValue(jsonWriter, mRecord);
        }
        jsonWriter.writeEndArray();
    }

    private boolean isDimension(SdkMetric<?> metric) {
        return this.config.dimensions().contains(metric);
    }

    private String getMetricUnit(Class<?> type) {
        if (Duration.class.isAssignableFrom(type)) {
            return "Milliseconds";
        }
        return null;
    }

    private boolean shouldReport(MetricRecord<?> metricRecord) {
        return this.isSupportedCategory(metricRecord) && this.isSupportedLevel(metricRecord);
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private boolean isSupportedCategory(MetricRecord<?> metricRecord) {
        if (this.metricCategoriesContainsAll) return true;
        if (!metricRecord.metric().categories().stream().anyMatch(this.config.metricCategories()::contains)) return false;
        return true;
    }

    private boolean isSupportedLevel(MetricRecord<?> metricRecord) {
        return this.config.metricLevel().includesLevel(metricRecord.metric().level());
    }
}

