/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticinference;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.elasticinference.model.BadRequestException;
import software.amazon.awssdk.services.elasticinference.model.DescribeAcceleratorOfferingsRequest;
import software.amazon.awssdk.services.elasticinference.model.DescribeAcceleratorOfferingsResponse;
import software.amazon.awssdk.services.elasticinference.model.DescribeAcceleratorTypesRequest;
import software.amazon.awssdk.services.elasticinference.model.DescribeAcceleratorTypesResponse;
import software.amazon.awssdk.services.elasticinference.model.DescribeAcceleratorsRequest;
import software.amazon.awssdk.services.elasticinference.model.DescribeAcceleratorsResponse;
import software.amazon.awssdk.services.elasticinference.model.ElasticInferenceException;
import software.amazon.awssdk.services.elasticinference.model.ElasticInferenceRequest;
import software.amazon.awssdk.services.elasticinference.model.InternalServerException;
import software.amazon.awssdk.services.elasticinference.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.elasticinference.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.elasticinference.model.ResourceNotFoundException;
import software.amazon.awssdk.services.elasticinference.model.TagResourceRequest;
import software.amazon.awssdk.services.elasticinference.model.TagResourceResponse;
import software.amazon.awssdk.services.elasticinference.model.UntagResourceRequest;
import software.amazon.awssdk.services.elasticinference.model.UntagResourceResponse;
import software.amazon.awssdk.services.elasticinference.paginators.DescribeAcceleratorsIterable;
import software.amazon.awssdk.services.elasticinference.transform.DescribeAcceleratorOfferingsRequestMarshaller;
import software.amazon.awssdk.services.elasticinference.transform.DescribeAcceleratorTypesRequestMarshaller;
import software.amazon.awssdk.services.elasticinference.transform.DescribeAcceleratorsRequestMarshaller;
import software.amazon.awssdk.services.elasticinference.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.elasticinference.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.elasticinference.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link ElasticInferenceClient}.
 *
 * @see ElasticInferenceClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultElasticInferenceClient implements ElasticInferenceClient {
    private static final Logger log = Logger.loggerFor(DefaultElasticInferenceClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultElasticInferenceClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Describes the locations in which a given accelerator type or set of types is present in a given region.
     * </p>
     *
     * @param describeAcceleratorOfferingsRequest
     * @return Result of the DescribeAcceleratorOfferings operation returned by the service.
     * @throws BadRequestException
     *         Raised when a malformed input has been provided to the API.
     * @throws ResourceNotFoundException
     *         Raised when the requested resource cannot be found.
     * @throws InternalServerException
     *         Raised when an unexpected error occurred during request processing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticInferenceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticInferenceClient.DescribeAcceleratorOfferings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elastic-inference-2017-07-25/DescribeAcceleratorOfferings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAcceleratorOfferingsResponse describeAcceleratorOfferings(
            DescribeAcceleratorOfferingsRequest describeAcceleratorOfferingsRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException,
            ElasticInferenceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeAcceleratorOfferingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeAcceleratorOfferingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAcceleratorOfferingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Elastic Inference");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAcceleratorOfferings");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeAcceleratorOfferingsRequest, DescribeAcceleratorOfferingsResponse>()
                            .withOperationName("DescribeAcceleratorOfferings").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeAcceleratorOfferingsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeAcceleratorOfferingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the accelerator types available in a given region, as well as their characteristics, such as memory and
     * throughput.
     * </p>
     *
     * @param describeAcceleratorTypesRequest
     * @return Result of the DescribeAcceleratorTypes operation returned by the service.
     * @throws InternalServerException
     *         Raised when an unexpected error occurred during request processing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticInferenceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticInferenceClient.DescribeAcceleratorTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elastic-inference-2017-07-25/DescribeAcceleratorTypes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAcceleratorTypesResponse describeAcceleratorTypes(
            DescribeAcceleratorTypesRequest describeAcceleratorTypesRequest) throws InternalServerException, AwsServiceException,
            SdkClientException, ElasticInferenceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeAcceleratorTypesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeAcceleratorTypesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAcceleratorTypesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Elastic Inference");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAcceleratorTypes");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeAcceleratorTypesRequest, DescribeAcceleratorTypesResponse>()
                            .withOperationName("DescribeAcceleratorTypes").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeAcceleratorTypesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeAcceleratorTypesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes information over a provided set of accelerators belonging to an account.
     * </p>
     *
     * @param describeAcceleratorsRequest
     * @return Result of the DescribeAccelerators operation returned by the service.
     * @throws BadRequestException
     *         Raised when a malformed input has been provided to the API.
     * @throws ResourceNotFoundException
     *         Raised when the requested resource cannot be found.
     * @throws InternalServerException
     *         Raised when an unexpected error occurred during request processing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticInferenceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticInferenceClient.DescribeAccelerators
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elastic-inference-2017-07-25/DescribeAccelerators"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAcceleratorsResponse describeAccelerators(DescribeAcceleratorsRequest describeAcceleratorsRequest)
            throws BadRequestException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, ElasticInferenceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeAcceleratorsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeAcceleratorsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAcceleratorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Elastic Inference");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAccelerators");

            return clientHandler.execute(new ClientExecutionParams<DescribeAcceleratorsRequest, DescribeAcceleratorsResponse>()
                    .withOperationName("DescribeAccelerators").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeAcceleratorsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeAcceleratorsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes information over a provided set of accelerators belonging to an account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeAccelerators(software.amazon.awssdk.services.elasticinference.model.DescribeAcceleratorsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.elasticinference.paginators.DescribeAcceleratorsIterable responses = client.describeAcceleratorsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.elasticinference.paginators.DescribeAcceleratorsIterable responses = client
     *             .describeAcceleratorsPaginator(request);
     *     for (software.amazon.awssdk.services.elasticinference.model.DescribeAcceleratorsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.elasticinference.paginators.DescribeAcceleratorsIterable responses = client.describeAcceleratorsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeAccelerators(software.amazon.awssdk.services.elasticinference.model.DescribeAcceleratorsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeAcceleratorsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BadRequestException
     *         Raised when a malformed input has been provided to the API.
     * @throws ResourceNotFoundException
     *         Raised when the requested resource cannot be found.
     * @throws InternalServerException
     *         Raised when an unexpected error occurred during request processing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticInferenceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticInferenceClient.DescribeAccelerators
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elastic-inference-2017-07-25/DescribeAccelerators"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAcceleratorsIterable describeAcceleratorsPaginator(DescribeAcceleratorsRequest describeAcceleratorsRequest)
            throws BadRequestException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, ElasticInferenceException {
        return new DescribeAcceleratorsIterable(this, applyPaginatorUserAgent(describeAcceleratorsRequest));
    }

    /**
     * <p>
     * Returns all tags of an Elastic Inference Accelerator.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws BadRequestException
     *         Raised when a malformed input has been provided to the API.
     * @throws ResourceNotFoundException
     *         Raised when the requested resource cannot be found.
     * @throws InternalServerException
     *         Raised when an unexpected error occurred during request processing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticInferenceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticInferenceClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elastic-inference-2017-07-25/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws BadRequestException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, ElasticInferenceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Elastic Inference");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds the specified tags to an Elastic Inference Accelerator.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws BadRequestException
     *         Raised when a malformed input has been provided to the API.
     * @throws ResourceNotFoundException
     *         Raised when the requested resource cannot be found.
     * @throws InternalServerException
     *         Raised when an unexpected error occurred during request processing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticInferenceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticInferenceClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elastic-inference-2017-07-25/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException,
            ElasticInferenceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Elastic Inference");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the specified tags from an Elastic Inference Accelerator.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws BadRequestException
     *         Raised when a malformed input has been provided to the API.
     * @throws ResourceNotFoundException
     *         Raised when the requested resource cannot be found.
     * @throws InternalServerException
     *         Raised when an unexpected error occurred during request processing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticInferenceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticInferenceClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elastic-inference-2017-07-25/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException,
            ElasticInferenceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Elastic Inference");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ElasticInferenceException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends ElasticInferenceRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
