/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.eks;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.eks.internal.EksServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.eks.model.AccessDeniedException;
import software.amazon.awssdk.services.eks.model.AssociateAccessPolicyRequest;
import software.amazon.awssdk.services.eks.model.AssociateAccessPolicyResponse;
import software.amazon.awssdk.services.eks.model.AssociateEncryptionConfigRequest;
import software.amazon.awssdk.services.eks.model.AssociateEncryptionConfigResponse;
import software.amazon.awssdk.services.eks.model.AssociateIdentityProviderConfigRequest;
import software.amazon.awssdk.services.eks.model.AssociateIdentityProviderConfigResponse;
import software.amazon.awssdk.services.eks.model.BadRequestException;
import software.amazon.awssdk.services.eks.model.ClientException;
import software.amazon.awssdk.services.eks.model.CreateAccessEntryRequest;
import software.amazon.awssdk.services.eks.model.CreateAccessEntryResponse;
import software.amazon.awssdk.services.eks.model.CreateAddonRequest;
import software.amazon.awssdk.services.eks.model.CreateAddonResponse;
import software.amazon.awssdk.services.eks.model.CreateClusterRequest;
import software.amazon.awssdk.services.eks.model.CreateClusterResponse;
import software.amazon.awssdk.services.eks.model.CreateEksAnywhereSubscriptionRequest;
import software.amazon.awssdk.services.eks.model.CreateEksAnywhereSubscriptionResponse;
import software.amazon.awssdk.services.eks.model.CreateFargateProfileRequest;
import software.amazon.awssdk.services.eks.model.CreateFargateProfileResponse;
import software.amazon.awssdk.services.eks.model.CreateNodegroupRequest;
import software.amazon.awssdk.services.eks.model.CreateNodegroupResponse;
import software.amazon.awssdk.services.eks.model.CreatePodIdentityAssociationRequest;
import software.amazon.awssdk.services.eks.model.CreatePodIdentityAssociationResponse;
import software.amazon.awssdk.services.eks.model.DeleteAccessEntryRequest;
import software.amazon.awssdk.services.eks.model.DeleteAccessEntryResponse;
import software.amazon.awssdk.services.eks.model.DeleteAddonRequest;
import software.amazon.awssdk.services.eks.model.DeleteAddonResponse;
import software.amazon.awssdk.services.eks.model.DeleteClusterRequest;
import software.amazon.awssdk.services.eks.model.DeleteClusterResponse;
import software.amazon.awssdk.services.eks.model.DeleteEksAnywhereSubscriptionRequest;
import software.amazon.awssdk.services.eks.model.DeleteEksAnywhereSubscriptionResponse;
import software.amazon.awssdk.services.eks.model.DeleteFargateProfileRequest;
import software.amazon.awssdk.services.eks.model.DeleteFargateProfileResponse;
import software.amazon.awssdk.services.eks.model.DeleteNodegroupRequest;
import software.amazon.awssdk.services.eks.model.DeleteNodegroupResponse;
import software.amazon.awssdk.services.eks.model.DeletePodIdentityAssociationRequest;
import software.amazon.awssdk.services.eks.model.DeletePodIdentityAssociationResponse;
import software.amazon.awssdk.services.eks.model.DeregisterClusterRequest;
import software.amazon.awssdk.services.eks.model.DeregisterClusterResponse;
import software.amazon.awssdk.services.eks.model.DescribeAccessEntryRequest;
import software.amazon.awssdk.services.eks.model.DescribeAccessEntryResponse;
import software.amazon.awssdk.services.eks.model.DescribeAddonConfigurationRequest;
import software.amazon.awssdk.services.eks.model.DescribeAddonConfigurationResponse;
import software.amazon.awssdk.services.eks.model.DescribeAddonRequest;
import software.amazon.awssdk.services.eks.model.DescribeAddonResponse;
import software.amazon.awssdk.services.eks.model.DescribeAddonVersionsRequest;
import software.amazon.awssdk.services.eks.model.DescribeAddonVersionsResponse;
import software.amazon.awssdk.services.eks.model.DescribeClusterRequest;
import software.amazon.awssdk.services.eks.model.DescribeClusterResponse;
import software.amazon.awssdk.services.eks.model.DescribeEksAnywhereSubscriptionRequest;
import software.amazon.awssdk.services.eks.model.DescribeEksAnywhereSubscriptionResponse;
import software.amazon.awssdk.services.eks.model.DescribeFargateProfileRequest;
import software.amazon.awssdk.services.eks.model.DescribeFargateProfileResponse;
import software.amazon.awssdk.services.eks.model.DescribeIdentityProviderConfigRequest;
import software.amazon.awssdk.services.eks.model.DescribeIdentityProviderConfigResponse;
import software.amazon.awssdk.services.eks.model.DescribeInsightRequest;
import software.amazon.awssdk.services.eks.model.DescribeInsightResponse;
import software.amazon.awssdk.services.eks.model.DescribeNodegroupRequest;
import software.amazon.awssdk.services.eks.model.DescribeNodegroupResponse;
import software.amazon.awssdk.services.eks.model.DescribePodIdentityAssociationRequest;
import software.amazon.awssdk.services.eks.model.DescribePodIdentityAssociationResponse;
import software.amazon.awssdk.services.eks.model.DescribeUpdateRequest;
import software.amazon.awssdk.services.eks.model.DescribeUpdateResponse;
import software.amazon.awssdk.services.eks.model.DisassociateAccessPolicyRequest;
import software.amazon.awssdk.services.eks.model.DisassociateAccessPolicyResponse;
import software.amazon.awssdk.services.eks.model.DisassociateIdentityProviderConfigRequest;
import software.amazon.awssdk.services.eks.model.DisassociateIdentityProviderConfigResponse;
import software.amazon.awssdk.services.eks.model.EksException;
import software.amazon.awssdk.services.eks.model.InvalidParameterException;
import software.amazon.awssdk.services.eks.model.InvalidRequestException;
import software.amazon.awssdk.services.eks.model.ListAccessEntriesRequest;
import software.amazon.awssdk.services.eks.model.ListAccessEntriesResponse;
import software.amazon.awssdk.services.eks.model.ListAccessPoliciesRequest;
import software.amazon.awssdk.services.eks.model.ListAccessPoliciesResponse;
import software.amazon.awssdk.services.eks.model.ListAddonsRequest;
import software.amazon.awssdk.services.eks.model.ListAddonsResponse;
import software.amazon.awssdk.services.eks.model.ListAssociatedAccessPoliciesRequest;
import software.amazon.awssdk.services.eks.model.ListAssociatedAccessPoliciesResponse;
import software.amazon.awssdk.services.eks.model.ListClustersRequest;
import software.amazon.awssdk.services.eks.model.ListClustersResponse;
import software.amazon.awssdk.services.eks.model.ListEksAnywhereSubscriptionsRequest;
import software.amazon.awssdk.services.eks.model.ListEksAnywhereSubscriptionsResponse;
import software.amazon.awssdk.services.eks.model.ListFargateProfilesRequest;
import software.amazon.awssdk.services.eks.model.ListFargateProfilesResponse;
import software.amazon.awssdk.services.eks.model.ListIdentityProviderConfigsRequest;
import software.amazon.awssdk.services.eks.model.ListIdentityProviderConfigsResponse;
import software.amazon.awssdk.services.eks.model.ListInsightsRequest;
import software.amazon.awssdk.services.eks.model.ListInsightsResponse;
import software.amazon.awssdk.services.eks.model.ListNodegroupsRequest;
import software.amazon.awssdk.services.eks.model.ListNodegroupsResponse;
import software.amazon.awssdk.services.eks.model.ListPodIdentityAssociationsRequest;
import software.amazon.awssdk.services.eks.model.ListPodIdentityAssociationsResponse;
import software.amazon.awssdk.services.eks.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.eks.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.eks.model.ListUpdatesRequest;
import software.amazon.awssdk.services.eks.model.ListUpdatesResponse;
import software.amazon.awssdk.services.eks.model.NotFoundException;
import software.amazon.awssdk.services.eks.model.RegisterClusterRequest;
import software.amazon.awssdk.services.eks.model.RegisterClusterResponse;
import software.amazon.awssdk.services.eks.model.ResourceInUseException;
import software.amazon.awssdk.services.eks.model.ResourceLimitExceededException;
import software.amazon.awssdk.services.eks.model.ResourceNotFoundException;
import software.amazon.awssdk.services.eks.model.ResourcePropagationDelayException;
import software.amazon.awssdk.services.eks.model.ServerException;
import software.amazon.awssdk.services.eks.model.ServiceUnavailableException;
import software.amazon.awssdk.services.eks.model.TagResourceRequest;
import software.amazon.awssdk.services.eks.model.TagResourceResponse;
import software.amazon.awssdk.services.eks.model.UnsupportedAvailabilityZoneException;
import software.amazon.awssdk.services.eks.model.UntagResourceRequest;
import software.amazon.awssdk.services.eks.model.UntagResourceResponse;
import software.amazon.awssdk.services.eks.model.UpdateAccessEntryRequest;
import software.amazon.awssdk.services.eks.model.UpdateAccessEntryResponse;
import software.amazon.awssdk.services.eks.model.UpdateAddonRequest;
import software.amazon.awssdk.services.eks.model.UpdateAddonResponse;
import software.amazon.awssdk.services.eks.model.UpdateClusterConfigRequest;
import software.amazon.awssdk.services.eks.model.UpdateClusterConfigResponse;
import software.amazon.awssdk.services.eks.model.UpdateClusterVersionRequest;
import software.amazon.awssdk.services.eks.model.UpdateClusterVersionResponse;
import software.amazon.awssdk.services.eks.model.UpdateEksAnywhereSubscriptionRequest;
import software.amazon.awssdk.services.eks.model.UpdateEksAnywhereSubscriptionResponse;
import software.amazon.awssdk.services.eks.model.UpdateNodegroupConfigRequest;
import software.amazon.awssdk.services.eks.model.UpdateNodegroupConfigResponse;
import software.amazon.awssdk.services.eks.model.UpdateNodegroupVersionRequest;
import software.amazon.awssdk.services.eks.model.UpdateNodegroupVersionResponse;
import software.amazon.awssdk.services.eks.model.UpdatePodIdentityAssociationRequest;
import software.amazon.awssdk.services.eks.model.UpdatePodIdentityAssociationResponse;
import software.amazon.awssdk.services.eks.transform.AssociateAccessPolicyRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.AssociateEncryptionConfigRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.AssociateIdentityProviderConfigRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.CreateAccessEntryRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.CreateAddonRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.CreateClusterRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.CreateEksAnywhereSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.CreateFargateProfileRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.CreateNodegroupRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.CreatePodIdentityAssociationRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.DeleteAccessEntryRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.DeleteAddonRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.DeleteClusterRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.DeleteEksAnywhereSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.DeleteFargateProfileRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.DeleteNodegroupRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.DeletePodIdentityAssociationRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.DeregisterClusterRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.DescribeAccessEntryRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.DescribeAddonConfigurationRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.DescribeAddonRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.DescribeAddonVersionsRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.DescribeClusterRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.DescribeEksAnywhereSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.DescribeFargateProfileRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.DescribeIdentityProviderConfigRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.DescribeInsightRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.DescribeNodegroupRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.DescribePodIdentityAssociationRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.DescribeUpdateRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.DisassociateAccessPolicyRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.DisassociateIdentityProviderConfigRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.ListAccessEntriesRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.ListAccessPoliciesRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.ListAddonsRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.ListAssociatedAccessPoliciesRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.ListClustersRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.ListEksAnywhereSubscriptionsRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.ListFargateProfilesRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.ListIdentityProviderConfigsRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.ListInsightsRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.ListNodegroupsRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.ListPodIdentityAssociationsRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.ListUpdatesRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.RegisterClusterRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.UpdateAccessEntryRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.UpdateAddonRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.UpdateClusterConfigRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.UpdateClusterVersionRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.UpdateEksAnywhereSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.UpdateNodegroupConfigRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.UpdateNodegroupVersionRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.UpdatePodIdentityAssociationRequestMarshaller;
import software.amazon.awssdk.services.eks.waiters.EksAsyncWaiter;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link EksAsyncClient}.
 *
 * @see EksAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultEksAsyncClient implements EksAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultEksAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final ScheduledExecutorService executorService;

    protected DefaultEksAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
        this.executorService = clientConfiguration.option(SdkClientOption.SCHEDULED_EXECUTOR_SERVICE);
    }

    /**
     * <p>
     * Associates an access policy and its scope to an access entry. For more information about associating access
     * policies, see <a href="https://docs.aws.amazon.com/eks/latest/userguide/access-policies.html">Associating and
     * disassociating access policies to and from access entries</a> in the <i>Amazon EKS User Guide</i>.
     * </p>
     *
     * @param associateAccessPolicyRequest
     * @return A Java Future containing the result of the AssociateAccessPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>InvalidRequestException The request is invalid given the state of the cluster. Check the state of the
     *         cluster and the associated operations.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.AssociateAccessPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/AssociateAccessPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateAccessPolicyResponse> associateAccessPolicy(
            AssociateAccessPolicyRequest associateAccessPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateAccessPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateAccessPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateAccessPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateAccessPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateAccessPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateAccessPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateAccessPolicyRequest, AssociateAccessPolicyResponse>()
                            .withOperationName("AssociateAccessPolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AssociateAccessPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(associateAccessPolicyRequest));
            CompletableFuture<AssociateAccessPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associates an encryption configuration to an existing cluster.
     * </p>
     * <p>
     * Use this API to enable encryption on existing clusters that don't already have encryption enabled. This allows
     * you to implement a defense-in-depth security strategy without migrating applications to new Amazon EKS clusters.
     * </p>
     *
     * @param associateEncryptionConfigRequest
     * @return A Java Future containing the result of the AssociateEncryptionConfig operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of an <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>InvalidRequestException The request is invalid given the state of the cluster. Check the state of the
     *         cluster and the associated operations.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.AssociateEncryptionConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/AssociateEncryptionConfig" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateEncryptionConfigResponse> associateEncryptionConfig(
            AssociateEncryptionConfigRequest associateEncryptionConfigRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateEncryptionConfigRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateEncryptionConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateEncryptionConfig");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateEncryptionConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateEncryptionConfigResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateEncryptionConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateEncryptionConfigRequest, AssociateEncryptionConfigResponse>()
                            .withOperationName("AssociateEncryptionConfig").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AssociateEncryptionConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(associateEncryptionConfigRequest));
            CompletableFuture<AssociateEncryptionConfigResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associates an identity provider configuration to a cluster.
     * </p>
     * <p>
     * If you want to authenticate identities using an identity provider, you can create an identity provider
     * configuration and associate it to your cluster. After configuring authentication to your cluster you can create
     * Kubernetes <code>Role</code> and <code>ClusterRole</code> objects, assign permissions to them, and then bind them
     * to the identities using Kubernetes <code>RoleBinding</code> and <code>ClusterRoleBinding</code> objects. For more
     * information see <a href="https://kubernetes.io/docs/reference/access-authn-authz/rbac/">Using RBAC
     * Authorization</a> in the Kubernetes documentation.
     * </p>
     *
     * @param associateIdentityProviderConfigRequest
     * @return A Java Future containing the result of the AssociateIdentityProviderConfig operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of an <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>InvalidRequestException The request is invalid given the state of the cluster. Check the state of the
     *         cluster and the associated operations.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.AssociateIdentityProviderConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/AssociateIdentityProviderConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateIdentityProviderConfigResponse> associateIdentityProviderConfig(
            AssociateIdentityProviderConfigRequest associateIdentityProviderConfigRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateIdentityProviderConfigRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                associateIdentityProviderConfigRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateIdentityProviderConfig");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateIdentityProviderConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateIdentityProviderConfigResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateIdentityProviderConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateIdentityProviderConfigRequest, AssociateIdentityProviderConfigResponse>()
                            .withOperationName("AssociateIdentityProviderConfig").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AssociateIdentityProviderConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(associateIdentityProviderConfigRequest));
            CompletableFuture<AssociateIdentityProviderConfigResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an access entry.
     * </p>
     * <p>
     * An access entry allows an IAM principal to access your cluster. Access entries can replace the need to maintain
     * entries in the <code>aws-auth</code> <code>ConfigMap</code> for authentication. You have the following options
     * for authorizing an IAM principal to access Kubernetes objects on your cluster: Kubernetes role-based access
     * control (RBAC), Amazon EKS, or both. Kubernetes RBAC authorization requires you to create and manage Kubernetes
     * <code>Role</code>, <code>ClusterRole</code>, <code>RoleBinding</code>, and <code>ClusterRoleBinding</code>
     * objects, in addition to managing access entries. If you use Amazon EKS authorization exclusively, you don't need
     * to create and manage Kubernetes <code>Role</code>, <code>ClusterRole</code>, <code>RoleBinding</code>, and
     * <code>ClusterRoleBinding</code> objects.
     * </p>
     * <p>
     * For more information about access entries, see <a
     * href="https://docs.aws.amazon.com/eks/latest/userguide/access-entries.html">Access entries</a> in the <i>Amazon
     * EKS User Guide</i>.
     * </p>
     *
     * @param createAccessEntryRequest
     * @return A Java Future containing the result of the CreateAccessEntry operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>InvalidRequestException The request is invalid given the state of the cluster. Check the state of the
     *         cluster and the associated operations.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ResourceLimitExceededException You have encountered a service limit on the specified resource.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.CreateAccessEntry
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/CreateAccessEntry" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAccessEntryResponse> createAccessEntry(CreateAccessEntryRequest createAccessEntryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createAccessEntryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAccessEntryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAccessEntry");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAccessEntryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateAccessEntryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAccessEntryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAccessEntryRequest, CreateAccessEntryResponse>()
                            .withOperationName("CreateAccessEntry").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateAccessEntryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createAccessEntryRequest));
            CompletableFuture<CreateAccessEntryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Amazon EKS add-on.
     * </p>
     * <p>
     * Amazon EKS add-ons help to automate the provisioning and lifecycle management of common operational software for
     * Amazon EKS clusters. For more information, see <a
     * href="https://docs.aws.amazon.com/eks/latest/userguide/eks-add-ons.html">Amazon EKS add-ons</a> in the <i>Amazon
     * EKS User Guide</i>.
     * </p>
     *
     * @param createAddonRequest
     * @return A Java Future containing the result of the CreateAddon operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>InvalidRequestException The request is invalid given the state of the cluster. Check the state of the
     *         cluster and the associated operations.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of an <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.CreateAddon
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/CreateAddon" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAddonResponse> createAddon(CreateAddonRequest createAddonRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createAddonRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAddonRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAddon");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAddonResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateAddonResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAddonResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAddonRequest, CreateAddonResponse>()
                            .withOperationName("CreateAddon").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateAddonRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createAddonRequest));
            CompletableFuture<CreateAddonResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Amazon EKS control plane.
     * </p>
     * <p>
     * The Amazon EKS control plane consists of control plane instances that run the Kubernetes software, such as
     * <code>etcd</code> and the API server. The control plane runs in an account managed by Amazon Web Services, and
     * the Kubernetes API is exposed by the Amazon EKS API server endpoint. Each Amazon EKS cluster control plane is
     * single tenant and unique. It runs on its own set of Amazon EC2 instances.
     * </p>
     * <p>
     * The cluster control plane is provisioned across multiple Availability Zones and fronted by an Elastic Load
     * Balancing Network Load Balancer. Amazon EKS also provisions elastic network interfaces in your VPC subnets to
     * provide connectivity from the control plane instances to the nodes (for example, to support
     * <code>kubectl exec</code>, <code>logs</code>, and <code>proxy</code> data flows).
     * </p>
     * <p>
     * Amazon EKS nodes run in your Amazon Web Services account and connect to your cluster's control plane over the
     * Kubernetes API server endpoint and a certificate file that is created for your cluster.
     * </p>
     * <p>
     * You can use the <code>endpointPublicAccess</code> and <code>endpointPrivateAccess</code> parameters to enable or
     * disable public and private access to your cluster's Kubernetes API server endpoint. By default, public access is
     * enabled, and private access is disabled. For more information, see <a
     * href="https://docs.aws.amazon.com/eks/latest/userguide/cluster-endpoint.html">Amazon EKS Cluster Endpoint Access
     * Control</a> in the <i> <i>Amazon EKS User Guide</i> </i>.
     * </p>
     * <p>
     * You can use the <code>logging</code> parameter to enable or disable exporting the Kubernetes control plane logs
     * for your cluster to CloudWatch Logs. By default, cluster control plane logs aren't exported to CloudWatch Logs.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/eks/latest/userguide/control-plane-logs.html">Amazon EKS Cluster Control Plane
     * Logs</a> in the <i> <i>Amazon EKS User Guide</i> </i>.
     * </p>
     * <note>
     * <p>
     * CloudWatch Logs ingestion, archive storage, and data scanning rates apply to exported control plane logs. For
     * more information, see <a href="http://aws.amazon.com/cloudwatch/pricing/">CloudWatch Pricing</a>.
     * </p>
     * </note>
     * <p>
     * In most cases, it takes several minutes to create a cluster. After you create an Amazon EKS cluster, you must
     * configure your Kubernetes tooling to communicate with the API server and launch nodes into your cluster. For more
     * information, see <a href="https://docs.aws.amazon.com/eks/latest/userguide/managing-auth.html">Managing Cluster
     * Authentication</a> and <a href="https://docs.aws.amazon.com/eks/latest/userguide/launch-workers.html">Launching
     * Amazon EKS nodes</a> in the <i>Amazon EKS User Guide</i>.
     * </p>
     *
     * @param createClusterRequest
     * @return A Java Future containing the result of the CreateCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>ResourceLimitExceededException You have encountered a service limit on the specified resource.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of an <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Back off and retry the operation.</li>
     *         <li>UnsupportedAvailabilityZoneException At least one of your specified cluster subnets is in an
     *         Availability Zone that does not support Amazon EKS. The exception output specifies the supported
     *         Availability Zones for your account, from which you can choose subnets for your cluster.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.CreateCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/CreateCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateClusterResponse> createCluster(CreateClusterRequest createClusterRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createClusterRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCluster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateClusterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateClusterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateClusterRequest, CreateClusterResponse>()
                            .withOperationName("CreateCluster").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateClusterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createClusterRequest));
            CompletableFuture<CreateClusterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an EKS Anywhere subscription. When a subscription is created, it is a contract agreement for the length
     * of the term specified in the request. Licenses that are used to validate support are provisioned in Amazon Web
     * Services License Manager and the caller account is granted access to EKS Anywhere Curated Packages.
     * </p>
     *
     * @param createEksAnywhereSubscriptionRequest
     * @return A Java Future containing the result of the CreateEksAnywhereSubscription operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceLimitExceededException You have encountered a service limit on the specified resource.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of an <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Back off and retry the operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.CreateEksAnywhereSubscription
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/CreateEksAnywhereSubscription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateEksAnywhereSubscriptionResponse> createEksAnywhereSubscription(
            CreateEksAnywhereSubscriptionRequest createEksAnywhereSubscriptionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createEksAnywhereSubscriptionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createEksAnywhereSubscriptionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateEksAnywhereSubscription");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateEksAnywhereSubscriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateEksAnywhereSubscriptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateEksAnywhereSubscriptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateEksAnywhereSubscriptionRequest, CreateEksAnywhereSubscriptionResponse>()
                            .withOperationName("CreateEksAnywhereSubscription").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateEksAnywhereSubscriptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createEksAnywhereSubscriptionRequest));
            CompletableFuture<CreateEksAnywhereSubscriptionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Fargate profile for your Amazon EKS cluster. You must have at least one Fargate profile in a cluster
     * to be able to run pods on Fargate.
     * </p>
     * <p>
     * The Fargate profile allows an administrator to declare which pods run on Fargate and specify which pods run on
     * which Fargate profile. This declaration is done through the profile’s selectors. Each profile can have up to five
     * selectors that contain a namespace and labels. A namespace is required for every selector. The label field
     * consists of multiple optional key-value pairs. Pods that match the selectors are scheduled on Fargate. If a
     * to-be-scheduled pod matches any of the selectors in the Fargate profile, then that pod is run on Fargate.
     * </p>
     * <p>
     * When you create a Fargate profile, you must specify a pod execution role to use with the pods that are scheduled
     * with the profile. This role is added to the cluster's Kubernetes <a
     * href="https://kubernetes.io/docs/reference/access-authn-authz/rbac/">Role Based Access Control</a> (RBAC) for
     * authorization so that the <code>kubelet</code> that is running on the Fargate infrastructure can register with
     * your Amazon EKS cluster so that it can appear in your cluster as a node. The pod execution role also provides IAM
     * permissions to the Fargate infrastructure to allow read access to Amazon ECR image repositories. For more
     * information, see <a href="https://docs.aws.amazon.com/eks/latest/userguide/pod-execution-role.html">Pod Execution
     * Role</a> in the <i>Amazon EKS User Guide</i>.
     * </p>
     * <p>
     * Fargate profiles are immutable. However, you can create a new updated profile to replace an existing profile and
     * then delete the original after the updated profile has finished creating.
     * </p>
     * <p>
     * If any Fargate profiles in a cluster are in the <code>DELETING</code> status, you must wait for that Fargate
     * profile to finish deleting before you can create any other profiles in that cluster.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/eks/latest/userguide/fargate-profile.html">Fargate
     * profile</a> in the <i>Amazon EKS User Guide</i>.
     * </p>
     *
     * @param createFargateProfileRequest
     * @return A Java Future containing the result of the CreateFargateProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>InvalidRequestException The request is invalid given the state of the cluster. Check the state of the
     *         cluster and the associated operations.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of an <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceLimitExceededException You have encountered a service limit on the specified resource.</li>
     *         <li>UnsupportedAvailabilityZoneException At least one of your specified cluster subnets is in an
     *         Availability Zone that does not support Amazon EKS. The exception output specifies the supported
     *         Availability Zones for your account, from which you can choose subnets for your cluster.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.CreateFargateProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/CreateFargateProfile" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateFargateProfileResponse> createFargateProfile(
            CreateFargateProfileRequest createFargateProfileRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createFargateProfileRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createFargateProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateFargateProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateFargateProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateFargateProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateFargateProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateFargateProfileRequest, CreateFargateProfileResponse>()
                            .withOperationName("CreateFargateProfile").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateFargateProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createFargateProfileRequest));
            CompletableFuture<CreateFargateProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a managed node group for an Amazon EKS cluster.
     * </p>
     * <p>
     * You can only create a node group for your cluster that is equal to the current Kubernetes version for the
     * cluster. All node groups are created with the latest AMI release version for the respective minor Kubernetes
     * version of the cluster, unless you deploy a custom AMI using a launch template. For more information about using
     * launch templates, see <a href="https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html">Launch
     * template support</a>.
     * </p>
     * <p>
     * An Amazon EKS managed node group is an Amazon EC2 Auto Scaling group and associated Amazon EC2 instances that are
     * managed by Amazon Web Services for an Amazon EKS cluster. For more information, see <a
     * href="https://docs.aws.amazon.com/eks/latest/userguide/managed-node-groups.html">Managed node groups</a> in the
     * <i>Amazon EKS User Guide</i>.
     * </p>
     * <note>
     * <p>
     * Windows AMI types are only supported for commercial Amazon Web Services Regions that support Windows on Amazon
     * EKS.
     * </p>
     * </note>
     *
     * @param createNodegroupRequest
     * @return A Java Future containing the result of the CreateNodegroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>ResourceLimitExceededException You have encountered a service limit on the specified resource.</li>
     *         <li>InvalidRequestException The request is invalid given the state of the cluster. Check the state of the
     *         cluster and the associated operations.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of an <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Back off and retry the operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.CreateNodegroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/CreateNodegroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateNodegroupResponse> createNodegroup(CreateNodegroupRequest createNodegroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createNodegroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createNodegroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateNodegroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateNodegroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateNodegroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateNodegroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateNodegroupRequest, CreateNodegroupResponse>()
                            .withOperationName("CreateNodegroup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateNodegroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createNodegroupRequest));
            CompletableFuture<CreateNodegroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an EKS Pod Identity association between a service account in an Amazon EKS cluster and an IAM role with
     * <i>EKS Pod Identity</i>. Use EKS Pod Identity to give temporary IAM credentials to pods and the credentials are
     * rotated automatically.
     * </p>
     * <p>
     * Amazon EKS Pod Identity associations provide the ability to manage credentials for your applications, similar to
     * the way that Amazon EC2 instance profiles provide credentials to Amazon EC2 instances.
     * </p>
     * <p>
     * If a pod uses a service account that has an association, Amazon EKS sets environment variables in the containers
     * of the pod. The environment variables configure the Amazon Web Services SDKs, including the Command Line
     * Interface, to use the EKS Pod Identity credentials.
     * </p>
     * <p>
     * Pod Identity is a simpler method than <i>IAM roles for service accounts</i>, as this method doesn't use OIDC
     * identity providers. Additionally, you can configure a role for Pod Identity once, and reuse it across clusters.
     * </p>
     *
     * @param createPodIdentityAssociationRequest
     * @return A Java Future containing the result of the CreatePodIdentityAssociation operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>InvalidRequestException The request is invalid given the state of the cluster. Check the state of the
     *         cluster and the associated operations.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ResourceLimitExceededException You have encountered a service limit on the specified resource.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.CreatePodIdentityAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/CreatePodIdentityAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePodIdentityAssociationResponse> createPodIdentityAssociation(
            CreatePodIdentityAssociationRequest createPodIdentityAssociationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPodIdentityAssociationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPodIdentityAssociationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePodIdentityAssociation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePodIdentityAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePodIdentityAssociationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePodIdentityAssociationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePodIdentityAssociationRequest, CreatePodIdentityAssociationResponse>()
                            .withOperationName("CreatePodIdentityAssociation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreatePodIdentityAssociationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createPodIdentityAssociationRequest));
            CompletableFuture<CreatePodIdentityAssociationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an access entry.
     * </p>
     * <p>
     * Deleting an access entry of a type other than <code>Standard</code> can cause your cluster to function
     * improperly. If you delete an access entry in error, you can recreate it.
     * </p>
     *
     * @param deleteAccessEntryRequest
     * @return A Java Future containing the result of the DeleteAccessEntry operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>InvalidRequestException The request is invalid given the state of the cluster. Check the state of the
     *         cluster and the associated operations.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.DeleteAccessEntry
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/DeleteAccessEntry" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAccessEntryResponse> deleteAccessEntry(DeleteAccessEntryRequest deleteAccessEntryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAccessEntryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAccessEntryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAccessEntry");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAccessEntryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteAccessEntryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAccessEntryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAccessEntryRequest, DeleteAccessEntryResponse>()
                            .withOperationName("DeleteAccessEntry").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteAccessEntryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteAccessEntryRequest));
            CompletableFuture<DeleteAccessEntryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Amazon EKS add-on.
     * </p>
     * <p>
     * When you remove an add-on, it's deleted from the cluster. You can always manually start an add-on on the cluster
     * using the Kubernetes API.
     * </p>
     *
     * @param deleteAddonRequest
     * @return A Java Future containing the result of the DeleteAddon operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>InvalidRequestException The request is invalid given the state of the cluster. Check the state of the
     *         cluster and the associated operations.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of an <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.DeleteAddon
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/DeleteAddon" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAddonResponse> deleteAddon(DeleteAddonRequest deleteAddonRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAddonRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAddonRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAddon");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAddonResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteAddonResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAddonResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAddonRequest, DeleteAddonResponse>()
                            .withOperationName("DeleteAddon").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteAddonRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteAddonRequest));
            CompletableFuture<DeleteAddonResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Amazon EKS cluster control plane.
     * </p>
     * <p>
     * If you have active services in your cluster that are associated with a load balancer, you must delete those
     * services before deleting the cluster so that the load balancers are deleted properly. Otherwise, you can have
     * orphaned resources in your VPC that prevent you from being able to delete the VPC. For more information, see <a
     * href="https://docs.aws.amazon.com/eks/latest/userguide/delete-cluster.html">Deleting a cluster</a> in the
     * <i>Amazon EKS User Guide</i>.
     * </p>
     * <p>
     * If you have managed node groups or Fargate profiles attached to the cluster, you must delete them first. For more
     * information, see <code>DeleteNodgroup</code> and <code>DeleteFargateProfile</code>.
     * </p>
     *
     * @param deleteClusterRequest
     * @return A Java Future containing the result of the DeleteCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of an <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Back off and retry the operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.DeleteCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/DeleteCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteClusterResponse> deleteCluster(DeleteClusterRequest deleteClusterRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteClusterRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCluster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteClusterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteClusterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteClusterRequest, DeleteClusterResponse>()
                            .withOperationName("DeleteCluster").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteClusterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteClusterRequest));
            CompletableFuture<DeleteClusterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an expired or inactive subscription. Deleting inactive subscriptions removes them from the Amazon Web
     * Services Management Console view and from list/describe API responses. Subscriptions can only be cancelled within
     * 7 days of creation and are cancelled by creating a ticket in the Amazon Web Services Support Center.
     * </p>
     *
     * @param deleteEksAnywhereSubscriptionRequest
     * @return A Java Future containing the result of the DeleteEksAnywhereSubscription operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of an <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.</li>
     *         <li>InvalidRequestException The request is invalid given the state of the cluster. Check the state of the
     *         cluster and the associated operations.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.DeleteEksAnywhereSubscription
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/DeleteEksAnywhereSubscription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteEksAnywhereSubscriptionResponse> deleteEksAnywhereSubscription(
            DeleteEksAnywhereSubscriptionRequest deleteEksAnywhereSubscriptionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteEksAnywhereSubscriptionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteEksAnywhereSubscriptionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteEksAnywhereSubscription");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteEksAnywhereSubscriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteEksAnywhereSubscriptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteEksAnywhereSubscriptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteEksAnywhereSubscriptionRequest, DeleteEksAnywhereSubscriptionResponse>()
                            .withOperationName("DeleteEksAnywhereSubscription").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteEksAnywhereSubscriptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteEksAnywhereSubscriptionRequest));
            CompletableFuture<DeleteEksAnywhereSubscriptionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Fargate profile.
     * </p>
     * <p>
     * When you delete a Fargate profile, any <code>Pod</code> running on Fargate that was created with the profile is
     * deleted. If the <code>Pod</code> matches another Fargate profile, then it is scheduled on Fargate with that
     * profile. If it no longer matches any Fargate profiles, then it's not scheduled on Fargate and may remain in a
     * pending state.
     * </p>
     * <p>
     * Only one Fargate profile in a cluster can be in the <code>DELETING</code> status at a time. You must wait for a
     * Fargate profile to finish deleting before you can delete any other profiles in that cluster.
     * </p>
     *
     * @param deleteFargateProfileRequest
     * @return A Java Future containing the result of the DeleteFargateProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of an <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.DeleteFargateProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/DeleteFargateProfile" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteFargateProfileResponse> deleteFargateProfile(
            DeleteFargateProfileRequest deleteFargateProfileRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteFargateProfileRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteFargateProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteFargateProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteFargateProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteFargateProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteFargateProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteFargateProfileRequest, DeleteFargateProfileResponse>()
                            .withOperationName("DeleteFargateProfile").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteFargateProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteFargateProfileRequest));
            CompletableFuture<DeleteFargateProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a managed node group.
     * </p>
     *
     * @param deleteNodegroupRequest
     * @return A Java Future containing the result of the DeleteNodegroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of an <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Back off and retry the operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.DeleteNodegroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/DeleteNodegroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteNodegroupResponse> deleteNodegroup(DeleteNodegroupRequest deleteNodegroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteNodegroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteNodegroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteNodegroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteNodegroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteNodegroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteNodegroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteNodegroupRequest, DeleteNodegroupResponse>()
                            .withOperationName("DeleteNodegroup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteNodegroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteNodegroupRequest));
            CompletableFuture<DeleteNodegroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a EKS Pod Identity association.
     * </p>
     * <p>
     * The temporary Amazon Web Services credentials from the previous IAM role session might still be valid until the
     * session expiry. If you need to immediately revoke the temporary session credentials, then go to the role in the
     * IAM console.
     * </p>
     *
     * @param deletePodIdentityAssociationRequest
     * @return A Java Future containing the result of the DeletePodIdentityAssociation operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>InvalidRequestException The request is invalid given the state of the cluster. Check the state of the
     *         cluster and the associated operations.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.DeletePodIdentityAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/DeletePodIdentityAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePodIdentityAssociationResponse> deletePodIdentityAssociation(
            DeletePodIdentityAssociationRequest deletePodIdentityAssociationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePodIdentityAssociationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePodIdentityAssociationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePodIdentityAssociation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePodIdentityAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeletePodIdentityAssociationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePodIdentityAssociationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePodIdentityAssociationRequest, DeletePodIdentityAssociationResponse>()
                            .withOperationName("DeletePodIdentityAssociation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeletePodIdentityAssociationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deletePodIdentityAssociationRequest));
            CompletableFuture<DeletePodIdentityAssociationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deregisters a connected cluster to remove it from the Amazon EKS control plane.
     * </p>
     * <p>
     * A connected cluster is a Kubernetes cluster that you've connected to your control plane using the <a
     * href="https://docs.aws.amazon.com/eks/latest/userguide/eks-connector.html">Amazon EKS Connector</a>.
     * </p>
     *
     * @param deregisterClusterRequest
     * @return A Java Future containing the result of the DeregisterCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of an <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Back off and retry the operation.</li>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.DeregisterCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/DeregisterCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeregisterClusterResponse> deregisterCluster(DeregisterClusterRequest deregisterClusterRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deregisterClusterRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deregisterClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeregisterCluster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeregisterClusterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeregisterClusterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeregisterClusterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeregisterClusterRequest, DeregisterClusterResponse>()
                            .withOperationName("DeregisterCluster").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeregisterClusterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deregisterClusterRequest));
            CompletableFuture<DeregisterClusterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes an access entry.
     * </p>
     *
     * @param describeAccessEntryRequest
     * @return A Java Future containing the result of the DescribeAccessEntry operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>InvalidRequestException The request is invalid given the state of the cluster. Check the state of the
     *         cluster and the associated operations.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.DescribeAccessEntry
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/DescribeAccessEntry" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAccessEntryResponse> describeAccessEntry(
            DescribeAccessEntryRequest describeAccessEntryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeAccessEntryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAccessEntryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAccessEntry");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAccessEntryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeAccessEntryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAccessEntryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAccessEntryRequest, DescribeAccessEntryResponse>()
                            .withOperationName("DescribeAccessEntry").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeAccessEntryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeAccessEntryRequest));
            CompletableFuture<DescribeAccessEntryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes an Amazon EKS add-on.
     * </p>
     *
     * @param describeAddonRequest
     * @return A Java Future containing the result of the DescribeAddon operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>InvalidRequestException The request is invalid given the state of the cluster. Check the state of the
     *         cluster and the associated operations.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of an <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.DescribeAddon
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/DescribeAddon" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAddonResponse> describeAddon(DescribeAddonRequest describeAddonRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeAddonRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAddonRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAddon");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAddonResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeAddonResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAddonResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAddonRequest, DescribeAddonResponse>()
                            .withOperationName("DescribeAddon").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeAddonRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeAddonRequest));
            CompletableFuture<DescribeAddonResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns configuration options.
     * </p>
     *
     * @param describeAddonConfigurationRequest
     * @return A Java Future containing the result of the DescribeAddonConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.DescribeAddonConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/DescribeAddonConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAddonConfigurationResponse> describeAddonConfiguration(
            DescribeAddonConfigurationRequest describeAddonConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeAddonConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAddonConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAddonConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAddonConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeAddonConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAddonConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAddonConfigurationRequest, DescribeAddonConfigurationResponse>()
                            .withOperationName("DescribeAddonConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeAddonConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeAddonConfigurationRequest));
            CompletableFuture<DescribeAddonConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the versions for an add-on.
     * </p>
     * <p>
     * Information such as the Kubernetes versions that you can use the add-on with, the <code>owner</code>,
     * <code>publisher</code>, and the <code>type</code> of the add-on are returned.
     * </p>
     *
     * @param describeAddonVersionsRequest
     * @return A Java Future containing the result of the DescribeAddonVersions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.DescribeAddonVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/DescribeAddonVersions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAddonVersionsResponse> describeAddonVersions(
            DescribeAddonVersionsRequest describeAddonVersionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeAddonVersionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAddonVersionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAddonVersions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAddonVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeAddonVersionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAddonVersionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAddonVersionsRequest, DescribeAddonVersionsResponse>()
                            .withOperationName("DescribeAddonVersions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeAddonVersionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeAddonVersionsRequest));
            CompletableFuture<DescribeAddonVersionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes an Amazon EKS cluster.
     * </p>
     * <p>
     * The API server endpoint and certificate authority data returned by this operation are required for
     * <code>kubelet</code> and <code>kubectl</code> to communicate with your Kubernetes API server. For more
     * information, see <a href="https://docs.aws.amazon.com/eks/latest/userguide/create-kubeconfig.html">Creating or
     * updating a <code>kubeconfig</code> file for an Amazon EKS cluster</a>.
     * </p>
     * <note>
     * <p>
     * The API server endpoint and certificate authority data aren't available until the cluster reaches the
     * <code>ACTIVE</code> state.
     * </p>
     * </note>
     *
     * @param describeClusterRequest
     * @return A Java Future containing the result of the DescribeCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of an <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Back off and retry the operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.DescribeCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/DescribeCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeClusterResponse> describeCluster(DescribeClusterRequest describeClusterRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeClusterRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeCluster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeClusterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeClusterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeClusterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeClusterRequest, DescribeClusterResponse>()
                            .withOperationName("DescribeCluster").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeClusterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeClusterRequest));
            CompletableFuture<DescribeClusterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns descriptive information about a subscription.
     * </p>
     *
     * @param describeEksAnywhereSubscriptionRequest
     * @return A Java Future containing the result of the DescribeEksAnywhereSubscription operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of an <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Back off and retry the operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.DescribeEksAnywhereSubscription
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/DescribeEksAnywhereSubscription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEksAnywhereSubscriptionResponse> describeEksAnywhereSubscription(
            DescribeEksAnywhereSubscriptionRequest describeEksAnywhereSubscriptionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeEksAnywhereSubscriptionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeEksAnywhereSubscriptionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEksAnywhereSubscription");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEksAnywhereSubscriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeEksAnywhereSubscriptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEksAnywhereSubscriptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEksAnywhereSubscriptionRequest, DescribeEksAnywhereSubscriptionResponse>()
                            .withOperationName("DescribeEksAnywhereSubscription").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeEksAnywhereSubscriptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeEksAnywhereSubscriptionRequest));
            CompletableFuture<DescribeEksAnywhereSubscriptionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes an Fargate profile.
     * </p>
     *
     * @param describeFargateProfileRequest
     * @return A Java Future containing the result of the DescribeFargateProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of an <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.DescribeFargateProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/DescribeFargateProfile" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeFargateProfileResponse> describeFargateProfile(
            DescribeFargateProfileRequest describeFargateProfileRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeFargateProfileRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFargateProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFargateProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeFargateProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeFargateProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeFargateProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeFargateProfileRequest, DescribeFargateProfileResponse>()
                            .withOperationName("DescribeFargateProfile").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeFargateProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeFargateProfileRequest));
            CompletableFuture<DescribeFargateProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes an identity provider configuration.
     * </p>
     *
     * @param describeIdentityProviderConfigRequest
     * @return A Java Future containing the result of the DescribeIdentityProviderConfig operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of an <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Back off and retry the operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.DescribeIdentityProviderConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/DescribeIdentityProviderConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeIdentityProviderConfigResponse> describeIdentityProviderConfig(
            DescribeIdentityProviderConfigRequest describeIdentityProviderConfigRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeIdentityProviderConfigRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeIdentityProviderConfigRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeIdentityProviderConfig");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeIdentityProviderConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeIdentityProviderConfigResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeIdentityProviderConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeIdentityProviderConfigRequest, DescribeIdentityProviderConfigResponse>()
                            .withOperationName("DescribeIdentityProviderConfig").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeIdentityProviderConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeIdentityProviderConfigRequest));
            CompletableFuture<DescribeIdentityProviderConfigResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns details about an insight that you specify using its ID.
     * </p>
     *
     * @param describeInsightRequest
     * @return A Java Future containing the result of the DescribeInsight operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>InvalidRequestException The request is invalid given the state of the cluster. Check the state of the
     *         cluster and the associated operations.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.DescribeInsight
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/DescribeInsight" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeInsightResponse> describeInsight(DescribeInsightRequest describeInsightRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeInsightRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeInsightRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeInsight");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeInsightResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeInsightResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeInsightResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeInsightRequest, DescribeInsightResponse>()
                            .withOperationName("DescribeInsight").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeInsightRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeInsightRequest));
            CompletableFuture<DescribeInsightResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes a managed node group.
     * </p>
     *
     * @param describeNodegroupRequest
     * @return A Java Future containing the result of the DescribeNodegroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of an <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Back off and retry the operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.DescribeNodegroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/DescribeNodegroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeNodegroupResponse> describeNodegroup(DescribeNodegroupRequest describeNodegroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeNodegroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeNodegroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeNodegroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeNodegroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeNodegroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeNodegroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeNodegroupRequest, DescribeNodegroupResponse>()
                            .withOperationName("DescribeNodegroup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeNodegroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeNodegroupRequest));
            CompletableFuture<DescribeNodegroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns descriptive information about an EKS Pod Identity association.
     * </p>
     * <p>
     * This action requires the ID of the association. You can get the ID from the response to the
     * <code>CreatePodIdentityAssocation</code> for newly created associations. Or, you can list the IDs for
     * associations with <code>ListPodIdentityAssociations</code> and filter the list by namespace or service account.
     * </p>
     *
     * @param describePodIdentityAssociationRequest
     * @return A Java Future containing the result of the DescribePodIdentityAssociation operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>InvalidRequestException The request is invalid given the state of the cluster. Check the state of the
     *         cluster and the associated operations.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.DescribePodIdentityAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/DescribePodIdentityAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribePodIdentityAssociationResponse> describePodIdentityAssociation(
            DescribePodIdentityAssociationRequest describePodIdentityAssociationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describePodIdentityAssociationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describePodIdentityAssociationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePodIdentityAssociation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribePodIdentityAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribePodIdentityAssociationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribePodIdentityAssociationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribePodIdentityAssociationRequest, DescribePodIdentityAssociationResponse>()
                            .withOperationName("DescribePodIdentityAssociation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribePodIdentityAssociationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describePodIdentityAssociationRequest));
            CompletableFuture<DescribePodIdentityAssociationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes an update to an Amazon EKS resource.
     * </p>
     * <p>
     * When the status of the update is <code>Succeeded</code>, the update is complete. If an update fails, the status
     * is <code>Failed</code>, and an error detail explains the reason for the failure.
     * </p>
     *
     * @param describeUpdateRequest
     *        Describes an update request.
     * @return A Java Future containing the result of the DescribeUpdate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of an <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.DescribeUpdate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/DescribeUpdate" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeUpdateResponse> describeUpdate(DescribeUpdateRequest describeUpdateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeUpdateRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeUpdateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeUpdate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeUpdateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeUpdateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeUpdateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeUpdateRequest, DescribeUpdateResponse>()
                            .withOperationName("DescribeUpdate").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeUpdateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeUpdateRequest));
            CompletableFuture<DescribeUpdateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disassociates an access policy from an access entry.
     * </p>
     *
     * @param disassociateAccessPolicyRequest
     * @return A Java Future containing the result of the DisassociateAccessPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>InvalidRequestException The request is invalid given the state of the cluster. Check the state of the
     *         cluster and the associated operations.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.DisassociateAccessPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/DisassociateAccessPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateAccessPolicyResponse> disassociateAccessPolicy(
            DisassociateAccessPolicyRequest disassociateAccessPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateAccessPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateAccessPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateAccessPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateAccessPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateAccessPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateAccessPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateAccessPolicyRequest, DisassociateAccessPolicyResponse>()
                            .withOperationName("DisassociateAccessPolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisassociateAccessPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disassociateAccessPolicyRequest));
            CompletableFuture<DisassociateAccessPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disassociates an identity provider configuration from a cluster.
     * </p>
     * <p>
     * If you disassociate an identity provider from your cluster, users included in the provider can no longer access
     * the cluster. However, you can still access the cluster with IAM principals.
     * </p>
     *
     * @param disassociateIdentityProviderConfigRequest
     * @return A Java Future containing the result of the DisassociateIdentityProviderConfig operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of an <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>InvalidRequestException The request is invalid given the state of the cluster. Check the state of the
     *         cluster and the associated operations.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.DisassociateIdentityProviderConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/DisassociateIdentityProviderConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateIdentityProviderConfigResponse> disassociateIdentityProviderConfig(
            DisassociateIdentityProviderConfigRequest disassociateIdentityProviderConfigRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateIdentityProviderConfigRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateIdentityProviderConfigRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateIdentityProviderConfig");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateIdentityProviderConfigResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DisassociateIdentityProviderConfigResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateIdentityProviderConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateIdentityProviderConfigRequest, DisassociateIdentityProviderConfigResponse>()
                            .withOperationName("DisassociateIdentityProviderConfig").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisassociateIdentityProviderConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disassociateIdentityProviderConfigRequest));
            CompletableFuture<DisassociateIdentityProviderConfigResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the access entries for your cluster.
     * </p>
     *
     * @param listAccessEntriesRequest
     * @return A Java Future containing the result of the ListAccessEntries operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>InvalidRequestException The request is invalid given the state of the cluster. Check the state of the
     *         cluster and the associated operations.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.ListAccessEntries
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/ListAccessEntries" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListAccessEntriesResponse> listAccessEntries(ListAccessEntriesRequest listAccessEntriesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAccessEntriesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAccessEntriesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAccessEntries");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAccessEntriesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAccessEntriesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAccessEntriesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAccessEntriesRequest, ListAccessEntriesResponse>()
                            .withOperationName("ListAccessEntries").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListAccessEntriesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listAccessEntriesRequest));
            CompletableFuture<ListAccessEntriesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the available access policies.
     * </p>
     *
     * @param listAccessPoliciesRequest
     * @return A Java Future containing the result of the ListAccessPolicies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.ListAccessPolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/ListAccessPolicies" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListAccessPoliciesResponse> listAccessPolicies(ListAccessPoliciesRequest listAccessPoliciesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAccessPoliciesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAccessPoliciesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAccessPolicies");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAccessPoliciesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAccessPoliciesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAccessPoliciesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAccessPoliciesRequest, ListAccessPoliciesResponse>()
                            .withOperationName("ListAccessPolicies").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListAccessPoliciesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listAccessPoliciesRequest));
            CompletableFuture<ListAccessPoliciesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the installed add-ons.
     * </p>
     *
     * @param listAddonsRequest
     * @return A Java Future containing the result of the ListAddons operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>InvalidRequestException The request is invalid given the state of the cluster. Check the state of the
     *         cluster and the associated operations.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of an <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.ListAddons
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/ListAddons" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListAddonsResponse> listAddons(ListAddonsRequest listAddonsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAddonsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAddonsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAddons");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAddonsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListAddonsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAddonsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAddonsRequest, ListAddonsResponse>().withOperationName("ListAddons")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListAddonsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listAddonsRequest));
            CompletableFuture<ListAddonsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the access policies associated with an access entry.
     * </p>
     *
     * @param listAssociatedAccessPoliciesRequest
     * @return A Java Future containing the result of the ListAssociatedAccessPolicies operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>InvalidRequestException The request is invalid given the state of the cluster. Check the state of the
     *         cluster and the associated operations.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.ListAssociatedAccessPolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/ListAssociatedAccessPolicies"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAssociatedAccessPoliciesResponse> listAssociatedAccessPolicies(
            ListAssociatedAccessPoliciesRequest listAssociatedAccessPoliciesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAssociatedAccessPoliciesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAssociatedAccessPoliciesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAssociatedAccessPolicies");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAssociatedAccessPoliciesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAssociatedAccessPoliciesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAssociatedAccessPoliciesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAssociatedAccessPoliciesRequest, ListAssociatedAccessPoliciesResponse>()
                            .withOperationName("ListAssociatedAccessPolicies").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListAssociatedAccessPoliciesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listAssociatedAccessPoliciesRequest));
            CompletableFuture<ListAssociatedAccessPoliciesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the Amazon EKS clusters in your Amazon Web Services account in the specified Amazon Web Services Region.
     * </p>
     *
     * @param listClustersRequest
     * @return A Java Future containing the result of the ListClusters operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of an <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Back off and retry the operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.ListClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/ListClusters" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListClustersResponse> listClusters(ListClustersRequest listClustersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listClustersRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listClustersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListClusters");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListClustersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListClustersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListClustersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListClustersRequest, ListClustersResponse>()
                            .withOperationName("ListClusters").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListClustersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listClustersRequest));
            CompletableFuture<ListClustersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Displays the full description of the subscription.
     * </p>
     *
     * @param listEksAnywhereSubscriptionsRequest
     * @return A Java Future containing the result of the ListEksAnywhereSubscriptions operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of an <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Back off and retry the operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.ListEksAnywhereSubscriptions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/ListEksAnywhereSubscriptions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListEksAnywhereSubscriptionsResponse> listEksAnywhereSubscriptions(
            ListEksAnywhereSubscriptionsRequest listEksAnywhereSubscriptionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listEksAnywhereSubscriptionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEksAnywhereSubscriptionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEksAnywhereSubscriptions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEksAnywhereSubscriptionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListEksAnywhereSubscriptionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListEksAnywhereSubscriptionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEksAnywhereSubscriptionsRequest, ListEksAnywhereSubscriptionsResponse>()
                            .withOperationName("ListEksAnywhereSubscriptions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListEksAnywhereSubscriptionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listEksAnywhereSubscriptionsRequest));
            CompletableFuture<ListEksAnywhereSubscriptionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the Fargate profiles associated with the specified cluster in your Amazon Web Services account in the
     * specified Amazon Web Services Region.
     * </p>
     *
     * @param listFargateProfilesRequest
     * @return A Java Future containing the result of the ListFargateProfiles operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of an <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.ListFargateProfiles
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/ListFargateProfiles" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListFargateProfilesResponse> listFargateProfiles(
            ListFargateProfilesRequest listFargateProfilesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listFargateProfilesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFargateProfilesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFargateProfiles");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListFargateProfilesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListFargateProfilesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListFargateProfilesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListFargateProfilesRequest, ListFargateProfilesResponse>()
                            .withOperationName("ListFargateProfiles").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListFargateProfilesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listFargateProfilesRequest));
            CompletableFuture<ListFargateProfilesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the identity provider configurations for your cluster.
     * </p>
     *
     * @param listIdentityProviderConfigsRequest
     * @return A Java Future containing the result of the ListIdentityProviderConfigs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of an <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Back off and retry the operation.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.ListIdentityProviderConfigs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/ListIdentityProviderConfigs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListIdentityProviderConfigsResponse> listIdentityProviderConfigs(
            ListIdentityProviderConfigsRequest listIdentityProviderConfigsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listIdentityProviderConfigsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listIdentityProviderConfigsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListIdentityProviderConfigs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListIdentityProviderConfigsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListIdentityProviderConfigsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListIdentityProviderConfigsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListIdentityProviderConfigsRequest, ListIdentityProviderConfigsResponse>()
                            .withOperationName("ListIdentityProviderConfigs").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListIdentityProviderConfigsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listIdentityProviderConfigsRequest));
            CompletableFuture<ListIdentityProviderConfigsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of all insights checked for against the specified cluster. You can filter which insights are
     * returned by category, associated Kubernetes version, and status.
     * </p>
     *
     * @param listInsightsRequest
     * @return A Java Future containing the result of the ListInsights operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>InvalidRequestException The request is invalid given the state of the cluster. Check the state of the
     *         cluster and the associated operations.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.ListInsights
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/ListInsights" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListInsightsResponse> listInsights(ListInsightsRequest listInsightsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listInsightsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInsightsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInsights");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListInsightsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListInsightsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListInsightsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListInsightsRequest, ListInsightsResponse>()
                            .withOperationName("ListInsights").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListInsightsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listInsightsRequest));
            CompletableFuture<ListInsightsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the managed node groups associated with the specified cluster in your Amazon Web Services account in the
     * specified Amazon Web Services Region. Self-managed node groups aren't listed.
     * </p>
     *
     * @param listNodegroupsRequest
     * @return A Java Future containing the result of the ListNodegroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of an <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Back off and retry the operation.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.ListNodegroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/ListNodegroups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListNodegroupsResponse> listNodegroups(ListNodegroupsRequest listNodegroupsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listNodegroupsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listNodegroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListNodegroups");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListNodegroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListNodegroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListNodegroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListNodegroupsRequest, ListNodegroupsResponse>()
                            .withOperationName("ListNodegroups").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListNodegroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listNodegroupsRequest));
            CompletableFuture<ListNodegroupsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List the EKS Pod Identity associations in a cluster. You can filter the list by the namespace that the
     * association is in or the service account that the association uses.
     * </p>
     *
     * @param listPodIdentityAssociationsRequest
     * @return A Java Future containing the result of the ListPodIdentityAssociations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>InvalidRequestException The request is invalid given the state of the cluster. Check the state of the
     *         cluster and the associated operations.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.ListPodIdentityAssociations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/ListPodIdentityAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPodIdentityAssociationsResponse> listPodIdentityAssociations(
            ListPodIdentityAssociationsRequest listPodIdentityAssociationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPodIdentityAssociationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPodIdentityAssociationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPodIdentityAssociations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPodIdentityAssociationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPodIdentityAssociationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPodIdentityAssociationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPodIdentityAssociationsRequest, ListPodIdentityAssociationsResponse>()
                            .withOperationName("ListPodIdentityAssociations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListPodIdentityAssociationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPodIdentityAssociationsRequest));
            CompletableFuture<ListPodIdentityAssociationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List the tags for an Amazon EKS resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException This exception is thrown if the request contains a semantic error. The precise
     *         meaning will depend on the API, and will be documented in the error message.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the updates associated with an Amazon EKS resource in your Amazon Web Services account, in the specified
     * Amazon Web Services Region.
     * </p>
     *
     * @param listUpdatesRequest
     * @return A Java Future containing the result of the ListUpdates operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of an <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.ListUpdates
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/ListUpdates" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListUpdatesResponse> listUpdates(ListUpdatesRequest listUpdatesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listUpdatesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listUpdatesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListUpdates");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListUpdatesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListUpdatesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListUpdatesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListUpdatesRequest, ListUpdatesResponse>()
                            .withOperationName("ListUpdates").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListUpdatesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listUpdatesRequest));
            CompletableFuture<ListUpdatesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Connects a Kubernetes cluster to the Amazon EKS control plane.
     * </p>
     * <p>
     * Any Kubernetes cluster can be connected to the Amazon EKS control plane to view current information about the
     * cluster and its nodes.
     * </p>
     * <p>
     * Cluster connection requires two steps. First, send a <code> <a>RegisterClusterRequest</a> </code> to add it to
     * the Amazon EKS control plane.
     * </p>
     * <p>
     * Second, a <a href=
     * "https://amazon-eks.s3.us-west-2.amazonaws.com/eks-connector/manifests/eks-connector/latest/eks-connector.yaml"
     * >Manifest</a> containing the <code>activationID</code> and <code>activationCode</code> must be applied to the
     * Kubernetes cluster through it's native provider to provide visibility.
     * </p>
     * <p>
     * After the manifest is updated and applied, the connected cluster is visible to the Amazon EKS control plane. If
     * the manifest isn't applied within three days, the connected cluster will no longer be visible and must be
     * deregistered using <code>DeregisterCluster</code>.
     * </p>
     *
     * @param registerClusterRequest
     * @return A Java Future containing the result of the RegisterCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceLimitExceededException You have encountered a service limit on the specified resource.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of an <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Back off and retry the operation.</li>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>ResourcePropagationDelayException Required resources (such as service-linked roles) were created and
     *         are still propagating. Retry later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.RegisterCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/RegisterCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<RegisterClusterResponse> registerCluster(RegisterClusterRequest registerClusterRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(registerClusterRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, registerClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterCluster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RegisterClusterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RegisterClusterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RegisterClusterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RegisterClusterRequest, RegisterClusterResponse>()
                            .withOperationName("RegisterCluster").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RegisterClusterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(registerClusterRequest));
            CompletableFuture<RegisterClusterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associates the specified tags to an Amazon EKS resource with the specified <code>resourceArn</code>. If existing
     * tags on a resource are not specified in the request parameters, they aren't changed. When a resource is deleted,
     * the tags associated with that resource are also deleted. Tags that you create for Amazon EKS resources don't
     * propagate to any other resources associated with the cluster. For example, if you tag a cluster with this
     * operation, that tag doesn't automatically propagate to the subnets and nodes associated with the cluster.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException This exception is thrown if the request contains a semantic error. The precise
     *         meaning will depend on the API, and will be documented in the error message.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes specified tags from an Amazon EKS resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException This exception is thrown if the request contains a semantic error. The precise
     *         meaning will depend on the API, and will be documented in the error message.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an access entry.
     * </p>
     *
     * @param updateAccessEntryRequest
     * @return A Java Future containing the result of the UpdateAccessEntry operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>InvalidRequestException The request is invalid given the state of the cluster. Check the state of the
     *         cluster and the associated operations.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.UpdateAccessEntry
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/UpdateAccessEntry" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAccessEntryResponse> updateAccessEntry(UpdateAccessEntryRequest updateAccessEntryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateAccessEntryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAccessEntryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAccessEntry");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAccessEntryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateAccessEntryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateAccessEntryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAccessEntryRequest, UpdateAccessEntryResponse>()
                            .withOperationName("UpdateAccessEntry").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateAccessEntryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateAccessEntryRequest));
            CompletableFuture<UpdateAccessEntryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an Amazon EKS add-on.
     * </p>
     *
     * @param updateAddonRequest
     * @return A Java Future containing the result of the UpdateAddon operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>InvalidRequestException The request is invalid given the state of the cluster. Check the state of the
     *         cluster and the associated operations.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of an <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.UpdateAddon
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/UpdateAddon" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAddonResponse> updateAddon(UpdateAddonRequest updateAddonRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateAddonRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAddonRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAddon");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAddonResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateAddonResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateAddonResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAddonRequest, UpdateAddonResponse>()
                            .withOperationName("UpdateAddon").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateAddonRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateAddonRequest));
            CompletableFuture<UpdateAddonResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an Amazon EKS cluster configuration. Your cluster continues to function during the update. The response
     * output includes an update ID that you can use to track the status of your cluster update with
     * <code>DescribeUpdate</code>"/&gt;.
     * </p>
     * <p>
     * You can use this API operation to enable or disable exporting the Kubernetes control plane logs for your cluster
     * to CloudWatch Logs. By default, cluster control plane logs aren't exported to CloudWatch Logs. For more
     * information, see <a href="https://docs.aws.amazon.com/eks/latest/userguide/control-plane-logs.html">Amazon EKS
     * Cluster control plane logs</a> in the <i> <i>Amazon EKS User Guide</i> </i>.
     * </p>
     * <note>
     * <p>
     * CloudWatch Logs ingestion, archive storage, and data scanning rates apply to exported control plane logs. For
     * more information, see <a href="http://aws.amazon.com/cloudwatch/pricing/">CloudWatch Pricing</a>.
     * </p>
     * </note>
     * <p>
     * You can also use this API operation to enable or disable public and private access to your cluster's Kubernetes
     * API server endpoint. By default, public access is enabled, and private access is disabled. For more information,
     * see <a href="https://docs.aws.amazon.com/eks/latest/userguide/cluster-endpoint.html">Amazon EKS cluster endpoint
     * access control</a> in the <i> <i>Amazon EKS User Guide</i> </i>.
     * </p>
     * <p>
     * You can also use this API operation to choose different subnets and security groups for the cluster. You must
     * specify at least two subnets that are in different Availability Zones. You can't change which VPC the subnets are
     * from, the subnets must be in the same VPC as the subnets that the cluster was created with. For more information
     * about the VPC requirements, see <a
     * href="https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html">https
     * ://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html</a> in the <i> <i>Amazon EKS User Guide</i> </i>.
     * </p>
     * <p>
     * Cluster updates are asynchronous, and they should finish within a few minutes. During an update, the cluster
     * status moves to <code>UPDATING</code> (this status transition is eventually consistent). When the update is
     * complete (either <code>Failed</code> or <code>Successful</code>), the cluster status moves to <code>Active</code>
     * .
     * </p>
     *
     * @param updateClusterConfigRequest
     * @return A Java Future containing the result of the UpdateClusterConfig operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of an <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>InvalidRequestException The request is invalid given the state of the cluster. Check the state of the
     *         cluster and the associated operations.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.UpdateClusterConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/UpdateClusterConfig" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateClusterConfigResponse> updateClusterConfig(
            UpdateClusterConfigRequest updateClusterConfigRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateClusterConfigRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateClusterConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateClusterConfig");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateClusterConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateClusterConfigResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateClusterConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateClusterConfigRequest, UpdateClusterConfigResponse>()
                            .withOperationName("UpdateClusterConfig").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateClusterConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateClusterConfigRequest));
            CompletableFuture<UpdateClusterConfigResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an Amazon EKS cluster to the specified Kubernetes version. Your cluster continues to function during the
     * update. The response output includes an update ID that you can use to track the status of your cluster update
     * with the <a>DescribeUpdate</a> API operation.
     * </p>
     * <p>
     * Cluster updates are asynchronous, and they should finish within a few minutes. During an update, the cluster
     * status moves to <code>UPDATING</code> (this status transition is eventually consistent). When the update is
     * complete (either <code>Failed</code> or <code>Successful</code>), the cluster status moves to <code>Active</code>
     * .
     * </p>
     * <p>
     * If your cluster has managed node groups attached to it, all of your node groups’ Kubernetes versions must match
     * the cluster’s Kubernetes version in order to update the cluster to a new Kubernetes version.
     * </p>
     *
     * @param updateClusterVersionRequest
     * @return A Java Future containing the result of the UpdateClusterVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of an <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>InvalidRequestException The request is invalid given the state of the cluster. Check the state of the
     *         cluster and the associated operations.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.UpdateClusterVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/UpdateClusterVersion" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateClusterVersionResponse> updateClusterVersion(
            UpdateClusterVersionRequest updateClusterVersionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateClusterVersionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateClusterVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateClusterVersion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateClusterVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateClusterVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateClusterVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateClusterVersionRequest, UpdateClusterVersionResponse>()
                            .withOperationName("UpdateClusterVersion").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateClusterVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateClusterVersionRequest));
            CompletableFuture<UpdateClusterVersionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update an EKS Anywhere Subscription. Only auto renewal and tags can be updated after subscription creation.
     * </p>
     *
     * @param updateEksAnywhereSubscriptionRequest
     * @return A Java Future containing the result of the UpdateEksAnywhereSubscription operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of an <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>InvalidRequestException The request is invalid given the state of the cluster. Check the state of the
     *         cluster and the associated operations.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.UpdateEksAnywhereSubscription
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/UpdateEksAnywhereSubscription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateEksAnywhereSubscriptionResponse> updateEksAnywhereSubscription(
            UpdateEksAnywhereSubscriptionRequest updateEksAnywhereSubscriptionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateEksAnywhereSubscriptionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateEksAnywhereSubscriptionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateEksAnywhereSubscription");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateEksAnywhereSubscriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateEksAnywhereSubscriptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateEksAnywhereSubscriptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateEksAnywhereSubscriptionRequest, UpdateEksAnywhereSubscriptionResponse>()
                            .withOperationName("UpdateEksAnywhereSubscription").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateEksAnywhereSubscriptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateEksAnywhereSubscriptionRequest));
            CompletableFuture<UpdateEksAnywhereSubscriptionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an Amazon EKS managed node group configuration. Your node group continues to function during the update.
     * The response output includes an update ID that you can use to track the status of your node group update with the
     * <a>DescribeUpdate</a> API operation. Currently you can update the Kubernetes labels for a node group or the
     * scaling configuration.
     * </p>
     *
     * @param updateNodegroupConfigRequest
     * @return A Java Future containing the result of the UpdateNodegroupConfig operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of an <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>InvalidRequestException The request is invalid given the state of the cluster. Check the state of the
     *         cluster and the associated operations.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.UpdateNodegroupConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/UpdateNodegroupConfig" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateNodegroupConfigResponse> updateNodegroupConfig(
            UpdateNodegroupConfigRequest updateNodegroupConfigRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateNodegroupConfigRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateNodegroupConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateNodegroupConfig");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateNodegroupConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateNodegroupConfigResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateNodegroupConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateNodegroupConfigRequest, UpdateNodegroupConfigResponse>()
                            .withOperationName("UpdateNodegroupConfig").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateNodegroupConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateNodegroupConfigRequest));
            CompletableFuture<UpdateNodegroupConfigResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the Kubernetes version or AMI version of an Amazon EKS managed node group.
     * </p>
     * <p>
     * You can update a node group using a launch template only if the node group was originally deployed with a launch
     * template. If you need to update a custom AMI in a node group that was deployed with a launch template, then
     * update your custom AMI, specify the new ID in a new version of the launch template, and then update the node
     * group to the new version of the launch template.
     * </p>
     * <p>
     * If you update without a launch template, then you can update to the latest available AMI version of a node
     * group's current Kubernetes version by not specifying a Kubernetes version in the request. You can update to the
     * latest AMI version of your cluster's current Kubernetes version by specifying your cluster's Kubernetes version
     * in the request. For information about Linux versions, see <a
     * href="https://docs.aws.amazon.com/eks/latest/userguide/eks-linux-ami-versions.html">Amazon EKS optimized Amazon
     * Linux AMI versions</a> in the <i>Amazon EKS User Guide</i>. For information about Windows versions, see <a
     * href="https://docs.aws.amazon.com/eks/latest/userguide/eks-ami-versions-windows.html">Amazon EKS optimized
     * Windows AMI versions</a> in the <i>Amazon EKS User Guide</i>.
     * </p>
     * <p>
     * You cannot roll back a node group to an earlier Kubernetes version or AMI version.
     * </p>
     * <p>
     * When a node in a managed node group is terminated due to a scaling action or update, every <code>Pod</code> on
     * that node is drained first. Amazon EKS attempts to drain the nodes gracefully and will fail if it is unable to do
     * so. You can <code>force</code> the update if Amazon EKS is unable to drain the nodes as a result of a
     * <code>Pod</code> disruption budget issue.
     * </p>
     *
     * @param updateNodegroupVersionRequest
     * @return A Java Future containing the result of the UpdateNodegroupVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of an <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html">IAM
     *         principal</a> that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>InvalidRequestException The request is invalid given the state of the cluster. Check the state of the
     *         cluster and the associated operations.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.UpdateNodegroupVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/UpdateNodegroupVersion" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateNodegroupVersionResponse> updateNodegroupVersion(
            UpdateNodegroupVersionRequest updateNodegroupVersionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateNodegroupVersionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateNodegroupVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateNodegroupVersion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateNodegroupVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateNodegroupVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateNodegroupVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateNodegroupVersionRequest, UpdateNodegroupVersionResponse>()
                            .withOperationName("UpdateNodegroupVersion").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateNodegroupVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateNodegroupVersionRequest));
            CompletableFuture<UpdateNodegroupVersionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a EKS Pod Identity association. Only the IAM role can be changed; an association can't be moved between
     * clusters, namespaces, or service accounts. If you need to edit the namespace or service account, you need to
     * delete the association and then create a new association with your desired settings.
     * </p>
     *
     * @param updatePodIdentityAssociationRequest
     * @return A Java Future containing the result of the UpdatePodIdentityAssociation operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <code>ListClusters</code>. You can view your available managed node groups with
     *         <code>ListNodegroups</code>. Amazon EKS clusters and node groups are Amazon Web Services Region specific.
     *         </li>
     *         <li>InvalidRequestException The request is invalid given the state of the cluster. Check the state of the
     *         cluster and the associated operations.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.UpdatePodIdentityAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/UpdatePodIdentityAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdatePodIdentityAssociationResponse> updatePodIdentityAssociation(
            UpdatePodIdentityAssociationRequest updatePodIdentityAssociationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePodIdentityAssociationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePodIdentityAssociationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EKS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePodIdentityAssociation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdatePodIdentityAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdatePodIdentityAssociationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdatePodIdentityAssociationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdatePodIdentityAssociationRequest, UpdatePodIdentityAssociationResponse>()
                            .withOperationName("UpdatePodIdentityAssociation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdatePodIdentityAssociationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updatePodIdentityAssociationRequest));
            CompletableFuture<UpdatePodIdentityAssociationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public EksAsyncWaiter waiter() {
        return EksAsyncWaiter.builder().client(this).scheduledExecutorService(executorService).build();
    }

    @Override
    public final EksServiceClientConfiguration serviceClientConfiguration() {
        return new EksServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(EksException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ClientException")
                                .exceptionBuilderSupplier(ClientException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                                .exceptionBuilderSupplier(ResourceLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUseException")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedAvailabilityZoneException")
                                .exceptionBuilderSupplier(UnsupportedAvailabilityZoneException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServerException")
                                .exceptionBuilderSupplier(ServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourcePropagationDelayException")
                                .exceptionBuilderSupplier(ResourcePropagationDelayException::builder).httpStatusCode(428).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        EksServiceClientConfigurationBuilder serviceConfigBuilder = new EksServiceClientConfigurationBuilder(configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
