/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Port mappings allow containers to access ports on the host container instance to send or receive traffic. Port
 * mappings are specified as part of the container definition.
 * </p>
 * <p>
 * If you use containers in a task with the <code>awsvpc</code> or <code>host</code> network mode, specify the exposed
 * ports using <code>containerPort</code>. The <code>hostPort</code> can be left blank or it must be the same value as
 * the <code>containerPort</code>.
 * </p>
 * <p>
 * Most fields of this parameter (<code>containerPort</code>, <code>hostPort</code>, <code>protocol</code>) maps to
 * <code>PortBindings</code> in the docker conainer create command and the <code>--publish</code> option to
 * <code>docker run</code>. If the network mode of a task definition is set to <code>host</code>, host ports must either
 * be undefined or match the container port in the port mapping.
 * </p>
 * <note>
 * <p>
 * You can't expose the same container port for multiple protocols. If you attempt this, an error is returned.
 * </p>
 * </note>
 * <p>
 * After a task reaches the <code>RUNNING</code> status, manual and automatic host and container port assignments are
 * visible in the <code>networkBindings</code> section of <a
 * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTasks.html">DescribeTasks</a> API
 * responses.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PortMapping implements SdkPojo, Serializable, ToCopyableBuilder<PortMapping.Builder, PortMapping> {
    private static final SdkField<Integer> CONTAINER_PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("containerPort").getter(getter(PortMapping::containerPort)).setter(setter(Builder::containerPort))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("containerPort").build()).build();

    private static final SdkField<Integer> HOST_PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("hostPort").getter(getter(PortMapping::hostPort)).setter(setter(Builder::hostPort))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("hostPort").build()).build();

    private static final SdkField<String> PROTOCOL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("protocol").getter(getter(PortMapping::protocolAsString)).setter(setter(Builder::protocol))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("protocol").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(PortMapping::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> APP_PROTOCOL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("appProtocol").getter(getter(PortMapping::appProtocolAsString)).setter(setter(Builder::appProtocol))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("appProtocol").build()).build();

    private static final SdkField<String> CONTAINER_PORT_RANGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("containerPortRange").getter(getter(PortMapping::containerPortRange))
            .setter(setter(Builder::containerPortRange))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("containerPortRange").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CONTAINER_PORT_FIELD,
            HOST_PORT_FIELD, PROTOCOL_FIELD, NAME_FIELD, APP_PROTOCOL_FIELD, CONTAINER_PORT_RANGE_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer containerPort;

    private final Integer hostPort;

    private final String protocol;

    private final String name;

    private final String appProtocol;

    private final String containerPortRange;

    private PortMapping(BuilderImpl builder) {
        this.containerPort = builder.containerPort;
        this.hostPort = builder.hostPort;
        this.protocol = builder.protocol;
        this.name = builder.name;
        this.appProtocol = builder.appProtocol;
        this.containerPortRange = builder.containerPortRange;
    }

    /**
     * <p>
     * The port number on the container that's bound to the user-specified or automatically assigned host port.
     * </p>
     * <p>
     * If you use containers in a task with the <code>awsvpc</code> or <code>host</code> network mode, specify the
     * exposed ports using <code>containerPort</code>.
     * </p>
     * <p>
     * If you use containers in a task with the <code>bridge</code> network mode and you specify a container port and
     * not a host port, your container automatically receives a host port in the ephemeral port range. For more
     * information, see <code>hostPort</code>. Port mappings that are automatically assigned in this way do not count
     * toward the 100 reserved ports limit of a container instance.
     * </p>
     * 
     * @return The port number on the container that's bound to the user-specified or automatically assigned host
     *         port.</p>
     *         <p>
     *         If you use containers in a task with the <code>awsvpc</code> or <code>host</code> network mode, specify
     *         the exposed ports using <code>containerPort</code>.
     *         </p>
     *         <p>
     *         If you use containers in a task with the <code>bridge</code> network mode and you specify a container
     *         port and not a host port, your container automatically receives a host port in the ephemeral port range.
     *         For more information, see <code>hostPort</code>. Port mappings that are automatically assigned in this
     *         way do not count toward the 100 reserved ports limit of a container instance.
     */
    public final Integer containerPort() {
        return containerPort;
    }

    /**
     * <p>
     * The port number on the container instance to reserve for your container.
     * </p>
     * <p>
     * If you specify a <code>containerPortRange</code>, leave this field empty and the value of the
     * <code>hostPort</code> is set as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For containers in a task with the <code>awsvpc</code> network mode, the <code>hostPort</code> is set to the same
     * value as the <code>containerPort</code>. This is a static mapping strategy.
     * </p>
     * </li>
     * <li>
     * <p>
     * For containers in a task with the <code>bridge</code> network mode, the Amazon ECS agent finds open ports on the
     * host and automatically binds them to the container ports. This is a dynamic mapping strategy.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you use containers in a task with the <code>awsvpc</code> or <code>host</code> network mode, the
     * <code>hostPort</code> can either be left blank or set to the same value as the <code>containerPort</code>.
     * </p>
     * <p>
     * If you use containers in a task with the <code>bridge</code> network mode, you can specify a non-reserved host
     * port for your container port mapping, or you can omit the <code>hostPort</code> (or set it to <code>0</code>)
     * while specifying a <code>containerPort</code> and your container automatically receives a port in the ephemeral
     * port range for your container instance operating system and Docker version.
     * </p>
     * <p>
     * The default ephemeral port range for Docker version 1.6.0 and later is listed on the instance under
     * <code>/proc/sys/net/ipv4/ip_local_port_range</code>. If this kernel parameter is unavailable, the default
     * ephemeral port range from 49153 through 65535 (Linux) or 49152 through 65535 (Windows) is used. Do not attempt to
     * specify a host port in the ephemeral port range as these are reserved for automatic assignment. In general, ports
     * below 32768 are outside of the ephemeral port range.
     * </p>
     * <p>
     * The default reserved ports are 22 for SSH, the Docker ports 2375 and 2376, and the Amazon ECS container agent
     * ports 51678-51680. Any host port that was previously specified in a running task is also reserved while the task
     * is running. That is, after a task stops, the host port is released. The current reserved ports are displayed in
     * the <code>remainingResources</code> of <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeContainerInstances.html"
     * >DescribeContainerInstances</a> output. A container instance can have up to 100 reserved ports at a time. This
     * number includes the default reserved ports. Automatically assigned ports aren't included in the 100 reserved
     * ports quota.
     * </p>
     * 
     * @return The port number on the container instance to reserve for your container.</p>
     *         <p>
     *         If you specify a <code>containerPortRange</code>, leave this field empty and the value of the
     *         <code>hostPort</code> is set as follows:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For containers in a task with the <code>awsvpc</code> network mode, the <code>hostPort</code> is set to
     *         the same value as the <code>containerPort</code>. This is a static mapping strategy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For containers in a task with the <code>bridge</code> network mode, the Amazon ECS agent finds open ports
     *         on the host and automatically binds them to the container ports. This is a dynamic mapping strategy.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you use containers in a task with the <code>awsvpc</code> or <code>host</code> network mode, the
     *         <code>hostPort</code> can either be left blank or set to the same value as the <code>containerPort</code>
     *         .
     *         </p>
     *         <p>
     *         If you use containers in a task with the <code>bridge</code> network mode, you can specify a non-reserved
     *         host port for your container port mapping, or you can omit the <code>hostPort</code> (or set it to
     *         <code>0</code>) while specifying a <code>containerPort</code> and your container automatically receives a
     *         port in the ephemeral port range for your container instance operating system and Docker version.
     *         </p>
     *         <p>
     *         The default ephemeral port range for Docker version 1.6.0 and later is listed on the instance under
     *         <code>/proc/sys/net/ipv4/ip_local_port_range</code>. If this kernel parameter is unavailable, the default
     *         ephemeral port range from 49153 through 65535 (Linux) or 49152 through 65535 (Windows) is used. Do not
     *         attempt to specify a host port in the ephemeral port range as these are reserved for automatic
     *         assignment. In general, ports below 32768 are outside of the ephemeral port range.
     *         </p>
     *         <p>
     *         The default reserved ports are 22 for SSH, the Docker ports 2375 and 2376, and the Amazon ECS container
     *         agent ports 51678-51680. Any host port that was previously specified in a running task is also reserved
     *         while the task is running. That is, after a task stops, the host port is released. The current reserved
     *         ports are displayed in the <code>remainingResources</code> of <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeContainerInstances.html"
     *         >DescribeContainerInstances</a> output. A container instance can have up to 100 reserved ports at a time.
     *         This number includes the default reserved ports. Automatically assigned ports aren't included in the 100
     *         reserved ports quota.
     */
    public final Integer hostPort() {
        return hostPort;
    }

    /**
     * <p>
     * The protocol used for the port mapping. Valid values are <code>tcp</code> and <code>udp</code>. The default is
     * <code>tcp</code>. <code>protocol</code> is immutable in a Service Connect service. Updating this field requires a
     * service deletion and redeployment.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #protocol} will
     * return {@link TransportProtocol#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #protocolAsString}.
     * </p>
     * 
     * @return The protocol used for the port mapping. Valid values are <code>tcp</code> and <code>udp</code>. The
     *         default is <code>tcp</code>. <code>protocol</code> is immutable in a Service Connect service. Updating
     *         this field requires a service deletion and redeployment.
     * @see TransportProtocol
     */
    public final TransportProtocol protocol() {
        return TransportProtocol.fromValue(protocol);
    }

    /**
     * <p>
     * The protocol used for the port mapping. Valid values are <code>tcp</code> and <code>udp</code>. The default is
     * <code>tcp</code>. <code>protocol</code> is immutable in a Service Connect service. Updating this field requires a
     * service deletion and redeployment.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #protocol} will
     * return {@link TransportProtocol#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #protocolAsString}.
     * </p>
     * 
     * @return The protocol used for the port mapping. Valid values are <code>tcp</code> and <code>udp</code>. The
     *         default is <code>tcp</code>. <code>protocol</code> is immutable in a Service Connect service. Updating
     *         this field requires a service deletion and redeployment.
     * @see TransportProtocol
     */
    public final String protocolAsString() {
        return protocol;
    }

    /**
     * <p>
     * The name that's used for the port mapping. This parameter only applies to Service Connect. This parameter is the
     * name that you use in the <code>serviceConnectConfiguration</code> of a service. The name can include up to 64
     * characters. The characters can include lowercase letters, numbers, underscores (_), and hyphens (-). The name
     * can't start with a hyphen.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-connect.html">Service Connect</a> in
     * the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * 
     * @return The name that's used for the port mapping. This parameter only applies to Service Connect. This parameter
     *         is the name that you use in the <code>serviceConnectConfiguration</code> of a service. The name can
     *         include up to 64 characters. The characters can include lowercase letters, numbers, underscores (_), and
     *         hyphens (-). The name can't start with a hyphen.</p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-connect.html">Service
     *         Connect</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The application protocol that's used for the port mapping. This parameter only applies to Service Connect. We
     * recommend that you set this parameter to be consistent with the protocol that your application uses. If you set
     * this parameter, Amazon ECS adds protocol-specific connection handling to the Service Connect proxy. If you set
     * this parameter, Amazon ECS adds protocol-specific telemetry in the Amazon ECS console and CloudWatch.
     * </p>
     * <p>
     * If you don't set a value for this parameter, then TCP is used. However, Amazon ECS doesn't add protocol-specific
     * telemetry for TCP.
     * </p>
     * <p>
     * <code>appProtocol</code> is immutable in a Service Connect service. Updating this field requires a service
     * deletion and redeployment.
     * </p>
     * <p>
     * Tasks that run in a namespace can use short names to connect to services in the namespace. Tasks can connect to
     * services across all of the clusters in the namespace. Tasks connect through a managed proxy container that
     * collects logs and metrics for increased visibility. Only the tasks that Amazon ECS services create are supported
     * with Service Connect. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-connect.html">Service Connect</a> in
     * the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #appProtocol} will
     * return {@link ApplicationProtocol#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #appProtocolAsString}.
     * </p>
     * 
     * @return The application protocol that's used for the port mapping. This parameter only applies to Service
     *         Connect. We recommend that you set this parameter to be consistent with the protocol that your
     *         application uses. If you set this parameter, Amazon ECS adds protocol-specific connection handling to the
     *         Service Connect proxy. If you set this parameter, Amazon ECS adds protocol-specific telemetry in the
     *         Amazon ECS console and CloudWatch.</p>
     *         <p>
     *         If you don't set a value for this parameter, then TCP is used. However, Amazon ECS doesn't add
     *         protocol-specific telemetry for TCP.
     *         </p>
     *         <p>
     *         <code>appProtocol</code> is immutable in a Service Connect service. Updating this field requires a
     *         service deletion and redeployment.
     *         </p>
     *         <p>
     *         Tasks that run in a namespace can use short names to connect to services in the namespace. Tasks can
     *         connect to services across all of the clusters in the namespace. Tasks connect through a managed proxy
     *         container that collects logs and metrics for increased visibility. Only the tasks that Amazon ECS
     *         services create are supported with Service Connect. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-connect.html">Service
     *         Connect</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * @see ApplicationProtocol
     */
    public final ApplicationProtocol appProtocol() {
        return ApplicationProtocol.fromValue(appProtocol);
    }

    /**
     * <p>
     * The application protocol that's used for the port mapping. This parameter only applies to Service Connect. We
     * recommend that you set this parameter to be consistent with the protocol that your application uses. If you set
     * this parameter, Amazon ECS adds protocol-specific connection handling to the Service Connect proxy. If you set
     * this parameter, Amazon ECS adds protocol-specific telemetry in the Amazon ECS console and CloudWatch.
     * </p>
     * <p>
     * If you don't set a value for this parameter, then TCP is used. However, Amazon ECS doesn't add protocol-specific
     * telemetry for TCP.
     * </p>
     * <p>
     * <code>appProtocol</code> is immutable in a Service Connect service. Updating this field requires a service
     * deletion and redeployment.
     * </p>
     * <p>
     * Tasks that run in a namespace can use short names to connect to services in the namespace. Tasks can connect to
     * services across all of the clusters in the namespace. Tasks connect through a managed proxy container that
     * collects logs and metrics for increased visibility. Only the tasks that Amazon ECS services create are supported
     * with Service Connect. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-connect.html">Service Connect</a> in
     * the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #appProtocol} will
     * return {@link ApplicationProtocol#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #appProtocolAsString}.
     * </p>
     * 
     * @return The application protocol that's used for the port mapping. This parameter only applies to Service
     *         Connect. We recommend that you set this parameter to be consistent with the protocol that your
     *         application uses. If you set this parameter, Amazon ECS adds protocol-specific connection handling to the
     *         Service Connect proxy. If you set this parameter, Amazon ECS adds protocol-specific telemetry in the
     *         Amazon ECS console and CloudWatch.</p>
     *         <p>
     *         If you don't set a value for this parameter, then TCP is used. However, Amazon ECS doesn't add
     *         protocol-specific telemetry for TCP.
     *         </p>
     *         <p>
     *         <code>appProtocol</code> is immutable in a Service Connect service. Updating this field requires a
     *         service deletion and redeployment.
     *         </p>
     *         <p>
     *         Tasks that run in a namespace can use short names to connect to services in the namespace. Tasks can
     *         connect to services across all of the clusters in the namespace. Tasks connect through a managed proxy
     *         container that collects logs and metrics for increased visibility. Only the tasks that Amazon ECS
     *         services create are supported with Service Connect. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-connect.html">Service
     *         Connect</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * @see ApplicationProtocol
     */
    public final String appProtocolAsString() {
        return appProtocol;
    }

    /**
     * <p>
     * The port number range on the container that's bound to the dynamically mapped host port range.
     * </p>
     * <p>
     * The following rules apply when you specify a <code>containerPortRange</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * You must use either the <code>bridge</code> network mode or the <code>awsvpc</code> network mode.
     * </p>
     * </li>
     * <li>
     * <p>
     * This parameter is available for both the EC2 and Fargate launch types.
     * </p>
     * </li>
     * <li>
     * <p>
     * This parameter is available for both the Linux and Windows operating systems.
     * </p>
     * </li>
     * <li>
     * <p>
     * The container instance must have at least version 1.67.0 of the container agent and at least version 1.67.0-1 of
     * the <code>ecs-init</code> package
     * </p>
     * </li>
     * <li>
     * <p>
     * You can specify a maximum of 100 port ranges per container.
     * </p>
     * </li>
     * <li>
     * <p>
     * You do not specify a <code>hostPortRange</code>. The value of the <code>hostPortRange</code> is set as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For containers in a task with the <code>awsvpc</code> network mode, the <code>hostPortRange</code> is set to the
     * same value as the <code>containerPortRange</code>. This is a static mapping strategy.
     * </p>
     * </li>
     * <li>
     * <p>
     * For containers in a task with the <code>bridge</code> network mode, the Amazon ECS agent finds open host ports
     * from the default ephemeral range and passes it to docker to bind them to the container ports.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * The <code>containerPortRange</code> valid values are between 1 and 65535.
     * </p>
     * </li>
     * <li>
     * <p>
     * A port can only be included in one port mapping per container.
     * </p>
     * </li>
     * <li>
     * <p>
     * You cannot specify overlapping port ranges.
     * </p>
     * </li>
     * <li>
     * <p>
     * The first port in the range must be less than last port in the range.
     * </p>
     * </li>
     * <li>
     * <p>
     * Docker recommends that you turn off the docker-proxy in the Docker daemon config file when you have a large
     * number of ports.
     * </p>
     * <p>
     * For more information, see <a href="https://github.com/moby/moby/issues/11185"> Issue #11185</a> on the Github
     * website.
     * </p>
     * <p>
     * For information about how to turn off the docker-proxy in the Docker daemon config file, see <a href=
     * "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/bootstrap_container_instance.html#bootstrap_docker_daemon"
     * >Docker daemon</a> in the <i>Amazon ECS Developer Guide</i>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * You can call <a href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTasks.html">
     * <code>DescribeTasks</code> </a> to view the <code>hostPortRange</code> which are the host ports that are bound to
     * the container ports.
     * </p>
     * 
     * @return The port number range on the container that's bound to the dynamically mapped host port range. </p>
     *         <p>
     *         The following rules apply when you specify a <code>containerPortRange</code>:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You must use either the <code>bridge</code> network mode or the <code>awsvpc</code> network mode.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         This parameter is available for both the EC2 and Fargate launch types.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         This parameter is available for both the Linux and Windows operating systems.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The container instance must have at least version 1.67.0 of the container agent and at least version
     *         1.67.0-1 of the <code>ecs-init</code> package
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You can specify a maximum of 100 port ranges per container.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You do not specify a <code>hostPortRange</code>. The value of the <code>hostPortRange</code> is set as
     *         follows:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For containers in a task with the <code>awsvpc</code> network mode, the <code>hostPortRange</code> is set
     *         to the same value as the <code>containerPortRange</code>. This is a static mapping strategy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For containers in a task with the <code>bridge</code> network mode, the Amazon ECS agent finds open host
     *         ports from the default ephemeral range and passes it to docker to bind them to the container ports.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>containerPortRange</code> valid values are between 1 and 65535.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A port can only be included in one port mapping per container.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You cannot specify overlapping port ranges.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The first port in the range must be less than last port in the range.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Docker recommends that you turn off the docker-proxy in the Docker daemon config file when you have a
     *         large number of ports.
     *         </p>
     *         <p>
     *         For more information, see <a href="https://github.com/moby/moby/issues/11185"> Issue #11185</a> on the
     *         Github website.
     *         </p>
     *         <p>
     *         For information about how to turn off the docker-proxy in the Docker daemon config file, see <a href=
     *         "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/bootstrap_container_instance.html#bootstrap_docker_daemon"
     *         >Docker daemon</a> in the <i>Amazon ECS Developer Guide</i>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         You can call <a href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTasks.html">
     *         <code>DescribeTasks</code> </a> to view the <code>hostPortRange</code> which are the host ports that are
     *         bound to the container ports.
     */
    public final String containerPortRange() {
        return containerPortRange;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(containerPort());
        hashCode = 31 * hashCode + Objects.hashCode(hostPort());
        hashCode = 31 * hashCode + Objects.hashCode(protocolAsString());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(appProtocolAsString());
        hashCode = 31 * hashCode + Objects.hashCode(containerPortRange());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PortMapping)) {
            return false;
        }
        PortMapping other = (PortMapping) obj;
        return Objects.equals(containerPort(), other.containerPort()) && Objects.equals(hostPort(), other.hostPort())
                && Objects.equals(protocolAsString(), other.protocolAsString()) && Objects.equals(name(), other.name())
                && Objects.equals(appProtocolAsString(), other.appProtocolAsString())
                && Objects.equals(containerPortRange(), other.containerPortRange());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PortMapping").add("ContainerPort", containerPort()).add("HostPort", hostPort())
                .add("Protocol", protocolAsString()).add("Name", name()).add("AppProtocol", appProtocolAsString())
                .add("ContainerPortRange", containerPortRange()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "containerPort":
            return Optional.ofNullable(clazz.cast(containerPort()));
        case "hostPort":
            return Optional.ofNullable(clazz.cast(hostPort()));
        case "protocol":
            return Optional.ofNullable(clazz.cast(protocolAsString()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "appProtocol":
            return Optional.ofNullable(clazz.cast(appProtocolAsString()));
        case "containerPortRange":
            return Optional.ofNullable(clazz.cast(containerPortRange()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PortMapping, T> g) {
        return obj -> g.apply((PortMapping) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PortMapping> {
        /**
         * <p>
         * The port number on the container that's bound to the user-specified or automatically assigned host port.
         * </p>
         * <p>
         * If you use containers in a task with the <code>awsvpc</code> or <code>host</code> network mode, specify the
         * exposed ports using <code>containerPort</code>.
         * </p>
         * <p>
         * If you use containers in a task with the <code>bridge</code> network mode and you specify a container port
         * and not a host port, your container automatically receives a host port in the ephemeral port range. For more
         * information, see <code>hostPort</code>. Port mappings that are automatically assigned in this way do not
         * count toward the 100 reserved ports limit of a container instance.
         * </p>
         * 
         * @param containerPort
         *        The port number on the container that's bound to the user-specified or automatically assigned host
         *        port.</p>
         *        <p>
         *        If you use containers in a task with the <code>awsvpc</code> or <code>host</code> network mode,
         *        specify the exposed ports using <code>containerPort</code>.
         *        </p>
         *        <p>
         *        If you use containers in a task with the <code>bridge</code> network mode and you specify a container
         *        port and not a host port, your container automatically receives a host port in the ephemeral port
         *        range. For more information, see <code>hostPort</code>. Port mappings that are automatically assigned
         *        in this way do not count toward the 100 reserved ports limit of a container instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerPort(Integer containerPort);

        /**
         * <p>
         * The port number on the container instance to reserve for your container.
         * </p>
         * <p>
         * If you specify a <code>containerPortRange</code>, leave this field empty and the value of the
         * <code>hostPort</code> is set as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * For containers in a task with the <code>awsvpc</code> network mode, the <code>hostPort</code> is set to the
         * same value as the <code>containerPort</code>. This is a static mapping strategy.
         * </p>
         * </li>
         * <li>
         * <p>
         * For containers in a task with the <code>bridge</code> network mode, the Amazon ECS agent finds open ports on
         * the host and automatically binds them to the container ports. This is a dynamic mapping strategy.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you use containers in a task with the <code>awsvpc</code> or <code>host</code> network mode, the
         * <code>hostPort</code> can either be left blank or set to the same value as the <code>containerPort</code>.
         * </p>
         * <p>
         * If you use containers in a task with the <code>bridge</code> network mode, you can specify a non-reserved
         * host port for your container port mapping, or you can omit the <code>hostPort</code> (or set it to
         * <code>0</code>) while specifying a <code>containerPort</code> and your container automatically receives a
         * port in the ephemeral port range for your container instance operating system and Docker version.
         * </p>
         * <p>
         * The default ephemeral port range for Docker version 1.6.0 and later is listed on the instance under
         * <code>/proc/sys/net/ipv4/ip_local_port_range</code>. If this kernel parameter is unavailable, the default
         * ephemeral port range from 49153 through 65535 (Linux) or 49152 through 65535 (Windows) is used. Do not
         * attempt to specify a host port in the ephemeral port range as these are reserved for automatic assignment. In
         * general, ports below 32768 are outside of the ephemeral port range.
         * </p>
         * <p>
         * The default reserved ports are 22 for SSH, the Docker ports 2375 and 2376, and the Amazon ECS container agent
         * ports 51678-51680. Any host port that was previously specified in a running task is also reserved while the
         * task is running. That is, after a task stops, the host port is released. The current reserved ports are
         * displayed in the <code>remainingResources</code> of <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeContainerInstances.html"
         * >DescribeContainerInstances</a> output. A container instance can have up to 100 reserved ports at a time.
         * This number includes the default reserved ports. Automatically assigned ports aren't included in the 100
         * reserved ports quota.
         * </p>
         * 
         * @param hostPort
         *        The port number on the container instance to reserve for your container.</p>
         *        <p>
         *        If you specify a <code>containerPortRange</code>, leave this field empty and the value of the
         *        <code>hostPort</code> is set as follows:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For containers in a task with the <code>awsvpc</code> network mode, the <code>hostPort</code> is set
         *        to the same value as the <code>containerPort</code>. This is a static mapping strategy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For containers in a task with the <code>bridge</code> network mode, the Amazon ECS agent finds open
         *        ports on the host and automatically binds them to the container ports. This is a dynamic mapping
         *        strategy.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you use containers in a task with the <code>awsvpc</code> or <code>host</code> network mode, the
         *        <code>hostPort</code> can either be left blank or set to the same value as the
         *        <code>containerPort</code>.
         *        </p>
         *        <p>
         *        If you use containers in a task with the <code>bridge</code> network mode, you can specify a
         *        non-reserved host port for your container port mapping, or you can omit the <code>hostPort</code> (or
         *        set it to <code>0</code>) while specifying a <code>containerPort</code> and your container
         *        automatically receives a port in the ephemeral port range for your container instance operating system
         *        and Docker version.
         *        </p>
         *        <p>
         *        The default ephemeral port range for Docker version 1.6.0 and later is listed on the instance under
         *        <code>/proc/sys/net/ipv4/ip_local_port_range</code>. If this kernel parameter is unavailable, the
         *        default ephemeral port range from 49153 through 65535 (Linux) or 49152 through 65535 (Windows) is
         *        used. Do not attempt to specify a host port in the ephemeral port range as these are reserved for
         *        automatic assignment. In general, ports below 32768 are outside of the ephemeral port range.
         *        </p>
         *        <p>
         *        The default reserved ports are 22 for SSH, the Docker ports 2375 and 2376, and the Amazon ECS
         *        container agent ports 51678-51680. Any host port that was previously specified in a running task is
         *        also reserved while the task is running. That is, after a task stops, the host port is released. The
         *        current reserved ports are displayed in the <code>remainingResources</code> of <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeContainerInstances.html"
         *        >DescribeContainerInstances</a> output. A container instance can have up to 100 reserved ports at a
         *        time. This number includes the default reserved ports. Automatically assigned ports aren't included in
         *        the 100 reserved ports quota.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hostPort(Integer hostPort);

        /**
         * <p>
         * The protocol used for the port mapping. Valid values are <code>tcp</code> and <code>udp</code>. The default
         * is <code>tcp</code>. <code>protocol</code> is immutable in a Service Connect service. Updating this field
         * requires a service deletion and redeployment.
         * </p>
         * 
         * @param protocol
         *        The protocol used for the port mapping. Valid values are <code>tcp</code> and <code>udp</code>. The
         *        default is <code>tcp</code>. <code>protocol</code> is immutable in a Service Connect service. Updating
         *        this field requires a service deletion and redeployment.
         * @see TransportProtocol
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TransportProtocol
         */
        Builder protocol(String protocol);

        /**
         * <p>
         * The protocol used for the port mapping. Valid values are <code>tcp</code> and <code>udp</code>. The default
         * is <code>tcp</code>. <code>protocol</code> is immutable in a Service Connect service. Updating this field
         * requires a service deletion and redeployment.
         * </p>
         * 
         * @param protocol
         *        The protocol used for the port mapping. Valid values are <code>tcp</code> and <code>udp</code>. The
         *        default is <code>tcp</code>. <code>protocol</code> is immutable in a Service Connect service. Updating
         *        this field requires a service deletion and redeployment.
         * @see TransportProtocol
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TransportProtocol
         */
        Builder protocol(TransportProtocol protocol);

        /**
         * <p>
         * The name that's used for the port mapping. This parameter only applies to Service Connect. This parameter is
         * the name that you use in the <code>serviceConnectConfiguration</code> of a service. The name can include up
         * to 64 characters. The characters can include lowercase letters, numbers, underscores (_), and hyphens (-).
         * The name can't start with a hyphen.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-connect.html">Service Connect</a>
         * in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param name
         *        The name that's used for the port mapping. This parameter only applies to Service Connect. This
         *        parameter is the name that you use in the <code>serviceConnectConfiguration</code> of a service. The
         *        name can include up to 64 characters. The characters can include lowercase letters, numbers,
         *        underscores (_), and hyphens (-). The name can't start with a hyphen.</p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-connect.html">Service
         *        Connect</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The application protocol that's used for the port mapping. This parameter only applies to Service Connect. We
         * recommend that you set this parameter to be consistent with the protocol that your application uses. If you
         * set this parameter, Amazon ECS adds protocol-specific connection handling to the Service Connect proxy. If
         * you set this parameter, Amazon ECS adds protocol-specific telemetry in the Amazon ECS console and CloudWatch.
         * </p>
         * <p>
         * If you don't set a value for this parameter, then TCP is used. However, Amazon ECS doesn't add
         * protocol-specific telemetry for TCP.
         * </p>
         * <p>
         * <code>appProtocol</code> is immutable in a Service Connect service. Updating this field requires a service
         * deletion and redeployment.
         * </p>
         * <p>
         * Tasks that run in a namespace can use short names to connect to services in the namespace. Tasks can connect
         * to services across all of the clusters in the namespace. Tasks connect through a managed proxy container that
         * collects logs and metrics for increased visibility. Only the tasks that Amazon ECS services create are
         * supported with Service Connect. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-connect.html">Service Connect</a>
         * in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param appProtocol
         *        The application protocol that's used for the port mapping. This parameter only applies to Service
         *        Connect. We recommend that you set this parameter to be consistent with the protocol that your
         *        application uses. If you set this parameter, Amazon ECS adds protocol-specific connection handling to
         *        the Service Connect proxy. If you set this parameter, Amazon ECS adds protocol-specific telemetry in
         *        the Amazon ECS console and CloudWatch.</p>
         *        <p>
         *        If you don't set a value for this parameter, then TCP is used. However, Amazon ECS doesn't add
         *        protocol-specific telemetry for TCP.
         *        </p>
         *        <p>
         *        <code>appProtocol</code> is immutable in a Service Connect service. Updating this field requires a
         *        service deletion and redeployment.
         *        </p>
         *        <p>
         *        Tasks that run in a namespace can use short names to connect to services in the namespace. Tasks can
         *        connect to services across all of the clusters in the namespace. Tasks connect through a managed proxy
         *        container that collects logs and metrics for increased visibility. Only the tasks that Amazon ECS
         *        services create are supported with Service Connect. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-connect.html">Service
         *        Connect</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @see ApplicationProtocol
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ApplicationProtocol
         */
        Builder appProtocol(String appProtocol);

        /**
         * <p>
         * The application protocol that's used for the port mapping. This parameter only applies to Service Connect. We
         * recommend that you set this parameter to be consistent with the protocol that your application uses. If you
         * set this parameter, Amazon ECS adds protocol-specific connection handling to the Service Connect proxy. If
         * you set this parameter, Amazon ECS adds protocol-specific telemetry in the Amazon ECS console and CloudWatch.
         * </p>
         * <p>
         * If you don't set a value for this parameter, then TCP is used. However, Amazon ECS doesn't add
         * protocol-specific telemetry for TCP.
         * </p>
         * <p>
         * <code>appProtocol</code> is immutable in a Service Connect service. Updating this field requires a service
         * deletion and redeployment.
         * </p>
         * <p>
         * Tasks that run in a namespace can use short names to connect to services in the namespace. Tasks can connect
         * to services across all of the clusters in the namespace. Tasks connect through a managed proxy container that
         * collects logs and metrics for increased visibility. Only the tasks that Amazon ECS services create are
         * supported with Service Connect. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-connect.html">Service Connect</a>
         * in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param appProtocol
         *        The application protocol that's used for the port mapping. This parameter only applies to Service
         *        Connect. We recommend that you set this parameter to be consistent with the protocol that your
         *        application uses. If you set this parameter, Amazon ECS adds protocol-specific connection handling to
         *        the Service Connect proxy. If you set this parameter, Amazon ECS adds protocol-specific telemetry in
         *        the Amazon ECS console and CloudWatch.</p>
         *        <p>
         *        If you don't set a value for this parameter, then TCP is used. However, Amazon ECS doesn't add
         *        protocol-specific telemetry for TCP.
         *        </p>
         *        <p>
         *        <code>appProtocol</code> is immutable in a Service Connect service. Updating this field requires a
         *        service deletion and redeployment.
         *        </p>
         *        <p>
         *        Tasks that run in a namespace can use short names to connect to services in the namespace. Tasks can
         *        connect to services across all of the clusters in the namespace. Tasks connect through a managed proxy
         *        container that collects logs and metrics for increased visibility. Only the tasks that Amazon ECS
         *        services create are supported with Service Connect. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-connect.html">Service
         *        Connect</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @see ApplicationProtocol
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ApplicationProtocol
         */
        Builder appProtocol(ApplicationProtocol appProtocol);

        /**
         * <p>
         * The port number range on the container that's bound to the dynamically mapped host port range.
         * </p>
         * <p>
         * The following rules apply when you specify a <code>containerPortRange</code>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * You must use either the <code>bridge</code> network mode or the <code>awsvpc</code> network mode.
         * </p>
         * </li>
         * <li>
         * <p>
         * This parameter is available for both the EC2 and Fargate launch types.
         * </p>
         * </li>
         * <li>
         * <p>
         * This parameter is available for both the Linux and Windows operating systems.
         * </p>
         * </li>
         * <li>
         * <p>
         * The container instance must have at least version 1.67.0 of the container agent and at least version 1.67.0-1
         * of the <code>ecs-init</code> package
         * </p>
         * </li>
         * <li>
         * <p>
         * You can specify a maximum of 100 port ranges per container.
         * </p>
         * </li>
         * <li>
         * <p>
         * You do not specify a <code>hostPortRange</code>. The value of the <code>hostPortRange</code> is set as
         * follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * For containers in a task with the <code>awsvpc</code> network mode, the <code>hostPortRange</code> is set to
         * the same value as the <code>containerPortRange</code>. This is a static mapping strategy.
         * </p>
         * </li>
         * <li>
         * <p>
         * For containers in a task with the <code>bridge</code> network mode, the Amazon ECS agent finds open host
         * ports from the default ephemeral range and passes it to docker to bind them to the container ports.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * The <code>containerPortRange</code> valid values are between 1 and 65535.
         * </p>
         * </li>
         * <li>
         * <p>
         * A port can only be included in one port mapping per container.
         * </p>
         * </li>
         * <li>
         * <p>
         * You cannot specify overlapping port ranges.
         * </p>
         * </li>
         * <li>
         * <p>
         * The first port in the range must be less than last port in the range.
         * </p>
         * </li>
         * <li>
         * <p>
         * Docker recommends that you turn off the docker-proxy in the Docker daemon config file when you have a large
         * number of ports.
         * </p>
         * <p>
         * For more information, see <a href="https://github.com/moby/moby/issues/11185"> Issue #11185</a> on the Github
         * website.
         * </p>
         * <p>
         * For information about how to turn off the docker-proxy in the Docker daemon config file, see <a href=
         * "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/bootstrap_container_instance.html#bootstrap_docker_daemon"
         * >Docker daemon</a> in the <i>Amazon ECS Developer Guide</i>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * You can call <a href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTasks.html">
         * <code>DescribeTasks</code> </a> to view the <code>hostPortRange</code> which are the host ports that are
         * bound to the container ports.
         * </p>
         * 
         * @param containerPortRange
         *        The port number range on the container that's bound to the dynamically mapped host port range. </p>
         *        <p>
         *        The following rules apply when you specify a <code>containerPortRange</code>:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        You must use either the <code>bridge</code> network mode or the <code>awsvpc</code> network mode.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        This parameter is available for both the EC2 and Fargate launch types.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        This parameter is available for both the Linux and Windows operating systems.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The container instance must have at least version 1.67.0 of the container agent and at least version
         *        1.67.0-1 of the <code>ecs-init</code> package
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        You can specify a maximum of 100 port ranges per container.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        You do not specify a <code>hostPortRange</code>. The value of the <code>hostPortRange</code> is set as
         *        follows:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For containers in a task with the <code>awsvpc</code> network mode, the <code>hostPortRange</code> is
         *        set to the same value as the <code>containerPortRange</code>. This is a static mapping strategy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For containers in a task with the <code>bridge</code> network mode, the Amazon ECS agent finds open
         *        host ports from the default ephemeral range and passes it to docker to bind them to the container
         *        ports.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        The <code>containerPortRange</code> valid values are between 1 and 65535.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A port can only be included in one port mapping per container.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        You cannot specify overlapping port ranges.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The first port in the range must be less than last port in the range.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Docker recommends that you turn off the docker-proxy in the Docker daemon config file when you have a
         *        large number of ports.
         *        </p>
         *        <p>
         *        For more information, see <a href="https://github.com/moby/moby/issues/11185"> Issue #11185</a> on the
         *        Github website.
         *        </p>
         *        <p>
         *        For information about how to turn off the docker-proxy in the Docker daemon config file, see <a href=
         *        "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/bootstrap_container_instance.html#bootstrap_docker_daemon"
         *        >Docker daemon</a> in the <i>Amazon ECS Developer Guide</i>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        You can call <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTasks.html">
         *        <code>DescribeTasks</code> </a> to view the <code>hostPortRange</code> which are the host ports that
         *        are bound to the container ports.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerPortRange(String containerPortRange);
    }

    static final class BuilderImpl implements Builder {
        private Integer containerPort;

        private Integer hostPort;

        private String protocol;

        private String name;

        private String appProtocol;

        private String containerPortRange;

        private BuilderImpl() {
        }

        private BuilderImpl(PortMapping model) {
            containerPort(model.containerPort);
            hostPort(model.hostPort);
            protocol(model.protocol);
            name(model.name);
            appProtocol(model.appProtocol);
            containerPortRange(model.containerPortRange);
        }

        public final Integer getContainerPort() {
            return containerPort;
        }

        public final void setContainerPort(Integer containerPort) {
            this.containerPort = containerPort;
        }

        @Override
        public final Builder containerPort(Integer containerPort) {
            this.containerPort = containerPort;
            return this;
        }

        public final Integer getHostPort() {
            return hostPort;
        }

        public final void setHostPort(Integer hostPort) {
            this.hostPort = hostPort;
        }

        @Override
        public final Builder hostPort(Integer hostPort) {
            this.hostPort = hostPort;
            return this;
        }

        public final String getProtocol() {
            return protocol;
        }

        public final void setProtocol(String protocol) {
            this.protocol = protocol;
        }

        @Override
        public final Builder protocol(String protocol) {
            this.protocol = protocol;
            return this;
        }

        @Override
        public final Builder protocol(TransportProtocol protocol) {
            this.protocol(protocol == null ? null : protocol.toString());
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getAppProtocol() {
            return appProtocol;
        }

        public final void setAppProtocol(String appProtocol) {
            this.appProtocol = appProtocol;
        }

        @Override
        public final Builder appProtocol(String appProtocol) {
            this.appProtocol = appProtocol;
            return this;
        }

        @Override
        public final Builder appProtocol(ApplicationProtocol appProtocol) {
            this.appProtocol(appProtocol == null ? null : appProtocol.toString());
            return this;
        }

        public final String getContainerPortRange() {
            return containerPortRange;
        }

        public final void setContainerPortRange(String containerPortRange) {
            this.containerPortRange = containerPortRange;
        }

        @Override
        public final Builder containerPortRange(String containerPortRange) {
            this.containerPortRange = containerPortRange;
            return this;
        }

        @Override
        public PortMapping build() {
            return new PortMapping(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
