/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration details for the App Mesh proxy.
 * </p>
 * <p>
 * For tasks that use the EC2 launch type, the container instances require at least version 1.26.0 of the container
 * agent and at least version 1.26.0-1 of the <code>ecs-init</code> package to enable a proxy configuration. If your
 * container instances are launched from the Amazon ECS optimized AMI version <code>20190301</code> or later, then they
 * contain the required versions of the container agent and <code>ecs-init</code>. For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html">Amazon ECS-optimized Linux
 * AMI</a>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ProxyConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<ProxyConfiguration.Builder, ProxyConfiguration> {
    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(ProxyConfiguration::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<String> CONTAINER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("containerName").getter(getter(ProxyConfiguration::containerName)).setter(setter(Builder::containerName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("containerName").build()).build();

    private static final SdkField<List<KeyValuePair>> PROPERTIES_FIELD = SdkField
            .<List<KeyValuePair>> builder(MarshallingType.LIST)
            .memberName("properties")
            .getter(getter(ProxyConfiguration::properties))
            .setter(setter(Builder::properties))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("properties").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<KeyValuePair> builder(MarshallingType.SDK_POJO)
                                            .constructor(KeyValuePair::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_FIELD,
            CONTAINER_NAME_FIELD, PROPERTIES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String type;

    private final String containerName;

    private final List<KeyValuePair> properties;

    private ProxyConfiguration(BuilderImpl builder) {
        this.type = builder.type;
        this.containerName = builder.containerName;
        this.properties = builder.properties;
    }

    /**
     * <p>
     * The proxy type. The only supported value is <code>APPMESH</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ProxyConfigurationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The proxy type. The only supported value is <code>APPMESH</code>.
     * @see ProxyConfigurationType
     */
    public final ProxyConfigurationType type() {
        return ProxyConfigurationType.fromValue(type);
    }

    /**
     * <p>
     * The proxy type. The only supported value is <code>APPMESH</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ProxyConfigurationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The proxy type. The only supported value is <code>APPMESH</code>.
     * @see ProxyConfigurationType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The name of the container that will serve as the App Mesh proxy.
     * </p>
     * 
     * @return The name of the container that will serve as the App Mesh proxy.
     */
    public final String containerName() {
        return containerName;
    }

    /**
     * For responses, this returns true if the service returned a value for the Properties property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasProperties() {
        return properties != null && !(properties instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The set of network configuration parameters to provide the Container Network Interface (CNI) plugin, specified as
     * key-value pairs.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>IgnoredUID</code> - (Required) The user ID (UID) of the proxy container as defined by the <code>user</code>
     * parameter in a container definition. This is used to ensure the proxy ignores its own traffic. If
     * <code>IgnoredGID</code> is specified, this field can be empty.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IgnoredGID</code> - (Required) The group ID (GID) of the proxy container as defined by the
     * <code>user</code> parameter in a container definition. This is used to ensure the proxy ignores its own traffic.
     * If <code>IgnoredUID</code> is specified, this field can be empty.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AppPorts</code> - (Required) The list of ports that the application uses. Network traffic to these ports is
     * forwarded to the <code>ProxyIngressPort</code> and <code>ProxyEgressPort</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ProxyIngressPort</code> - (Required) Specifies the port that incoming traffic to the <code>AppPorts</code>
     * is directed to.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ProxyEgressPort</code> - (Required) Specifies the port that outgoing traffic from the <code>AppPorts</code>
     * is directed to.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>EgressIgnoredPorts</code> - (Required) The egress traffic going to the specified ports is ignored and not
     * redirected to the <code>ProxyEgressPort</code>. It can be an empty list.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>EgressIgnoredIPs</code> - (Required) The egress traffic going to the specified IP addresses is ignored and
     * not redirected to the <code>ProxyEgressPort</code>. It can be an empty list.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasProperties} method.
     * </p>
     * 
     * @return The set of network configuration parameters to provide the Container Network Interface (CNI) plugin,
     *         specified as key-value pairs.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>IgnoredUID</code> - (Required) The user ID (UID) of the proxy container as defined by the
     *         <code>user</code> parameter in a container definition. This is used to ensure the proxy ignores its own
     *         traffic. If <code>IgnoredGID</code> is specified, this field can be empty.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IgnoredGID</code> - (Required) The group ID (GID) of the proxy container as defined by the
     *         <code>user</code> parameter in a container definition. This is used to ensure the proxy ignores its own
     *         traffic. If <code>IgnoredUID</code> is specified, this field can be empty.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AppPorts</code> - (Required) The list of ports that the application uses. Network traffic to these
     *         ports is forwarded to the <code>ProxyIngressPort</code> and <code>ProxyEgressPort</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ProxyIngressPort</code> - (Required) Specifies the port that incoming traffic to the
     *         <code>AppPorts</code> is directed to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ProxyEgressPort</code> - (Required) Specifies the port that outgoing traffic from the
     *         <code>AppPorts</code> is directed to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>EgressIgnoredPorts</code> - (Required) The egress traffic going to the specified ports is ignored
     *         and not redirected to the <code>ProxyEgressPort</code>. It can be an empty list.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>EgressIgnoredIPs</code> - (Required) The egress traffic going to the specified IP addresses is
     *         ignored and not redirected to the <code>ProxyEgressPort</code>. It can be an empty list.
     *         </p>
     *         </li>
     */
    public final List<KeyValuePair> properties() {
        return properties;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(containerName());
        hashCode = 31 * hashCode + Objects.hashCode(hasProperties() ? properties() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ProxyConfiguration)) {
            return false;
        }
        ProxyConfiguration other = (ProxyConfiguration) obj;
        return Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(containerName(), other.containerName())
                && hasProperties() == other.hasProperties() && Objects.equals(properties(), other.properties());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ProxyConfiguration").add("Type", typeAsString()).add("ContainerName", containerName())
                .add("Properties", hasProperties() ? properties() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "containerName":
            return Optional.ofNullable(clazz.cast(containerName()));
        case "properties":
            return Optional.ofNullable(clazz.cast(properties()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ProxyConfiguration, T> g) {
        return obj -> g.apply((ProxyConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ProxyConfiguration> {
        /**
         * <p>
         * The proxy type. The only supported value is <code>APPMESH</code>.
         * </p>
         * 
         * @param type
         *        The proxy type. The only supported value is <code>APPMESH</code>.
         * @see ProxyConfigurationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProxyConfigurationType
         */
        Builder type(String type);

        /**
         * <p>
         * The proxy type. The only supported value is <code>APPMESH</code>.
         * </p>
         * 
         * @param type
         *        The proxy type. The only supported value is <code>APPMESH</code>.
         * @see ProxyConfigurationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProxyConfigurationType
         */
        Builder type(ProxyConfigurationType type);

        /**
         * <p>
         * The name of the container that will serve as the App Mesh proxy.
         * </p>
         * 
         * @param containerName
         *        The name of the container that will serve as the App Mesh proxy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerName(String containerName);

        /**
         * <p>
         * The set of network configuration parameters to provide the Container Network Interface (CNI) plugin,
         * specified as key-value pairs.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>IgnoredUID</code> - (Required) The user ID (UID) of the proxy container as defined by the
         * <code>user</code> parameter in a container definition. This is used to ensure the proxy ignores its own
         * traffic. If <code>IgnoredGID</code> is specified, this field can be empty.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IgnoredGID</code> - (Required) The group ID (GID) of the proxy container as defined by the
         * <code>user</code> parameter in a container definition. This is used to ensure the proxy ignores its own
         * traffic. If <code>IgnoredUID</code> is specified, this field can be empty.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AppPorts</code> - (Required) The list of ports that the application uses. Network traffic to these
         * ports is forwarded to the <code>ProxyIngressPort</code> and <code>ProxyEgressPort</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ProxyIngressPort</code> - (Required) Specifies the port that incoming traffic to the
         * <code>AppPorts</code> is directed to.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ProxyEgressPort</code> - (Required) Specifies the port that outgoing traffic from the
         * <code>AppPorts</code> is directed to.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>EgressIgnoredPorts</code> - (Required) The egress traffic going to the specified ports is ignored and
         * not redirected to the <code>ProxyEgressPort</code>. It can be an empty list.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>EgressIgnoredIPs</code> - (Required) The egress traffic going to the specified IP addresses is ignored
         * and not redirected to the <code>ProxyEgressPort</code>. It can be an empty list.
         * </p>
         * </li>
         * </ul>
         * 
         * @param properties
         *        The set of network configuration parameters to provide the Container Network Interface (CNI) plugin,
         *        specified as key-value pairs.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>IgnoredUID</code> - (Required) The user ID (UID) of the proxy container as defined by the
         *        <code>user</code> parameter in a container definition. This is used to ensure the proxy ignores its
         *        own traffic. If <code>IgnoredGID</code> is specified, this field can be empty.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IgnoredGID</code> - (Required) The group ID (GID) of the proxy container as defined by the
         *        <code>user</code> parameter in a container definition. This is used to ensure the proxy ignores its
         *        own traffic. If <code>IgnoredUID</code> is specified, this field can be empty.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AppPorts</code> - (Required) The list of ports that the application uses. Network traffic to
         *        these ports is forwarded to the <code>ProxyIngressPort</code> and <code>ProxyEgressPort</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ProxyIngressPort</code> - (Required) Specifies the port that incoming traffic to the
         *        <code>AppPorts</code> is directed to.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ProxyEgressPort</code> - (Required) Specifies the port that outgoing traffic from the
         *        <code>AppPorts</code> is directed to.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>EgressIgnoredPorts</code> - (Required) The egress traffic going to the specified ports is
         *        ignored and not redirected to the <code>ProxyEgressPort</code>. It can be an empty list.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>EgressIgnoredIPs</code> - (Required) The egress traffic going to the specified IP addresses is
         *        ignored and not redirected to the <code>ProxyEgressPort</code>. It can be an empty list.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder properties(Collection<KeyValuePair> properties);

        /**
         * <p>
         * The set of network configuration parameters to provide the Container Network Interface (CNI) plugin,
         * specified as key-value pairs.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>IgnoredUID</code> - (Required) The user ID (UID) of the proxy container as defined by the
         * <code>user</code> parameter in a container definition. This is used to ensure the proxy ignores its own
         * traffic. If <code>IgnoredGID</code> is specified, this field can be empty.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IgnoredGID</code> - (Required) The group ID (GID) of the proxy container as defined by the
         * <code>user</code> parameter in a container definition. This is used to ensure the proxy ignores its own
         * traffic. If <code>IgnoredUID</code> is specified, this field can be empty.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AppPorts</code> - (Required) The list of ports that the application uses. Network traffic to these
         * ports is forwarded to the <code>ProxyIngressPort</code> and <code>ProxyEgressPort</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ProxyIngressPort</code> - (Required) Specifies the port that incoming traffic to the
         * <code>AppPorts</code> is directed to.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ProxyEgressPort</code> - (Required) Specifies the port that outgoing traffic from the
         * <code>AppPorts</code> is directed to.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>EgressIgnoredPorts</code> - (Required) The egress traffic going to the specified ports is ignored and
         * not redirected to the <code>ProxyEgressPort</code>. It can be an empty list.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>EgressIgnoredIPs</code> - (Required) The egress traffic going to the specified IP addresses is ignored
         * and not redirected to the <code>ProxyEgressPort</code>. It can be an empty list.
         * </p>
         * </li>
         * </ul>
         * 
         * @param properties
         *        The set of network configuration parameters to provide the Container Network Interface (CNI) plugin,
         *        specified as key-value pairs.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>IgnoredUID</code> - (Required) The user ID (UID) of the proxy container as defined by the
         *        <code>user</code> parameter in a container definition. This is used to ensure the proxy ignores its
         *        own traffic. If <code>IgnoredGID</code> is specified, this field can be empty.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IgnoredGID</code> - (Required) The group ID (GID) of the proxy container as defined by the
         *        <code>user</code> parameter in a container definition. This is used to ensure the proxy ignores its
         *        own traffic. If <code>IgnoredUID</code> is specified, this field can be empty.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AppPorts</code> - (Required) The list of ports that the application uses. Network traffic to
         *        these ports is forwarded to the <code>ProxyIngressPort</code> and <code>ProxyEgressPort</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ProxyIngressPort</code> - (Required) Specifies the port that incoming traffic to the
         *        <code>AppPorts</code> is directed to.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ProxyEgressPort</code> - (Required) Specifies the port that outgoing traffic from the
         *        <code>AppPorts</code> is directed to.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>EgressIgnoredPorts</code> - (Required) The egress traffic going to the specified ports is
         *        ignored and not redirected to the <code>ProxyEgressPort</code>. It can be an empty list.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>EgressIgnoredIPs</code> - (Required) The egress traffic going to the specified IP addresses is
         *        ignored and not redirected to the <code>ProxyEgressPort</code>. It can be an empty list.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder properties(KeyValuePair... properties);

        /**
         * <p>
         * The set of network configuration parameters to provide the Container Network Interface (CNI) plugin,
         * specified as key-value pairs.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>IgnoredUID</code> - (Required) The user ID (UID) of the proxy container as defined by the
         * <code>user</code> parameter in a container definition. This is used to ensure the proxy ignores its own
         * traffic. If <code>IgnoredGID</code> is specified, this field can be empty.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IgnoredGID</code> - (Required) The group ID (GID) of the proxy container as defined by the
         * <code>user</code> parameter in a container definition. This is used to ensure the proxy ignores its own
         * traffic. If <code>IgnoredUID</code> is specified, this field can be empty.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AppPorts</code> - (Required) The list of ports that the application uses. Network traffic to these
         * ports is forwarded to the <code>ProxyIngressPort</code> and <code>ProxyEgressPort</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ProxyIngressPort</code> - (Required) Specifies the port that incoming traffic to the
         * <code>AppPorts</code> is directed to.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ProxyEgressPort</code> - (Required) Specifies the port that outgoing traffic from the
         * <code>AppPorts</code> is directed to.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>EgressIgnoredPorts</code> - (Required) The egress traffic going to the specified ports is ignored and
         * not redirected to the <code>ProxyEgressPort</code>. It can be an empty list.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>EgressIgnoredIPs</code> - (Required) The egress traffic going to the specified IP addresses is ignored
         * and not redirected to the <code>ProxyEgressPort</code>. It can be an empty list.
         * </p>
         * </li>
         * </ul>
         * This is a convenience that creates an instance of the {@link List<KeyValuePair>.Builder} avoiding the need to
         * create one manually via {@link List<KeyValuePair>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<KeyValuePair>.Builder#build()} is called immediately and its
         * result is passed to {@link #properties(List<KeyValuePair>)}.
         * 
         * @param properties
         *        a consumer that will call methods on {@link List<KeyValuePair>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #properties(List<KeyValuePair>)
         */
        Builder properties(Consumer<KeyValuePair.Builder>... properties);
    }

    static final class BuilderImpl implements Builder {
        private String type;

        private String containerName;

        private List<KeyValuePair> properties = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ProxyConfiguration model) {
            type(model.type);
            containerName(model.containerName);
            properties(model.properties);
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        @Transient
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        @Transient
        public final Builder type(ProxyConfigurationType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getContainerName() {
            return containerName;
        }

        public final void setContainerName(String containerName) {
            this.containerName = containerName;
        }

        @Override
        @Transient
        public final Builder containerName(String containerName) {
            this.containerName = containerName;
            return this;
        }

        public final List<KeyValuePair.Builder> getProperties() {
            List<KeyValuePair.Builder> result = ProxyConfigurationPropertiesCopier.copyToBuilder(this.properties);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setProperties(Collection<KeyValuePair.BuilderImpl> properties) {
            this.properties = ProxyConfigurationPropertiesCopier.copyFromBuilder(properties);
        }

        @Override
        @Transient
        public final Builder properties(Collection<KeyValuePair> properties) {
            this.properties = ProxyConfigurationPropertiesCopier.copy(properties);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder properties(KeyValuePair... properties) {
            properties(Arrays.asList(properties));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder properties(Consumer<KeyValuePair.Builder>... properties) {
            properties(Stream.of(properties).map(c -> KeyValuePair.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public ProxyConfiguration build() {
            return new ProxyConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
