/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a condition within a CIDR selection rule. Conditions define the criteria for selecting CIDRs from IPAM's
 * database based on resource attributes.
 * </p>
 * <p>
 * CIDR selection rules define the business logic for selecting CIDRs from IPAM. If a CIDR matches any of the rules, it
 * will be included. If a rule has multiple conditions, the CIDR has to match every condition of that rule. You can
 * create a prefix list resolver without any CIDR selection rules, but it will generate empty versions (containing no
 * CIDRs) until you add rules.
 * </p>
 * <p>
 * There are three rule types. Only 2 of the 3 rule types support conditions - <b>IPAM pool CIDR</b> and <b>Scope
 * resource CIDR</b>. <b>Static CIDR</b> rules cannot have conditions.
 * </p>
 * <ul>
 * <li>
 * <p>
 * <b>Static CIDR</b>: A fixed list of CIDRs that do not change (like a manual list replicated across Regions)
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>IPAM pool CIDR</b>: CIDRs from specific IPAM pools (like all CIDRs from your IPAM production pool)
 * </p>
 * <p>
 * If you choose this option, choose the following:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <b>IPAM scope</b>: Select the IPAM scope to search for resources
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Conditions:</b>
 * </p>
 * <ul>
 * <li>
 * <p>
 * <b>Property</b>
 * </p>
 * <ul>
 * <li>
 * <p>
 * <b>IPAM pool ID</b>: Select an IPAM pool that contains the resources
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>CIDR</b> (like 10.24.34.0/23)
 * </p>
 * </li>
 * </ul>
 * </li>
 * <li>
 * <p>
 * <b>Operation</b>: Equals/Not equals
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Value</b>: The value on which to match the condition
 * </p>
 * </li>
 * </ul>
 * </li>
 * </ul>
 * </li>
 * <li>
 * <p>
 * <b>Scope resource CIDR</b>: CIDRs from Amazon Web Services resources like VPCs, subnets, EIPs within an IPAM scope
 * </p>
 * <p>
 * If you choose this option, choose the following:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <b>IPAM scope</b>: Select the IPAM scope to search for resources
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Resource type</b>: Select a resource, like a VPC or subnet.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Conditions</b>:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <b>Property</b>:
 * </p>
 * <ul>
 * <li>
 * <p>
 * Resource ID: The unique ID of a resource (like vpc-1234567890abcdef0)
 * </p>
 * </li>
 * <li>
 * <p>
 * Resource owner (like 111122223333)
 * </p>
 * </li>
 * <li>
 * <p>
 * Resource region (like us-east-1)
 * </p>
 * </li>
 * <li>
 * <p>
 * Resource tag (like key: name, value: dev-vpc-1)
 * </p>
 * </li>
 * <li>
 * <p>
 * CIDR (like 10.24.34.0/23)
 * </p>
 * </li>
 * </ul>
 * </li>
 * <li>
 * <p>
 * <b>Operation</b>: Equals/Not equals
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Value</b>: The value on which to match the condition
 * </p>
 * </li>
 * </ul>
 * </li>
 * </ul>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class IpamPrefixListResolverRuleCondition implements SdkPojo, Serializable,
        ToCopyableBuilder<IpamPrefixListResolverRuleCondition.Builder, IpamPrefixListResolverRuleCondition> {
    private static final SdkField<String> OPERATION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Operation")
            .getter(getter(IpamPrefixListResolverRuleCondition::operationAsString))
            .setter(setter(Builder::operation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Operation")
                    .unmarshallLocationName("operation").build()).build();

    private static final SdkField<String> IPAM_POOL_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamPoolId")
            .getter(getter(IpamPrefixListResolverRuleCondition::ipamPoolId))
            .setter(setter(Builder::ipamPoolId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamPoolId")
                    .unmarshallLocationName("ipamPoolId").build()).build();

    private static final SdkField<String> RESOURCE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ResourceId")
            .getter(getter(IpamPrefixListResolverRuleCondition::resourceId))
            .setter(setter(Builder::resourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceId")
                    .unmarshallLocationName("resourceId").build()).build();

    private static final SdkField<String> RESOURCE_OWNER_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ResourceOwner")
            .getter(getter(IpamPrefixListResolverRuleCondition::resourceOwner))
            .setter(setter(Builder::resourceOwner))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceOwner")
                    .unmarshallLocationName("resourceOwner").build()).build();

    private static final SdkField<String> RESOURCE_REGION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ResourceRegion")
            .getter(getter(IpamPrefixListResolverRuleCondition::resourceRegion))
            .setter(setter(Builder::resourceRegion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceRegion")
                    .unmarshallLocationName("resourceRegion").build()).build();

    private static final SdkField<IpamResourceTag> RESOURCE_TAG_FIELD = SdkField
            .<IpamResourceTag> builder(MarshallingType.SDK_POJO)
            .memberName("ResourceTag")
            .getter(getter(IpamPrefixListResolverRuleCondition::resourceTag))
            .setter(setter(Builder::resourceTag))
            .constructor(IpamResourceTag::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceTag")
                    .unmarshallLocationName("resourceTag").build()).build();

    private static final SdkField<String> CIDR_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Cidr")
            .getter(getter(IpamPrefixListResolverRuleCondition::cidr))
            .setter(setter(Builder::cidr))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Cidr")
                    .unmarshallLocationName("cidr").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(OPERATION_FIELD,
            IPAM_POOL_ID_FIELD, RESOURCE_ID_FIELD, RESOURCE_OWNER_FIELD, RESOURCE_REGION_FIELD, RESOURCE_TAG_FIELD, CIDR_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String operation;

    private final String ipamPoolId;

    private final String resourceId;

    private final String resourceOwner;

    private final String resourceRegion;

    private final IpamResourceTag resourceTag;

    private final String cidr;

    private IpamPrefixListResolverRuleCondition(BuilderImpl builder) {
        this.operation = builder.operation;
        this.ipamPoolId = builder.ipamPoolId;
        this.resourceId = builder.resourceId;
        this.resourceOwner = builder.resourceOwner;
        this.resourceRegion = builder.resourceRegion;
        this.resourceTag = builder.resourceTag;
        this.cidr = builder.cidr;
    }

    /**
     * <p>
     * The operation to perform when evaluating this condition. Valid values include <code>equals</code>,
     * <code>not-equals</code>, <code>contains</code>, and <code>not-contains</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operation} will
     * return {@link IpamPrefixListResolverRuleConditionOperation#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #operationAsString}.
     * </p>
     * 
     * @return The operation to perform when evaluating this condition. Valid values include <code>equals</code>,
     *         <code>not-equals</code>, <code>contains</code>, and <code>not-contains</code>.
     * @see IpamPrefixListResolverRuleConditionOperation
     */
    public final IpamPrefixListResolverRuleConditionOperation operation() {
        return IpamPrefixListResolverRuleConditionOperation.fromValue(operation);
    }

    /**
     * <p>
     * The operation to perform when evaluating this condition. Valid values include <code>equals</code>,
     * <code>not-equals</code>, <code>contains</code>, and <code>not-contains</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operation} will
     * return {@link IpamPrefixListResolverRuleConditionOperation#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #operationAsString}.
     * </p>
     * 
     * @return The operation to perform when evaluating this condition. Valid values include <code>equals</code>,
     *         <code>not-equals</code>, <code>contains</code>, and <code>not-contains</code>.
     * @see IpamPrefixListResolverRuleConditionOperation
     */
    public final String operationAsString() {
        return operation;
    }

    /**
     * <p>
     * The ID of the IPAM pool to match against. This condition selects CIDRs that belong to the specified IPAM pool.
     * </p>
     * 
     * @return The ID of the IPAM pool to match against. This condition selects CIDRs that belong to the specified IPAM
     *         pool.
     */
    public final String ipamPoolId() {
        return ipamPoolId;
    }

    /**
     * <p>
     * The ID of the Amazon Web Services resource to match against. This condition selects CIDRs associated with the
     * specified resource.
     * </p>
     * 
     * @return The ID of the Amazon Web Services resource to match against. This condition selects CIDRs associated with
     *         the specified resource.
     */
    public final String resourceId() {
        return resourceId;
    }

    /**
     * <p>
     * The Amazon Web Services account ID that owns the resources to match against. This condition selects CIDRs from
     * resources owned by the specified account.
     * </p>
     * 
     * @return The Amazon Web Services account ID that owns the resources to match against. This condition selects CIDRs
     *         from resources owned by the specified account.
     */
    public final String resourceOwner() {
        return resourceOwner;
    }

    /**
     * <p>
     * The Amazon Web Services Region where the resources are located. This condition selects CIDRs from resources in
     * the specified Region.
     * </p>
     * 
     * @return The Amazon Web Services Region where the resources are located. This condition selects CIDRs from
     *         resources in the specified Region.
     */
    public final String resourceRegion() {
        return resourceRegion;
    }

    /**
     * <p>
     * A tag key-value pair to match against. This condition selects CIDRs from resources that have the specified tag.
     * </p>
     * 
     * @return A tag key-value pair to match against. This condition selects CIDRs from resources that have the
     *         specified tag.
     */
    public final IpamResourceTag resourceTag() {
        return resourceTag;
    }

    /**
     * <p>
     * A CIDR block to match against. This condition selects CIDRs that fall within or match the specified CIDR range.
     * </p>
     * 
     * @return A CIDR block to match against. This condition selects CIDRs that fall within or match the specified CIDR
     *         range.
     */
    public final String cidr() {
        return cidr;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(operationAsString());
        hashCode = 31 * hashCode + Objects.hashCode(ipamPoolId());
        hashCode = 31 * hashCode + Objects.hashCode(resourceId());
        hashCode = 31 * hashCode + Objects.hashCode(resourceOwner());
        hashCode = 31 * hashCode + Objects.hashCode(resourceRegion());
        hashCode = 31 * hashCode + Objects.hashCode(resourceTag());
        hashCode = 31 * hashCode + Objects.hashCode(cidr());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof IpamPrefixListResolverRuleCondition)) {
            return false;
        }
        IpamPrefixListResolverRuleCondition other = (IpamPrefixListResolverRuleCondition) obj;
        return Objects.equals(operationAsString(), other.operationAsString()) && Objects.equals(ipamPoolId(), other.ipamPoolId())
                && Objects.equals(resourceId(), other.resourceId()) && Objects.equals(resourceOwner(), other.resourceOwner())
                && Objects.equals(resourceRegion(), other.resourceRegion()) && Objects.equals(resourceTag(), other.resourceTag())
                && Objects.equals(cidr(), other.cidr());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("IpamPrefixListResolverRuleCondition").add("Operation", operationAsString())
                .add("IpamPoolId", ipamPoolId()).add("ResourceId", resourceId()).add("ResourceOwner", resourceOwner())
                .add("ResourceRegion", resourceRegion()).add("ResourceTag", resourceTag()).add("Cidr", cidr()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Operation":
            return Optional.ofNullable(clazz.cast(operationAsString()));
        case "IpamPoolId":
            return Optional.ofNullable(clazz.cast(ipamPoolId()));
        case "ResourceId":
            return Optional.ofNullable(clazz.cast(resourceId()));
        case "ResourceOwner":
            return Optional.ofNullable(clazz.cast(resourceOwner()));
        case "ResourceRegion":
            return Optional.ofNullable(clazz.cast(resourceRegion()));
        case "ResourceTag":
            return Optional.ofNullable(clazz.cast(resourceTag()));
        case "Cidr":
            return Optional.ofNullable(clazz.cast(cidr()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Operation", OPERATION_FIELD);
        map.put("IpamPoolId", IPAM_POOL_ID_FIELD);
        map.put("ResourceId", RESOURCE_ID_FIELD);
        map.put("ResourceOwner", RESOURCE_OWNER_FIELD);
        map.put("ResourceRegion", RESOURCE_REGION_FIELD);
        map.put("ResourceTag", RESOURCE_TAG_FIELD);
        map.put("Cidr", CIDR_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<IpamPrefixListResolverRuleCondition, T> g) {
        return obj -> g.apply((IpamPrefixListResolverRuleCondition) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, IpamPrefixListResolverRuleCondition> {
        /**
         * <p>
         * The operation to perform when evaluating this condition. Valid values include <code>equals</code>,
         * <code>not-equals</code>, <code>contains</code>, and <code>not-contains</code>.
         * </p>
         * 
         * @param operation
         *        The operation to perform when evaluating this condition. Valid values include <code>equals</code>,
         *        <code>not-equals</code>, <code>contains</code>, and <code>not-contains</code>.
         * @see IpamPrefixListResolverRuleConditionOperation
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamPrefixListResolverRuleConditionOperation
         */
        Builder operation(String operation);

        /**
         * <p>
         * The operation to perform when evaluating this condition. Valid values include <code>equals</code>,
         * <code>not-equals</code>, <code>contains</code>, and <code>not-contains</code>.
         * </p>
         * 
         * @param operation
         *        The operation to perform when evaluating this condition. Valid values include <code>equals</code>,
         *        <code>not-equals</code>, <code>contains</code>, and <code>not-contains</code>.
         * @see IpamPrefixListResolverRuleConditionOperation
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamPrefixListResolverRuleConditionOperation
         */
        Builder operation(IpamPrefixListResolverRuleConditionOperation operation);

        /**
         * <p>
         * The ID of the IPAM pool to match against. This condition selects CIDRs that belong to the specified IPAM
         * pool.
         * </p>
         * 
         * @param ipamPoolId
         *        The ID of the IPAM pool to match against. This condition selects CIDRs that belong to the specified
         *        IPAM pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamPoolId(String ipamPoolId);

        /**
         * <p>
         * The ID of the Amazon Web Services resource to match against. This condition selects CIDRs associated with the
         * specified resource.
         * </p>
         * 
         * @param resourceId
         *        The ID of the Amazon Web Services resource to match against. This condition selects CIDRs associated
         *        with the specified resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceId(String resourceId);

        /**
         * <p>
         * The Amazon Web Services account ID that owns the resources to match against. This condition selects CIDRs
         * from resources owned by the specified account.
         * </p>
         * 
         * @param resourceOwner
         *        The Amazon Web Services account ID that owns the resources to match against. This condition selects
         *        CIDRs from resources owned by the specified account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceOwner(String resourceOwner);

        /**
         * <p>
         * The Amazon Web Services Region where the resources are located. This condition selects CIDRs from resources
         * in the specified Region.
         * </p>
         * 
         * @param resourceRegion
         *        The Amazon Web Services Region where the resources are located. This condition selects CIDRs from
         *        resources in the specified Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceRegion(String resourceRegion);

        /**
         * <p>
         * A tag key-value pair to match against. This condition selects CIDRs from resources that have the specified
         * tag.
         * </p>
         * 
         * @param resourceTag
         *        A tag key-value pair to match against. This condition selects CIDRs from resources that have the
         *        specified tag.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTag(IpamResourceTag resourceTag);

        /**
         * <p>
         * A tag key-value pair to match against. This condition selects CIDRs from resources that have the specified
         * tag.
         * </p>
         * This is a convenience method that creates an instance of the {@link IpamResourceTag.Builder} avoiding the
         * need to create one manually via {@link IpamResourceTag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link IpamResourceTag.Builder#build()} is called immediately and its
         * result is passed to {@link #resourceTag(IpamResourceTag)}.
         * 
         * @param resourceTag
         *        a consumer that will call methods on {@link IpamResourceTag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resourceTag(IpamResourceTag)
         */
        default Builder resourceTag(Consumer<IpamResourceTag.Builder> resourceTag) {
            return resourceTag(IpamResourceTag.builder().applyMutation(resourceTag).build());
        }

        /**
         * <p>
         * A CIDR block to match against. This condition selects CIDRs that fall within or match the specified CIDR
         * range.
         * </p>
         * 
         * @param cidr
         *        A CIDR block to match against. This condition selects CIDRs that fall within or match the specified
         *        CIDR range.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cidr(String cidr);
    }

    static final class BuilderImpl implements Builder {
        private String operation;

        private String ipamPoolId;

        private String resourceId;

        private String resourceOwner;

        private String resourceRegion;

        private IpamResourceTag resourceTag;

        private String cidr;

        private BuilderImpl() {
        }

        private BuilderImpl(IpamPrefixListResolverRuleCondition model) {
            operation(model.operation);
            ipamPoolId(model.ipamPoolId);
            resourceId(model.resourceId);
            resourceOwner(model.resourceOwner);
            resourceRegion(model.resourceRegion);
            resourceTag(model.resourceTag);
            cidr(model.cidr);
        }

        public final String getOperation() {
            return operation;
        }

        public final void setOperation(String operation) {
            this.operation = operation;
        }

        @Override
        public final Builder operation(String operation) {
            this.operation = operation;
            return this;
        }

        @Override
        public final Builder operation(IpamPrefixListResolverRuleConditionOperation operation) {
            this.operation(operation == null ? null : operation.toString());
            return this;
        }

        public final String getIpamPoolId() {
            return ipamPoolId;
        }

        public final void setIpamPoolId(String ipamPoolId) {
            this.ipamPoolId = ipamPoolId;
        }

        @Override
        public final Builder ipamPoolId(String ipamPoolId) {
            this.ipamPoolId = ipamPoolId;
            return this;
        }

        public final String getResourceId() {
            return resourceId;
        }

        public final void setResourceId(String resourceId) {
            this.resourceId = resourceId;
        }

        @Override
        public final Builder resourceId(String resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public final String getResourceOwner() {
            return resourceOwner;
        }

        public final void setResourceOwner(String resourceOwner) {
            this.resourceOwner = resourceOwner;
        }

        @Override
        public final Builder resourceOwner(String resourceOwner) {
            this.resourceOwner = resourceOwner;
            return this;
        }

        public final String getResourceRegion() {
            return resourceRegion;
        }

        public final void setResourceRegion(String resourceRegion) {
            this.resourceRegion = resourceRegion;
        }

        @Override
        public final Builder resourceRegion(String resourceRegion) {
            this.resourceRegion = resourceRegion;
            return this;
        }

        public final IpamResourceTag.Builder getResourceTag() {
            return resourceTag != null ? resourceTag.toBuilder() : null;
        }

        public final void setResourceTag(IpamResourceTag.BuilderImpl resourceTag) {
            this.resourceTag = resourceTag != null ? resourceTag.build() : null;
        }

        @Override
        public final Builder resourceTag(IpamResourceTag resourceTag) {
            this.resourceTag = resourceTag;
            return this;
        }

        public final String getCidr() {
            return cidr;
        }

        public final void setCidr(String cidr) {
            this.cidr = cidr;
        }

        @Override
        public final Builder cidr(String cidr) {
            this.cidr = cidr;
            return this;
        }

        @Override
        public IpamPrefixListResolverRuleCondition build() {
            return new IpamPrefixListResolverRuleCondition(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
