/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The criteria that are evaluated to determine which AMIs are discoverable and usable in your account for the specified
 * Amazon Web Services Region.
 * </p>
 * <p>
 * For more information, see <a
 * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-allowed-amis.html#how-allowed-amis-works">How Allowed
 * AMIs works</a> in the <i>Amazon EC2 User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ImageCriterion implements SdkPojo, Serializable, ToCopyableBuilder<ImageCriterion.Builder, ImageCriterion> {
    private static final SdkField<List<String>> IMAGE_PROVIDERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ImageProviders")
            .getter(getter(ImageCriterion::imageProviders))
            .setter(setter(Builder::imageProviders))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImageProviderSet")
                    .unmarshallLocationName("imageProviderSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<List<String>> MARKETPLACE_PRODUCT_CODES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("MarketplaceProductCodes")
            .getter(getter(ImageCriterion::marketplaceProductCodes))
            .setter(setter(Builder::marketplaceProductCodes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MarketplaceProductCodeSet")
                    .unmarshallLocationName("marketplaceProductCodeSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<List<String>> IMAGE_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ImageNames")
            .getter(getter(ImageCriterion::imageNames))
            .setter(setter(Builder::imageNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImageNameSet")
                    .unmarshallLocationName("imageNameSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<DeprecationTimeCondition> DEPRECATION_TIME_CONDITION_FIELD = SdkField
            .<DeprecationTimeCondition> builder(MarshallingType.SDK_POJO)
            .memberName("DeprecationTimeCondition")
            .getter(getter(ImageCriterion::deprecationTimeCondition))
            .setter(setter(Builder::deprecationTimeCondition))
            .constructor(DeprecationTimeCondition::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeprecationTimeCondition")
                    .unmarshallLocationName("deprecationTimeCondition").build()).build();

    private static final SdkField<CreationDateCondition> CREATION_DATE_CONDITION_FIELD = SdkField
            .<CreationDateCondition> builder(MarshallingType.SDK_POJO)
            .memberName("CreationDateCondition")
            .getter(getter(ImageCriterion::creationDateCondition))
            .setter(setter(Builder::creationDateCondition))
            .constructor(CreationDateCondition::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationDateCondition")
                    .unmarshallLocationName("creationDateCondition").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(IMAGE_PROVIDERS_FIELD,
            MARKETPLACE_PRODUCT_CODES_FIELD, IMAGE_NAMES_FIELD, DEPRECATION_TIME_CONDITION_FIELD, CREATION_DATE_CONDITION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<String> imageProviders;

    private final List<String> marketplaceProductCodes;

    private final List<String> imageNames;

    private final DeprecationTimeCondition deprecationTimeCondition;

    private final CreationDateCondition creationDateCondition;

    private ImageCriterion(BuilderImpl builder) {
        this.imageProviders = builder.imageProviders;
        this.marketplaceProductCodes = builder.marketplaceProductCodes;
        this.imageNames = builder.imageNames;
        this.deprecationTimeCondition = builder.deprecationTimeCondition;
        this.creationDateCondition = builder.creationDateCondition;
    }

    /**
     * For responses, this returns true if the service returned a value for the ImageProviders property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasImageProviders() {
        return imageProviders != null && !(imageProviders instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The image providers whose images are allowed.
     * </p>
     * <p>
     * Possible values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>amazon</code>: Allow AMIs created by Amazon or verified providers.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>aws-marketplace</code>: Allow AMIs created by verified providers in the Amazon Web Services Marketplace.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>aws-backup-vault</code>: Allow AMIs created by Amazon Web Services Backup.
     * </p>
     * </li>
     * <li>
     * <p>
     * 12-digit account ID: Allow AMIs created by this account. One or more account IDs can be specified.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>none</code>: Allow AMIs created by your own account only.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Maximum: 200 values
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasImageProviders} method.
     * </p>
     * 
     * @return The image providers whose images are allowed.</p>
     *         <p>
     *         Possible values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>amazon</code>: Allow AMIs created by Amazon or verified providers.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>aws-marketplace</code>: Allow AMIs created by verified providers in the Amazon Web Services
     *         Marketplace.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>aws-backup-vault</code>: Allow AMIs created by Amazon Web Services Backup.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         12-digit account ID: Allow AMIs created by this account. One or more account IDs can be specified.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>none</code>: Allow AMIs created by your own account only.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Maximum: 200 values
     */
    public final List<String> imageProviders() {
        return imageProviders;
    }

    /**
     * For responses, this returns true if the service returned a value for the MarketplaceProductCodes property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasMarketplaceProductCodes() {
        return marketplaceProductCodes != null && !(marketplaceProductCodes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon Web Services Marketplace product codes for allowed images.
     * </p>
     * <p>
     * Length: 1-25 characters
     * </p>
     * <p>
     * Valid characters: Letters (<code>A–Z, a–z</code>) and numbers (<code>0–9</code>)
     * </p>
     * <p>
     * Maximum: 50 values
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMarketplaceProductCodes} method.
     * </p>
     * 
     * @return The Amazon Web Services Marketplace product codes for allowed images.</p>
     *         <p>
     *         Length: 1-25 characters
     *         </p>
     *         <p>
     *         Valid characters: Letters (<code>A–Z, a–z</code>) and numbers (<code>0–9</code>)
     *         </p>
     *         <p>
     *         Maximum: 50 values
     */
    public final List<String> marketplaceProductCodes() {
        return marketplaceProductCodes;
    }

    /**
     * For responses, this returns true if the service returned a value for the ImageNames property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasImageNames() {
        return imageNames != null && !(imageNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The names of allowed images. Names can include wildcards (<code>?</code> and <code>*</code>).
     * </p>
     * <p>
     * Length: 1–128 characters. With <code>?</code>, the minimum is 3 characters.
     * </p>
     * <p>
     * Valid characters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Letters: <code>A–Z, a–z</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Numbers: <code>0–9</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Special characters: <code>( ) [ ] . / - ' @ _ * ?</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Spaces
     * </p>
     * </li>
     * </ul>
     * <p>
     * Maximum: 50 values
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasImageNames} method.
     * </p>
     * 
     * @return The names of allowed images. Names can include wildcards (<code>?</code> and <code>*</code>).</p>
     *         <p>
     *         Length: 1–128 characters. With <code>?</code>, the minimum is 3 characters.
     *         </p>
     *         <p>
     *         Valid characters:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Letters: <code>A–Z, a–z</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Numbers: <code>0–9</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Special characters: <code>( ) [ ] . / - ' @ _ * ?</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Spaces
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Maximum: 50 values
     */
    public final List<String> imageNames() {
        return imageNames;
    }

    /**
     * <p>
     * The maximum period since deprecation for allowed images.
     * </p>
     * 
     * @return The maximum period since deprecation for allowed images.
     */
    public final DeprecationTimeCondition deprecationTimeCondition() {
        return deprecationTimeCondition;
    }

    /**
     * <p>
     * The maximum age for allowed images.
     * </p>
     * 
     * @return The maximum age for allowed images.
     */
    public final CreationDateCondition creationDateCondition() {
        return creationDateCondition;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasImageProviders() ? imageProviders() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasMarketplaceProductCodes() ? marketplaceProductCodes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasImageNames() ? imageNames() : null);
        hashCode = 31 * hashCode + Objects.hashCode(deprecationTimeCondition());
        hashCode = 31 * hashCode + Objects.hashCode(creationDateCondition());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ImageCriterion)) {
            return false;
        }
        ImageCriterion other = (ImageCriterion) obj;
        return hasImageProviders() == other.hasImageProviders() && Objects.equals(imageProviders(), other.imageProviders())
                && hasMarketplaceProductCodes() == other.hasMarketplaceProductCodes()
                && Objects.equals(marketplaceProductCodes(), other.marketplaceProductCodes())
                && hasImageNames() == other.hasImageNames() && Objects.equals(imageNames(), other.imageNames())
                && Objects.equals(deprecationTimeCondition(), other.deprecationTimeCondition())
                && Objects.equals(creationDateCondition(), other.creationDateCondition());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ImageCriterion").add("ImageProviders", hasImageProviders() ? imageProviders() : null)
                .add("MarketplaceProductCodes", hasMarketplaceProductCodes() ? marketplaceProductCodes() : null)
                .add("ImageNames", hasImageNames() ? imageNames() : null)
                .add("DeprecationTimeCondition", deprecationTimeCondition())
                .add("CreationDateCondition", creationDateCondition()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ImageProviders":
            return Optional.ofNullable(clazz.cast(imageProviders()));
        case "MarketplaceProductCodes":
            return Optional.ofNullable(clazz.cast(marketplaceProductCodes()));
        case "ImageNames":
            return Optional.ofNullable(clazz.cast(imageNames()));
        case "DeprecationTimeCondition":
            return Optional.ofNullable(clazz.cast(deprecationTimeCondition()));
        case "CreationDateCondition":
            return Optional.ofNullable(clazz.cast(creationDateCondition()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ImageProviderSet", IMAGE_PROVIDERS_FIELD);
        map.put("MarketplaceProductCodeSet", MARKETPLACE_PRODUCT_CODES_FIELD);
        map.put("ImageNameSet", IMAGE_NAMES_FIELD);
        map.put("DeprecationTimeCondition", DEPRECATION_TIME_CONDITION_FIELD);
        map.put("CreationDateCondition", CREATION_DATE_CONDITION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ImageCriterion, T> g) {
        return obj -> g.apply((ImageCriterion) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ImageCriterion> {
        /**
         * <p>
         * The image providers whose images are allowed.
         * </p>
         * <p>
         * Possible values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>amazon</code>: Allow AMIs created by Amazon or verified providers.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>aws-marketplace</code>: Allow AMIs created by verified providers in the Amazon Web Services
         * Marketplace.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>aws-backup-vault</code>: Allow AMIs created by Amazon Web Services Backup.
         * </p>
         * </li>
         * <li>
         * <p>
         * 12-digit account ID: Allow AMIs created by this account. One or more account IDs can be specified.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>none</code>: Allow AMIs created by your own account only.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Maximum: 200 values
         * </p>
         * 
         * @param imageProviders
         *        The image providers whose images are allowed.</p>
         *        <p>
         *        Possible values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>amazon</code>: Allow AMIs created by Amazon or verified providers.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>aws-marketplace</code>: Allow AMIs created by verified providers in the Amazon Web Services
         *        Marketplace.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>aws-backup-vault</code>: Allow AMIs created by Amazon Web Services Backup.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        12-digit account ID: Allow AMIs created by this account. One or more account IDs can be specified.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>none</code>: Allow AMIs created by your own account only.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Maximum: 200 values
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageProviders(Collection<String> imageProviders);

        /**
         * <p>
         * The image providers whose images are allowed.
         * </p>
         * <p>
         * Possible values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>amazon</code>: Allow AMIs created by Amazon or verified providers.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>aws-marketplace</code>: Allow AMIs created by verified providers in the Amazon Web Services
         * Marketplace.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>aws-backup-vault</code>: Allow AMIs created by Amazon Web Services Backup.
         * </p>
         * </li>
         * <li>
         * <p>
         * 12-digit account ID: Allow AMIs created by this account. One or more account IDs can be specified.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>none</code>: Allow AMIs created by your own account only.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Maximum: 200 values
         * </p>
         * 
         * @param imageProviders
         *        The image providers whose images are allowed.</p>
         *        <p>
         *        Possible values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>amazon</code>: Allow AMIs created by Amazon or verified providers.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>aws-marketplace</code>: Allow AMIs created by verified providers in the Amazon Web Services
         *        Marketplace.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>aws-backup-vault</code>: Allow AMIs created by Amazon Web Services Backup.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        12-digit account ID: Allow AMIs created by this account. One or more account IDs can be specified.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>none</code>: Allow AMIs created by your own account only.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Maximum: 200 values
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageProviders(String... imageProviders);

        /**
         * <p>
         * The Amazon Web Services Marketplace product codes for allowed images.
         * </p>
         * <p>
         * Length: 1-25 characters
         * </p>
         * <p>
         * Valid characters: Letters (<code>A–Z, a–z</code>) and numbers (<code>0–9</code>)
         * </p>
         * <p>
         * Maximum: 50 values
         * </p>
         * 
         * @param marketplaceProductCodes
         *        The Amazon Web Services Marketplace product codes for allowed images.</p>
         *        <p>
         *        Length: 1-25 characters
         *        </p>
         *        <p>
         *        Valid characters: Letters (<code>A–Z, a–z</code>) and numbers (<code>0–9</code>)
         *        </p>
         *        <p>
         *        Maximum: 50 values
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder marketplaceProductCodes(Collection<String> marketplaceProductCodes);

        /**
         * <p>
         * The Amazon Web Services Marketplace product codes for allowed images.
         * </p>
         * <p>
         * Length: 1-25 characters
         * </p>
         * <p>
         * Valid characters: Letters (<code>A–Z, a–z</code>) and numbers (<code>0–9</code>)
         * </p>
         * <p>
         * Maximum: 50 values
         * </p>
         * 
         * @param marketplaceProductCodes
         *        The Amazon Web Services Marketplace product codes for allowed images.</p>
         *        <p>
         *        Length: 1-25 characters
         *        </p>
         *        <p>
         *        Valid characters: Letters (<code>A–Z, a–z</code>) and numbers (<code>0–9</code>)
         *        </p>
         *        <p>
         *        Maximum: 50 values
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder marketplaceProductCodes(String... marketplaceProductCodes);

        /**
         * <p>
         * The names of allowed images. Names can include wildcards (<code>?</code> and <code>*</code>).
         * </p>
         * <p>
         * Length: 1–128 characters. With <code>?</code>, the minimum is 3 characters.
         * </p>
         * <p>
         * Valid characters:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Letters: <code>A–Z, a–z</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Numbers: <code>0–9</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Special characters: <code>( ) [ ] . / - ' @ _ * ?</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Spaces
         * </p>
         * </li>
         * </ul>
         * <p>
         * Maximum: 50 values
         * </p>
         * 
         * @param imageNames
         *        The names of allowed images. Names can include wildcards (<code>?</code> and <code>*</code>).</p>
         *        <p>
         *        Length: 1–128 characters. With <code>?</code>, the minimum is 3 characters.
         *        </p>
         *        <p>
         *        Valid characters:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Letters: <code>A–Z, a–z</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Numbers: <code>0–9</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Special characters: <code>( ) [ ] . / - ' @ _ * ?</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Spaces
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Maximum: 50 values
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageNames(Collection<String> imageNames);

        /**
         * <p>
         * The names of allowed images. Names can include wildcards (<code>?</code> and <code>*</code>).
         * </p>
         * <p>
         * Length: 1–128 characters. With <code>?</code>, the minimum is 3 characters.
         * </p>
         * <p>
         * Valid characters:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Letters: <code>A–Z, a–z</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Numbers: <code>0–9</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Special characters: <code>( ) [ ] . / - ' @ _ * ?</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Spaces
         * </p>
         * </li>
         * </ul>
         * <p>
         * Maximum: 50 values
         * </p>
         * 
         * @param imageNames
         *        The names of allowed images. Names can include wildcards (<code>?</code> and <code>*</code>).</p>
         *        <p>
         *        Length: 1–128 characters. With <code>?</code>, the minimum is 3 characters.
         *        </p>
         *        <p>
         *        Valid characters:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Letters: <code>A–Z, a–z</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Numbers: <code>0–9</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Special characters: <code>( ) [ ] . / - ' @ _ * ?</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Spaces
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Maximum: 50 values
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageNames(String... imageNames);

        /**
         * <p>
         * The maximum period since deprecation for allowed images.
         * </p>
         * 
         * @param deprecationTimeCondition
         *        The maximum period since deprecation for allowed images.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deprecationTimeCondition(DeprecationTimeCondition deprecationTimeCondition);

        /**
         * <p>
         * The maximum period since deprecation for allowed images.
         * </p>
         * This is a convenience method that creates an instance of the {@link DeprecationTimeCondition.Builder}
         * avoiding the need to create one manually via {@link DeprecationTimeCondition#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DeprecationTimeCondition.Builder#build()} is called immediately
         * and its result is passed to {@link #deprecationTimeCondition(DeprecationTimeCondition)}.
         * 
         * @param deprecationTimeCondition
         *        a consumer that will call methods on {@link DeprecationTimeCondition.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #deprecationTimeCondition(DeprecationTimeCondition)
         */
        default Builder deprecationTimeCondition(Consumer<DeprecationTimeCondition.Builder> deprecationTimeCondition) {
            return deprecationTimeCondition(DeprecationTimeCondition.builder().applyMutation(deprecationTimeCondition).build());
        }

        /**
         * <p>
         * The maximum age for allowed images.
         * </p>
         * 
         * @param creationDateCondition
         *        The maximum age for allowed images.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationDateCondition(CreationDateCondition creationDateCondition);

        /**
         * <p>
         * The maximum age for allowed images.
         * </p>
         * This is a convenience method that creates an instance of the {@link CreationDateCondition.Builder} avoiding
         * the need to create one manually via {@link CreationDateCondition#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CreationDateCondition.Builder#build()} is called immediately and
         * its result is passed to {@link #creationDateCondition(CreationDateCondition)}.
         * 
         * @param creationDateCondition
         *        a consumer that will call methods on {@link CreationDateCondition.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #creationDateCondition(CreationDateCondition)
         */
        default Builder creationDateCondition(Consumer<CreationDateCondition.Builder> creationDateCondition) {
            return creationDateCondition(CreationDateCondition.builder().applyMutation(creationDateCondition).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private List<String> imageProviders = DefaultSdkAutoConstructList.getInstance();

        private List<String> marketplaceProductCodes = DefaultSdkAutoConstructList.getInstance();

        private List<String> imageNames = DefaultSdkAutoConstructList.getInstance();

        private DeprecationTimeCondition deprecationTimeCondition;

        private CreationDateCondition creationDateCondition;

        private BuilderImpl() {
        }

        private BuilderImpl(ImageCriterion model) {
            imageProviders(model.imageProviders);
            marketplaceProductCodes(model.marketplaceProductCodes);
            imageNames(model.imageNames);
            deprecationTimeCondition(model.deprecationTimeCondition);
            creationDateCondition(model.creationDateCondition);
        }

        public final Collection<String> getImageProviders() {
            if (imageProviders instanceof SdkAutoConstructList) {
                return null;
            }
            return imageProviders;
        }

        public final void setImageProviders(Collection<String> imageProviders) {
            this.imageProviders = ImageProviderListCopier.copy(imageProviders);
        }

        @Override
        public final Builder imageProviders(Collection<String> imageProviders) {
            this.imageProviders = ImageProviderListCopier.copy(imageProviders);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder imageProviders(String... imageProviders) {
            imageProviders(Arrays.asList(imageProviders));
            return this;
        }

        public final Collection<String> getMarketplaceProductCodes() {
            if (marketplaceProductCodes instanceof SdkAutoConstructList) {
                return null;
            }
            return marketplaceProductCodes;
        }

        public final void setMarketplaceProductCodes(Collection<String> marketplaceProductCodes) {
            this.marketplaceProductCodes = MarketplaceProductCodeListCopier.copy(marketplaceProductCodes);
        }

        @Override
        public final Builder marketplaceProductCodes(Collection<String> marketplaceProductCodes) {
            this.marketplaceProductCodes = MarketplaceProductCodeListCopier.copy(marketplaceProductCodes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder marketplaceProductCodes(String... marketplaceProductCodes) {
            marketplaceProductCodes(Arrays.asList(marketplaceProductCodes));
            return this;
        }

        public final Collection<String> getImageNames() {
            if (imageNames instanceof SdkAutoConstructList) {
                return null;
            }
            return imageNames;
        }

        public final void setImageNames(Collection<String> imageNames) {
            this.imageNames = ImageNameListCopier.copy(imageNames);
        }

        @Override
        public final Builder imageNames(Collection<String> imageNames) {
            this.imageNames = ImageNameListCopier.copy(imageNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder imageNames(String... imageNames) {
            imageNames(Arrays.asList(imageNames));
            return this;
        }

        public final DeprecationTimeCondition.Builder getDeprecationTimeCondition() {
            return deprecationTimeCondition != null ? deprecationTimeCondition.toBuilder() : null;
        }

        public final void setDeprecationTimeCondition(DeprecationTimeCondition.BuilderImpl deprecationTimeCondition) {
            this.deprecationTimeCondition = deprecationTimeCondition != null ? deprecationTimeCondition.build() : null;
        }

        @Override
        public final Builder deprecationTimeCondition(DeprecationTimeCondition deprecationTimeCondition) {
            this.deprecationTimeCondition = deprecationTimeCondition;
            return this;
        }

        public final CreationDateCondition.Builder getCreationDateCondition() {
            return creationDateCondition != null ? creationDateCondition.toBuilder() : null;
        }

        public final void setCreationDateCondition(CreationDateCondition.BuilderImpl creationDateCondition) {
            this.creationDateCondition = creationDateCondition != null ? creationDateCondition.build() : null;
        }

        @Override
        public final Builder creationDateCondition(CreationDateCondition creationDateCondition) {
            this.creationDateCondition = creationDateCondition;
            return this;
        }

        @Override
        public ImageCriterion build() {
            return new ImageCriterion(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
