/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about the Capacity Reservation topology.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CapacityReservationTopology implements SdkPojo, Serializable,
        ToCopyableBuilder<CapacityReservationTopology.Builder, CapacityReservationTopology> {
    private static final SdkField<String> CAPACITY_RESERVATION_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("CapacityReservationId")
            .getter(getter(CapacityReservationTopology::capacityReservationId))
            .setter(setter(Builder::capacityReservationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CapacityReservationId")
                    .unmarshallLocationName("capacityReservationId").build()).build();

    private static final SdkField<String> CAPACITY_BLOCK_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("CapacityBlockId")
            .getter(getter(CapacityReservationTopology::capacityBlockId))
            .setter(setter(Builder::capacityBlockId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CapacityBlockId")
                    .unmarshallLocationName("capacityBlockId").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("State")
            .getter(getter(CapacityReservationTopology::state))
            .setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State")
                    .unmarshallLocationName("state").build()).build();

    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("InstanceType")
            .getter(getter(CapacityReservationTopology::instanceType))
            .setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceType")
                    .unmarshallLocationName("instanceType").build()).build();

    private static final SdkField<String> GROUP_NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("GroupName")
            .getter(getter(CapacityReservationTopology::groupName))
            .setter(setter(Builder::groupName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GroupName")
                    .unmarshallLocationName("groupName").build()).build();

    private static final SdkField<List<String>> NETWORK_NODES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("NetworkNodes")
            .getter(getter(CapacityReservationTopology::networkNodes))
            .setter(setter(Builder::networkNodes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkNodeSet")
                    .unmarshallLocationName("networkNodeSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<String> AVAILABILITY_ZONE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AvailabilityZoneId")
            .getter(getter(CapacityReservationTopology::availabilityZoneId))
            .setter(setter(Builder::availabilityZoneId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityZoneId")
                    .unmarshallLocationName("availabilityZoneId").build()).build();

    private static final SdkField<String> AVAILABILITY_ZONE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AvailabilityZone")
            .getter(getter(CapacityReservationTopology::availabilityZone))
            .setter(setter(Builder::availabilityZone))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityZone")
                    .unmarshallLocationName("availabilityZone").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CAPACITY_RESERVATION_ID_FIELD,
            CAPACITY_BLOCK_ID_FIELD, STATE_FIELD, INSTANCE_TYPE_FIELD, GROUP_NAME_FIELD, NETWORK_NODES_FIELD,
            AVAILABILITY_ZONE_ID_FIELD, AVAILABILITY_ZONE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String capacityReservationId;

    private final String capacityBlockId;

    private final String state;

    private final String instanceType;

    private final String groupName;

    private final List<String> networkNodes;

    private final String availabilityZoneId;

    private final String availabilityZone;

    private CapacityReservationTopology(BuilderImpl builder) {
        this.capacityReservationId = builder.capacityReservationId;
        this.capacityBlockId = builder.capacityBlockId;
        this.state = builder.state;
        this.instanceType = builder.instanceType;
        this.groupName = builder.groupName;
        this.networkNodes = builder.networkNodes;
        this.availabilityZoneId = builder.availabilityZoneId;
        this.availabilityZone = builder.availabilityZone;
    }

    /**
     * <p>
     * The ID of the Capacity Reservation.
     * </p>
     * 
     * @return The ID of the Capacity Reservation.
     */
    public final String capacityReservationId() {
        return capacityReservationId;
    }

    /**
     * <p>
     * The ID of the Capacity Block. This parameter is only supported for UltraServer instances and identifies instances
     * within the UltraServer domain.
     * </p>
     * 
     * @return The ID of the Capacity Block. This parameter is only supported for UltraServer instances and identifies
     *         instances within the UltraServer domain.
     */
    public final String capacityBlockId() {
        return capacityBlockId;
    }

    /**
     * <p>
     * The current state of the Capacity Reservation. For the list of possible states, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeCapacityReservations.html"
     * >DescribeCapacityReservations</a>.
     * </p>
     * 
     * @return The current state of the Capacity Reservation. For the list of possible states, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeCapacityReservations.html"
     *         >DescribeCapacityReservations</a>.
     */
    public final String state() {
        return state;
    }

    /**
     * <p>
     * The instance type.
     * </p>
     * 
     * @return The instance type.
     */
    public final String instanceType() {
        return instanceType;
    }

    /**
     * <p>
     * The name of the placement group that the Capacity Reservation is in.
     * </p>
     * 
     * @return The name of the placement group that the Capacity Reservation is in.
     */
    public final String groupName() {
        return groupName;
    }

    /**
     * For responses, this returns true if the service returned a value for the NetworkNodes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasNetworkNodes() {
        return networkNodes != null && !(networkNodes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The network nodes. The nodes are hashed based on your account. Capacity Reservations from different accounts
     * running under the same server will return a different hashed list of strings.
     * </p>
     * <p>
     * The value is <code>null</code> or empty if:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The instance type is not supported.
     * </p>
     * </li>
     * <li>
     * <p>
     * The Capacity Reservation is in a state other than <code>active</code> or <code>pending</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNetworkNodes} method.
     * </p>
     * 
     * @return The network nodes. The nodes are hashed based on your account. Capacity Reservations from different
     *         accounts running under the same server will return a different hashed list of strings.</p>
     *         <p>
     *         The value is <code>null</code> or empty if:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The instance type is not supported.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The Capacity Reservation is in a state other than <code>active</code> or <code>pending</code>.
     *         </p>
     *         </li>
     */
    public final List<String> networkNodes() {
        return networkNodes;
    }

    /**
     * <p>
     * The ID of the Availability Zone or Local Zone that the Capacity Reservation is in.
     * </p>
     * 
     * @return The ID of the Availability Zone or Local Zone that the Capacity Reservation is in.
     */
    public final String availabilityZoneId() {
        return availabilityZoneId;
    }

    /**
     * <p>
     * The name of the Availability Zone or Local Zone that the Capacity Reservation is in.
     * </p>
     * 
     * @return The name of the Availability Zone or Local Zone that the Capacity Reservation is in.
     */
    public final String availabilityZone() {
        return availabilityZone;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(capacityReservationId());
        hashCode = 31 * hashCode + Objects.hashCode(capacityBlockId());
        hashCode = 31 * hashCode + Objects.hashCode(state());
        hashCode = 31 * hashCode + Objects.hashCode(instanceType());
        hashCode = 31 * hashCode + Objects.hashCode(groupName());
        hashCode = 31 * hashCode + Objects.hashCode(hasNetworkNodes() ? networkNodes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(availabilityZoneId());
        hashCode = 31 * hashCode + Objects.hashCode(availabilityZone());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CapacityReservationTopology)) {
            return false;
        }
        CapacityReservationTopology other = (CapacityReservationTopology) obj;
        return Objects.equals(capacityReservationId(), other.capacityReservationId())
                && Objects.equals(capacityBlockId(), other.capacityBlockId()) && Objects.equals(state(), other.state())
                && Objects.equals(instanceType(), other.instanceType()) && Objects.equals(groupName(), other.groupName())
                && hasNetworkNodes() == other.hasNetworkNodes() && Objects.equals(networkNodes(), other.networkNodes())
                && Objects.equals(availabilityZoneId(), other.availabilityZoneId())
                && Objects.equals(availabilityZone(), other.availabilityZone());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CapacityReservationTopology").add("CapacityReservationId", capacityReservationId())
                .add("CapacityBlockId", capacityBlockId()).add("State", state()).add("InstanceType", instanceType())
                .add("GroupName", groupName()).add("NetworkNodes", hasNetworkNodes() ? networkNodes() : null)
                .add("AvailabilityZoneId", availabilityZoneId()).add("AvailabilityZone", availabilityZone()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CapacityReservationId":
            return Optional.ofNullable(clazz.cast(capacityReservationId()));
        case "CapacityBlockId":
            return Optional.ofNullable(clazz.cast(capacityBlockId()));
        case "State":
            return Optional.ofNullable(clazz.cast(state()));
        case "InstanceType":
            return Optional.ofNullable(clazz.cast(instanceType()));
        case "GroupName":
            return Optional.ofNullable(clazz.cast(groupName()));
        case "NetworkNodes":
            return Optional.ofNullable(clazz.cast(networkNodes()));
        case "AvailabilityZoneId":
            return Optional.ofNullable(clazz.cast(availabilityZoneId()));
        case "AvailabilityZone":
            return Optional.ofNullable(clazz.cast(availabilityZone()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("CapacityReservationId", CAPACITY_RESERVATION_ID_FIELD);
        map.put("CapacityBlockId", CAPACITY_BLOCK_ID_FIELD);
        map.put("State", STATE_FIELD);
        map.put("InstanceType", INSTANCE_TYPE_FIELD);
        map.put("GroupName", GROUP_NAME_FIELD);
        map.put("NetworkNodeSet", NETWORK_NODES_FIELD);
        map.put("AvailabilityZoneId", AVAILABILITY_ZONE_ID_FIELD);
        map.put("AvailabilityZone", AVAILABILITY_ZONE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CapacityReservationTopology, T> g) {
        return obj -> g.apply((CapacityReservationTopology) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CapacityReservationTopology> {
        /**
         * <p>
         * The ID of the Capacity Reservation.
         * </p>
         * 
         * @param capacityReservationId
         *        The ID of the Capacity Reservation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capacityReservationId(String capacityReservationId);

        /**
         * <p>
         * The ID of the Capacity Block. This parameter is only supported for UltraServer instances and identifies
         * instances within the UltraServer domain.
         * </p>
         * 
         * @param capacityBlockId
         *        The ID of the Capacity Block. This parameter is only supported for UltraServer instances and
         *        identifies instances within the UltraServer domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capacityBlockId(String capacityBlockId);

        /**
         * <p>
         * The current state of the Capacity Reservation. For the list of possible states, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeCapacityReservations.html"
         * >DescribeCapacityReservations</a>.
         * </p>
         * 
         * @param state
         *        The current state of the Capacity Reservation. For the list of possible states, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeCapacityReservations.html"
         *        >DescribeCapacityReservations</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder state(String state);

        /**
         * <p>
         * The instance type.
         * </p>
         * 
         * @param instanceType
         *        The instance type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * The name of the placement group that the Capacity Reservation is in.
         * </p>
         * 
         * @param groupName
         *        The name of the placement group that the Capacity Reservation is in.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groupName(String groupName);

        /**
         * <p>
         * The network nodes. The nodes are hashed based on your account. Capacity Reservations from different accounts
         * running under the same server will return a different hashed list of strings.
         * </p>
         * <p>
         * The value is <code>null</code> or empty if:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The instance type is not supported.
         * </p>
         * </li>
         * <li>
         * <p>
         * The Capacity Reservation is in a state other than <code>active</code> or <code>pending</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param networkNodes
         *        The network nodes. The nodes are hashed based on your account. Capacity Reservations from different
         *        accounts running under the same server will return a different hashed list of strings.</p>
         *        <p>
         *        The value is <code>null</code> or empty if:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The instance type is not supported.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The Capacity Reservation is in a state other than <code>active</code> or <code>pending</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkNodes(Collection<String> networkNodes);

        /**
         * <p>
         * The network nodes. The nodes are hashed based on your account. Capacity Reservations from different accounts
         * running under the same server will return a different hashed list of strings.
         * </p>
         * <p>
         * The value is <code>null</code> or empty if:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The instance type is not supported.
         * </p>
         * </li>
         * <li>
         * <p>
         * The Capacity Reservation is in a state other than <code>active</code> or <code>pending</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param networkNodes
         *        The network nodes. The nodes are hashed based on your account. Capacity Reservations from different
         *        accounts running under the same server will return a different hashed list of strings.</p>
         *        <p>
         *        The value is <code>null</code> or empty if:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The instance type is not supported.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The Capacity Reservation is in a state other than <code>active</code> or <code>pending</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkNodes(String... networkNodes);

        /**
         * <p>
         * The ID of the Availability Zone or Local Zone that the Capacity Reservation is in.
         * </p>
         * 
         * @param availabilityZoneId
         *        The ID of the Availability Zone or Local Zone that the Capacity Reservation is in.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZoneId(String availabilityZoneId);

        /**
         * <p>
         * The name of the Availability Zone or Local Zone that the Capacity Reservation is in.
         * </p>
         * 
         * @param availabilityZone
         *        The name of the Availability Zone or Local Zone that the Capacity Reservation is in.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZone(String availabilityZone);
    }

    static final class BuilderImpl implements Builder {
        private String capacityReservationId;

        private String capacityBlockId;

        private String state;

        private String instanceType;

        private String groupName;

        private List<String> networkNodes = DefaultSdkAutoConstructList.getInstance();

        private String availabilityZoneId;

        private String availabilityZone;

        private BuilderImpl() {
        }

        private BuilderImpl(CapacityReservationTopology model) {
            capacityReservationId(model.capacityReservationId);
            capacityBlockId(model.capacityBlockId);
            state(model.state);
            instanceType(model.instanceType);
            groupName(model.groupName);
            networkNodes(model.networkNodes);
            availabilityZoneId(model.availabilityZoneId);
            availabilityZone(model.availabilityZone);
        }

        public final String getCapacityReservationId() {
            return capacityReservationId;
        }

        public final void setCapacityReservationId(String capacityReservationId) {
            this.capacityReservationId = capacityReservationId;
        }

        @Override
        public final Builder capacityReservationId(String capacityReservationId) {
            this.capacityReservationId = capacityReservationId;
            return this;
        }

        public final String getCapacityBlockId() {
            return capacityBlockId;
        }

        public final void setCapacityBlockId(String capacityBlockId) {
            this.capacityBlockId = capacityBlockId;
        }

        @Override
        public final Builder capacityBlockId(String capacityBlockId) {
            this.capacityBlockId = capacityBlockId;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        public final String getInstanceType() {
            return instanceType;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        public final String getGroupName() {
            return groupName;
        }

        public final void setGroupName(String groupName) {
            this.groupName = groupName;
        }

        @Override
        public final Builder groupName(String groupName) {
            this.groupName = groupName;
            return this;
        }

        public final Collection<String> getNetworkNodes() {
            if (networkNodes instanceof SdkAutoConstructList) {
                return null;
            }
            return networkNodes;
        }

        public final void setNetworkNodes(Collection<String> networkNodes) {
            this.networkNodes = NetworkNodeSetCopier.copy(networkNodes);
        }

        @Override
        public final Builder networkNodes(Collection<String> networkNodes) {
            this.networkNodes = NetworkNodeSetCopier.copy(networkNodes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder networkNodes(String... networkNodes) {
            networkNodes(Arrays.asList(networkNodes));
            return this;
        }

        public final String getAvailabilityZoneId() {
            return availabilityZoneId;
        }

        public final void setAvailabilityZoneId(String availabilityZoneId) {
            this.availabilityZoneId = availabilityZoneId;
        }

        @Override
        public final Builder availabilityZoneId(String availabilityZoneId) {
            this.availabilityZoneId = availabilityZoneId;
            return this;
        }

        public final String getAvailabilityZone() {
            return availabilityZone;
        }

        public final void setAvailabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
        }

        @Override
        public final Builder availabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
            return this;
        }

        @Override
        public CapacityReservationTopology build() {
            return new CapacityReservationTopology(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
