/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a CIDR selection rule.
 * </p>
 * <p>
 * CIDR selection rules define the business logic for selecting CIDRs from IPAM. If a CIDR matches any of the rules, it
 * will be included. If a rule has multiple conditions, the CIDR has to match every condition of that rule. You can
 * create a prefix list resolver without any CIDR selection rules, but it will generate empty versions (containing no
 * CIDRs) until you add rules.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class IpamPrefixListResolverRule implements SdkPojo, Serializable,
        ToCopyableBuilder<IpamPrefixListResolverRule.Builder, IpamPrefixListResolverRule> {
    private static final SdkField<String> RULE_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("RuleType")
            .getter(getter(IpamPrefixListResolverRule::ruleTypeAsString))
            .setter(setter(Builder::ruleType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleType")
                    .unmarshallLocationName("ruleType").build()).build();

    private static final SdkField<String> STATIC_CIDR_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("StaticCidr")
            .getter(getter(IpamPrefixListResolverRule::staticCidr))
            .setter(setter(Builder::staticCidr))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StaticCidr")
                    .unmarshallLocationName("staticCidr").build()).build();

    private static final SdkField<String> IPAM_SCOPE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamScopeId")
            .getter(getter(IpamPrefixListResolverRule::ipamScopeId))
            .setter(setter(Builder::ipamScopeId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamScopeId")
                    .unmarshallLocationName("ipamScopeId").build()).build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ResourceType")
            .getter(getter(IpamPrefixListResolverRule::resourceTypeAsString))
            .setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceType")
                    .unmarshallLocationName("resourceType").build()).build();

    private static final SdkField<List<IpamPrefixListResolverRuleCondition>> CONDITIONS_FIELD = SdkField
            .<List<IpamPrefixListResolverRuleCondition>> builder(MarshallingType.LIST)
            .memberName("Conditions")
            .getter(getter(IpamPrefixListResolverRule::conditions))
            .setter(setter(Builder::conditions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConditionSet")
                    .unmarshallLocationName("conditionSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<IpamPrefixListResolverRuleCondition> builder(MarshallingType.SDK_POJO)
                                            .constructor(IpamPrefixListResolverRuleCondition::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RULE_TYPE_FIELD,
            STATIC_CIDR_FIELD, IPAM_SCOPE_ID_FIELD, RESOURCE_TYPE_FIELD, CONDITIONS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String ruleType;

    private final String staticCidr;

    private final String ipamScopeId;

    private final String resourceType;

    private final List<IpamPrefixListResolverRuleCondition> conditions;

    private IpamPrefixListResolverRule(BuilderImpl builder) {
        this.ruleType = builder.ruleType;
        this.staticCidr = builder.staticCidr;
        this.ipamScopeId = builder.ipamScopeId;
        this.resourceType = builder.resourceType;
        this.conditions = builder.conditions;
    }

    /**
     * <p>
     * The type of CIDR selection rule. Valid values include <code>include</code> for selecting CIDRs that match the
     * conditions, and <code>exclude</code> for excluding CIDRs that match the conditions.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ruleType} will
     * return {@link IpamPrefixListResolverRuleType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #ruleTypeAsString}.
     * </p>
     * 
     * @return The type of CIDR selection rule. Valid values include <code>include</code> for selecting CIDRs that match
     *         the conditions, and <code>exclude</code> for excluding CIDRs that match the conditions.
     * @see IpamPrefixListResolverRuleType
     */
    public final IpamPrefixListResolverRuleType ruleType() {
        return IpamPrefixListResolverRuleType.fromValue(ruleType);
    }

    /**
     * <p>
     * The type of CIDR selection rule. Valid values include <code>include</code> for selecting CIDRs that match the
     * conditions, and <code>exclude</code> for excluding CIDRs that match the conditions.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ruleType} will
     * return {@link IpamPrefixListResolverRuleType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #ruleTypeAsString}.
     * </p>
     * 
     * @return The type of CIDR selection rule. Valid values include <code>include</code> for selecting CIDRs that match
     *         the conditions, and <code>exclude</code> for excluding CIDRs that match the conditions.
     * @see IpamPrefixListResolverRuleType
     */
    public final String ruleTypeAsString() {
        return ruleType;
    }

    /**
     * <p>
     * A fixed list of CIDRs that do not change (like a manual list replicated across Regions).
     * </p>
     * 
     * @return A fixed list of CIDRs that do not change (like a manual list replicated across Regions).
     */
    public final String staticCidr() {
        return staticCidr;
    }

    /**
     * <p>
     * The ID of the IPAM scope from which to select CIDRs. This determines whether to select from public or private IP
     * address space.
     * </p>
     * 
     * @return The ID of the IPAM scope from which to select CIDRs. This determines whether to select from public or
     *         private IP address space.
     */
    public final String ipamScopeId() {
        return ipamScopeId;
    }

    /**
     * <p>
     * For rules of type <code>ipam-resource-cidr</code>, this is the resource type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link IpamResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return For rules of type <code>ipam-resource-cidr</code>, this is the resource type.
     * @see IpamResourceType
     */
    public final IpamResourceType resourceType() {
        return IpamResourceType.fromValue(resourceType);
    }

    /**
     * <p>
     * For rules of type <code>ipam-resource-cidr</code>, this is the resource type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link IpamResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return For rules of type <code>ipam-resource-cidr</code>, this is the resource type.
     * @see IpamResourceType
     */
    public final String resourceTypeAsString() {
        return resourceType;
    }

    /**
     * For responses, this returns true if the service returned a value for the Conditions property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasConditions() {
        return conditions != null && !(conditions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The conditions that determine which CIDRs are selected by this rule. Conditions specify criteria such as resource
     * type, tags, account IDs, and Regions.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasConditions} method.
     * </p>
     * 
     * @return The conditions that determine which CIDRs are selected by this rule. Conditions specify criteria such as
     *         resource type, tags, account IDs, and Regions.
     */
    public final List<IpamPrefixListResolverRuleCondition> conditions() {
        return conditions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(ruleTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(staticCidr());
        hashCode = 31 * hashCode + Objects.hashCode(ipamScopeId());
        hashCode = 31 * hashCode + Objects.hashCode(resourceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasConditions() ? conditions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof IpamPrefixListResolverRule)) {
            return false;
        }
        IpamPrefixListResolverRule other = (IpamPrefixListResolverRule) obj;
        return Objects.equals(ruleTypeAsString(), other.ruleTypeAsString()) && Objects.equals(staticCidr(), other.staticCidr())
                && Objects.equals(ipamScopeId(), other.ipamScopeId())
                && Objects.equals(resourceTypeAsString(), other.resourceTypeAsString())
                && hasConditions() == other.hasConditions() && Objects.equals(conditions(), other.conditions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("IpamPrefixListResolverRule").add("RuleType", ruleTypeAsString()).add("StaticCidr", staticCidr())
                .add("IpamScopeId", ipamScopeId()).add("ResourceType", resourceTypeAsString())
                .add("Conditions", hasConditions() ? conditions() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RuleType":
            return Optional.ofNullable(clazz.cast(ruleTypeAsString()));
        case "StaticCidr":
            return Optional.ofNullable(clazz.cast(staticCidr()));
        case "IpamScopeId":
            return Optional.ofNullable(clazz.cast(ipamScopeId()));
        case "ResourceType":
            return Optional.ofNullable(clazz.cast(resourceTypeAsString()));
        case "Conditions":
            return Optional.ofNullable(clazz.cast(conditions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("RuleType", RULE_TYPE_FIELD);
        map.put("StaticCidr", STATIC_CIDR_FIELD);
        map.put("IpamScopeId", IPAM_SCOPE_ID_FIELD);
        map.put("ResourceType", RESOURCE_TYPE_FIELD);
        map.put("ConditionSet", CONDITIONS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<IpamPrefixListResolverRule, T> g) {
        return obj -> g.apply((IpamPrefixListResolverRule) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, IpamPrefixListResolverRule> {
        /**
         * <p>
         * The type of CIDR selection rule. Valid values include <code>include</code> for selecting CIDRs that match the
         * conditions, and <code>exclude</code> for excluding CIDRs that match the conditions.
         * </p>
         * 
         * @param ruleType
         *        The type of CIDR selection rule. Valid values include <code>include</code> for selecting CIDRs that
         *        match the conditions, and <code>exclude</code> for excluding CIDRs that match the conditions.
         * @see IpamPrefixListResolverRuleType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamPrefixListResolverRuleType
         */
        Builder ruleType(String ruleType);

        /**
         * <p>
         * The type of CIDR selection rule. Valid values include <code>include</code> for selecting CIDRs that match the
         * conditions, and <code>exclude</code> for excluding CIDRs that match the conditions.
         * </p>
         * 
         * @param ruleType
         *        The type of CIDR selection rule. Valid values include <code>include</code> for selecting CIDRs that
         *        match the conditions, and <code>exclude</code> for excluding CIDRs that match the conditions.
         * @see IpamPrefixListResolverRuleType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamPrefixListResolverRuleType
         */
        Builder ruleType(IpamPrefixListResolverRuleType ruleType);

        /**
         * <p>
         * A fixed list of CIDRs that do not change (like a manual list replicated across Regions).
         * </p>
         * 
         * @param staticCidr
         *        A fixed list of CIDRs that do not change (like a manual list replicated across Regions).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder staticCidr(String staticCidr);

        /**
         * <p>
         * The ID of the IPAM scope from which to select CIDRs. This determines whether to select from public or private
         * IP address space.
         * </p>
         * 
         * @param ipamScopeId
         *        The ID of the IPAM scope from which to select CIDRs. This determines whether to select from public or
         *        private IP address space.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamScopeId(String ipamScopeId);

        /**
         * <p>
         * For rules of type <code>ipam-resource-cidr</code>, this is the resource type.
         * </p>
         * 
         * @param resourceType
         *        For rules of type <code>ipam-resource-cidr</code>, this is the resource type.
         * @see IpamResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamResourceType
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * For rules of type <code>ipam-resource-cidr</code>, this is the resource type.
         * </p>
         * 
         * @param resourceType
         *        For rules of type <code>ipam-resource-cidr</code>, this is the resource type.
         * @see IpamResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamResourceType
         */
        Builder resourceType(IpamResourceType resourceType);

        /**
         * <p>
         * The conditions that determine which CIDRs are selected by this rule. Conditions specify criteria such as
         * resource type, tags, account IDs, and Regions.
         * </p>
         * 
         * @param conditions
         *        The conditions that determine which CIDRs are selected by this rule. Conditions specify criteria such
         *        as resource type, tags, account IDs, and Regions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder conditions(Collection<IpamPrefixListResolverRuleCondition> conditions);

        /**
         * <p>
         * The conditions that determine which CIDRs are selected by this rule. Conditions specify criteria such as
         * resource type, tags, account IDs, and Regions.
         * </p>
         * 
         * @param conditions
         *        The conditions that determine which CIDRs are selected by this rule. Conditions specify criteria such
         *        as resource type, tags, account IDs, and Regions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder conditions(IpamPrefixListResolverRuleCondition... conditions);

        /**
         * <p>
         * The conditions that determine which CIDRs are selected by this rule. Conditions specify criteria such as
         * resource type, tags, account IDs, and Regions.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.IpamPrefixListResolverRuleCondition.Builder} avoiding the
         * need to create one manually via
         * {@link software.amazon.awssdk.services.ec2.model.IpamPrefixListResolverRuleCondition#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ec2.model.IpamPrefixListResolverRuleCondition.Builder#build()} is
         * called immediately and its result is passed to {@link #conditions(List<IpamPrefixListResolverRuleCondition>)}.
         * 
         * @param conditions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ec2.model.IpamPrefixListResolverRuleCondition.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #conditions(java.util.Collection<IpamPrefixListResolverRuleCondition>)
         */
        Builder conditions(Consumer<IpamPrefixListResolverRuleCondition.Builder>... conditions);
    }

    static final class BuilderImpl implements Builder {
        private String ruleType;

        private String staticCidr;

        private String ipamScopeId;

        private String resourceType;

        private List<IpamPrefixListResolverRuleCondition> conditions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(IpamPrefixListResolverRule model) {
            ruleType(model.ruleType);
            staticCidr(model.staticCidr);
            ipamScopeId(model.ipamScopeId);
            resourceType(model.resourceType);
            conditions(model.conditions);
        }

        public final String getRuleType() {
            return ruleType;
        }

        public final void setRuleType(String ruleType) {
            this.ruleType = ruleType;
        }

        @Override
        public final Builder ruleType(String ruleType) {
            this.ruleType = ruleType;
            return this;
        }

        @Override
        public final Builder ruleType(IpamPrefixListResolverRuleType ruleType) {
            this.ruleType(ruleType == null ? null : ruleType.toString());
            return this;
        }

        public final String getStaticCidr() {
            return staticCidr;
        }

        public final void setStaticCidr(String staticCidr) {
            this.staticCidr = staticCidr;
        }

        @Override
        public final Builder staticCidr(String staticCidr) {
            this.staticCidr = staticCidr;
            return this;
        }

        public final String getIpamScopeId() {
            return ipamScopeId;
        }

        public final void setIpamScopeId(String ipamScopeId) {
            this.ipamScopeId = ipamScopeId;
        }

        @Override
        public final Builder ipamScopeId(String ipamScopeId) {
            this.ipamScopeId = ipamScopeId;
            return this;
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        @Override
        public final Builder resourceType(IpamResourceType resourceType) {
            this.resourceType(resourceType == null ? null : resourceType.toString());
            return this;
        }

        public final List<IpamPrefixListResolverRuleCondition.Builder> getConditions() {
            List<IpamPrefixListResolverRuleCondition.Builder> result = IpamPrefixListResolverRuleConditionSetCopier
                    .copyToBuilder(this.conditions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setConditions(Collection<IpamPrefixListResolverRuleCondition.BuilderImpl> conditions) {
            this.conditions = IpamPrefixListResolverRuleConditionSetCopier.copyFromBuilder(conditions);
        }

        @Override
        public final Builder conditions(Collection<IpamPrefixListResolverRuleCondition> conditions) {
            this.conditions = IpamPrefixListResolverRuleConditionSetCopier.copy(conditions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder conditions(IpamPrefixListResolverRuleCondition... conditions) {
            conditions(Arrays.asList(conditions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder conditions(Consumer<IpamPrefixListResolverRuleCondition.Builder>... conditions) {
            conditions(Stream.of(conditions).map(c -> IpamPrefixListResolverRuleCondition.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public IpamPrefixListResolverRule build() {
            return new IpamPrefixListResolverRule(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
