/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes overrides for a launch template.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class FleetLaunchTemplateOverridesRequest implements SdkPojo, Serializable,
        ToCopyableBuilder<FleetLaunchTemplateOverridesRequest.Builder, FleetLaunchTemplateOverridesRequest> {
    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("InstanceType")
            .getter(getter(FleetLaunchTemplateOverridesRequest::instanceTypeAsString))
            .setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceType")
                    .unmarshallLocationName("InstanceType").build()).build();

    private static final SdkField<String> MAX_PRICE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("MaxPrice")
            .getter(getter(FleetLaunchTemplateOverridesRequest::maxPrice))
            .setter(setter(Builder::maxPrice))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxPrice")
                    .unmarshallLocationName("MaxPrice").build()).build();

    private static final SdkField<String> SUBNET_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SubnetId")
            .getter(getter(FleetLaunchTemplateOverridesRequest::subnetId))
            .setter(setter(Builder::subnetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetId")
                    .unmarshallLocationName("SubnetId").build()).build();

    private static final SdkField<String> AVAILABILITY_ZONE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AvailabilityZone")
            .getter(getter(FleetLaunchTemplateOverridesRequest::availabilityZone))
            .setter(setter(Builder::availabilityZone))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityZone")
                    .unmarshallLocationName("AvailabilityZone").build()).build();

    private static final SdkField<Double> WEIGHTED_CAPACITY_FIELD = SdkField
            .<Double> builder(MarshallingType.DOUBLE)
            .memberName("WeightedCapacity")
            .getter(getter(FleetLaunchTemplateOverridesRequest::weightedCapacity))
            .setter(setter(Builder::weightedCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WeightedCapacity")
                    .unmarshallLocationName("WeightedCapacity").build()).build();

    private static final SdkField<Double> PRIORITY_FIELD = SdkField
            .<Double> builder(MarshallingType.DOUBLE)
            .memberName("Priority")
            .getter(getter(FleetLaunchTemplateOverridesRequest::priority))
            .setter(setter(Builder::priority))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Priority")
                    .unmarshallLocationName("Priority").build()).build();

    private static final SdkField<Placement> PLACEMENT_FIELD = SdkField
            .<Placement> builder(MarshallingType.SDK_POJO)
            .memberName("Placement")
            .getter(getter(FleetLaunchTemplateOverridesRequest::placement))
            .setter(setter(Builder::placement))
            .constructor(Placement::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Placement")
                    .unmarshallLocationName("Placement").build()).build();

    private static final SdkField<List<FleetBlockDeviceMappingRequest>> BLOCK_DEVICE_MAPPINGS_FIELD = SdkField
            .<List<FleetBlockDeviceMappingRequest>> builder(MarshallingType.LIST)
            .memberName("BlockDeviceMappings")
            .getter(getter(FleetLaunchTemplateOverridesRequest::blockDeviceMappings))
            .setter(setter(Builder::blockDeviceMappings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BlockDeviceMapping")
                    .unmarshallLocationName("BlockDeviceMapping").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("BlockDeviceMapping")
                            .memberFieldInfo(
                                    SdkField.<FleetBlockDeviceMappingRequest> builder(MarshallingType.SDK_POJO)
                                            .constructor(FleetBlockDeviceMappingRequest::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("BlockDeviceMapping")
                                                    .unmarshallLocationName("BlockDeviceMapping").build()).build()).build())
            .build();

    private static final SdkField<InstanceRequirementsRequest> INSTANCE_REQUIREMENTS_FIELD = SdkField
            .<InstanceRequirementsRequest> builder(MarshallingType.SDK_POJO)
            .memberName("InstanceRequirements")
            .getter(getter(FleetLaunchTemplateOverridesRequest::instanceRequirements))
            .setter(setter(Builder::instanceRequirements))
            .constructor(InstanceRequirementsRequest::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceRequirements")
                    .unmarshallLocationName("InstanceRequirements").build()).build();

    private static final SdkField<String> IMAGE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ImageId")
            .getter(getter(FleetLaunchTemplateOverridesRequest::imageId))
            .setter(setter(Builder::imageId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImageId")
                    .unmarshallLocationName("ImageId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INSTANCE_TYPE_FIELD,
            MAX_PRICE_FIELD, SUBNET_ID_FIELD, AVAILABILITY_ZONE_FIELD, WEIGHTED_CAPACITY_FIELD, PRIORITY_FIELD, PLACEMENT_FIELD,
            BLOCK_DEVICE_MAPPINGS_FIELD, INSTANCE_REQUIREMENTS_FIELD, IMAGE_ID_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String instanceType;

    private final String maxPrice;

    private final String subnetId;

    private final String availabilityZone;

    private final Double weightedCapacity;

    private final Double priority;

    private final Placement placement;

    private final List<FleetBlockDeviceMappingRequest> blockDeviceMappings;

    private final InstanceRequirementsRequest instanceRequirements;

    private final String imageId;

    private FleetLaunchTemplateOverridesRequest(BuilderImpl builder) {
        this.instanceType = builder.instanceType;
        this.maxPrice = builder.maxPrice;
        this.subnetId = builder.subnetId;
        this.availabilityZone = builder.availabilityZone;
        this.weightedCapacity = builder.weightedCapacity;
        this.priority = builder.priority;
        this.placement = builder.placement;
        this.blockDeviceMappings = builder.blockDeviceMappings;
        this.instanceRequirements = builder.instanceRequirements;
        this.imageId = builder.imageId;
    }

    /**
     * <p>
     * The instance type.
     * </p>
     * <p>
     * <code>mac1.metal</code> is not supported as a launch template override.
     * </p>
     * <note>
     * <p>
     * If you specify <code>InstanceType</code>, you can't specify <code>InstanceRequirements</code>.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceType} will
     * return {@link InstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #instanceTypeAsString}.
     * </p>
     * 
     * @return The instance type.</p>
     *         <p>
     *         <code>mac1.metal</code> is not supported as a launch template override.
     *         </p>
     *         <note>
     *         <p>
     *         If you specify <code>InstanceType</code>, you can't specify <code>InstanceRequirements</code>.
     *         </p>
     * @see InstanceType
     */
    public final InstanceType instanceType() {
        return InstanceType.fromValue(instanceType);
    }

    /**
     * <p>
     * The instance type.
     * </p>
     * <p>
     * <code>mac1.metal</code> is not supported as a launch template override.
     * </p>
     * <note>
     * <p>
     * If you specify <code>InstanceType</code>, you can't specify <code>InstanceRequirements</code>.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceType} will
     * return {@link InstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #instanceTypeAsString}.
     * </p>
     * 
     * @return The instance type.</p>
     *         <p>
     *         <code>mac1.metal</code> is not supported as a launch template override.
     *         </p>
     *         <note>
     *         <p>
     *         If you specify <code>InstanceType</code>, you can't specify <code>InstanceRequirements</code>.
     *         </p>
     * @see InstanceType
     */
    public final String instanceTypeAsString() {
        return instanceType;
    }

    /**
     * <p>
     * The maximum price per unit hour that you are willing to pay for a Spot Instance. We do not recommend using this
     * parameter because it can lead to increased interruptions. If you do not specify this parameter, you will pay the
     * current Spot price.
     * </p>
     * <important>
     * <p>
     * If you specify a maximum price, your instances will be interrupted more frequently than if you do not specify
     * this parameter.
     * </p>
     * <p>
     * If you specify a maximum price, it must be more than USD $0.001. Specifying a value below USD $0.001 will result
     * in an <code>InvalidParameterValue</code> error message.
     * </p>
     * </important>
     * 
     * @return The maximum price per unit hour that you are willing to pay for a Spot Instance. We do not recommend
     *         using this parameter because it can lead to increased interruptions. If you do not specify this
     *         parameter, you will pay the current Spot price. </p> <important>
     *         <p>
     *         If you specify a maximum price, your instances will be interrupted more frequently than if you do not
     *         specify this parameter.
     *         </p>
     *         <p>
     *         If you specify a maximum price, it must be more than USD $0.001. Specifying a value below USD $0.001 will
     *         result in an <code>InvalidParameterValue</code> error message.
     *         </p>
     */
    public final String maxPrice() {
        return maxPrice;
    }

    /**
     * <p>
     * The IDs of the subnets in which to launch the instances. Separate multiple subnet IDs using commas (for example,
     * <code>subnet-1234abcdeexample1, subnet-0987cdef6example2</code>). A request of type <code>instant</code> can have
     * only one subnet ID.
     * </p>
     * 
     * @return The IDs of the subnets in which to launch the instances. Separate multiple subnet IDs using commas (for
     *         example, <code>subnet-1234abcdeexample1, subnet-0987cdef6example2</code>). A request of type
     *         <code>instant</code> can have only one subnet ID.
     */
    public final String subnetId() {
        return subnetId;
    }

    /**
     * <p>
     * The Availability Zone in which to launch the instances.
     * </p>
     * 
     * @return The Availability Zone in which to launch the instances.
     */
    public final String availabilityZone() {
        return availabilityZone;
    }

    /**
     * <p>
     * The number of units provided by the specified instance type. These are the same units that you chose to set the
     * target capacity in terms of instances, or a performance characteristic such as vCPUs, memory, or I/O.
     * </p>
     * <p>
     * If the target capacity divided by this value is not a whole number, Amazon EC2 rounds the number of instances to
     * the next whole number. If this value is not specified, the default is 1.
     * </p>
     * <note>
     * <p>
     * When specifying weights, the price used in the <code>lowest-price</code> and
     * <code>price-capacity-optimized</code> allocation strategies is per <i>unit</i> hour (where the instance price is
     * divided by the specified weight). However, if all the specified weights are above the requested
     * <code>TargetCapacity</code>, resulting in only 1 instance being launched, the price used is per <i>instance</i>
     * hour.
     * </p>
     * </note>
     * 
     * @return The number of units provided by the specified instance type. These are the same units that you chose to
     *         set the target capacity in terms of instances, or a performance characteristic such as vCPUs, memory, or
     *         I/O.</p>
     *         <p>
     *         If the target capacity divided by this value is not a whole number, Amazon EC2 rounds the number of
     *         instances to the next whole number. If this value is not specified, the default is 1.
     *         </p>
     *         <note>
     *         <p>
     *         When specifying weights, the price used in the <code>lowest-price</code> and
     *         <code>price-capacity-optimized</code> allocation strategies is per <i>unit</i> hour (where the instance
     *         price is divided by the specified weight). However, if all the specified weights are above the requested
     *         <code>TargetCapacity</code>, resulting in only 1 instance being launched, the price used is per
     *         <i>instance</i> hour.
     *         </p>
     */
    public final Double weightedCapacity() {
        return weightedCapacity;
    }

    /**
     * <p>
     * The priority for the launch template override. The highest priority is launched first.
     * </p>
     * <p>
     * If the On-Demand <code>AllocationStrategy</code> is set to <code>prioritized</code>, EC2 Fleet uses priority to
     * determine which launch template override to use first in fulfilling On-Demand capacity.
     * </p>
     * <p>
     * If the Spot <code>AllocationStrategy</code> is set to <code>capacity-optimized-prioritized</code>, EC2 Fleet uses
     * priority on a best-effort basis to determine which launch template override to use in fulfilling Spot capacity,
     * but optimizes for capacity first.
     * </p>
     * <p>
     * Valid values are whole numbers starting at <code>0</code>. The lower the number, the higher the priority. If no
     * number is set, the launch template override has the lowest priority. You can set the same priority for different
     * launch template overrides.
     * </p>
     * 
     * @return The priority for the launch template override. The highest priority is launched first.</p>
     *         <p>
     *         If the On-Demand <code>AllocationStrategy</code> is set to <code>prioritized</code>, EC2 Fleet uses
     *         priority to determine which launch template override to use first in fulfilling On-Demand capacity.
     *         </p>
     *         <p>
     *         If the Spot <code>AllocationStrategy</code> is set to <code>capacity-optimized-prioritized</code>, EC2
     *         Fleet uses priority on a best-effort basis to determine which launch template override to use in
     *         fulfilling Spot capacity, but optimizes for capacity first.
     *         </p>
     *         <p>
     *         Valid values are whole numbers starting at <code>0</code>. The lower the number, the higher the priority.
     *         If no number is set, the launch template override has the lowest priority. You can set the same priority
     *         for different launch template overrides.
     */
    public final Double priority() {
        return priority;
    }

    /**
     * <p>
     * The location where the instance launched, if applicable.
     * </p>
     * 
     * @return The location where the instance launched, if applicable.
     */
    public final Placement placement() {
        return placement;
    }

    /**
     * For responses, this returns true if the service returned a value for the BlockDeviceMappings property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasBlockDeviceMappings() {
        return blockDeviceMappings != null && !(blockDeviceMappings instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The block device mappings, which define the EBS volumes and instance store volumes to attach to the instance at
     * launch.
     * </p>
     * <p>
     * Supported only for fleets of type <code>instant</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/block-device-mapping-concepts.html">Block device
     * mappings for volumes on Amazon EC2 instances</a> in the <i>Amazon EC2 User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasBlockDeviceMappings} method.
     * </p>
     * 
     * @return The block device mappings, which define the EBS volumes and instance store volumes to attach to the
     *         instance at launch.</p>
     *         <p>
     *         Supported only for fleets of type <code>instant</code>.
     *         </p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/block-device-mapping-concepts.html">Block
     *         device mappings for volumes on Amazon EC2 instances</a> in the <i>Amazon EC2 User Guide</i>.
     */
    public final List<FleetBlockDeviceMappingRequest> blockDeviceMappings() {
        return blockDeviceMappings;
    }

    /**
     * <p>
     * The attributes for the instance types. When you specify instance attributes, Amazon EC2 will identify instance
     * types with those attributes.
     * </p>
     * <note>
     * <p>
     * If you specify <code>InstanceRequirements</code>, you can't specify <code>InstanceType</code>.
     * </p>
     * </note>
     * 
     * @return The attributes for the instance types. When you specify instance attributes, Amazon EC2 will identify
     *         instance types with those attributes.</p> <note>
     *         <p>
     *         If you specify <code>InstanceRequirements</code>, you can't specify <code>InstanceType</code>.
     *         </p>
     */
    public final InstanceRequirementsRequest instanceRequirements() {
        return instanceRequirements;
    }

    /**
     * <p>
     * The ID of the AMI in the format <code>ami-17characters00000</code>.
     * </p>
     * <p>
     * Alternatively, you can specify a Systems Manager parameter, using one of the following formats. The Systems
     * Manager parameter will resolve to an AMI ID on launch.
     * </p>
     * <p>
     * To reference a public parameter:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resolve:ssm:<i>public-parameter</i> </code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * To reference a parameter stored in the same account:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resolve:ssm:<i>parameter-name</i> </code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>resolve:ssm:<i>parameter-name:version-number</i> </code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>resolve:ssm:<i>parameter-name:label</i> </code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * To reference a parameter shared from another Amazon Web Services account:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resolve:ssm:<i>parameter-ARN</i> </code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>resolve:ssm:<i>parameter-ARN:version-number</i> </code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>resolve:ssm:<i>parameter-ARN:label</i> </code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/create-launch-template.html#use-an-ssm-parameter-instead-of-an-ami-id"
     * >Use a Systems Manager parameter instead of an AMI ID</a> in the <i>Amazon EC2 User Guide</i>.
     * </p>
     * <note>
     * <p>
     * This parameter is only available for fleets of type <code>instant</code>. For fleets of type
     * <code>maintain</code> and <code>request</code>, you must specify the AMI ID in the launch template.
     * </p>
     * </note>
     * 
     * @return The ID of the AMI in the format <code>ami-17characters00000</code>.</p>
     *         <p>
     *         Alternatively, you can specify a Systems Manager parameter, using one of the following formats. The
     *         Systems Manager parameter will resolve to an AMI ID on launch.
     *         </p>
     *         <p>
     *         To reference a public parameter:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>resolve:ssm:<i>public-parameter</i> </code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         To reference a parameter stored in the same account:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>resolve:ssm:<i>parameter-name</i> </code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>resolve:ssm:<i>parameter-name:version-number</i> </code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>resolve:ssm:<i>parameter-name:label</i> </code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         To reference a parameter shared from another Amazon Web Services account:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>resolve:ssm:<i>parameter-ARN</i> </code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>resolve:ssm:<i>parameter-ARN:version-number</i> </code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>resolve:ssm:<i>parameter-ARN:label</i> </code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/create-launch-template.html#use-an-ssm-parameter-instead-of-an-ami-id"
     *         >Use a Systems Manager parameter instead of an AMI ID</a> in the <i>Amazon EC2 User Guide</i>.
     *         </p>
     *         <note>
     *         <p>
     *         This parameter is only available for fleets of type <code>instant</code>. For fleets of type
     *         <code>maintain</code> and <code>request</code>, you must specify the AMI ID in the launch template.
     *         </p>
     */
    public final String imageId() {
        return imageId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(instanceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(maxPrice());
        hashCode = 31 * hashCode + Objects.hashCode(subnetId());
        hashCode = 31 * hashCode + Objects.hashCode(availabilityZone());
        hashCode = 31 * hashCode + Objects.hashCode(weightedCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(priority());
        hashCode = 31 * hashCode + Objects.hashCode(placement());
        hashCode = 31 * hashCode + Objects.hashCode(hasBlockDeviceMappings() ? blockDeviceMappings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(instanceRequirements());
        hashCode = 31 * hashCode + Objects.hashCode(imageId());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof FleetLaunchTemplateOverridesRequest)) {
            return false;
        }
        FleetLaunchTemplateOverridesRequest other = (FleetLaunchTemplateOverridesRequest) obj;
        return Objects.equals(instanceTypeAsString(), other.instanceTypeAsString())
                && Objects.equals(maxPrice(), other.maxPrice()) && Objects.equals(subnetId(), other.subnetId())
                && Objects.equals(availabilityZone(), other.availabilityZone())
                && Objects.equals(weightedCapacity(), other.weightedCapacity()) && Objects.equals(priority(), other.priority())
                && Objects.equals(placement(), other.placement()) && hasBlockDeviceMappings() == other.hasBlockDeviceMappings()
                && Objects.equals(blockDeviceMappings(), other.blockDeviceMappings())
                && Objects.equals(instanceRequirements(), other.instanceRequirements())
                && Objects.equals(imageId(), other.imageId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("FleetLaunchTemplateOverridesRequest").add("InstanceType", instanceTypeAsString())
                .add("MaxPrice", maxPrice()).add("SubnetId", subnetId()).add("AvailabilityZone", availabilityZone())
                .add("WeightedCapacity", weightedCapacity()).add("Priority", priority()).add("Placement", placement())
                .add("BlockDeviceMappings", hasBlockDeviceMappings() ? blockDeviceMappings() : null)
                .add("InstanceRequirements", instanceRequirements()).add("ImageId", imageId()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "InstanceType":
            return Optional.ofNullable(clazz.cast(instanceTypeAsString()));
        case "MaxPrice":
            return Optional.ofNullable(clazz.cast(maxPrice()));
        case "SubnetId":
            return Optional.ofNullable(clazz.cast(subnetId()));
        case "AvailabilityZone":
            return Optional.ofNullable(clazz.cast(availabilityZone()));
        case "WeightedCapacity":
            return Optional.ofNullable(clazz.cast(weightedCapacity()));
        case "Priority":
            return Optional.ofNullable(clazz.cast(priority()));
        case "Placement":
            return Optional.ofNullable(clazz.cast(placement()));
        case "BlockDeviceMappings":
            return Optional.ofNullable(clazz.cast(blockDeviceMappings()));
        case "InstanceRequirements":
            return Optional.ofNullable(clazz.cast(instanceRequirements()));
        case "ImageId":
            return Optional.ofNullable(clazz.cast(imageId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("InstanceType", INSTANCE_TYPE_FIELD);
        map.put("MaxPrice", MAX_PRICE_FIELD);
        map.put("SubnetId", SUBNET_ID_FIELD);
        map.put("AvailabilityZone", AVAILABILITY_ZONE_FIELD);
        map.put("WeightedCapacity", WEIGHTED_CAPACITY_FIELD);
        map.put("Priority", PRIORITY_FIELD);
        map.put("Placement", PLACEMENT_FIELD);
        map.put("BlockDeviceMapping", BLOCK_DEVICE_MAPPINGS_FIELD);
        map.put("InstanceRequirements", INSTANCE_REQUIREMENTS_FIELD);
        map.put("ImageId", IMAGE_ID_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<FleetLaunchTemplateOverridesRequest, T> g) {
        return obj -> g.apply((FleetLaunchTemplateOverridesRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, FleetLaunchTemplateOverridesRequest> {
        /**
         * <p>
         * The instance type.
         * </p>
         * <p>
         * <code>mac1.metal</code> is not supported as a launch template override.
         * </p>
         * <note>
         * <p>
         * If you specify <code>InstanceType</code>, you can't specify <code>InstanceRequirements</code>.
         * </p>
         * </note>
         * 
         * @param instanceType
         *        The instance type.</p>
         *        <p>
         *        <code>mac1.metal</code> is not supported as a launch template override.
         *        </p>
         *        <note>
         *        <p>
         *        If you specify <code>InstanceType</code>, you can't specify <code>InstanceRequirements</code>.
         *        </p>
         * @see InstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceType
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * The instance type.
         * </p>
         * <p>
         * <code>mac1.metal</code> is not supported as a launch template override.
         * </p>
         * <note>
         * <p>
         * If you specify <code>InstanceType</code>, you can't specify <code>InstanceRequirements</code>.
         * </p>
         * </note>
         * 
         * @param instanceType
         *        The instance type.</p>
         *        <p>
         *        <code>mac1.metal</code> is not supported as a launch template override.
         *        </p>
         *        <note>
         *        <p>
         *        If you specify <code>InstanceType</code>, you can't specify <code>InstanceRequirements</code>.
         *        </p>
         * @see InstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceType
         */
        Builder instanceType(InstanceType instanceType);

        /**
         * <p>
         * The maximum price per unit hour that you are willing to pay for a Spot Instance. We do not recommend using
         * this parameter because it can lead to increased interruptions. If you do not specify this parameter, you will
         * pay the current Spot price.
         * </p>
         * <important>
         * <p>
         * If you specify a maximum price, your instances will be interrupted more frequently than if you do not specify
         * this parameter.
         * </p>
         * <p>
         * If you specify a maximum price, it must be more than USD $0.001. Specifying a value below USD $0.001 will
         * result in an <code>InvalidParameterValue</code> error message.
         * </p>
         * </important>
         * 
         * @param maxPrice
         *        The maximum price per unit hour that you are willing to pay for a Spot Instance. We do not recommend
         *        using this parameter because it can lead to increased interruptions. If you do not specify this
         *        parameter, you will pay the current Spot price. </p> <important>
         *        <p>
         *        If you specify a maximum price, your instances will be interrupted more frequently than if you do not
         *        specify this parameter.
         *        </p>
         *        <p>
         *        If you specify a maximum price, it must be more than USD $0.001. Specifying a value below USD $0.001
         *        will result in an <code>InvalidParameterValue</code> error message.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxPrice(String maxPrice);

        /**
         * <p>
         * The IDs of the subnets in which to launch the instances. Separate multiple subnet IDs using commas (for
         * example, <code>subnet-1234abcdeexample1, subnet-0987cdef6example2</code>). A request of type
         * <code>instant</code> can have only one subnet ID.
         * </p>
         * 
         * @param subnetId
         *        The IDs of the subnets in which to launch the instances. Separate multiple subnet IDs using commas
         *        (for example, <code>subnet-1234abcdeexample1, subnet-0987cdef6example2</code>). A request of type
         *        <code>instant</code> can have only one subnet ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetId(String subnetId);

        /**
         * <p>
         * The Availability Zone in which to launch the instances.
         * </p>
         * 
         * @param availabilityZone
         *        The Availability Zone in which to launch the instances.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZone(String availabilityZone);

        /**
         * <p>
         * The number of units provided by the specified instance type. These are the same units that you chose to set
         * the target capacity in terms of instances, or a performance characteristic such as vCPUs, memory, or I/O.
         * </p>
         * <p>
         * If the target capacity divided by this value is not a whole number, Amazon EC2 rounds the number of instances
         * to the next whole number. If this value is not specified, the default is 1.
         * </p>
         * <note>
         * <p>
         * When specifying weights, the price used in the <code>lowest-price</code> and
         * <code>price-capacity-optimized</code> allocation strategies is per <i>unit</i> hour (where the instance price
         * is divided by the specified weight). However, if all the specified weights are above the requested
         * <code>TargetCapacity</code>, resulting in only 1 instance being launched, the price used is per
         * <i>instance</i> hour.
         * </p>
         * </note>
         * 
         * @param weightedCapacity
         *        The number of units provided by the specified instance type. These are the same units that you chose
         *        to set the target capacity in terms of instances, or a performance characteristic such as vCPUs,
         *        memory, or I/O.</p>
         *        <p>
         *        If the target capacity divided by this value is not a whole number, Amazon EC2 rounds the number of
         *        instances to the next whole number. If this value is not specified, the default is 1.
         *        </p>
         *        <note>
         *        <p>
         *        When specifying weights, the price used in the <code>lowest-price</code> and
         *        <code>price-capacity-optimized</code> allocation strategies is per <i>unit</i> hour (where the
         *        instance price is divided by the specified weight). However, if all the specified weights are above
         *        the requested <code>TargetCapacity</code>, resulting in only 1 instance being launched, the price used
         *        is per <i>instance</i> hour.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder weightedCapacity(Double weightedCapacity);

        /**
         * <p>
         * The priority for the launch template override. The highest priority is launched first.
         * </p>
         * <p>
         * If the On-Demand <code>AllocationStrategy</code> is set to <code>prioritized</code>, EC2 Fleet uses priority
         * to determine which launch template override to use first in fulfilling On-Demand capacity.
         * </p>
         * <p>
         * If the Spot <code>AllocationStrategy</code> is set to <code>capacity-optimized-prioritized</code>, EC2 Fleet
         * uses priority on a best-effort basis to determine which launch template override to use in fulfilling Spot
         * capacity, but optimizes for capacity first.
         * </p>
         * <p>
         * Valid values are whole numbers starting at <code>0</code>. The lower the number, the higher the priority. If
         * no number is set, the launch template override has the lowest priority. You can set the same priority for
         * different launch template overrides.
         * </p>
         * 
         * @param priority
         *        The priority for the launch template override. The highest priority is launched first.</p>
         *        <p>
         *        If the On-Demand <code>AllocationStrategy</code> is set to <code>prioritized</code>, EC2 Fleet uses
         *        priority to determine which launch template override to use first in fulfilling On-Demand capacity.
         *        </p>
         *        <p>
         *        If the Spot <code>AllocationStrategy</code> is set to <code>capacity-optimized-prioritized</code>, EC2
         *        Fleet uses priority on a best-effort basis to determine which launch template override to use in
         *        fulfilling Spot capacity, but optimizes for capacity first.
         *        </p>
         *        <p>
         *        Valid values are whole numbers starting at <code>0</code>. The lower the number, the higher the
         *        priority. If no number is set, the launch template override has the lowest priority. You can set the
         *        same priority for different launch template overrides.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder priority(Double priority);

        /**
         * <p>
         * The location where the instance launched, if applicable.
         * </p>
         * 
         * @param placement
         *        The location where the instance launched, if applicable.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder placement(Placement placement);

        /**
         * <p>
         * The location where the instance launched, if applicable.
         * </p>
         * This is a convenience method that creates an instance of the {@link Placement.Builder} avoiding the need to
         * create one manually via {@link Placement#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Placement.Builder#build()} is called immediately and its result
         * is passed to {@link #placement(Placement)}.
         * 
         * @param placement
         *        a consumer that will call methods on {@link Placement.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #placement(Placement)
         */
        default Builder placement(Consumer<Placement.Builder> placement) {
            return placement(Placement.builder().applyMutation(placement).build());
        }

        /**
         * <p>
         * The block device mappings, which define the EBS volumes and instance store volumes to attach to the instance
         * at launch.
         * </p>
         * <p>
         * Supported only for fleets of type <code>instant</code>.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/block-device-mapping-concepts.html">Block device
         * mappings for volumes on Amazon EC2 instances</a> in the <i>Amazon EC2 User Guide</i>.
         * </p>
         * 
         * @param blockDeviceMappings
         *        The block device mappings, which define the EBS volumes and instance store volumes to attach to the
         *        instance at launch.</p>
         *        <p>
         *        Supported only for fleets of type <code>instant</code>.
         *        </p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/block-device-mapping-concepts.html">Block
         *        device mappings for volumes on Amazon EC2 instances</a> in the <i>Amazon EC2 User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder blockDeviceMappings(Collection<FleetBlockDeviceMappingRequest> blockDeviceMappings);

        /**
         * <p>
         * The block device mappings, which define the EBS volumes and instance store volumes to attach to the instance
         * at launch.
         * </p>
         * <p>
         * Supported only for fleets of type <code>instant</code>.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/block-device-mapping-concepts.html">Block device
         * mappings for volumes on Amazon EC2 instances</a> in the <i>Amazon EC2 User Guide</i>.
         * </p>
         * 
         * @param blockDeviceMappings
         *        The block device mappings, which define the EBS volumes and instance store volumes to attach to the
         *        instance at launch.</p>
         *        <p>
         *        Supported only for fleets of type <code>instant</code>.
         *        </p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/block-device-mapping-concepts.html">Block
         *        device mappings for volumes on Amazon EC2 instances</a> in the <i>Amazon EC2 User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder blockDeviceMappings(FleetBlockDeviceMappingRequest... blockDeviceMappings);

        /**
         * <p>
         * The block device mappings, which define the EBS volumes and instance store volumes to attach to the instance
         * at launch.
         * </p>
         * <p>
         * Supported only for fleets of type <code>instant</code>.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/block-device-mapping-concepts.html">Block device
         * mappings for volumes on Amazon EC2 instances</a> in the <i>Amazon EC2 User Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.FleetBlockDeviceMappingRequest.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.ec2.model.FleetBlockDeviceMappingRequest#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ec2.model.FleetBlockDeviceMappingRequest.Builder#build()} is called
         * immediately and its result is passed to {@link #blockDeviceMappings(List<FleetBlockDeviceMappingRequest>)}.
         * 
         * @param blockDeviceMappings
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ec2.model.FleetBlockDeviceMappingRequest.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #blockDeviceMappings(java.util.Collection<FleetBlockDeviceMappingRequest>)
         */
        Builder blockDeviceMappings(Consumer<FleetBlockDeviceMappingRequest.Builder>... blockDeviceMappings);

        /**
         * <p>
         * The attributes for the instance types. When you specify instance attributes, Amazon EC2 will identify
         * instance types with those attributes.
         * </p>
         * <note>
         * <p>
         * If you specify <code>InstanceRequirements</code>, you can't specify <code>InstanceType</code>.
         * </p>
         * </note>
         * 
         * @param instanceRequirements
         *        The attributes for the instance types. When you specify instance attributes, Amazon EC2 will identify
         *        instance types with those attributes.</p> <note>
         *        <p>
         *        If you specify <code>InstanceRequirements</code>, you can't specify <code>InstanceType</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceRequirements(InstanceRequirementsRequest instanceRequirements);

        /**
         * <p>
         * The attributes for the instance types. When you specify instance attributes, Amazon EC2 will identify
         * instance types with those attributes.
         * </p>
         * <note>
         * <p>
         * If you specify <code>InstanceRequirements</code>, you can't specify <code>InstanceType</code>.
         * </p>
         * </note> This is a convenience method that creates an instance of the
         * {@link InstanceRequirementsRequest.Builder} avoiding the need to create one manually via
         * {@link InstanceRequirementsRequest#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link InstanceRequirementsRequest.Builder#build()} is called
         * immediately and its result is passed to {@link #instanceRequirements(InstanceRequirementsRequest)}.
         * 
         * @param instanceRequirements
         *        a consumer that will call methods on {@link InstanceRequirementsRequest.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #instanceRequirements(InstanceRequirementsRequest)
         */
        default Builder instanceRequirements(Consumer<InstanceRequirementsRequest.Builder> instanceRequirements) {
            return instanceRequirements(InstanceRequirementsRequest.builder().applyMutation(instanceRequirements).build());
        }

        /**
         * <p>
         * The ID of the AMI in the format <code>ami-17characters00000</code>.
         * </p>
         * <p>
         * Alternatively, you can specify a Systems Manager parameter, using one of the following formats. The Systems
         * Manager parameter will resolve to an AMI ID on launch.
         * </p>
         * <p>
         * To reference a public parameter:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>resolve:ssm:<i>public-parameter</i> </code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * To reference a parameter stored in the same account:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>resolve:ssm:<i>parameter-name</i> </code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>resolve:ssm:<i>parameter-name:version-number</i> </code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>resolve:ssm:<i>parameter-name:label</i> </code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * To reference a parameter shared from another Amazon Web Services account:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>resolve:ssm:<i>parameter-ARN</i> </code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>resolve:ssm:<i>parameter-ARN:version-number</i> </code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>resolve:ssm:<i>parameter-ARN:label</i> </code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/create-launch-template.html#use-an-ssm-parameter-instead-of-an-ami-id"
         * >Use a Systems Manager parameter instead of an AMI ID</a> in the <i>Amazon EC2 User Guide</i>.
         * </p>
         * <note>
         * <p>
         * This parameter is only available for fleets of type <code>instant</code>. For fleets of type
         * <code>maintain</code> and <code>request</code>, you must specify the AMI ID in the launch template.
         * </p>
         * </note>
         * 
         * @param imageId
         *        The ID of the AMI in the format <code>ami-17characters00000</code>.</p>
         *        <p>
         *        Alternatively, you can specify a Systems Manager parameter, using one of the following formats. The
         *        Systems Manager parameter will resolve to an AMI ID on launch.
         *        </p>
         *        <p>
         *        To reference a public parameter:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>resolve:ssm:<i>public-parameter</i> </code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        To reference a parameter stored in the same account:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>resolve:ssm:<i>parameter-name</i> </code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>resolve:ssm:<i>parameter-name:version-number</i> </code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>resolve:ssm:<i>parameter-name:label</i> </code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        To reference a parameter shared from another Amazon Web Services account:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>resolve:ssm:<i>parameter-ARN</i> </code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>resolve:ssm:<i>parameter-ARN:version-number</i> </code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>resolve:ssm:<i>parameter-ARN:label</i> </code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/create-launch-template.html#use-an-ssm-parameter-instead-of-an-ami-id"
         *        >Use a Systems Manager parameter instead of an AMI ID</a> in the <i>Amazon EC2 User Guide</i>.
         *        </p>
         *        <note>
         *        <p>
         *        This parameter is only available for fleets of type <code>instant</code>. For fleets of type
         *        <code>maintain</code> and <code>request</code>, you must specify the AMI ID in the launch template.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageId(String imageId);
    }

    static final class BuilderImpl implements Builder {
        private String instanceType;

        private String maxPrice;

        private String subnetId;

        private String availabilityZone;

        private Double weightedCapacity;

        private Double priority;

        private Placement placement;

        private List<FleetBlockDeviceMappingRequest> blockDeviceMappings = DefaultSdkAutoConstructList.getInstance();

        private InstanceRequirementsRequest instanceRequirements;

        private String imageId;

        private BuilderImpl() {
        }

        private BuilderImpl(FleetLaunchTemplateOverridesRequest model) {
            instanceType(model.instanceType);
            maxPrice(model.maxPrice);
            subnetId(model.subnetId);
            availabilityZone(model.availabilityZone);
            weightedCapacity(model.weightedCapacity);
            priority(model.priority);
            placement(model.placement);
            blockDeviceMappings(model.blockDeviceMappings);
            instanceRequirements(model.instanceRequirements);
            imageId(model.imageId);
        }

        public final String getInstanceType() {
            return instanceType;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        @Override
        public final Builder instanceType(InstanceType instanceType) {
            this.instanceType(instanceType == null ? null : instanceType.toString());
            return this;
        }

        public final String getMaxPrice() {
            return maxPrice;
        }

        public final void setMaxPrice(String maxPrice) {
            this.maxPrice = maxPrice;
        }

        @Override
        public final Builder maxPrice(String maxPrice) {
            this.maxPrice = maxPrice;
            return this;
        }

        public final String getSubnetId() {
            return subnetId;
        }

        public final void setSubnetId(String subnetId) {
            this.subnetId = subnetId;
        }

        @Override
        public final Builder subnetId(String subnetId) {
            this.subnetId = subnetId;
            return this;
        }

        public final String getAvailabilityZone() {
            return availabilityZone;
        }

        public final void setAvailabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
        }

        @Override
        public final Builder availabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
            return this;
        }

        public final Double getWeightedCapacity() {
            return weightedCapacity;
        }

        public final void setWeightedCapacity(Double weightedCapacity) {
            this.weightedCapacity = weightedCapacity;
        }

        @Override
        public final Builder weightedCapacity(Double weightedCapacity) {
            this.weightedCapacity = weightedCapacity;
            return this;
        }

        public final Double getPriority() {
            return priority;
        }

        public final void setPriority(Double priority) {
            this.priority = priority;
        }

        @Override
        public final Builder priority(Double priority) {
            this.priority = priority;
            return this;
        }

        public final Placement.Builder getPlacement() {
            return placement != null ? placement.toBuilder() : null;
        }

        public final void setPlacement(Placement.BuilderImpl placement) {
            this.placement = placement != null ? placement.build() : null;
        }

        @Override
        public final Builder placement(Placement placement) {
            this.placement = placement;
            return this;
        }

        public final List<FleetBlockDeviceMappingRequest.Builder> getBlockDeviceMappings() {
            List<FleetBlockDeviceMappingRequest.Builder> result = FleetBlockDeviceMappingRequestListCopier
                    .copyToBuilder(this.blockDeviceMappings);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setBlockDeviceMappings(Collection<FleetBlockDeviceMappingRequest.BuilderImpl> blockDeviceMappings) {
            this.blockDeviceMappings = FleetBlockDeviceMappingRequestListCopier.copyFromBuilder(blockDeviceMappings);
        }

        @Override
        public final Builder blockDeviceMappings(Collection<FleetBlockDeviceMappingRequest> blockDeviceMappings) {
            this.blockDeviceMappings = FleetBlockDeviceMappingRequestListCopier.copy(blockDeviceMappings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder blockDeviceMappings(FleetBlockDeviceMappingRequest... blockDeviceMappings) {
            blockDeviceMappings(Arrays.asList(blockDeviceMappings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder blockDeviceMappings(Consumer<FleetBlockDeviceMappingRequest.Builder>... blockDeviceMappings) {
            blockDeviceMappings(Stream.of(blockDeviceMappings)
                    .map(c -> FleetBlockDeviceMappingRequest.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final InstanceRequirementsRequest.Builder getInstanceRequirements() {
            return instanceRequirements != null ? instanceRequirements.toBuilder() : null;
        }

        public final void setInstanceRequirements(InstanceRequirementsRequest.BuilderImpl instanceRequirements) {
            this.instanceRequirements = instanceRequirements != null ? instanceRequirements.build() : null;
        }

        @Override
        public final Builder instanceRequirements(InstanceRequirementsRequest instanceRequirements) {
            this.instanceRequirements = instanceRequirements;
            return this;
        }

        public final String getImageId() {
            return imageId;
        }

        public final void setImageId(String imageId) {
            this.imageId = imageId;
        }

        @Override
        public final Builder imageId(String imageId) {
            this.imageId = imageId;
            return this;
        }

        @Override
        public FleetLaunchTemplateOverridesRequest build() {
            return new FleetLaunchTemplateOverridesRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
