/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.DefaultValueTrait;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateCapacityReservationFleetRequest extends Ec2Request implements
        ToCopyableBuilder<CreateCapacityReservationFleetRequest.Builder, CreateCapacityReservationFleetRequest> {
    private static final SdkField<String> ALLOCATION_STRATEGY_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AllocationStrategy")
            .getter(getter(CreateCapacityReservationFleetRequest::allocationStrategy))
            .setter(setter(Builder::allocationStrategy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllocationStrategy")
                    .unmarshallLocationName("AllocationStrategy").build()).build();

    private static final SdkField<String> CLIENT_TOKEN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ClientToken")
            .getter(getter(CreateCapacityReservationFleetRequest::clientToken))
            .setter(setter(Builder::clientToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientToken")
                    .unmarshallLocationName("ClientToken").build(), DefaultValueTrait.idempotencyToken()).build();

    private static final SdkField<List<ReservationFleetInstanceSpecification>> INSTANCE_TYPE_SPECIFICATIONS_FIELD = SdkField
            .<List<ReservationFleetInstanceSpecification>> builder(MarshallingType.LIST)
            .memberName("InstanceTypeSpecifications")
            .getter(getter(CreateCapacityReservationFleetRequest::instanceTypeSpecifications))
            .setter(setter(Builder::instanceTypeSpecifications))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceTypeSpecification")
                    .unmarshallLocationName("InstanceTypeSpecification").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ReservationFleetInstanceSpecification> builder(MarshallingType.SDK_POJO)
                                            .constructor(ReservationFleetInstanceSpecification::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").unmarshallLocationName("member").build()).build())
                            .build()).build();

    private static final SdkField<String> TENANCY_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Tenancy")
            .getter(getter(CreateCapacityReservationFleetRequest::tenancyAsString))
            .setter(setter(Builder::tenancy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tenancy")
                    .unmarshallLocationName("Tenancy").build()).build();

    private static final SdkField<Integer> TOTAL_TARGET_CAPACITY_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("TotalTargetCapacity")
            .getter(getter(CreateCapacityReservationFleetRequest::totalTargetCapacity))
            .setter(setter(Builder::totalTargetCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TotalTargetCapacity")
                    .unmarshallLocationName("TotalTargetCapacity").build()).build();

    private static final SdkField<Instant> END_DATE_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("EndDate")
            .getter(getter(CreateCapacityReservationFleetRequest::endDate))
            .setter(setter(Builder::endDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndDate")
                    .unmarshallLocationName("EndDate").build()).build();

    private static final SdkField<String> INSTANCE_MATCH_CRITERIA_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("InstanceMatchCriteria")
            .getter(getter(CreateCapacityReservationFleetRequest::instanceMatchCriteriaAsString))
            .setter(setter(Builder::instanceMatchCriteria))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceMatchCriteria")
                    .unmarshallLocationName("InstanceMatchCriteria").build()).build();

    private static final SdkField<List<TagSpecification>> TAG_SPECIFICATIONS_FIELD = SdkField
            .<List<TagSpecification>> builder(MarshallingType.LIST)
            .memberName("TagSpecifications")
            .getter(getter(CreateCapacityReservationFleetRequest::tagSpecifications))
            .setter(setter(Builder::tagSpecifications))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSpecification")
                    .unmarshallLocationName("TagSpecification").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<TagSpecification> builder(MarshallingType.SDK_POJO)
                                            .constructor(TagSpecification::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<Boolean> DRY_RUN_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("DryRun")
            .getter(getter(CreateCapacityReservationFleetRequest::dryRun))
            .setter(setter(Builder::dryRun))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DryRun")
                    .unmarshallLocationName("DryRun").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ALLOCATION_STRATEGY_FIELD,
            CLIENT_TOKEN_FIELD, INSTANCE_TYPE_SPECIFICATIONS_FIELD, TENANCY_FIELD, TOTAL_TARGET_CAPACITY_FIELD, END_DATE_FIELD,
            INSTANCE_MATCH_CRITERIA_FIELD, TAG_SPECIFICATIONS_FIELD, DRY_RUN_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String allocationStrategy;

    private final String clientToken;

    private final List<ReservationFleetInstanceSpecification> instanceTypeSpecifications;

    private final String tenancy;

    private final Integer totalTargetCapacity;

    private final Instant endDate;

    private final String instanceMatchCriteria;

    private final List<TagSpecification> tagSpecifications;

    private final Boolean dryRun;

    private CreateCapacityReservationFleetRequest(BuilderImpl builder) {
        super(builder);
        this.allocationStrategy = builder.allocationStrategy;
        this.clientToken = builder.clientToken;
        this.instanceTypeSpecifications = builder.instanceTypeSpecifications;
        this.tenancy = builder.tenancy;
        this.totalTargetCapacity = builder.totalTargetCapacity;
        this.endDate = builder.endDate;
        this.instanceMatchCriteria = builder.instanceMatchCriteria;
        this.tagSpecifications = builder.tagSpecifications;
        this.dryRun = builder.dryRun;
    }

    /**
     * <p>
     * The strategy used by the Capacity Reservation Fleet to determine which of the specified instance types to use.
     * Currently, only the <code>prioritized</code> allocation strategy is supported. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#allocation-strategy"> Allocation
     * strategy</a> in the <i>Amazon EC2 User Guide</i>.
     * </p>
     * <p>
     * Valid values: <code>prioritized</code>
     * </p>
     * 
     * @return The strategy used by the Capacity Reservation Fleet to determine which of the specified instance types to
     *         use. Currently, only the <code>prioritized</code> allocation strategy is supported. For more information,
     *         see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#allocation-strategy">
     *         Allocation strategy</a> in the <i>Amazon EC2 User Guide</i>.</p>
     *         <p>
     *         Valid values: <code>prioritized</code>
     */
    public final String allocationStrategy() {
        return allocationStrategy;
    }

    /**
     * <p>
     * Unique, case-sensitive identifier that you provide to ensure the idempotency of the request. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html">Ensure
     * Idempotency</a>.
     * </p>
     * 
     * @return Unique, case-sensitive identifier that you provide to ensure the idempotency of the request. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html">Ensure
     *         Idempotency</a>.
     */
    public final String clientToken() {
        return clientToken;
    }

    /**
     * For responses, this returns true if the service returned a value for the InstanceTypeSpecifications property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasInstanceTypeSpecifications() {
        return instanceTypeSpecifications != null && !(instanceTypeSpecifications instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Information about the instance types for which to reserve the capacity.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInstanceTypeSpecifications} method.
     * </p>
     * 
     * @return Information about the instance types for which to reserve the capacity.
     */
    public final List<ReservationFleetInstanceSpecification> instanceTypeSpecifications() {
        return instanceTypeSpecifications;
    }

    /**
     * <p>
     * Indicates the tenancy of the Capacity Reservation Fleet. All Capacity Reservations in the Fleet inherit this
     * tenancy. The Capacity Reservation Fleet can have one of the following tenancy settings:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>default</code> - The Capacity Reservation Fleet is created on hardware that is shared with other Amazon Web
     * Services accounts.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dedicated</code> - The Capacity Reservations are created on single-tenant hardware that is dedicated to a
     * single Amazon Web Services account.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #tenancy} will
     * return {@link FleetCapacityReservationTenancy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #tenancyAsString}.
     * </p>
     * 
     * @return Indicates the tenancy of the Capacity Reservation Fleet. All Capacity Reservations in the Fleet inherit
     *         this tenancy. The Capacity Reservation Fleet can have one of the following tenancy settings:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>default</code> - The Capacity Reservation Fleet is created on hardware that is shared with other
     *         Amazon Web Services accounts.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dedicated</code> - The Capacity Reservations are created on single-tenant hardware that is
     *         dedicated to a single Amazon Web Services account.
     *         </p>
     *         </li>
     * @see FleetCapacityReservationTenancy
     */
    public final FleetCapacityReservationTenancy tenancy() {
        return FleetCapacityReservationTenancy.fromValue(tenancy);
    }

    /**
     * <p>
     * Indicates the tenancy of the Capacity Reservation Fleet. All Capacity Reservations in the Fleet inherit this
     * tenancy. The Capacity Reservation Fleet can have one of the following tenancy settings:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>default</code> - The Capacity Reservation Fleet is created on hardware that is shared with other Amazon Web
     * Services accounts.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dedicated</code> - The Capacity Reservations are created on single-tenant hardware that is dedicated to a
     * single Amazon Web Services account.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #tenancy} will
     * return {@link FleetCapacityReservationTenancy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #tenancyAsString}.
     * </p>
     * 
     * @return Indicates the tenancy of the Capacity Reservation Fleet. All Capacity Reservations in the Fleet inherit
     *         this tenancy. The Capacity Reservation Fleet can have one of the following tenancy settings:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>default</code> - The Capacity Reservation Fleet is created on hardware that is shared with other
     *         Amazon Web Services accounts.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dedicated</code> - The Capacity Reservations are created on single-tenant hardware that is
     *         dedicated to a single Amazon Web Services account.
     *         </p>
     *         </li>
     * @see FleetCapacityReservationTenancy
     */
    public final String tenancyAsString() {
        return tenancy;
    }

    /**
     * <p>
     * The total number of capacity units to be reserved by the Capacity Reservation Fleet. This value, together with
     * the instance type weights that you assign to each instance type used by the Fleet determine the number of
     * instances for which the Fleet reserves capacity. Both values are based on units that make sense for your
     * workload. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#target-capacity">Total target
     * capacity</a> in the <i>Amazon EC2 User Guide</i>.
     * </p>
     * 
     * @return The total number of capacity units to be reserved by the Capacity Reservation Fleet. This value, together
     *         with the instance type weights that you assign to each instance type used by the Fleet determine the
     *         number of instances for which the Fleet reserves capacity. Both values are based on units that make sense
     *         for your workload. For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#target-capacity">Total
     *         target capacity</a> in the <i>Amazon EC2 User Guide</i>.
     */
    public final Integer totalTargetCapacity() {
        return totalTargetCapacity;
    }

    /**
     * <p>
     * The date and time at which the Capacity Reservation Fleet expires. When the Capacity Reservation Fleet expires,
     * its state changes to <code>expired</code> and all of the Capacity Reservations in the Fleet expire.
     * </p>
     * <p>
     * The Capacity Reservation Fleet expires within an hour after the specified time. For example, if you specify
     * <code>5/31/2019</code>, <code>13:30:55</code>, the Capacity Reservation Fleet is guaranteed to expire between
     * <code>13:30:55</code> and <code>14:30:55</code> on <code>5/31/2019</code>.
     * </p>
     * 
     * @return The date and time at which the Capacity Reservation Fleet expires. When the Capacity Reservation Fleet
     *         expires, its state changes to <code>expired</code> and all of the Capacity Reservations in the Fleet
     *         expire.</p>
     *         <p>
     *         The Capacity Reservation Fleet expires within an hour after the specified time. For example, if you
     *         specify <code>5/31/2019</code>, <code>13:30:55</code>, the Capacity Reservation Fleet is guaranteed to
     *         expire between <code>13:30:55</code> and <code>14:30:55</code> on <code>5/31/2019</code>.
     */
    public final Instant endDate() {
        return endDate;
    }

    /**
     * <p>
     * Indicates the type of instance launches that the Capacity Reservation Fleet accepts. All Capacity Reservations in
     * the Fleet inherit this instance matching criteria.
     * </p>
     * <p>
     * Currently, Capacity Reservation Fleets support <code>open</code> instance matching criteria only. This means that
     * instances that have matching attributes (instance type, platform, and Availability Zone) run in the Capacity
     * Reservations automatically. Instances do not need to explicitly target a Capacity Reservation Fleet to use its
     * reserved capacity.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #instanceMatchCriteria} will return {@link FleetInstanceMatchCriteria#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #instanceMatchCriteriaAsString}.
     * </p>
     * 
     * @return Indicates the type of instance launches that the Capacity Reservation Fleet accepts. All Capacity
     *         Reservations in the Fleet inherit this instance matching criteria.</p>
     *         <p>
     *         Currently, Capacity Reservation Fleets support <code>open</code> instance matching criteria only. This
     *         means that instances that have matching attributes (instance type, platform, and Availability Zone) run
     *         in the Capacity Reservations automatically. Instances do not need to explicitly target a Capacity
     *         Reservation Fleet to use its reserved capacity.
     * @see FleetInstanceMatchCriteria
     */
    public final FleetInstanceMatchCriteria instanceMatchCriteria() {
        return FleetInstanceMatchCriteria.fromValue(instanceMatchCriteria);
    }

    /**
     * <p>
     * Indicates the type of instance launches that the Capacity Reservation Fleet accepts. All Capacity Reservations in
     * the Fleet inherit this instance matching criteria.
     * </p>
     * <p>
     * Currently, Capacity Reservation Fleets support <code>open</code> instance matching criteria only. This means that
     * instances that have matching attributes (instance type, platform, and Availability Zone) run in the Capacity
     * Reservations automatically. Instances do not need to explicitly target a Capacity Reservation Fleet to use its
     * reserved capacity.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #instanceMatchCriteria} will return {@link FleetInstanceMatchCriteria#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #instanceMatchCriteriaAsString}.
     * </p>
     * 
     * @return Indicates the type of instance launches that the Capacity Reservation Fleet accepts. All Capacity
     *         Reservations in the Fleet inherit this instance matching criteria.</p>
     *         <p>
     *         Currently, Capacity Reservation Fleets support <code>open</code> instance matching criteria only. This
     *         means that instances that have matching attributes (instance type, platform, and Availability Zone) run
     *         in the Capacity Reservations automatically. Instances do not need to explicitly target a Capacity
     *         Reservation Fleet to use its reserved capacity.
     * @see FleetInstanceMatchCriteria
     */
    public final String instanceMatchCriteriaAsString() {
        return instanceMatchCriteria;
    }

    /**
     * For responses, this returns true if the service returned a value for the TagSpecifications property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTagSpecifications() {
        return tagSpecifications != null && !(tagSpecifications instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags to assign to the Capacity Reservation Fleet. The tags are automatically assigned to the Capacity
     * Reservations in the Fleet.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTagSpecifications} method.
     * </p>
     * 
     * @return The tags to assign to the Capacity Reservation Fleet. The tags are automatically assigned to the Capacity
     *         Reservations in the Fleet.
     */
    public final List<TagSpecification> tagSpecifications() {
        return tagSpecifications;
    }

    /**
     * <p>
     * Checks whether you have the required permissions for the action, without actually making the request, and
     * provides an error response. If you have the required permissions, the error response is
     * <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
     * </p>
     * 
     * @return Checks whether you have the required permissions for the action, without actually making the request, and
     *         provides an error response. If you have the required permissions, the error response is
     *         <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
     */
    public final Boolean dryRun() {
        return dryRun;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(allocationStrategy());
        hashCode = 31 * hashCode + Objects.hashCode(clientToken());
        hashCode = 31 * hashCode + Objects.hashCode(hasInstanceTypeSpecifications() ? instanceTypeSpecifications() : null);
        hashCode = 31 * hashCode + Objects.hashCode(tenancyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(totalTargetCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(endDate());
        hashCode = 31 * hashCode + Objects.hashCode(instanceMatchCriteriaAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTagSpecifications() ? tagSpecifications() : null);
        hashCode = 31 * hashCode + Objects.hashCode(dryRun());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateCapacityReservationFleetRequest)) {
            return false;
        }
        CreateCapacityReservationFleetRequest other = (CreateCapacityReservationFleetRequest) obj;
        return Objects.equals(allocationStrategy(), other.allocationStrategy())
                && Objects.equals(clientToken(), other.clientToken())
                && hasInstanceTypeSpecifications() == other.hasInstanceTypeSpecifications()
                && Objects.equals(instanceTypeSpecifications(), other.instanceTypeSpecifications())
                && Objects.equals(tenancyAsString(), other.tenancyAsString())
                && Objects.equals(totalTargetCapacity(), other.totalTargetCapacity())
                && Objects.equals(endDate(), other.endDate())
                && Objects.equals(instanceMatchCriteriaAsString(), other.instanceMatchCriteriaAsString())
                && hasTagSpecifications() == other.hasTagSpecifications()
                && Objects.equals(tagSpecifications(), other.tagSpecifications()) && Objects.equals(dryRun(), other.dryRun());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateCapacityReservationFleetRequest").add("AllocationStrategy", allocationStrategy())
                .add("ClientToken", clientToken())
                .add("InstanceTypeSpecifications", hasInstanceTypeSpecifications() ? instanceTypeSpecifications() : null)
                .add("Tenancy", tenancyAsString()).add("TotalTargetCapacity", totalTargetCapacity()).add("EndDate", endDate())
                .add("InstanceMatchCriteria", instanceMatchCriteriaAsString())
                .add("TagSpecifications", hasTagSpecifications() ? tagSpecifications() : null).add("DryRun", dryRun()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AllocationStrategy":
            return Optional.ofNullable(clazz.cast(allocationStrategy()));
        case "ClientToken":
            return Optional.ofNullable(clazz.cast(clientToken()));
        case "InstanceTypeSpecifications":
            return Optional.ofNullable(clazz.cast(instanceTypeSpecifications()));
        case "Tenancy":
            return Optional.ofNullable(clazz.cast(tenancyAsString()));
        case "TotalTargetCapacity":
            return Optional.ofNullable(clazz.cast(totalTargetCapacity()));
        case "EndDate":
            return Optional.ofNullable(clazz.cast(endDate()));
        case "InstanceMatchCriteria":
            return Optional.ofNullable(clazz.cast(instanceMatchCriteriaAsString()));
        case "TagSpecifications":
            return Optional.ofNullable(clazz.cast(tagSpecifications()));
        case "DryRun":
            return Optional.ofNullable(clazz.cast(dryRun()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("AllocationStrategy", ALLOCATION_STRATEGY_FIELD);
        map.put("ClientToken", CLIENT_TOKEN_FIELD);
        map.put("InstanceTypeSpecification", INSTANCE_TYPE_SPECIFICATIONS_FIELD);
        map.put("Tenancy", TENANCY_FIELD);
        map.put("TotalTargetCapacity", TOTAL_TARGET_CAPACITY_FIELD);
        map.put("EndDate", END_DATE_FIELD);
        map.put("InstanceMatchCriteria", INSTANCE_MATCH_CRITERIA_FIELD);
        map.put("TagSpecification", TAG_SPECIFICATIONS_FIELD);
        map.put("DryRun", DRY_RUN_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CreateCapacityReservationFleetRequest, T> g) {
        return obj -> g.apply((CreateCapacityReservationFleetRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends Ec2Request.Builder, SdkPojo, CopyableBuilder<Builder, CreateCapacityReservationFleetRequest> {
        /**
         * <p>
         * The strategy used by the Capacity Reservation Fleet to determine which of the specified instance types to
         * use. Currently, only the <code>prioritized</code> allocation strategy is supported. For more information, see
         * <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#allocation-strategy">
         * Allocation strategy</a> in the <i>Amazon EC2 User Guide</i>.
         * </p>
         * <p>
         * Valid values: <code>prioritized</code>
         * </p>
         * 
         * @param allocationStrategy
         *        The strategy used by the Capacity Reservation Fleet to determine which of the specified instance types
         *        to use. Currently, only the <code>prioritized</code> allocation strategy is supported. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#allocation-strategy">
         *        Allocation strategy</a> in the <i>Amazon EC2 User Guide</i>.</p>
         *        <p>
         *        Valid values: <code>prioritized</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allocationStrategy(String allocationStrategy);

        /**
         * <p>
         * Unique, case-sensitive identifier that you provide to ensure the idempotency of the request. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html">Ensure
         * Idempotency</a>.
         * </p>
         * 
         * @param clientToken
         *        Unique, case-sensitive identifier that you provide to ensure the idempotency of the request. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html">Ensure
         *        Idempotency</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientToken(String clientToken);

        /**
         * <p>
         * Information about the instance types for which to reserve the capacity.
         * </p>
         * 
         * @param instanceTypeSpecifications
         *        Information about the instance types for which to reserve the capacity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceTypeSpecifications(Collection<ReservationFleetInstanceSpecification> instanceTypeSpecifications);

        /**
         * <p>
         * Information about the instance types for which to reserve the capacity.
         * </p>
         * 
         * @param instanceTypeSpecifications
         *        Information about the instance types for which to reserve the capacity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceTypeSpecifications(ReservationFleetInstanceSpecification... instanceTypeSpecifications);

        /**
         * <p>
         * Information about the instance types for which to reserve the capacity.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.ReservationFleetInstanceSpecification.Builder} avoiding the
         * need to create one manually via
         * {@link software.amazon.awssdk.services.ec2.model.ReservationFleetInstanceSpecification#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ec2.model.ReservationFleetInstanceSpecification.Builder#build()} is
         * called immediately and its result is passed to {@link
         * #instanceTypeSpecifications(List<ReservationFleetInstanceSpecification>)}.
         * 
         * @param instanceTypeSpecifications
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ec2.model.ReservationFleetInstanceSpecification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #instanceTypeSpecifications(java.util.Collection<ReservationFleetInstanceSpecification>)
         */
        Builder instanceTypeSpecifications(Consumer<ReservationFleetInstanceSpecification.Builder>... instanceTypeSpecifications);

        /**
         * <p>
         * Indicates the tenancy of the Capacity Reservation Fleet. All Capacity Reservations in the Fleet inherit this
         * tenancy. The Capacity Reservation Fleet can have one of the following tenancy settings:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>default</code> - The Capacity Reservation Fleet is created on hardware that is shared with other Amazon
         * Web Services accounts.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dedicated</code> - The Capacity Reservations are created on single-tenant hardware that is dedicated to
         * a single Amazon Web Services account.
         * </p>
         * </li>
         * </ul>
         * 
         * @param tenancy
         *        Indicates the tenancy of the Capacity Reservation Fleet. All Capacity Reservations in the Fleet
         *        inherit this tenancy. The Capacity Reservation Fleet can have one of the following tenancy
         *        settings:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>default</code> - The Capacity Reservation Fleet is created on hardware that is shared with other
         *        Amazon Web Services accounts.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dedicated</code> - The Capacity Reservations are created on single-tenant hardware that is
         *        dedicated to a single Amazon Web Services account.
         *        </p>
         *        </li>
         * @see FleetCapacityReservationTenancy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FleetCapacityReservationTenancy
         */
        Builder tenancy(String tenancy);

        /**
         * <p>
         * Indicates the tenancy of the Capacity Reservation Fleet. All Capacity Reservations in the Fleet inherit this
         * tenancy. The Capacity Reservation Fleet can have one of the following tenancy settings:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>default</code> - The Capacity Reservation Fleet is created on hardware that is shared with other Amazon
         * Web Services accounts.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dedicated</code> - The Capacity Reservations are created on single-tenant hardware that is dedicated to
         * a single Amazon Web Services account.
         * </p>
         * </li>
         * </ul>
         * 
         * @param tenancy
         *        Indicates the tenancy of the Capacity Reservation Fleet. All Capacity Reservations in the Fleet
         *        inherit this tenancy. The Capacity Reservation Fleet can have one of the following tenancy
         *        settings:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>default</code> - The Capacity Reservation Fleet is created on hardware that is shared with other
         *        Amazon Web Services accounts.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dedicated</code> - The Capacity Reservations are created on single-tenant hardware that is
         *        dedicated to a single Amazon Web Services account.
         *        </p>
         *        </li>
         * @see FleetCapacityReservationTenancy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FleetCapacityReservationTenancy
         */
        Builder tenancy(FleetCapacityReservationTenancy tenancy);

        /**
         * <p>
         * The total number of capacity units to be reserved by the Capacity Reservation Fleet. This value, together
         * with the instance type weights that you assign to each instance type used by the Fleet determine the number
         * of instances for which the Fleet reserves capacity. Both values are based on units that make sense for your
         * workload. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#target-capacity">Total target
         * capacity</a> in the <i>Amazon EC2 User Guide</i>.
         * </p>
         * 
         * @param totalTargetCapacity
         *        The total number of capacity units to be reserved by the Capacity Reservation Fleet. This value,
         *        together with the instance type weights that you assign to each instance type used by the Fleet
         *        determine the number of instances for which the Fleet reserves capacity. Both values are based on
         *        units that make sense for your workload. For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#target-capacity">Total
         *        target capacity</a> in the <i>Amazon EC2 User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder totalTargetCapacity(Integer totalTargetCapacity);

        /**
         * <p>
         * The date and time at which the Capacity Reservation Fleet expires. When the Capacity Reservation Fleet
         * expires, its state changes to <code>expired</code> and all of the Capacity Reservations in the Fleet expire.
         * </p>
         * <p>
         * The Capacity Reservation Fleet expires within an hour after the specified time. For example, if you specify
         * <code>5/31/2019</code>, <code>13:30:55</code>, the Capacity Reservation Fleet is guaranteed to expire between
         * <code>13:30:55</code> and <code>14:30:55</code> on <code>5/31/2019</code>.
         * </p>
         * 
         * @param endDate
         *        The date and time at which the Capacity Reservation Fleet expires. When the Capacity Reservation Fleet
         *        expires, its state changes to <code>expired</code> and all of the Capacity Reservations in the Fleet
         *        expire.</p>
         *        <p>
         *        The Capacity Reservation Fleet expires within an hour after the specified time. For example, if you
         *        specify <code>5/31/2019</code>, <code>13:30:55</code>, the Capacity Reservation Fleet is guaranteed to
         *        expire between <code>13:30:55</code> and <code>14:30:55</code> on <code>5/31/2019</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endDate(Instant endDate);

        /**
         * <p>
         * Indicates the type of instance launches that the Capacity Reservation Fleet accepts. All Capacity
         * Reservations in the Fleet inherit this instance matching criteria.
         * </p>
         * <p>
         * Currently, Capacity Reservation Fleets support <code>open</code> instance matching criteria only. This means
         * that instances that have matching attributes (instance type, platform, and Availability Zone) run in the
         * Capacity Reservations automatically. Instances do not need to explicitly target a Capacity Reservation Fleet
         * to use its reserved capacity.
         * </p>
         * 
         * @param instanceMatchCriteria
         *        Indicates the type of instance launches that the Capacity Reservation Fleet accepts. All Capacity
         *        Reservations in the Fleet inherit this instance matching criteria.</p>
         *        <p>
         *        Currently, Capacity Reservation Fleets support <code>open</code> instance matching criteria only. This
         *        means that instances that have matching attributes (instance type, platform, and Availability Zone)
         *        run in the Capacity Reservations automatically. Instances do not need to explicitly target a Capacity
         *        Reservation Fleet to use its reserved capacity.
         * @see FleetInstanceMatchCriteria
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FleetInstanceMatchCriteria
         */
        Builder instanceMatchCriteria(String instanceMatchCriteria);

        /**
         * <p>
         * Indicates the type of instance launches that the Capacity Reservation Fleet accepts. All Capacity
         * Reservations in the Fleet inherit this instance matching criteria.
         * </p>
         * <p>
         * Currently, Capacity Reservation Fleets support <code>open</code> instance matching criteria only. This means
         * that instances that have matching attributes (instance type, platform, and Availability Zone) run in the
         * Capacity Reservations automatically. Instances do not need to explicitly target a Capacity Reservation Fleet
         * to use its reserved capacity.
         * </p>
         * 
         * @param instanceMatchCriteria
         *        Indicates the type of instance launches that the Capacity Reservation Fleet accepts. All Capacity
         *        Reservations in the Fleet inherit this instance matching criteria.</p>
         *        <p>
         *        Currently, Capacity Reservation Fleets support <code>open</code> instance matching criteria only. This
         *        means that instances that have matching attributes (instance type, platform, and Availability Zone)
         *        run in the Capacity Reservations automatically. Instances do not need to explicitly target a Capacity
         *        Reservation Fleet to use its reserved capacity.
         * @see FleetInstanceMatchCriteria
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FleetInstanceMatchCriteria
         */
        Builder instanceMatchCriteria(FleetInstanceMatchCriteria instanceMatchCriteria);

        /**
         * <p>
         * The tags to assign to the Capacity Reservation Fleet. The tags are automatically assigned to the Capacity
         * Reservations in the Fleet.
         * </p>
         * 
         * @param tagSpecifications
         *        The tags to assign to the Capacity Reservation Fleet. The tags are automatically assigned to the
         *        Capacity Reservations in the Fleet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagSpecifications(Collection<TagSpecification> tagSpecifications);

        /**
         * <p>
         * The tags to assign to the Capacity Reservation Fleet. The tags are automatically assigned to the Capacity
         * Reservations in the Fleet.
         * </p>
         * 
         * @param tagSpecifications
         *        The tags to assign to the Capacity Reservation Fleet. The tags are automatically assigned to the
         *        Capacity Reservations in the Fleet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagSpecifications(TagSpecification... tagSpecifications);

        /**
         * <p>
         * The tags to assign to the Capacity Reservation Fleet. The tags are automatically assigned to the Capacity
         * Reservations in the Fleet.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.TagSpecification.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ec2.model.TagSpecification#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ec2.model.TagSpecification.Builder#build()} is called immediately and
         * its result is passed to {@link #tagSpecifications(List<TagSpecification>)}.
         * 
         * @param tagSpecifications
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ec2.model.TagSpecification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tagSpecifications(java.util.Collection<TagSpecification>)
         */
        Builder tagSpecifications(Consumer<TagSpecification.Builder>... tagSpecifications);

        /**
         * <p>
         * Checks whether you have the required permissions for the action, without actually making the request, and
         * provides an error response. If you have the required permissions, the error response is
         * <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
         * </p>
         * 
         * @param dryRun
         *        Checks whether you have the required permissions for the action, without actually making the request,
         *        and provides an error response. If you have the required permissions, the error response is
         *        <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dryRun(Boolean dryRun);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends Ec2Request.BuilderImpl implements Builder {
        private String allocationStrategy;

        private String clientToken;

        private List<ReservationFleetInstanceSpecification> instanceTypeSpecifications = DefaultSdkAutoConstructList
                .getInstance();

        private String tenancy;

        private Integer totalTargetCapacity;

        private Instant endDate;

        private String instanceMatchCriteria;

        private List<TagSpecification> tagSpecifications = DefaultSdkAutoConstructList.getInstance();

        private Boolean dryRun;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateCapacityReservationFleetRequest model) {
            super(model);
            allocationStrategy(model.allocationStrategy);
            clientToken(model.clientToken);
            instanceTypeSpecifications(model.instanceTypeSpecifications);
            tenancy(model.tenancy);
            totalTargetCapacity(model.totalTargetCapacity);
            endDate(model.endDate);
            instanceMatchCriteria(model.instanceMatchCriteria);
            tagSpecifications(model.tagSpecifications);
            dryRun(model.dryRun);
        }

        public final String getAllocationStrategy() {
            return allocationStrategy;
        }

        public final void setAllocationStrategy(String allocationStrategy) {
            this.allocationStrategy = allocationStrategy;
        }

        @Override
        public final Builder allocationStrategy(String allocationStrategy) {
            this.allocationStrategy = allocationStrategy;
            return this;
        }

        public final String getClientToken() {
            return clientToken;
        }

        public final void setClientToken(String clientToken) {
            this.clientToken = clientToken;
        }

        @Override
        public final Builder clientToken(String clientToken) {
            this.clientToken = clientToken;
            return this;
        }

        public final List<ReservationFleetInstanceSpecification.Builder> getInstanceTypeSpecifications() {
            List<ReservationFleetInstanceSpecification.Builder> result = ReservationFleetInstanceSpecificationListCopier
                    .copyToBuilder(this.instanceTypeSpecifications);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setInstanceTypeSpecifications(
                Collection<ReservationFleetInstanceSpecification.BuilderImpl> instanceTypeSpecifications) {
            this.instanceTypeSpecifications = ReservationFleetInstanceSpecificationListCopier
                    .copyFromBuilder(instanceTypeSpecifications);
        }

        @Override
        public final Builder instanceTypeSpecifications(
                Collection<ReservationFleetInstanceSpecification> instanceTypeSpecifications) {
            this.instanceTypeSpecifications = ReservationFleetInstanceSpecificationListCopier.copy(instanceTypeSpecifications);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceTypeSpecifications(ReservationFleetInstanceSpecification... instanceTypeSpecifications) {
            instanceTypeSpecifications(Arrays.asList(instanceTypeSpecifications));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceTypeSpecifications(
                Consumer<ReservationFleetInstanceSpecification.Builder>... instanceTypeSpecifications) {
            instanceTypeSpecifications(Stream.of(instanceTypeSpecifications)
                    .map(c -> ReservationFleetInstanceSpecification.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getTenancy() {
            return tenancy;
        }

        public final void setTenancy(String tenancy) {
            this.tenancy = tenancy;
        }

        @Override
        public final Builder tenancy(String tenancy) {
            this.tenancy = tenancy;
            return this;
        }

        @Override
        public final Builder tenancy(FleetCapacityReservationTenancy tenancy) {
            this.tenancy(tenancy == null ? null : tenancy.toString());
            return this;
        }

        public final Integer getTotalTargetCapacity() {
            return totalTargetCapacity;
        }

        public final void setTotalTargetCapacity(Integer totalTargetCapacity) {
            this.totalTargetCapacity = totalTargetCapacity;
        }

        @Override
        public final Builder totalTargetCapacity(Integer totalTargetCapacity) {
            this.totalTargetCapacity = totalTargetCapacity;
            return this;
        }

        public final Instant getEndDate() {
            return endDate;
        }

        public final void setEndDate(Instant endDate) {
            this.endDate = endDate;
        }

        @Override
        public final Builder endDate(Instant endDate) {
            this.endDate = endDate;
            return this;
        }

        public final String getInstanceMatchCriteria() {
            return instanceMatchCriteria;
        }

        public final void setInstanceMatchCriteria(String instanceMatchCriteria) {
            this.instanceMatchCriteria = instanceMatchCriteria;
        }

        @Override
        public final Builder instanceMatchCriteria(String instanceMatchCriteria) {
            this.instanceMatchCriteria = instanceMatchCriteria;
            return this;
        }

        @Override
        public final Builder instanceMatchCriteria(FleetInstanceMatchCriteria instanceMatchCriteria) {
            this.instanceMatchCriteria(instanceMatchCriteria == null ? null : instanceMatchCriteria.toString());
            return this;
        }

        public final List<TagSpecification.Builder> getTagSpecifications() {
            List<TagSpecification.Builder> result = TagSpecificationListCopier.copyToBuilder(this.tagSpecifications);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTagSpecifications(Collection<TagSpecification.BuilderImpl> tagSpecifications) {
            this.tagSpecifications = TagSpecificationListCopier.copyFromBuilder(tagSpecifications);
        }

        @Override
        public final Builder tagSpecifications(Collection<TagSpecification> tagSpecifications) {
            this.tagSpecifications = TagSpecificationListCopier.copy(tagSpecifications);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagSpecifications(TagSpecification... tagSpecifications) {
            tagSpecifications(Arrays.asList(tagSpecifications));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagSpecifications(Consumer<TagSpecification.Builder>... tagSpecifications) {
            tagSpecifications(Stream.of(tagSpecifications).map(c -> TagSpecification.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Boolean getDryRun() {
            return dryRun;
        }

        public final void setDryRun(Boolean dryRun) {
            this.dryRun = dryRun;
        }

        @Override
        public final Builder dryRun(Boolean dryRun) {
            this.dryRun = dryRun;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateCapacityReservationFleetRequest build() {
            return new CreateCapacityReservationFleetRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
