/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a snapshot.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SnapshotInfo implements SdkPojo, Serializable, ToCopyableBuilder<SnapshotInfo.Builder, SnapshotInfo> {
    private static final SdkField<String> DESCRIPTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Description")
            .getter(getter(SnapshotInfo::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description")
                    .unmarshallLocationName("description").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(SnapshotInfo::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSet")
                    .unmarshallLocationName("tagSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<Boolean> ENCRYPTED_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Encrypted")
            .getter(getter(SnapshotInfo::encrypted))
            .setter(setter(Builder::encrypted))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Encrypted")
                    .unmarshallLocationName("encrypted").build()).build();

    private static final SdkField<String> VOLUME_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("VolumeId")
            .getter(getter(SnapshotInfo::volumeId))
            .setter(setter(Builder::volumeId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VolumeId")
                    .unmarshallLocationName("volumeId").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("State")
            .getter(getter(SnapshotInfo::stateAsString))
            .setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State")
                    .unmarshallLocationName("state").build()).build();

    private static final SdkField<Integer> VOLUME_SIZE_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("VolumeSize")
            .getter(getter(SnapshotInfo::volumeSize))
            .setter(setter(Builder::volumeSize))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VolumeSize")
                    .unmarshallLocationName("volumeSize").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartTime")
            .getter(getter(SnapshotInfo::startTime))
            .setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartTime")
                    .unmarshallLocationName("startTime").build()).build();

    private static final SdkField<String> PROGRESS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Progress")
            .getter(getter(SnapshotInfo::progress))
            .setter(setter(Builder::progress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Progress")
                    .unmarshallLocationName("progress").build()).build();

    private static final SdkField<String> OWNER_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("OwnerId")
            .getter(getter(SnapshotInfo::ownerId))
            .setter(setter(Builder::ownerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OwnerId")
                    .unmarshallLocationName("ownerId").build()).build();

    private static final SdkField<String> SNAPSHOT_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SnapshotId")
            .getter(getter(SnapshotInfo::snapshotId))
            .setter(setter(Builder::snapshotId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnapshotId")
                    .unmarshallLocationName("snapshotId").build()).build();

    private static final SdkField<String> OUTPOST_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("OutpostArn")
            .getter(getter(SnapshotInfo::outpostArn))
            .setter(setter(Builder::outpostArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutpostArn")
                    .unmarshallLocationName("outpostArn").build()).build();

    private static final SdkField<String> SSE_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SseType")
            .getter(getter(SnapshotInfo::sseTypeAsString))
            .setter(setter(Builder::sseType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SseType")
                    .unmarshallLocationName("sseType").build()).build();

    private static final SdkField<String> AVAILABILITY_ZONE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AvailabilityZone")
            .getter(getter(SnapshotInfo::availabilityZone))
            .setter(setter(Builder::availabilityZone))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityZone")
                    .unmarshallLocationName("availabilityZone").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DESCRIPTION_FIELD, TAGS_FIELD,
            ENCRYPTED_FIELD, VOLUME_ID_FIELD, STATE_FIELD, VOLUME_SIZE_FIELD, START_TIME_FIELD, PROGRESS_FIELD, OWNER_ID_FIELD,
            SNAPSHOT_ID_FIELD, OUTPOST_ARN_FIELD, SSE_TYPE_FIELD, AVAILABILITY_ZONE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String description;

    private final List<Tag> tags;

    private final Boolean encrypted;

    private final String volumeId;

    private final String state;

    private final Integer volumeSize;

    private final Instant startTime;

    private final String progress;

    private final String ownerId;

    private final String snapshotId;

    private final String outpostArn;

    private final String sseType;

    private final String availabilityZone;

    private SnapshotInfo(BuilderImpl builder) {
        this.description = builder.description;
        this.tags = builder.tags;
        this.encrypted = builder.encrypted;
        this.volumeId = builder.volumeId;
        this.state = builder.state;
        this.volumeSize = builder.volumeSize;
        this.startTime = builder.startTime;
        this.progress = builder.progress;
        this.ownerId = builder.ownerId;
        this.snapshotId = builder.snapshotId;
        this.outpostArn = builder.outpostArn;
        this.sseType = builder.sseType;
        this.availabilityZone = builder.availabilityZone;
    }

    /**
     * <p>
     * Description specified by the CreateSnapshotRequest that has been applied to all snapshots.
     * </p>
     * 
     * @return Description specified by the CreateSnapshotRequest that has been applied to all snapshots.
     */
    public final String description() {
        return description;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Tags associated with this snapshot.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Tags associated with this snapshot.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * Indicates whether the snapshot is encrypted.
     * </p>
     * 
     * @return Indicates whether the snapshot is encrypted.
     */
    public final Boolean encrypted() {
        return encrypted;
    }

    /**
     * <p>
     * Source volume from which this snapshot was created.
     * </p>
     * 
     * @return Source volume from which this snapshot was created.
     */
    public final String volumeId() {
        return volumeId;
    }

    /**
     * <p>
     * Current state of the snapshot.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link SnapshotState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return Current state of the snapshot.
     * @see SnapshotState
     */
    public final SnapshotState state() {
        return SnapshotState.fromValue(state);
    }

    /**
     * <p>
     * Current state of the snapshot.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link SnapshotState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return Current state of the snapshot.
     * @see SnapshotState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * Size of the volume from which this snapshot was created.
     * </p>
     * 
     * @return Size of the volume from which this snapshot was created.
     */
    public final Integer volumeSize() {
        return volumeSize;
    }

    /**
     * <p>
     * Time this snapshot was started. This is the same for all snapshots initiated by the same request.
     * </p>
     * 
     * @return Time this snapshot was started. This is the same for all snapshots initiated by the same request.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * Progress this snapshot has made towards completing.
     * </p>
     * 
     * @return Progress this snapshot has made towards completing.
     */
    public final String progress() {
        return progress;
    }

    /**
     * <p>
     * Account id used when creating this snapshot.
     * </p>
     * 
     * @return Account id used when creating this snapshot.
     */
    public final String ownerId() {
        return ownerId;
    }

    /**
     * <p>
     * Snapshot id that can be used to describe this snapshot.
     * </p>
     * 
     * @return Snapshot id that can be used to describe this snapshot.
     */
    public final String snapshotId() {
        return snapshotId;
    }

    /**
     * <p>
     * The ARN of the Outpost on which the snapshot is stored. For more information, see <a
     * href="https://docs.aws.amazon.com/ebs/latest/userguide/snapshots-outposts.html">Amazon EBS local snapshots on
     * Outposts</a> in the <i>Amazon EBS User Guide</i>.
     * </p>
     * 
     * @return The ARN of the Outpost on which the snapshot is stored. For more information, see <a
     *         href="https://docs.aws.amazon.com/ebs/latest/userguide/snapshots-outposts.html">Amazon EBS local
     *         snapshots on Outposts</a> in the <i>Amazon EBS User Guide</i>.
     */
    public final String outpostArn() {
        return outpostArn;
    }

    /**
     * <p>
     * Reserved for future use.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #sseType} will
     * return {@link SSEType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #sseTypeAsString}.
     * </p>
     * 
     * @return Reserved for future use.
     * @see SSEType
     */
    public final SSEType sseType() {
        return SSEType.fromValue(sseType);
    }

    /**
     * <p>
     * Reserved for future use.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #sseType} will
     * return {@link SSEType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #sseTypeAsString}.
     * </p>
     * 
     * @return Reserved for future use.
     * @see SSEType
     */
    public final String sseTypeAsString() {
        return sseType;
    }

    /**
     * <p>
     * The Availability Zone or Local Zone of the snapshots. For example, <code>us-west-1a</code> (Availability Zone) or
     * <code>us-west-2-lax-1a</code> (Local Zone).
     * </p>
     * 
     * @return The Availability Zone or Local Zone of the snapshots. For example, <code>us-west-1a</code> (Availability
     *         Zone) or <code>us-west-2-lax-1a</code> (Local Zone).
     */
    public final String availabilityZone() {
        return availabilityZone;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(encrypted());
        hashCode = 31 * hashCode + Objects.hashCode(volumeId());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(volumeSize());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(progress());
        hashCode = 31 * hashCode + Objects.hashCode(ownerId());
        hashCode = 31 * hashCode + Objects.hashCode(snapshotId());
        hashCode = 31 * hashCode + Objects.hashCode(outpostArn());
        hashCode = 31 * hashCode + Objects.hashCode(sseTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(availabilityZone());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SnapshotInfo)) {
            return false;
        }
        SnapshotInfo other = (SnapshotInfo) obj;
        return Objects.equals(description(), other.description()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(encrypted(), other.encrypted())
                && Objects.equals(volumeId(), other.volumeId()) && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(volumeSize(), other.volumeSize()) && Objects.equals(startTime(), other.startTime())
                && Objects.equals(progress(), other.progress()) && Objects.equals(ownerId(), other.ownerId())
                && Objects.equals(snapshotId(), other.snapshotId()) && Objects.equals(outpostArn(), other.outpostArn())
                && Objects.equals(sseTypeAsString(), other.sseTypeAsString())
                && Objects.equals(availabilityZone(), other.availabilityZone());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SnapshotInfo").add("Description", description()).add("Tags", hasTags() ? tags() : null)
                .add("Encrypted", encrypted()).add("VolumeId", volumeId()).add("State", stateAsString())
                .add("VolumeSize", volumeSize()).add("StartTime", startTime()).add("Progress", progress())
                .add("OwnerId", ownerId()).add("SnapshotId", snapshotId()).add("OutpostArn", outpostArn())
                .add("SseType", sseTypeAsString()).add("AvailabilityZone", availabilityZone()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "Encrypted":
            return Optional.ofNullable(clazz.cast(encrypted()));
        case "VolumeId":
            return Optional.ofNullable(clazz.cast(volumeId()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "VolumeSize":
            return Optional.ofNullable(clazz.cast(volumeSize()));
        case "StartTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "Progress":
            return Optional.ofNullable(clazz.cast(progress()));
        case "OwnerId":
            return Optional.ofNullable(clazz.cast(ownerId()));
        case "SnapshotId":
            return Optional.ofNullable(clazz.cast(snapshotId()));
        case "OutpostArn":
            return Optional.ofNullable(clazz.cast(outpostArn()));
        case "SseType":
            return Optional.ofNullable(clazz.cast(sseTypeAsString()));
        case "AvailabilityZone":
            return Optional.ofNullable(clazz.cast(availabilityZone()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Description", DESCRIPTION_FIELD);
        map.put("TagSet", TAGS_FIELD);
        map.put("Encrypted", ENCRYPTED_FIELD);
        map.put("VolumeId", VOLUME_ID_FIELD);
        map.put("State", STATE_FIELD);
        map.put("VolumeSize", VOLUME_SIZE_FIELD);
        map.put("StartTime", START_TIME_FIELD);
        map.put("Progress", PROGRESS_FIELD);
        map.put("OwnerId", OWNER_ID_FIELD);
        map.put("SnapshotId", SNAPSHOT_ID_FIELD);
        map.put("OutpostArn", OUTPOST_ARN_FIELD);
        map.put("SseType", SSE_TYPE_FIELD);
        map.put("AvailabilityZone", AVAILABILITY_ZONE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<SnapshotInfo, T> g) {
        return obj -> g.apply((SnapshotInfo) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SnapshotInfo> {
        /**
         * <p>
         * Description specified by the CreateSnapshotRequest that has been applied to all snapshots.
         * </p>
         * 
         * @param description
         *        Description specified by the CreateSnapshotRequest that has been applied to all snapshots.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * Tags associated with this snapshot.
         * </p>
         * 
         * @param tags
         *        Tags associated with this snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Tags associated with this snapshot.
         * </p>
         * 
         * @param tags
         *        Tags associated with this snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Tags associated with this snapshot.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ec2.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ec2.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ec2.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * Indicates whether the snapshot is encrypted.
         * </p>
         * 
         * @param encrypted
         *        Indicates whether the snapshot is encrypted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encrypted(Boolean encrypted);

        /**
         * <p>
         * Source volume from which this snapshot was created.
         * </p>
         * 
         * @param volumeId
         *        Source volume from which this snapshot was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumeId(String volumeId);

        /**
         * <p>
         * Current state of the snapshot.
         * </p>
         * 
         * @param state
         *        Current state of the snapshot.
         * @see SnapshotState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SnapshotState
         */
        Builder state(String state);

        /**
         * <p>
         * Current state of the snapshot.
         * </p>
         * 
         * @param state
         *        Current state of the snapshot.
         * @see SnapshotState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SnapshotState
         */
        Builder state(SnapshotState state);

        /**
         * <p>
         * Size of the volume from which this snapshot was created.
         * </p>
         * 
         * @param volumeSize
         *        Size of the volume from which this snapshot was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumeSize(Integer volumeSize);

        /**
         * <p>
         * Time this snapshot was started. This is the same for all snapshots initiated by the same request.
         * </p>
         * 
         * @param startTime
         *        Time this snapshot was started. This is the same for all snapshots initiated by the same request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * Progress this snapshot has made towards completing.
         * </p>
         * 
         * @param progress
         *        Progress this snapshot has made towards completing.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder progress(String progress);

        /**
         * <p>
         * Account id used when creating this snapshot.
         * </p>
         * 
         * @param ownerId
         *        Account id used when creating this snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ownerId(String ownerId);

        /**
         * <p>
         * Snapshot id that can be used to describe this snapshot.
         * </p>
         * 
         * @param snapshotId
         *        Snapshot id that can be used to describe this snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snapshotId(String snapshotId);

        /**
         * <p>
         * The ARN of the Outpost on which the snapshot is stored. For more information, see <a
         * href="https://docs.aws.amazon.com/ebs/latest/userguide/snapshots-outposts.html">Amazon EBS local snapshots on
         * Outposts</a> in the <i>Amazon EBS User Guide</i>.
         * </p>
         * 
         * @param outpostArn
         *        The ARN of the Outpost on which the snapshot is stored. For more information, see <a
         *        href="https://docs.aws.amazon.com/ebs/latest/userguide/snapshots-outposts.html">Amazon EBS local
         *        snapshots on Outposts</a> in the <i>Amazon EBS User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outpostArn(String outpostArn);

        /**
         * <p>
         * Reserved for future use.
         * </p>
         * 
         * @param sseType
         *        Reserved for future use.
         * @see SSEType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SSEType
         */
        Builder sseType(String sseType);

        /**
         * <p>
         * Reserved for future use.
         * </p>
         * 
         * @param sseType
         *        Reserved for future use.
         * @see SSEType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SSEType
         */
        Builder sseType(SSEType sseType);

        /**
         * <p>
         * The Availability Zone or Local Zone of the snapshots. For example, <code>us-west-1a</code> (Availability
         * Zone) or <code>us-west-2-lax-1a</code> (Local Zone).
         * </p>
         * 
         * @param availabilityZone
         *        The Availability Zone or Local Zone of the snapshots. For example, <code>us-west-1a</code>
         *        (Availability Zone) or <code>us-west-2-lax-1a</code> (Local Zone).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZone(String availabilityZone);
    }

    static final class BuilderImpl implements Builder {
        private String description;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private Boolean encrypted;

        private String volumeId;

        private String state;

        private Integer volumeSize;

        private Instant startTime;

        private String progress;

        private String ownerId;

        private String snapshotId;

        private String outpostArn;

        private String sseType;

        private String availabilityZone;

        private BuilderImpl() {
        }

        private BuilderImpl(SnapshotInfo model) {
            description(model.description);
            tags(model.tags);
            encrypted(model.encrypted);
            volumeId(model.volumeId);
            state(model.state);
            volumeSize(model.volumeSize);
            startTime(model.startTime);
            progress(model.progress);
            ownerId(model.ownerId);
            snapshotId(model.snapshotId);
            outpostArn(model.outpostArn);
            sseType(model.sseType);
            availabilityZone(model.availabilityZone);
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Boolean getEncrypted() {
            return encrypted;
        }

        public final void setEncrypted(Boolean encrypted) {
            this.encrypted = encrypted;
        }

        @Override
        public final Builder encrypted(Boolean encrypted) {
            this.encrypted = encrypted;
            return this;
        }

        public final String getVolumeId() {
            return volumeId;
        }

        public final void setVolumeId(String volumeId) {
            this.volumeId = volumeId;
        }

        @Override
        public final Builder volumeId(String volumeId) {
            this.volumeId = volumeId;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(SnapshotState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final Integer getVolumeSize() {
            return volumeSize;
        }

        public final void setVolumeSize(Integer volumeSize) {
            this.volumeSize = volumeSize;
        }

        @Override
        public final Builder volumeSize(Integer volumeSize) {
            this.volumeSize = volumeSize;
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final String getProgress() {
            return progress;
        }

        public final void setProgress(String progress) {
            this.progress = progress;
        }

        @Override
        public final Builder progress(String progress) {
            this.progress = progress;
            return this;
        }

        public final String getOwnerId() {
            return ownerId;
        }

        public final void setOwnerId(String ownerId) {
            this.ownerId = ownerId;
        }

        @Override
        public final Builder ownerId(String ownerId) {
            this.ownerId = ownerId;
            return this;
        }

        public final String getSnapshotId() {
            return snapshotId;
        }

        public final void setSnapshotId(String snapshotId) {
            this.snapshotId = snapshotId;
        }

        @Override
        public final Builder snapshotId(String snapshotId) {
            this.snapshotId = snapshotId;
            return this;
        }

        public final String getOutpostArn() {
            return outpostArn;
        }

        public final void setOutpostArn(String outpostArn) {
            this.outpostArn = outpostArn;
        }

        @Override
        public final Builder outpostArn(String outpostArn) {
            this.outpostArn = outpostArn;
            return this;
        }

        public final String getSseType() {
            return sseType;
        }

        public final void setSseType(String sseType) {
            this.sseType = sseType;
        }

        @Override
        public final Builder sseType(String sseType) {
            this.sseType = sseType;
            return this;
        }

        @Override
        public final Builder sseType(SSEType sseType) {
            this.sseType(sseType == null ? null : sseType.toString());
            return this;
        }

        public final String getAvailabilityZone() {
            return availabilityZone;
        }

        public final void setAvailabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
        }

        @Override
        public final Builder availabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
            return this;
        }

        @Override
        public SnapshotInfo build() {
            return new SnapshotInfo(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
