/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an EC2 Fleet or Spot Fleet event.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EventInformation implements SdkPojo, Serializable,
        ToCopyableBuilder<EventInformation.Builder, EventInformation> {
    private static final SdkField<String> EVENT_DESCRIPTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("EventDescription")
            .getter(getter(EventInformation::eventDescription))
            .setter(setter(Builder::eventDescription))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventDescription")
                    .unmarshallLocationName("eventDescription").build()).build();

    private static final SdkField<String> EVENT_SUB_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("EventSubType")
            .getter(getter(EventInformation::eventSubType))
            .setter(setter(Builder::eventSubType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventSubType")
                    .unmarshallLocationName("eventSubType").build()).build();

    private static final SdkField<String> INSTANCE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("InstanceId")
            .getter(getter(EventInformation::instanceId))
            .setter(setter(Builder::instanceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceId")
                    .unmarshallLocationName("instanceId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EVENT_DESCRIPTION_FIELD,
            EVENT_SUB_TYPE_FIELD, INSTANCE_ID_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String eventDescription;

    private final String eventSubType;

    private final String instanceId;

    private EventInformation(BuilderImpl builder) {
        this.eventDescription = builder.eventDescription;
        this.eventSubType = builder.eventSubType;
        this.instanceId = builder.instanceId;
    }

    /**
     * <p>
     * The description of the event.
     * </p>
     * 
     * @return The description of the event.
     */
    public final String eventDescription() {
        return eventDescription;
    }

    /**
     * <p>
     * The event.
     * </p>
     * <p>
     * <code>error</code> events:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>iamFleetRoleInvalid</code> - The EC2 Fleet or Spot Fleet does not have the required permissions either to
     * launch or terminate an instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>allLaunchSpecsTemporarilyBlacklisted</code> - None of the configurations are valid, and several attempts to
     * launch instances have failed. For more information, see the description of the event.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>spotInstanceCountLimitExceeded</code> - You've reached the limit on the number of Spot Instances that you
     * can launch.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>spotFleetRequestConfigurationInvalid</code> - The configuration is not valid. For more information, see the
     * description of the event.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <code>fleetRequestChange</code> events:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>active</code> - The EC2 Fleet or Spot Fleet request has been validated and Amazon EC2 is attempting to
     * maintain the target number of running instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>deleted</code> (EC2 Fleet) / <code>cancelled</code> (Spot Fleet) - The EC2 Fleet is deleted or the Spot
     * Fleet request is canceled and has no running instances. The EC2 Fleet or Spot Fleet will be deleted two days
     * after its instances are terminated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>deleted_running</code> (EC2 Fleet) / <code>cancelled_running</code> (Spot Fleet) - The EC2 Fleet is deleted
     * or the Spot Fleet request is canceled and does not launch additional instances. Its existing instances continue
     * to run until they are interrupted or terminated. The request remains in this state until all instances are
     * interrupted or terminated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>deleted_terminating</code> (EC2 Fleet) / <code>cancelled_terminating</code> (Spot Fleet) - The EC2 Fleet is
     * deleted or the Spot Fleet request is canceled and its instances are terminating. The request remains in this
     * state until all instances are terminated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>expired</code> - The EC2 Fleet or Spot Fleet request has expired. If the request was created with
     * <code>TerminateInstancesWithExpiration</code> set, a subsequent <code>terminated</code> event indicates that the
     * instances are terminated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>modify_in_progress</code> - The EC2 Fleet or Spot Fleet request is being modified. The request remains in
     * this state until the modification is fully processed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>modify_succeeded</code> - The EC2 Fleet or Spot Fleet request was modified.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>submitted</code> - The EC2 Fleet or Spot Fleet request is being evaluated and Amazon EC2 is preparing to
     * launch the target number of instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>progress</code> - The EC2 Fleet or Spot Fleet request is in the process of being fulfilled.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <code>instanceChange</code> events:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>launched</code> - A new instance was launched.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>terminated</code> - An instance was terminated by the user.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>termination_notified</code> - An instance termination notification was sent when a Spot Instance was
     * terminated by Amazon EC2 during scale-down, when the target capacity of the fleet was modified down, for example,
     * from a target capacity of 4 to a target capacity of 3.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <code>Information</code> events:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>fleetProgressHalted</code> - The price in every launch specification is not valid because it is below the
     * Spot price (all the launch specifications have produced <code>launchSpecUnusable</code> events). A launch
     * specification might become valid if the Spot price changes.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>launchSpecTemporarilyBlacklisted</code> - The configuration is not valid and several attempts to launch
     * instances have failed. For more information, see the description of the event.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>launchSpecUnusable</code> - The price specified in a launch specification is not valid because it is below
     * the Spot price for the requested Spot pools.
     * </p>
     * <p>
     * Note: Even if a fleet with the <code>maintain</code> request type is in the process of being canceled, it may
     * still publish a <code>launchSpecUnusable</code> event. This does not mean that the canceled fleet is attempting
     * to launch a new instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>registerWithLoadBalancersFailed</code> - An attempt to register instances with load balancers failed. For
     * more information, see the description of the event.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The event.</p>
     *         <p>
     *         <code>error</code> events:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>iamFleetRoleInvalid</code> - The EC2 Fleet or Spot Fleet does not have the required permissions
     *         either to launch or terminate an instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>allLaunchSpecsTemporarilyBlacklisted</code> - None of the configurations are valid, and several
     *         attempts to launch instances have failed. For more information, see the description of the event.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>spotInstanceCountLimitExceeded</code> - You've reached the limit on the number of Spot Instances
     *         that you can launch.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>spotFleetRequestConfigurationInvalid</code> - The configuration is not valid. For more information,
     *         see the description of the event.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <code>fleetRequestChange</code> events:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>active</code> - The EC2 Fleet or Spot Fleet request has been validated and Amazon EC2 is attempting
     *         to maintain the target number of running instances.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>deleted</code> (EC2 Fleet) / <code>cancelled</code> (Spot Fleet) - The EC2 Fleet is deleted or the
     *         Spot Fleet request is canceled and has no running instances. The EC2 Fleet or Spot Fleet will be deleted
     *         two days after its instances are terminated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>deleted_running</code> (EC2 Fleet) / <code>cancelled_running</code> (Spot Fleet) - The EC2 Fleet is
     *         deleted or the Spot Fleet request is canceled and does not launch additional instances. Its existing
     *         instances continue to run until they are interrupted or terminated. The request remains in this state
     *         until all instances are interrupted or terminated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>deleted_terminating</code> (EC2 Fleet) / <code>cancelled_terminating</code> (Spot Fleet) - The EC2
     *         Fleet is deleted or the Spot Fleet request is canceled and its instances are terminating. The request
     *         remains in this state until all instances are terminated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>expired</code> - The EC2 Fleet or Spot Fleet request has expired. If the request was created with
     *         <code>TerminateInstancesWithExpiration</code> set, a subsequent <code>terminated</code> event indicates
     *         that the instances are terminated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>modify_in_progress</code> - The EC2 Fleet or Spot Fleet request is being modified. The request
     *         remains in this state until the modification is fully processed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>modify_succeeded</code> - The EC2 Fleet or Spot Fleet request was modified.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>submitted</code> - The EC2 Fleet or Spot Fleet request is being evaluated and Amazon EC2 is
     *         preparing to launch the target number of instances.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>progress</code> - The EC2 Fleet or Spot Fleet request is in the process of being fulfilled.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <code>instanceChange</code> events:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>launched</code> - A new instance was launched.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>terminated</code> - An instance was terminated by the user.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>termination_notified</code> - An instance termination notification was sent when a Spot Instance
     *         was terminated by Amazon EC2 during scale-down, when the target capacity of the fleet was modified down,
     *         for example, from a target capacity of 4 to a target capacity of 3.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <code>Information</code> events:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>fleetProgressHalted</code> - The price in every launch specification is not valid because it is
     *         below the Spot price (all the launch specifications have produced <code>launchSpecUnusable</code>
     *         events). A launch specification might become valid if the Spot price changes.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>launchSpecTemporarilyBlacklisted</code> - The configuration is not valid and several attempts to
     *         launch instances have failed. For more information, see the description of the event.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>launchSpecUnusable</code> - The price specified in a launch specification is not valid because it
     *         is below the Spot price for the requested Spot pools.
     *         </p>
     *         <p>
     *         Note: Even if a fleet with the <code>maintain</code> request type is in the process of being canceled, it
     *         may still publish a <code>launchSpecUnusable</code> event. This does not mean that the canceled fleet is
     *         attempting to launch a new instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>registerWithLoadBalancersFailed</code> - An attempt to register instances with load balancers
     *         failed. For more information, see the description of the event.
     *         </p>
     *         </li>
     */
    public final String eventSubType() {
        return eventSubType;
    }

    /**
     * <p>
     * The ID of the instance. This information is available only for <code>instanceChange</code> events.
     * </p>
     * 
     * @return The ID of the instance. This information is available only for <code>instanceChange</code> events.
     */
    public final String instanceId() {
        return instanceId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(eventDescription());
        hashCode = 31 * hashCode + Objects.hashCode(eventSubType());
        hashCode = 31 * hashCode + Objects.hashCode(instanceId());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EventInformation)) {
            return false;
        }
        EventInformation other = (EventInformation) obj;
        return Objects.equals(eventDescription(), other.eventDescription())
                && Objects.equals(eventSubType(), other.eventSubType()) && Objects.equals(instanceId(), other.instanceId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EventInformation").add("EventDescription", eventDescription())
                .add("EventSubType", eventSubType()).add("InstanceId", instanceId()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "EventDescription":
            return Optional.ofNullable(clazz.cast(eventDescription()));
        case "EventSubType":
            return Optional.ofNullable(clazz.cast(eventSubType()));
        case "InstanceId":
            return Optional.ofNullable(clazz.cast(instanceId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("EventDescription", EVENT_DESCRIPTION_FIELD);
        map.put("EventSubType", EVENT_SUB_TYPE_FIELD);
        map.put("InstanceId", INSTANCE_ID_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<EventInformation, T> g) {
        return obj -> g.apply((EventInformation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EventInformation> {
        /**
         * <p>
         * The description of the event.
         * </p>
         * 
         * @param eventDescription
         *        The description of the event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventDescription(String eventDescription);

        /**
         * <p>
         * The event.
         * </p>
         * <p>
         * <code>error</code> events:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>iamFleetRoleInvalid</code> - The EC2 Fleet or Spot Fleet does not have the required permissions either
         * to launch or terminate an instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>allLaunchSpecsTemporarilyBlacklisted</code> - None of the configurations are valid, and several
         * attempts to launch instances have failed. For more information, see the description of the event.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>spotInstanceCountLimitExceeded</code> - You've reached the limit on the number of Spot Instances that
         * you can launch.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>spotFleetRequestConfigurationInvalid</code> - The configuration is not valid. For more information, see
         * the description of the event.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <code>fleetRequestChange</code> events:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>active</code> - The EC2 Fleet or Spot Fleet request has been validated and Amazon EC2 is attempting to
         * maintain the target number of running instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>deleted</code> (EC2 Fleet) / <code>cancelled</code> (Spot Fleet) - The EC2 Fleet is deleted or the Spot
         * Fleet request is canceled and has no running instances. The EC2 Fleet or Spot Fleet will be deleted two days
         * after its instances are terminated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>deleted_running</code> (EC2 Fleet) / <code>cancelled_running</code> (Spot Fleet) - The EC2 Fleet is
         * deleted or the Spot Fleet request is canceled and does not launch additional instances. Its existing
         * instances continue to run until they are interrupted or terminated. The request remains in this state until
         * all instances are interrupted or terminated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>deleted_terminating</code> (EC2 Fleet) / <code>cancelled_terminating</code> (Spot Fleet) - The EC2
         * Fleet is deleted or the Spot Fleet request is canceled and its instances are terminating. The request remains
         * in this state until all instances are terminated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>expired</code> - The EC2 Fleet or Spot Fleet request has expired. If the request was created with
         * <code>TerminateInstancesWithExpiration</code> set, a subsequent <code>terminated</code> event indicates that
         * the instances are terminated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>modify_in_progress</code> - The EC2 Fleet or Spot Fleet request is being modified. The request remains
         * in this state until the modification is fully processed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>modify_succeeded</code> - The EC2 Fleet or Spot Fleet request was modified.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>submitted</code> - The EC2 Fleet or Spot Fleet request is being evaluated and Amazon EC2 is preparing
         * to launch the target number of instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>progress</code> - The EC2 Fleet or Spot Fleet request is in the process of being fulfilled.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <code>instanceChange</code> events:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>launched</code> - A new instance was launched.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>terminated</code> - An instance was terminated by the user.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>termination_notified</code> - An instance termination notification was sent when a Spot Instance was
         * terminated by Amazon EC2 during scale-down, when the target capacity of the fleet was modified down, for
         * example, from a target capacity of 4 to a target capacity of 3.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <code>Information</code> events:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>fleetProgressHalted</code> - The price in every launch specification is not valid because it is below
         * the Spot price (all the launch specifications have produced <code>launchSpecUnusable</code> events). A launch
         * specification might become valid if the Spot price changes.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>launchSpecTemporarilyBlacklisted</code> - The configuration is not valid and several attempts to launch
         * instances have failed. For more information, see the description of the event.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>launchSpecUnusable</code> - The price specified in a launch specification is not valid because it is
         * below the Spot price for the requested Spot pools.
         * </p>
         * <p>
         * Note: Even if a fleet with the <code>maintain</code> request type is in the process of being canceled, it may
         * still publish a <code>launchSpecUnusable</code> event. This does not mean that the canceled fleet is
         * attempting to launch a new instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>registerWithLoadBalancersFailed</code> - An attempt to register instances with load balancers failed.
         * For more information, see the description of the event.
         * </p>
         * </li>
         * </ul>
         * 
         * @param eventSubType
         *        The event.</p>
         *        <p>
         *        <code>error</code> events:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>iamFleetRoleInvalid</code> - The EC2 Fleet or Spot Fleet does not have the required permissions
         *        either to launch or terminate an instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>allLaunchSpecsTemporarilyBlacklisted</code> - None of the configurations are valid, and several
         *        attempts to launch instances have failed. For more information, see the description of the event.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>spotInstanceCountLimitExceeded</code> - You've reached the limit on the number of Spot Instances
         *        that you can launch.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>spotFleetRequestConfigurationInvalid</code> - The configuration is not valid. For more
         *        information, see the description of the event.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <code>fleetRequestChange</code> events:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>active</code> - The EC2 Fleet or Spot Fleet request has been validated and Amazon EC2 is
         *        attempting to maintain the target number of running instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>deleted</code> (EC2 Fleet) / <code>cancelled</code> (Spot Fleet) - The EC2 Fleet is deleted or
         *        the Spot Fleet request is canceled and has no running instances. The EC2 Fleet or Spot Fleet will be
         *        deleted two days after its instances are terminated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>deleted_running</code> (EC2 Fleet) / <code>cancelled_running</code> (Spot Fleet) - The EC2 Fleet
         *        is deleted or the Spot Fleet request is canceled and does not launch additional instances. Its
         *        existing instances continue to run until they are interrupted or terminated. The request remains in
         *        this state until all instances are interrupted or terminated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>deleted_terminating</code> (EC2 Fleet) / <code>cancelled_terminating</code> (Spot Fleet) - The
         *        EC2 Fleet is deleted or the Spot Fleet request is canceled and its instances are terminating. The
         *        request remains in this state until all instances are terminated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>expired</code> - The EC2 Fleet or Spot Fleet request has expired. If the request was created
         *        with <code>TerminateInstancesWithExpiration</code> set, a subsequent <code>terminated</code> event
         *        indicates that the instances are terminated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>modify_in_progress</code> - The EC2 Fleet or Spot Fleet request is being modified. The request
         *        remains in this state until the modification is fully processed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>modify_succeeded</code> - The EC2 Fleet or Spot Fleet request was modified.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>submitted</code> - The EC2 Fleet or Spot Fleet request is being evaluated and Amazon EC2 is
         *        preparing to launch the target number of instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>progress</code> - The EC2 Fleet or Spot Fleet request is in the process of being fulfilled.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <code>instanceChange</code> events:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>launched</code> - A new instance was launched.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>terminated</code> - An instance was terminated by the user.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>termination_notified</code> - An instance termination notification was sent when a Spot Instance
         *        was terminated by Amazon EC2 during scale-down, when the target capacity of the fleet was modified
         *        down, for example, from a target capacity of 4 to a target capacity of 3.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <code>Information</code> events:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>fleetProgressHalted</code> - The price in every launch specification is not valid because it is
         *        below the Spot price (all the launch specifications have produced <code>launchSpecUnusable</code>
         *        events). A launch specification might become valid if the Spot price changes.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>launchSpecTemporarilyBlacklisted</code> - The configuration is not valid and several attempts to
         *        launch instances have failed. For more information, see the description of the event.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>launchSpecUnusable</code> - The price specified in a launch specification is not valid because
         *        it is below the Spot price for the requested Spot pools.
         *        </p>
         *        <p>
         *        Note: Even if a fleet with the <code>maintain</code> request type is in the process of being canceled,
         *        it may still publish a <code>launchSpecUnusable</code> event. This does not mean that the canceled
         *        fleet is attempting to launch a new instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>registerWithLoadBalancersFailed</code> - An attempt to register instances with load balancers
         *        failed. For more information, see the description of the event.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventSubType(String eventSubType);

        /**
         * <p>
         * The ID of the instance. This information is available only for <code>instanceChange</code> events.
         * </p>
         * 
         * @param instanceId
         *        The ID of the instance. This information is available only for <code>instanceChange</code> events.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceId(String instanceId);
    }

    static final class BuilderImpl implements Builder {
        private String eventDescription;

        private String eventSubType;

        private String instanceId;

        private BuilderImpl() {
        }

        private BuilderImpl(EventInformation model) {
            eventDescription(model.eventDescription);
            eventSubType(model.eventSubType);
            instanceId(model.instanceId);
        }

        public final String getEventDescription() {
            return eventDescription;
        }

        public final void setEventDescription(String eventDescription) {
            this.eventDescription = eventDescription;
        }

        @Override
        public final Builder eventDescription(String eventDescription) {
            this.eventDescription = eventDescription;
            return this;
        }

        public final String getEventSubType() {
            return eventSubType;
        }

        public final void setEventSubType(String eventSubType) {
            this.eventSubType = eventSubType;
        }

        @Override
        public final Builder eventSubType(String eventSubType) {
            this.eventSubType = eventSubType;
            return this;
        }

        public final String getInstanceId() {
            return instanceId;
        }

        public final void setInstanceId(String instanceId) {
            this.instanceId = instanceId;
        }

        @Override
        public final Builder instanceId(String instanceId) {
            this.instanceId = instanceId;
            return this;
        }

        @Override
        public EventInformation build() {
            return new EventInformation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
