/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class DescribeInstanceTypesRequest extends Ec2Request implements
        ToCopyableBuilder<DescribeInstanceTypesRequest.Builder, DescribeInstanceTypesRequest> {
    private static final SdkField<Boolean> DRY_RUN_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("DryRun")
            .getter(getter(DescribeInstanceTypesRequest::dryRun))
            .setter(setter(Builder::dryRun))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DryRun")
                    .unmarshallLocationName("DryRun").build()).build();

    private static final SdkField<List<String>> INSTANCE_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("InstanceTypes")
            .getter(getter(DescribeInstanceTypesRequest::instanceTypesAsStrings))
            .setter(setter(Builder::instanceTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceType")
                    .unmarshallLocationName("InstanceType").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").unmarshallLocationName("member").build()).build())
                            .build()).build();

    private static final SdkField<List<Filter>> FILTERS_FIELD = SdkField
            .<List<Filter>> builder(MarshallingType.LIST)
            .memberName("Filters")
            .getter(getter(DescribeInstanceTypesRequest::filters))
            .setter(setter(Builder::filters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Filter")
                    .unmarshallLocationName("Filter").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("Filter")
                            .memberFieldInfo(
                                    SdkField.<Filter> builder(MarshallingType.SDK_POJO)
                                            .constructor(Filter::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Filter").unmarshallLocationName("Filter").build()).build())
                            .build()).build();

    private static final SdkField<Integer> MAX_RESULTS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxResults")
            .getter(getter(DescribeInstanceTypesRequest::maxResults))
            .setter(setter(Builder::maxResults))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxResults")
                    .unmarshallLocationName("MaxResults").build()).build();

    private static final SdkField<String> NEXT_TOKEN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("NextToken")
            .getter(getter(DescribeInstanceTypesRequest::nextToken))
            .setter(setter(Builder::nextToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NextToken")
                    .unmarshallLocationName("NextToken").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DRY_RUN_FIELD,
            INSTANCE_TYPES_FIELD, FILTERS_FIELD, MAX_RESULTS_FIELD, NEXT_TOKEN_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("DryRun", DRY_RUN_FIELD);
                    put("InstanceType", INSTANCE_TYPES_FIELD);
                    put("Filter", FILTERS_FIELD);
                    put("MaxResults", MAX_RESULTS_FIELD);
                    put("NextToken", NEXT_TOKEN_FIELD);
                }
            });

    private final Boolean dryRun;

    private final List<String> instanceTypes;

    private final List<Filter> filters;

    private final Integer maxResults;

    private final String nextToken;

    private DescribeInstanceTypesRequest(BuilderImpl builder) {
        super(builder);
        this.dryRun = builder.dryRun;
        this.instanceTypes = builder.instanceTypes;
        this.filters = builder.filters;
        this.maxResults = builder.maxResults;
        this.nextToken = builder.nextToken;
    }

    /**
     * <p>
     * Checks whether you have the required permissions for the action, without actually making the request, and
     * provides an error response. If you have the required permissions, the error response is
     * <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
     * </p>
     * 
     * @return Checks whether you have the required permissions for the action, without actually making the request, and
     *         provides an error response. If you have the required permissions, the error response is
     *         <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
     */
    public final Boolean dryRun() {
        return dryRun;
    }

    /**
     * <p>
     * The instance types.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInstanceTypes} method.
     * </p>
     * 
     * @return The instance types.
     */
    public final List<InstanceType> instanceTypes() {
        return RequestInstanceTypeListCopier.copyStringToEnum(instanceTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the InstanceTypes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasInstanceTypes() {
        return instanceTypes != null && !(instanceTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The instance types.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInstanceTypes} method.
     * </p>
     * 
     * @return The instance types.
     */
    public final List<String> instanceTypesAsStrings() {
        return instanceTypes;
    }

    /**
     * For responses, this returns true if the service returned a value for the Filters property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasFilters() {
        return filters != null && !(filters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * One or more filters. Filter names and values are case-sensitive.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>auto-recovery-supported</code> - Indicates whether Amazon CloudWatch action based recovery is supported (
     * <code>true</code> | <code>false</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>bare-metal</code> - Indicates whether it is a bare metal instance type (<code>true</code> |
     * <code>false</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>burstable-performance-supported</code> - Indicates whether the instance type is a burstable performance T
     * instance type (<code>true</code> | <code>false</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>current-generation</code> - Indicates whether this instance type is the latest generation instance type of
     * an instance family (<code>true</code> | <code>false</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ebs-info.ebs-optimized-info.baseline-bandwidth-in-mbps</code> - The baseline bandwidth performance for an
     * EBS-optimized instance type, in Mbps.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ebs-info.ebs-optimized-info.baseline-iops</code> - The baseline input/output storage operations per second
     * for an EBS-optimized instance type.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ebs-info.ebs-optimized-info.baseline-throughput-in-mbps</code> - The baseline throughput performance for an
     * EBS-optimized instance type, in MB/s.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ebs-info.ebs-optimized-info.maximum-bandwidth-in-mbps</code> - The maximum bandwidth performance for an
     * EBS-optimized instance type, in Mbps.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ebs-info.ebs-optimized-info.maximum-iops</code> - The maximum input/output storage operations per second
     * for an EBS-optimized instance type.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ebs-info.ebs-optimized-info.maximum-throughput-in-mbps</code> - The maximum throughput performance for an
     * EBS-optimized instance type, in MB/s.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ebs-info.ebs-optimized-support</code> - Indicates whether the instance type is EBS-optimized (
     * <code>supported</code> | <code>unsupported</code> | <code>default</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ebs-info.encryption-support</code> - Indicates whether EBS encryption is supported (<code>supported</code>
     * | <code>unsupported</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ebs-info.nvme-support</code> - Indicates whether non-volatile memory express (NVMe) is supported for EBS
     * volumes (<code>required</code> | <code>supported</code> | <code>unsupported</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>free-tier-eligible</code> - Indicates whether the instance type is eligible to use in the free tier (
     * <code>true</code> | <code>false</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>hibernation-supported</code> - Indicates whether On-Demand hibernation is supported (<code>true</code> |
     * <code>false</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>hypervisor</code> - The hypervisor (<code>nitro</code> | <code>xen</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>instance-storage-info.disk.count</code> - The number of local disks.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>instance-storage-info.disk.size-in-gb</code> - The storage size of each instance storage disk, in GB.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>instance-storage-info.disk.type</code> - The storage technology for the local instance storage disks (
     * <code>hdd</code> | <code>ssd</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>instance-storage-info.encryption-support</code> - Indicates whether data is encrypted at rest (
     * <code>required</code> | <code>supported</code> | <code>unsupported</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>instance-storage-info.nvme-support</code> - Indicates whether non-volatile memory express (NVMe) is
     * supported for instance store (<code>required</code> | <code>supported</code> | <code>unsupported</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>instance-storage-info.total-size-in-gb</code> - The total amount of storage available from all local
     * instance storage, in GB.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>instance-storage-supported</code> - Indicates whether the instance type has local instance storage (
     * <code>true</code> | <code>false</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>instance-type</code> - The instance type (for example <code>c5.2xlarge</code> or c5*).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>memory-info.size-in-mib</code> - The memory size.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>network-info.efa-info.maximum-efa-interfaces</code> - The maximum number of Elastic Fabric Adapters (EFAs)
     * per instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>network-info.efa-supported</code> - Indicates whether the instance type supports Elastic Fabric Adapter
     * (EFA) (<code>true</code> | <code>false</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>network-info.ena-support</code> - Indicates whether Elastic Network Adapter (ENA) is supported or required
     * (<code>required</code> | <code>supported</code> | <code>unsupported</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>network-info.encryption-in-transit-supported</code> - Indicates whether the instance type automatically
     * encrypts in-transit traffic between instances (<code>true</code> | <code>false</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>network-info.ipv4-addresses-per-interface</code> - The maximum number of private IPv4 addresses per network
     * interface.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>network-info.ipv6-addresses-per-interface</code> - The maximum number of private IPv6 addresses per network
     * interface.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>network-info.ipv6-supported</code> - Indicates whether the instance type supports IPv6 (<code>true</code> |
     * <code>false</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>network-info.maximum-network-cards</code> - The maximum number of network cards per instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>network-info.maximum-network-interfaces</code> - The maximum number of network interfaces per instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>network-info.network-performance</code> - The network performance (for example, "25 Gigabit").
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>nitro-enclaves-support</code> - Indicates whether Nitro Enclaves is supported (<code>supported</code> |
     * <code>unsupported</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>nitro-tpm-support</code> - Indicates whether NitroTPM is supported (<code>supported</code> |
     * <code>unsupported</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>nitro-tpm-info.supported-versions</code> - The supported NitroTPM version (<code>2.0</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>processor-info.supported-architecture</code> - The CPU architecture (<code>arm64</code> | <code>i386</code>
     * | <code>x86_64</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>processor-info.sustained-clock-speed-in-ghz</code> - The CPU clock speed, in GHz.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>processor-info.supported-features</code> - The supported CPU features (<code>amd-sev-snp</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>supported-boot-mode</code> - The boot mode (<code>legacy-bios</code> | <code>uefi</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>supported-root-device-type</code> - The root device type (<code>ebs</code> | <code>instance-store</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>supported-usage-class</code> - The usage class (<code>on-demand</code> | <code>spot</code> |
     * <code>capacity-block</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>supported-virtualization-type</code> - The virtualization type (<code>hvm</code> | <code>paravirtual</code>
     * ).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>vcpu-info.default-cores</code> - The default number of cores for the instance type.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>vcpu-info.default-threads-per-core</code> - The default number of threads per core for the instance type.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>vcpu-info.default-vcpus</code> - The default number of vCPUs for the instance type.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>vcpu-info.valid-cores</code> - The number of cores that can be configured for the instance type.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>vcpu-info.valid-threads-per-core</code> - The number of threads per core that can be configured for the
     * instance type. For example, "1" or "1,2".
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFilters} method.
     * </p>
     * 
     * @return One or more filters. Filter names and values are case-sensitive.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>auto-recovery-supported</code> - Indicates whether Amazon CloudWatch action based recovery is
     *         supported (<code>true</code> | <code>false</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>bare-metal</code> - Indicates whether it is a bare metal instance type (<code>true</code> |
     *         <code>false</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>burstable-performance-supported</code> - Indicates whether the instance type is a burstable
     *         performance T instance type (<code>true</code> | <code>false</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>current-generation</code> - Indicates whether this instance type is the latest generation instance
     *         type of an instance family (<code>true</code> | <code>false</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ebs-info.ebs-optimized-info.baseline-bandwidth-in-mbps</code> - The baseline bandwidth performance
     *         for an EBS-optimized instance type, in Mbps.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ebs-info.ebs-optimized-info.baseline-iops</code> - The baseline input/output storage operations per
     *         second for an EBS-optimized instance type.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ebs-info.ebs-optimized-info.baseline-throughput-in-mbps</code> - The baseline throughput
     *         performance for an EBS-optimized instance type, in MB/s.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ebs-info.ebs-optimized-info.maximum-bandwidth-in-mbps</code> - The maximum bandwidth performance
     *         for an EBS-optimized instance type, in Mbps.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ebs-info.ebs-optimized-info.maximum-iops</code> - The maximum input/output storage operations per
     *         second for an EBS-optimized instance type.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ebs-info.ebs-optimized-info.maximum-throughput-in-mbps</code> - The maximum throughput performance
     *         for an EBS-optimized instance type, in MB/s.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ebs-info.ebs-optimized-support</code> - Indicates whether the instance type is EBS-optimized (
     *         <code>supported</code> | <code>unsupported</code> | <code>default</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ebs-info.encryption-support</code> - Indicates whether EBS encryption is supported (
     *         <code>supported</code> | <code>unsupported</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ebs-info.nvme-support</code> - Indicates whether non-volatile memory express (NVMe) is supported
     *         for EBS volumes (<code>required</code> | <code>supported</code> | <code>unsupported</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>free-tier-eligible</code> - Indicates whether the instance type is eligible to use in the free tier
     *         (<code>true</code> | <code>false</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>hibernation-supported</code> - Indicates whether On-Demand hibernation is supported (
     *         <code>true</code> | <code>false</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>hypervisor</code> - The hypervisor (<code>nitro</code> | <code>xen</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>instance-storage-info.disk.count</code> - The number of local disks.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>instance-storage-info.disk.size-in-gb</code> - The storage size of each instance storage disk, in
     *         GB.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>instance-storage-info.disk.type</code> - The storage technology for the local instance storage
     *         disks (<code>hdd</code> | <code>ssd</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>instance-storage-info.encryption-support</code> - Indicates whether data is encrypted at rest (
     *         <code>required</code> | <code>supported</code> | <code>unsupported</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>instance-storage-info.nvme-support</code> - Indicates whether non-volatile memory express (NVMe) is
     *         supported for instance store (<code>required</code> | <code>supported</code> | <code>unsupported</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>instance-storage-info.total-size-in-gb</code> - The total amount of storage available from all
     *         local instance storage, in GB.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>instance-storage-supported</code> - Indicates whether the instance type has local instance storage
     *         (<code>true</code> | <code>false</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>instance-type</code> - The instance type (for example <code>c5.2xlarge</code> or c5*).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>memory-info.size-in-mib</code> - The memory size.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>network-info.efa-info.maximum-efa-interfaces</code> - The maximum number of Elastic Fabric Adapters
     *         (EFAs) per instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>network-info.efa-supported</code> - Indicates whether the instance type supports Elastic Fabric
     *         Adapter (EFA) (<code>true</code> | <code>false</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>network-info.ena-support</code> - Indicates whether Elastic Network Adapter (ENA) is supported or
     *         required (<code>required</code> | <code>supported</code> | <code>unsupported</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>network-info.encryption-in-transit-supported</code> - Indicates whether the instance type
     *         automatically encrypts in-transit traffic between instances (<code>true</code> | <code>false</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>network-info.ipv4-addresses-per-interface</code> - The maximum number of private IPv4 addresses per
     *         network interface.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>network-info.ipv6-addresses-per-interface</code> - The maximum number of private IPv6 addresses per
     *         network interface.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>network-info.ipv6-supported</code> - Indicates whether the instance type supports IPv6 (
     *         <code>true</code> | <code>false</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>network-info.maximum-network-cards</code> - The maximum number of network cards per instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>network-info.maximum-network-interfaces</code> - The maximum number of network interfaces per
     *         instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>network-info.network-performance</code> - The network performance (for example, "25 Gigabit").
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>nitro-enclaves-support</code> - Indicates whether Nitro Enclaves is supported (
     *         <code>supported</code> | <code>unsupported</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>nitro-tpm-support</code> - Indicates whether NitroTPM is supported (<code>supported</code> |
     *         <code>unsupported</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>nitro-tpm-info.supported-versions</code> - The supported NitroTPM version (<code>2.0</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>processor-info.supported-architecture</code> - The CPU architecture (<code>arm64</code> |
     *         <code>i386</code> | <code>x86_64</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>processor-info.sustained-clock-speed-in-ghz</code> - The CPU clock speed, in GHz.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>processor-info.supported-features</code> - The supported CPU features (<code>amd-sev-snp</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>supported-boot-mode</code> - The boot mode (<code>legacy-bios</code> | <code>uefi</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>supported-root-device-type</code> - The root device type (<code>ebs</code> |
     *         <code>instance-store</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>supported-usage-class</code> - The usage class (<code>on-demand</code> | <code>spot</code> |
     *         <code>capacity-block</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>supported-virtualization-type</code> - The virtualization type (<code>hvm</code> |
     *         <code>paravirtual</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>vcpu-info.default-cores</code> - The default number of cores for the instance type.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>vcpu-info.default-threads-per-core</code> - The default number of threads per core for the instance
     *         type.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>vcpu-info.default-vcpus</code> - The default number of vCPUs for the instance type.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>vcpu-info.valid-cores</code> - The number of cores that can be configured for the instance type.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>vcpu-info.valid-threads-per-core</code> - The number of threads per core that can be configured for
     *         the instance type. For example, "1" or "1,2".
     *         </p>
     *         </li>
     */
    public final List<Filter> filters() {
        return filters;
    }

    /**
     * <p>
     * The maximum number of items to return for this request. To get the next page of items, make another request with
     * the token returned in the output. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Query-Requests.html#api-pagination">Pagination</a>.
     * </p>
     * 
     * @return The maximum number of items to return for this request. To get the next page of items, make another
     *         request with the token returned in the output. For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Query-Requests.html#api-pagination"
     *         >Pagination</a>.
     */
    public final Integer maxResults() {
        return maxResults;
    }

    /**
     * <p>
     * The token returned from a previous paginated request. Pagination continues from the end of the items returned by
     * the previous request.
     * </p>
     * 
     * @return The token returned from a previous paginated request. Pagination continues from the end of the items
     *         returned by the previous request.
     */
    public final String nextToken() {
        return nextToken;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(dryRun());
        hashCode = 31 * hashCode + Objects.hashCode(hasInstanceTypes() ? instanceTypesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasFilters() ? filters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(maxResults());
        hashCode = 31 * hashCode + Objects.hashCode(nextToken());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DescribeInstanceTypesRequest)) {
            return false;
        }
        DescribeInstanceTypesRequest other = (DescribeInstanceTypesRequest) obj;
        return Objects.equals(dryRun(), other.dryRun()) && hasInstanceTypes() == other.hasInstanceTypes()
                && Objects.equals(instanceTypesAsStrings(), other.instanceTypesAsStrings()) && hasFilters() == other.hasFilters()
                && Objects.equals(filters(), other.filters()) && Objects.equals(maxResults(), other.maxResults())
                && Objects.equals(nextToken(), other.nextToken());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DescribeInstanceTypesRequest").add("DryRun", dryRun())
                .add("InstanceTypes", hasInstanceTypes() ? instanceTypesAsStrings() : null)
                .add("Filters", hasFilters() ? filters() : null).add("MaxResults", maxResults()).add("NextToken", nextToken())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DryRun":
            return Optional.ofNullable(clazz.cast(dryRun()));
        case "InstanceTypes":
            return Optional.ofNullable(clazz.cast(instanceTypesAsStrings()));
        case "Filters":
            return Optional.ofNullable(clazz.cast(filters()));
        case "MaxResults":
            return Optional.ofNullable(clazz.cast(maxResults()));
        case "NextToken":
            return Optional.ofNullable(clazz.cast(nextToken()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<DescribeInstanceTypesRequest, T> g) {
        return obj -> g.apply((DescribeInstanceTypesRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends Ec2Request.Builder, SdkPojo, CopyableBuilder<Builder, DescribeInstanceTypesRequest> {
        /**
         * <p>
         * Checks whether you have the required permissions for the action, without actually making the request, and
         * provides an error response. If you have the required permissions, the error response is
         * <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
         * </p>
         * 
         * @param dryRun
         *        Checks whether you have the required permissions for the action, without actually making the request,
         *        and provides an error response. If you have the required permissions, the error response is
         *        <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dryRun(Boolean dryRun);

        /**
         * <p>
         * The instance types.
         * </p>
         * 
         * @param instanceTypes
         *        The instance types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceTypesWithStrings(Collection<String> instanceTypes);

        /**
         * <p>
         * The instance types.
         * </p>
         * 
         * @param instanceTypes
         *        The instance types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceTypesWithStrings(String... instanceTypes);

        /**
         * <p>
         * The instance types.
         * </p>
         * 
         * @param instanceTypes
         *        The instance types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceTypes(Collection<InstanceType> instanceTypes);

        /**
         * <p>
         * The instance types.
         * </p>
         * 
         * @param instanceTypes
         *        The instance types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceTypes(InstanceType... instanceTypes);

        /**
         * <p>
         * One or more filters. Filter names and values are case-sensitive.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>auto-recovery-supported</code> - Indicates whether Amazon CloudWatch action based recovery is supported
         * (<code>true</code> | <code>false</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>bare-metal</code> - Indicates whether it is a bare metal instance type (<code>true</code> |
         * <code>false</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>burstable-performance-supported</code> - Indicates whether the instance type is a burstable performance
         * T instance type (<code>true</code> | <code>false</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>current-generation</code> - Indicates whether this instance type is the latest generation instance type
         * of an instance family (<code>true</code> | <code>false</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ebs-info.ebs-optimized-info.baseline-bandwidth-in-mbps</code> - The baseline bandwidth performance for
         * an EBS-optimized instance type, in Mbps.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ebs-info.ebs-optimized-info.baseline-iops</code> - The baseline input/output storage operations per
         * second for an EBS-optimized instance type.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ebs-info.ebs-optimized-info.baseline-throughput-in-mbps</code> - The baseline throughput performance
         * for an EBS-optimized instance type, in MB/s.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ebs-info.ebs-optimized-info.maximum-bandwidth-in-mbps</code> - The maximum bandwidth performance for an
         * EBS-optimized instance type, in Mbps.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ebs-info.ebs-optimized-info.maximum-iops</code> - The maximum input/output storage operations per
         * second for an EBS-optimized instance type.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ebs-info.ebs-optimized-info.maximum-throughput-in-mbps</code> - The maximum throughput performance for
         * an EBS-optimized instance type, in MB/s.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ebs-info.ebs-optimized-support</code> - Indicates whether the instance type is EBS-optimized (
         * <code>supported</code> | <code>unsupported</code> | <code>default</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ebs-info.encryption-support</code> - Indicates whether EBS encryption is supported (
         * <code>supported</code> | <code>unsupported</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ebs-info.nvme-support</code> - Indicates whether non-volatile memory express (NVMe) is supported for
         * EBS volumes (<code>required</code> | <code>supported</code> | <code>unsupported</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>free-tier-eligible</code> - Indicates whether the instance type is eligible to use in the free tier (
         * <code>true</code> | <code>false</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>hibernation-supported</code> - Indicates whether On-Demand hibernation is supported (<code>true</code>
         * | <code>false</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>hypervisor</code> - The hypervisor (<code>nitro</code> | <code>xen</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>instance-storage-info.disk.count</code> - The number of local disks.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>instance-storage-info.disk.size-in-gb</code> - The storage size of each instance storage disk, in GB.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>instance-storage-info.disk.type</code> - The storage technology for the local instance storage disks (
         * <code>hdd</code> | <code>ssd</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>instance-storage-info.encryption-support</code> - Indicates whether data is encrypted at rest (
         * <code>required</code> | <code>supported</code> | <code>unsupported</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>instance-storage-info.nvme-support</code> - Indicates whether non-volatile memory express (NVMe) is
         * supported for instance store (<code>required</code> | <code>supported</code> | <code>unsupported</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>instance-storage-info.total-size-in-gb</code> - The total amount of storage available from all local
         * instance storage, in GB.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>instance-storage-supported</code> - Indicates whether the instance type has local instance storage (
         * <code>true</code> | <code>false</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>instance-type</code> - The instance type (for example <code>c5.2xlarge</code> or c5*).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>memory-info.size-in-mib</code> - The memory size.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>network-info.efa-info.maximum-efa-interfaces</code> - The maximum number of Elastic Fabric Adapters
         * (EFAs) per instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>network-info.efa-supported</code> - Indicates whether the instance type supports Elastic Fabric Adapter
         * (EFA) (<code>true</code> | <code>false</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>network-info.ena-support</code> - Indicates whether Elastic Network Adapter (ENA) is supported or
         * required (<code>required</code> | <code>supported</code> | <code>unsupported</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>network-info.encryption-in-transit-supported</code> - Indicates whether the instance type automatically
         * encrypts in-transit traffic between instances (<code>true</code> | <code>false</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>network-info.ipv4-addresses-per-interface</code> - The maximum number of private IPv4 addresses per
         * network interface.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>network-info.ipv6-addresses-per-interface</code> - The maximum number of private IPv6 addresses per
         * network interface.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>network-info.ipv6-supported</code> - Indicates whether the instance type supports IPv6 (
         * <code>true</code> | <code>false</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>network-info.maximum-network-cards</code> - The maximum number of network cards per instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>network-info.maximum-network-interfaces</code> - The maximum number of network interfaces per instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>network-info.network-performance</code> - The network performance (for example, "25 Gigabit").
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>nitro-enclaves-support</code> - Indicates whether Nitro Enclaves is supported (<code>supported</code> |
         * <code>unsupported</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>nitro-tpm-support</code> - Indicates whether NitroTPM is supported (<code>supported</code> |
         * <code>unsupported</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>nitro-tpm-info.supported-versions</code> - The supported NitroTPM version (<code>2.0</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>processor-info.supported-architecture</code> - The CPU architecture (<code>arm64</code> |
         * <code>i386</code> | <code>x86_64</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>processor-info.sustained-clock-speed-in-ghz</code> - The CPU clock speed, in GHz.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>processor-info.supported-features</code> - The supported CPU features (<code>amd-sev-snp</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>supported-boot-mode</code> - The boot mode (<code>legacy-bios</code> | <code>uefi</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>supported-root-device-type</code> - The root device type (<code>ebs</code> |
         * <code>instance-store</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>supported-usage-class</code> - The usage class (<code>on-demand</code> | <code>spot</code> |
         * <code>capacity-block</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>supported-virtualization-type</code> - The virtualization type (<code>hvm</code> |
         * <code>paravirtual</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>vcpu-info.default-cores</code> - The default number of cores for the instance type.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>vcpu-info.default-threads-per-core</code> - The default number of threads per core for the instance
         * type.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>vcpu-info.default-vcpus</code> - The default number of vCPUs for the instance type.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>vcpu-info.valid-cores</code> - The number of cores that can be configured for the instance type.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>vcpu-info.valid-threads-per-core</code> - The number of threads per core that can be configured for the
         * instance type. For example, "1" or "1,2".
         * </p>
         * </li>
         * </ul>
         * 
         * @param filters
         *        One or more filters. Filter names and values are case-sensitive.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>auto-recovery-supported</code> - Indicates whether Amazon CloudWatch action based recovery is
         *        supported (<code>true</code> | <code>false</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>bare-metal</code> - Indicates whether it is a bare metal instance type (<code>true</code> |
         *        <code>false</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>burstable-performance-supported</code> - Indicates whether the instance type is a burstable
         *        performance T instance type (<code>true</code> | <code>false</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>current-generation</code> - Indicates whether this instance type is the latest generation
         *        instance type of an instance family (<code>true</code> | <code>false</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ebs-info.ebs-optimized-info.baseline-bandwidth-in-mbps</code> - The baseline bandwidth
         *        performance for an EBS-optimized instance type, in Mbps.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ebs-info.ebs-optimized-info.baseline-iops</code> - The baseline input/output storage operations
         *        per second for an EBS-optimized instance type.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ebs-info.ebs-optimized-info.baseline-throughput-in-mbps</code> - The baseline throughput
         *        performance for an EBS-optimized instance type, in MB/s.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ebs-info.ebs-optimized-info.maximum-bandwidth-in-mbps</code> - The maximum bandwidth performance
         *        for an EBS-optimized instance type, in Mbps.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ebs-info.ebs-optimized-info.maximum-iops</code> - The maximum input/output storage operations
         *        per second for an EBS-optimized instance type.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ebs-info.ebs-optimized-info.maximum-throughput-in-mbps</code> - The maximum throughput
         *        performance for an EBS-optimized instance type, in MB/s.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ebs-info.ebs-optimized-support</code> - Indicates whether the instance type is EBS-optimized (
         *        <code>supported</code> | <code>unsupported</code> | <code>default</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ebs-info.encryption-support</code> - Indicates whether EBS encryption is supported (
         *        <code>supported</code> | <code>unsupported</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ebs-info.nvme-support</code> - Indicates whether non-volatile memory express (NVMe) is supported
         *        for EBS volumes (<code>required</code> | <code>supported</code> | <code>unsupported</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>free-tier-eligible</code> - Indicates whether the instance type is eligible to use in the free
         *        tier (<code>true</code> | <code>false</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>hibernation-supported</code> - Indicates whether On-Demand hibernation is supported (
         *        <code>true</code> | <code>false</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>hypervisor</code> - The hypervisor (<code>nitro</code> | <code>xen</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>instance-storage-info.disk.count</code> - The number of local disks.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>instance-storage-info.disk.size-in-gb</code> - The storage size of each instance storage disk,
         *        in GB.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>instance-storage-info.disk.type</code> - The storage technology for the local instance storage
         *        disks (<code>hdd</code> | <code>ssd</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>instance-storage-info.encryption-support</code> - Indicates whether data is encrypted at rest (
         *        <code>required</code> | <code>supported</code> | <code>unsupported</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>instance-storage-info.nvme-support</code> - Indicates whether non-volatile memory express (NVMe)
         *        is supported for instance store (<code>required</code> | <code>supported</code> |
         *        <code>unsupported</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>instance-storage-info.total-size-in-gb</code> - The total amount of storage available from all
         *        local instance storage, in GB.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>instance-storage-supported</code> - Indicates whether the instance type has local instance
         *        storage (<code>true</code> | <code>false</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>instance-type</code> - The instance type (for example <code>c5.2xlarge</code> or c5*).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>memory-info.size-in-mib</code> - The memory size.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>network-info.efa-info.maximum-efa-interfaces</code> - The maximum number of Elastic Fabric
         *        Adapters (EFAs) per instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>network-info.efa-supported</code> - Indicates whether the instance type supports Elastic Fabric
         *        Adapter (EFA) (<code>true</code> | <code>false</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>network-info.ena-support</code> - Indicates whether Elastic Network Adapter (ENA) is supported
         *        or required (<code>required</code> | <code>supported</code> | <code>unsupported</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>network-info.encryption-in-transit-supported</code> - Indicates whether the instance type
         *        automatically encrypts in-transit traffic between instances (<code>true</code> | <code>false</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>network-info.ipv4-addresses-per-interface</code> - The maximum number of private IPv4 addresses
         *        per network interface.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>network-info.ipv6-addresses-per-interface</code> - The maximum number of private IPv6 addresses
         *        per network interface.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>network-info.ipv6-supported</code> - Indicates whether the instance type supports IPv6 (
         *        <code>true</code> | <code>false</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>network-info.maximum-network-cards</code> - The maximum number of network cards per instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>network-info.maximum-network-interfaces</code> - The maximum number of network interfaces per
         *        instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>network-info.network-performance</code> - The network performance (for example, "25 Gigabit").
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>nitro-enclaves-support</code> - Indicates whether Nitro Enclaves is supported (
         *        <code>supported</code> | <code>unsupported</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>nitro-tpm-support</code> - Indicates whether NitroTPM is supported (<code>supported</code> |
         *        <code>unsupported</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>nitro-tpm-info.supported-versions</code> - The supported NitroTPM version (<code>2.0</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>processor-info.supported-architecture</code> - The CPU architecture (<code>arm64</code> |
         *        <code>i386</code> | <code>x86_64</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>processor-info.sustained-clock-speed-in-ghz</code> - The CPU clock speed, in GHz.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>processor-info.supported-features</code> - The supported CPU features (<code>amd-sev-snp</code>
         *        ).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>supported-boot-mode</code> - The boot mode (<code>legacy-bios</code> | <code>uefi</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>supported-root-device-type</code> - The root device type (<code>ebs</code> |
         *        <code>instance-store</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>supported-usage-class</code> - The usage class (<code>on-demand</code> | <code>spot</code> |
         *        <code>capacity-block</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>supported-virtualization-type</code> - The virtualization type (<code>hvm</code> |
         *        <code>paravirtual</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>vcpu-info.default-cores</code> - The default number of cores for the instance type.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>vcpu-info.default-threads-per-core</code> - The default number of threads per core for the
         *        instance type.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>vcpu-info.default-vcpus</code> - The default number of vCPUs for the instance type.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>vcpu-info.valid-cores</code> - The number of cores that can be configured for the instance type.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>vcpu-info.valid-threads-per-core</code> - The number of threads per core that can be configured
         *        for the instance type. For example, "1" or "1,2".
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filters(Collection<Filter> filters);

        /**
         * <p>
         * One or more filters. Filter names and values are case-sensitive.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>auto-recovery-supported</code> - Indicates whether Amazon CloudWatch action based recovery is supported
         * (<code>true</code> | <code>false</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>bare-metal</code> - Indicates whether it is a bare metal instance type (<code>true</code> |
         * <code>false</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>burstable-performance-supported</code> - Indicates whether the instance type is a burstable performance
         * T instance type (<code>true</code> | <code>false</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>current-generation</code> - Indicates whether this instance type is the latest generation instance type
         * of an instance family (<code>true</code> | <code>false</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ebs-info.ebs-optimized-info.baseline-bandwidth-in-mbps</code> - The baseline bandwidth performance for
         * an EBS-optimized instance type, in Mbps.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ebs-info.ebs-optimized-info.baseline-iops</code> - The baseline input/output storage operations per
         * second for an EBS-optimized instance type.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ebs-info.ebs-optimized-info.baseline-throughput-in-mbps</code> - The baseline throughput performance
         * for an EBS-optimized instance type, in MB/s.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ebs-info.ebs-optimized-info.maximum-bandwidth-in-mbps</code> - The maximum bandwidth performance for an
         * EBS-optimized instance type, in Mbps.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ebs-info.ebs-optimized-info.maximum-iops</code> - The maximum input/output storage operations per
         * second for an EBS-optimized instance type.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ebs-info.ebs-optimized-info.maximum-throughput-in-mbps</code> - The maximum throughput performance for
         * an EBS-optimized instance type, in MB/s.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ebs-info.ebs-optimized-support</code> - Indicates whether the instance type is EBS-optimized (
         * <code>supported</code> | <code>unsupported</code> | <code>default</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ebs-info.encryption-support</code> - Indicates whether EBS encryption is supported (
         * <code>supported</code> | <code>unsupported</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ebs-info.nvme-support</code> - Indicates whether non-volatile memory express (NVMe) is supported for
         * EBS volumes (<code>required</code> | <code>supported</code> | <code>unsupported</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>free-tier-eligible</code> - Indicates whether the instance type is eligible to use in the free tier (
         * <code>true</code> | <code>false</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>hibernation-supported</code> - Indicates whether On-Demand hibernation is supported (<code>true</code>
         * | <code>false</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>hypervisor</code> - The hypervisor (<code>nitro</code> | <code>xen</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>instance-storage-info.disk.count</code> - The number of local disks.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>instance-storage-info.disk.size-in-gb</code> - The storage size of each instance storage disk, in GB.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>instance-storage-info.disk.type</code> - The storage technology for the local instance storage disks (
         * <code>hdd</code> | <code>ssd</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>instance-storage-info.encryption-support</code> - Indicates whether data is encrypted at rest (
         * <code>required</code> | <code>supported</code> | <code>unsupported</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>instance-storage-info.nvme-support</code> - Indicates whether non-volatile memory express (NVMe) is
         * supported for instance store (<code>required</code> | <code>supported</code> | <code>unsupported</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>instance-storage-info.total-size-in-gb</code> - The total amount of storage available from all local
         * instance storage, in GB.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>instance-storage-supported</code> - Indicates whether the instance type has local instance storage (
         * <code>true</code> | <code>false</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>instance-type</code> - The instance type (for example <code>c5.2xlarge</code> or c5*).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>memory-info.size-in-mib</code> - The memory size.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>network-info.efa-info.maximum-efa-interfaces</code> - The maximum number of Elastic Fabric Adapters
         * (EFAs) per instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>network-info.efa-supported</code> - Indicates whether the instance type supports Elastic Fabric Adapter
         * (EFA) (<code>true</code> | <code>false</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>network-info.ena-support</code> - Indicates whether Elastic Network Adapter (ENA) is supported or
         * required (<code>required</code> | <code>supported</code> | <code>unsupported</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>network-info.encryption-in-transit-supported</code> - Indicates whether the instance type automatically
         * encrypts in-transit traffic between instances (<code>true</code> | <code>false</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>network-info.ipv4-addresses-per-interface</code> - The maximum number of private IPv4 addresses per
         * network interface.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>network-info.ipv6-addresses-per-interface</code> - The maximum number of private IPv6 addresses per
         * network interface.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>network-info.ipv6-supported</code> - Indicates whether the instance type supports IPv6 (
         * <code>true</code> | <code>false</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>network-info.maximum-network-cards</code> - The maximum number of network cards per instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>network-info.maximum-network-interfaces</code> - The maximum number of network interfaces per instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>network-info.network-performance</code> - The network performance (for example, "25 Gigabit").
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>nitro-enclaves-support</code> - Indicates whether Nitro Enclaves is supported (<code>supported</code> |
         * <code>unsupported</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>nitro-tpm-support</code> - Indicates whether NitroTPM is supported (<code>supported</code> |
         * <code>unsupported</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>nitro-tpm-info.supported-versions</code> - The supported NitroTPM version (<code>2.0</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>processor-info.supported-architecture</code> - The CPU architecture (<code>arm64</code> |
         * <code>i386</code> | <code>x86_64</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>processor-info.sustained-clock-speed-in-ghz</code> - The CPU clock speed, in GHz.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>processor-info.supported-features</code> - The supported CPU features (<code>amd-sev-snp</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>supported-boot-mode</code> - The boot mode (<code>legacy-bios</code> | <code>uefi</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>supported-root-device-type</code> - The root device type (<code>ebs</code> |
         * <code>instance-store</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>supported-usage-class</code> - The usage class (<code>on-demand</code> | <code>spot</code> |
         * <code>capacity-block</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>supported-virtualization-type</code> - The virtualization type (<code>hvm</code> |
         * <code>paravirtual</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>vcpu-info.default-cores</code> - The default number of cores for the instance type.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>vcpu-info.default-threads-per-core</code> - The default number of threads per core for the instance
         * type.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>vcpu-info.default-vcpus</code> - The default number of vCPUs for the instance type.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>vcpu-info.valid-cores</code> - The number of cores that can be configured for the instance type.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>vcpu-info.valid-threads-per-core</code> - The number of threads per core that can be configured for the
         * instance type. For example, "1" or "1,2".
         * </p>
         * </li>
         * </ul>
         * 
         * @param filters
         *        One or more filters. Filter names and values are case-sensitive.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>auto-recovery-supported</code> - Indicates whether Amazon CloudWatch action based recovery is
         *        supported (<code>true</code> | <code>false</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>bare-metal</code> - Indicates whether it is a bare metal instance type (<code>true</code> |
         *        <code>false</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>burstable-performance-supported</code> - Indicates whether the instance type is a burstable
         *        performance T instance type (<code>true</code> | <code>false</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>current-generation</code> - Indicates whether this instance type is the latest generation
         *        instance type of an instance family (<code>true</code> | <code>false</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ebs-info.ebs-optimized-info.baseline-bandwidth-in-mbps</code> - The baseline bandwidth
         *        performance for an EBS-optimized instance type, in Mbps.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ebs-info.ebs-optimized-info.baseline-iops</code> - The baseline input/output storage operations
         *        per second for an EBS-optimized instance type.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ebs-info.ebs-optimized-info.baseline-throughput-in-mbps</code> - The baseline throughput
         *        performance for an EBS-optimized instance type, in MB/s.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ebs-info.ebs-optimized-info.maximum-bandwidth-in-mbps</code> - The maximum bandwidth performance
         *        for an EBS-optimized instance type, in Mbps.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ebs-info.ebs-optimized-info.maximum-iops</code> - The maximum input/output storage operations
         *        per second for an EBS-optimized instance type.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ebs-info.ebs-optimized-info.maximum-throughput-in-mbps</code> - The maximum throughput
         *        performance for an EBS-optimized instance type, in MB/s.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ebs-info.ebs-optimized-support</code> - Indicates whether the instance type is EBS-optimized (
         *        <code>supported</code> | <code>unsupported</code> | <code>default</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ebs-info.encryption-support</code> - Indicates whether EBS encryption is supported (
         *        <code>supported</code> | <code>unsupported</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ebs-info.nvme-support</code> - Indicates whether non-volatile memory express (NVMe) is supported
         *        for EBS volumes (<code>required</code> | <code>supported</code> | <code>unsupported</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>free-tier-eligible</code> - Indicates whether the instance type is eligible to use in the free
         *        tier (<code>true</code> | <code>false</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>hibernation-supported</code> - Indicates whether On-Demand hibernation is supported (
         *        <code>true</code> | <code>false</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>hypervisor</code> - The hypervisor (<code>nitro</code> | <code>xen</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>instance-storage-info.disk.count</code> - The number of local disks.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>instance-storage-info.disk.size-in-gb</code> - The storage size of each instance storage disk,
         *        in GB.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>instance-storage-info.disk.type</code> - The storage technology for the local instance storage
         *        disks (<code>hdd</code> | <code>ssd</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>instance-storage-info.encryption-support</code> - Indicates whether data is encrypted at rest (
         *        <code>required</code> | <code>supported</code> | <code>unsupported</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>instance-storage-info.nvme-support</code> - Indicates whether non-volatile memory express (NVMe)
         *        is supported for instance store (<code>required</code> | <code>supported</code> |
         *        <code>unsupported</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>instance-storage-info.total-size-in-gb</code> - The total amount of storage available from all
         *        local instance storage, in GB.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>instance-storage-supported</code> - Indicates whether the instance type has local instance
         *        storage (<code>true</code> | <code>false</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>instance-type</code> - The instance type (for example <code>c5.2xlarge</code> or c5*).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>memory-info.size-in-mib</code> - The memory size.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>network-info.efa-info.maximum-efa-interfaces</code> - The maximum number of Elastic Fabric
         *        Adapters (EFAs) per instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>network-info.efa-supported</code> - Indicates whether the instance type supports Elastic Fabric
         *        Adapter (EFA) (<code>true</code> | <code>false</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>network-info.ena-support</code> - Indicates whether Elastic Network Adapter (ENA) is supported
         *        or required (<code>required</code> | <code>supported</code> | <code>unsupported</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>network-info.encryption-in-transit-supported</code> - Indicates whether the instance type
         *        automatically encrypts in-transit traffic between instances (<code>true</code> | <code>false</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>network-info.ipv4-addresses-per-interface</code> - The maximum number of private IPv4 addresses
         *        per network interface.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>network-info.ipv6-addresses-per-interface</code> - The maximum number of private IPv6 addresses
         *        per network interface.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>network-info.ipv6-supported</code> - Indicates whether the instance type supports IPv6 (
         *        <code>true</code> | <code>false</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>network-info.maximum-network-cards</code> - The maximum number of network cards per instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>network-info.maximum-network-interfaces</code> - The maximum number of network interfaces per
         *        instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>network-info.network-performance</code> - The network performance (for example, "25 Gigabit").
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>nitro-enclaves-support</code> - Indicates whether Nitro Enclaves is supported (
         *        <code>supported</code> | <code>unsupported</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>nitro-tpm-support</code> - Indicates whether NitroTPM is supported (<code>supported</code> |
         *        <code>unsupported</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>nitro-tpm-info.supported-versions</code> - The supported NitroTPM version (<code>2.0</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>processor-info.supported-architecture</code> - The CPU architecture (<code>arm64</code> |
         *        <code>i386</code> | <code>x86_64</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>processor-info.sustained-clock-speed-in-ghz</code> - The CPU clock speed, in GHz.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>processor-info.supported-features</code> - The supported CPU features (<code>amd-sev-snp</code>
         *        ).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>supported-boot-mode</code> - The boot mode (<code>legacy-bios</code> | <code>uefi</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>supported-root-device-type</code> - The root device type (<code>ebs</code> |
         *        <code>instance-store</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>supported-usage-class</code> - The usage class (<code>on-demand</code> | <code>spot</code> |
         *        <code>capacity-block</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>supported-virtualization-type</code> - The virtualization type (<code>hvm</code> |
         *        <code>paravirtual</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>vcpu-info.default-cores</code> - The default number of cores for the instance type.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>vcpu-info.default-threads-per-core</code> - The default number of threads per core for the
         *        instance type.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>vcpu-info.default-vcpus</code> - The default number of vCPUs for the instance type.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>vcpu-info.valid-cores</code> - The number of cores that can be configured for the instance type.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>vcpu-info.valid-threads-per-core</code> - The number of threads per core that can be configured
         *        for the instance type. For example, "1" or "1,2".
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filters(Filter... filters);

        /**
         * <p>
         * One or more filters. Filter names and values are case-sensitive.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>auto-recovery-supported</code> - Indicates whether Amazon CloudWatch action based recovery is supported
         * (<code>true</code> | <code>false</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>bare-metal</code> - Indicates whether it is a bare metal instance type (<code>true</code> |
         * <code>false</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>burstable-performance-supported</code> - Indicates whether the instance type is a burstable performance
         * T instance type (<code>true</code> | <code>false</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>current-generation</code> - Indicates whether this instance type is the latest generation instance type
         * of an instance family (<code>true</code> | <code>false</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ebs-info.ebs-optimized-info.baseline-bandwidth-in-mbps</code> - The baseline bandwidth performance for
         * an EBS-optimized instance type, in Mbps.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ebs-info.ebs-optimized-info.baseline-iops</code> - The baseline input/output storage operations per
         * second for an EBS-optimized instance type.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ebs-info.ebs-optimized-info.baseline-throughput-in-mbps</code> - The baseline throughput performance
         * for an EBS-optimized instance type, in MB/s.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ebs-info.ebs-optimized-info.maximum-bandwidth-in-mbps</code> - The maximum bandwidth performance for an
         * EBS-optimized instance type, in Mbps.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ebs-info.ebs-optimized-info.maximum-iops</code> - The maximum input/output storage operations per
         * second for an EBS-optimized instance type.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ebs-info.ebs-optimized-info.maximum-throughput-in-mbps</code> - The maximum throughput performance for
         * an EBS-optimized instance type, in MB/s.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ebs-info.ebs-optimized-support</code> - Indicates whether the instance type is EBS-optimized (
         * <code>supported</code> | <code>unsupported</code> | <code>default</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ebs-info.encryption-support</code> - Indicates whether EBS encryption is supported (
         * <code>supported</code> | <code>unsupported</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ebs-info.nvme-support</code> - Indicates whether non-volatile memory express (NVMe) is supported for
         * EBS volumes (<code>required</code> | <code>supported</code> | <code>unsupported</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>free-tier-eligible</code> - Indicates whether the instance type is eligible to use in the free tier (
         * <code>true</code> | <code>false</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>hibernation-supported</code> - Indicates whether On-Demand hibernation is supported (<code>true</code>
         * | <code>false</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>hypervisor</code> - The hypervisor (<code>nitro</code> | <code>xen</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>instance-storage-info.disk.count</code> - The number of local disks.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>instance-storage-info.disk.size-in-gb</code> - The storage size of each instance storage disk, in GB.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>instance-storage-info.disk.type</code> - The storage technology for the local instance storage disks (
         * <code>hdd</code> | <code>ssd</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>instance-storage-info.encryption-support</code> - Indicates whether data is encrypted at rest (
         * <code>required</code> | <code>supported</code> | <code>unsupported</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>instance-storage-info.nvme-support</code> - Indicates whether non-volatile memory express (NVMe) is
         * supported for instance store (<code>required</code> | <code>supported</code> | <code>unsupported</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>instance-storage-info.total-size-in-gb</code> - The total amount of storage available from all local
         * instance storage, in GB.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>instance-storage-supported</code> - Indicates whether the instance type has local instance storage (
         * <code>true</code> | <code>false</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>instance-type</code> - The instance type (for example <code>c5.2xlarge</code> or c5*).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>memory-info.size-in-mib</code> - The memory size.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>network-info.efa-info.maximum-efa-interfaces</code> - The maximum number of Elastic Fabric Adapters
         * (EFAs) per instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>network-info.efa-supported</code> - Indicates whether the instance type supports Elastic Fabric Adapter
         * (EFA) (<code>true</code> | <code>false</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>network-info.ena-support</code> - Indicates whether Elastic Network Adapter (ENA) is supported or
         * required (<code>required</code> | <code>supported</code> | <code>unsupported</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>network-info.encryption-in-transit-supported</code> - Indicates whether the instance type automatically
         * encrypts in-transit traffic between instances (<code>true</code> | <code>false</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>network-info.ipv4-addresses-per-interface</code> - The maximum number of private IPv4 addresses per
         * network interface.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>network-info.ipv6-addresses-per-interface</code> - The maximum number of private IPv6 addresses per
         * network interface.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>network-info.ipv6-supported</code> - Indicates whether the instance type supports IPv6 (
         * <code>true</code> | <code>false</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>network-info.maximum-network-cards</code> - The maximum number of network cards per instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>network-info.maximum-network-interfaces</code> - The maximum number of network interfaces per instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>network-info.network-performance</code> - The network performance (for example, "25 Gigabit").
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>nitro-enclaves-support</code> - Indicates whether Nitro Enclaves is supported (<code>supported</code> |
         * <code>unsupported</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>nitro-tpm-support</code> - Indicates whether NitroTPM is supported (<code>supported</code> |
         * <code>unsupported</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>nitro-tpm-info.supported-versions</code> - The supported NitroTPM version (<code>2.0</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>processor-info.supported-architecture</code> - The CPU architecture (<code>arm64</code> |
         * <code>i386</code> | <code>x86_64</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>processor-info.sustained-clock-speed-in-ghz</code> - The CPU clock speed, in GHz.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>processor-info.supported-features</code> - The supported CPU features (<code>amd-sev-snp</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>supported-boot-mode</code> - The boot mode (<code>legacy-bios</code> | <code>uefi</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>supported-root-device-type</code> - The root device type (<code>ebs</code> |
         * <code>instance-store</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>supported-usage-class</code> - The usage class (<code>on-demand</code> | <code>spot</code> |
         * <code>capacity-block</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>supported-virtualization-type</code> - The virtualization type (<code>hvm</code> |
         * <code>paravirtual</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>vcpu-info.default-cores</code> - The default number of cores for the instance type.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>vcpu-info.default-threads-per-core</code> - The default number of threads per core for the instance
         * type.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>vcpu-info.default-vcpus</code> - The default number of vCPUs for the instance type.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>vcpu-info.valid-cores</code> - The number of cores that can be configured for the instance type.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>vcpu-info.valid-threads-per-core</code> - The number of threads per core that can be configured for the
         * instance type. For example, "1" or "1,2".
         * </p>
         * </li>
         * </ul>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.Filter.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ec2.model.Filter#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ec2.model.Filter.Builder#build()}
         * is called immediately and its result is passed to {@link #filters(List<Filter>)}.
         * 
         * @param filters
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ec2.model.Filter.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #filters(java.util.Collection<Filter>)
         */
        Builder filters(Consumer<Filter.Builder>... filters);

        /**
         * <p>
         * The maximum number of items to return for this request. To get the next page of items, make another request
         * with the token returned in the output. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Query-Requests.html#api-pagination"
         * >Pagination</a>.
         * </p>
         * 
         * @param maxResults
         *        The maximum number of items to return for this request. To get the next page of items, make another
         *        request with the token returned in the output. For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Query-Requests.html#api-pagination"
         *        >Pagination</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxResults(Integer maxResults);

        /**
         * <p>
         * The token returned from a previous paginated request. Pagination continues from the end of the items returned
         * by the previous request.
         * </p>
         * 
         * @param nextToken
         *        The token returned from a previous paginated request. Pagination continues from the end of the items
         *        returned by the previous request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nextToken(String nextToken);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends Ec2Request.BuilderImpl implements Builder {
        private Boolean dryRun;

        private List<String> instanceTypes = DefaultSdkAutoConstructList.getInstance();

        private List<Filter> filters = DefaultSdkAutoConstructList.getInstance();

        private Integer maxResults;

        private String nextToken;

        private BuilderImpl() {
        }

        private BuilderImpl(DescribeInstanceTypesRequest model) {
            super(model);
            dryRun(model.dryRun);
            instanceTypesWithStrings(model.instanceTypes);
            filters(model.filters);
            maxResults(model.maxResults);
            nextToken(model.nextToken);
        }

        public final Boolean getDryRun() {
            return dryRun;
        }

        public final void setDryRun(Boolean dryRun) {
            this.dryRun = dryRun;
        }

        @Override
        public final Builder dryRun(Boolean dryRun) {
            this.dryRun = dryRun;
            return this;
        }

        public final Collection<String> getInstanceTypes() {
            if (instanceTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return instanceTypes;
        }

        public final void setInstanceTypes(Collection<String> instanceTypes) {
            this.instanceTypes = RequestInstanceTypeListCopier.copy(instanceTypes);
        }

        @Override
        public final Builder instanceTypesWithStrings(Collection<String> instanceTypes) {
            this.instanceTypes = RequestInstanceTypeListCopier.copy(instanceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceTypesWithStrings(String... instanceTypes) {
            instanceTypesWithStrings(Arrays.asList(instanceTypes));
            return this;
        }

        @Override
        public final Builder instanceTypes(Collection<InstanceType> instanceTypes) {
            this.instanceTypes = RequestInstanceTypeListCopier.copyEnumToString(instanceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceTypes(InstanceType... instanceTypes) {
            instanceTypes(Arrays.asList(instanceTypes));
            return this;
        }

        public final List<Filter.Builder> getFilters() {
            List<Filter.Builder> result = FilterListCopier.copyToBuilder(this.filters);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setFilters(Collection<Filter.BuilderImpl> filters) {
            this.filters = FilterListCopier.copyFromBuilder(filters);
        }

        @Override
        public final Builder filters(Collection<Filter> filters) {
            this.filters = FilterListCopier.copy(filters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder filters(Filter... filters) {
            filters(Arrays.asList(filters));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder filters(Consumer<Filter.Builder>... filters) {
            filters(Stream.of(filters).map(c -> Filter.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Integer getMaxResults() {
            return maxResults;
        }

        public final void setMaxResults(Integer maxResults) {
            this.maxResults = maxResults;
        }

        @Override
        public final Builder maxResults(Integer maxResults) {
            this.maxResults = maxResults;
            return this;
        }

        public final String getNextToken() {
            return nextToken;
        }

        public final void setNextToken(String nextToken) {
            this.nextToken = nextToken;
        }

        @Override
        public final Builder nextToken(String nextToken) {
            this.nextToken = nextToken;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public DescribeInstanceTypesRequest build() {
            return new DescribeInstanceTypesRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
