/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.directory.model;

import java.beans.Transient;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Initiates the creation of a conditional forwarder for your Directory Service for Microsoft Active Directory.
 * Conditional forwarders are required in order to set up a trust relationship with another domain.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateConditionalForwarderRequest extends DirectoryRequest implements
        ToCopyableBuilder<CreateConditionalForwarderRequest.Builder, CreateConditionalForwarderRequest> {
    private static final SdkField<String> DIRECTORY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DirectoryId").getter(getter(CreateConditionalForwarderRequest::directoryId))
            .setter(setter(Builder::directoryId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DirectoryId").build()).build();

    private static final SdkField<String> REMOTE_DOMAIN_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RemoteDomainName").getter(getter(CreateConditionalForwarderRequest::remoteDomainName))
            .setter(setter(Builder::remoteDomainName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RemoteDomainName").build()).build();

    private static final SdkField<List<String>> DNS_IP_ADDRS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("DnsIpAddrs")
            .getter(getter(CreateConditionalForwarderRequest::dnsIpAddrs))
            .setter(setter(Builder::dnsIpAddrs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DnsIpAddrs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DIRECTORY_ID_FIELD,
            REMOTE_DOMAIN_NAME_FIELD, DNS_IP_ADDRS_FIELD));

    private final String directoryId;

    private final String remoteDomainName;

    private final List<String> dnsIpAddrs;

    private CreateConditionalForwarderRequest(BuilderImpl builder) {
        super(builder);
        this.directoryId = builder.directoryId;
        this.remoteDomainName = builder.remoteDomainName;
        this.dnsIpAddrs = builder.dnsIpAddrs;
    }

    /**
     * <p>
     * The directory ID of the Amazon Web Services directory for which you are creating the conditional forwarder.
     * </p>
     * 
     * @return The directory ID of the Amazon Web Services directory for which you are creating the conditional
     *         forwarder.
     */
    public final String directoryId() {
        return directoryId;
    }

    /**
     * <p>
     * The fully qualified domain name (FQDN) of the remote domain with which you will set up a trust relationship.
     * </p>
     * 
     * @return The fully qualified domain name (FQDN) of the remote domain with which you will set up a trust
     *         relationship.
     */
    public final String remoteDomainName() {
        return remoteDomainName;
    }

    /**
     * For responses, this returns true if the service returned a value for the DnsIpAddrs property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDnsIpAddrs() {
        return dnsIpAddrs != null && !(dnsIpAddrs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IP addresses of the remote DNS server associated with RemoteDomainName.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDnsIpAddrs} method.
     * </p>
     * 
     * @return The IP addresses of the remote DNS server associated with RemoteDomainName.
     */
    public final List<String> dnsIpAddrs() {
        return dnsIpAddrs;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(directoryId());
        hashCode = 31 * hashCode + Objects.hashCode(remoteDomainName());
        hashCode = 31 * hashCode + Objects.hashCode(hasDnsIpAddrs() ? dnsIpAddrs() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateConditionalForwarderRequest)) {
            return false;
        }
        CreateConditionalForwarderRequest other = (CreateConditionalForwarderRequest) obj;
        return Objects.equals(directoryId(), other.directoryId()) && Objects.equals(remoteDomainName(), other.remoteDomainName())
                && hasDnsIpAddrs() == other.hasDnsIpAddrs() && Objects.equals(dnsIpAddrs(), other.dnsIpAddrs());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateConditionalForwarderRequest").add("DirectoryId", directoryId())
                .add("RemoteDomainName", remoteDomainName()).add("DnsIpAddrs", hasDnsIpAddrs() ? dnsIpAddrs() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DirectoryId":
            return Optional.ofNullable(clazz.cast(directoryId()));
        case "RemoteDomainName":
            return Optional.ofNullable(clazz.cast(remoteDomainName()));
        case "DnsIpAddrs":
            return Optional.ofNullable(clazz.cast(dnsIpAddrs()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateConditionalForwarderRequest, T> g) {
        return obj -> g.apply((CreateConditionalForwarderRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends DirectoryRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, CreateConditionalForwarderRequest> {
        /**
         * <p>
         * The directory ID of the Amazon Web Services directory for which you are creating the conditional forwarder.
         * </p>
         * 
         * @param directoryId
         *        The directory ID of the Amazon Web Services directory for which you are creating the conditional
         *        forwarder.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder directoryId(String directoryId);

        /**
         * <p>
         * The fully qualified domain name (FQDN) of the remote domain with which you will set up a trust relationship.
         * </p>
         * 
         * @param remoteDomainName
         *        The fully qualified domain name (FQDN) of the remote domain with which you will set up a trust
         *        relationship.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder remoteDomainName(String remoteDomainName);

        /**
         * <p>
         * The IP addresses of the remote DNS server associated with RemoteDomainName.
         * </p>
         * 
         * @param dnsIpAddrs
         *        The IP addresses of the remote DNS server associated with RemoteDomainName.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsIpAddrs(Collection<String> dnsIpAddrs);

        /**
         * <p>
         * The IP addresses of the remote DNS server associated with RemoteDomainName.
         * </p>
         * 
         * @param dnsIpAddrs
         *        The IP addresses of the remote DNS server associated with RemoteDomainName.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsIpAddrs(String... dnsIpAddrs);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends DirectoryRequest.BuilderImpl implements Builder {
        private String directoryId;

        private String remoteDomainName;

        private List<String> dnsIpAddrs = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateConditionalForwarderRequest model) {
            super(model);
            directoryId(model.directoryId);
            remoteDomainName(model.remoteDomainName);
            dnsIpAddrs(model.dnsIpAddrs);
        }

        public final String getDirectoryId() {
            return directoryId;
        }

        public final void setDirectoryId(String directoryId) {
            this.directoryId = directoryId;
        }

        @Override
        @Transient
        public final Builder directoryId(String directoryId) {
            this.directoryId = directoryId;
            return this;
        }

        public final String getRemoteDomainName() {
            return remoteDomainName;
        }

        public final void setRemoteDomainName(String remoteDomainName) {
            this.remoteDomainName = remoteDomainName;
        }

        @Override
        @Transient
        public final Builder remoteDomainName(String remoteDomainName) {
            this.remoteDomainName = remoteDomainName;
            return this;
        }

        public final Collection<String> getDnsIpAddrs() {
            if (dnsIpAddrs instanceof SdkAutoConstructList) {
                return null;
            }
            return dnsIpAddrs;
        }

        public final void setDnsIpAddrs(Collection<String> dnsIpAddrs) {
            this.dnsIpAddrs = DnsIpAddrsCopier.copy(dnsIpAddrs);
        }

        @Override
        @Transient
        public final Builder dnsIpAddrs(Collection<String> dnsIpAddrs) {
            this.dnsIpAddrs = DnsIpAddrsCopier.copy(dnsIpAddrs);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder dnsIpAddrs(String... dnsIpAddrs) {
            dnsIpAddrs(Arrays.asList(dnsIpAddrs));
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateConditionalForwarderRequest build() {
            return new CreateConditionalForwarderRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
