/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudtrail.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A storage lake of event data against which you can run complex SQL-based queries. An event data store can include
 * events that you have logged on your account. To select events for an event data store, use <a
 * href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/cloudtrail-lake-concepts.html#adv-event-selectors"
 * >advanced event selectors</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EventDataStore implements SdkPojo, Serializable, ToCopyableBuilder<EventDataStore.Builder, EventDataStore> {
    private static final SdkField<String> EVENT_DATA_STORE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EventDataStoreArn").getter(getter(EventDataStore::eventDataStoreArn))
            .setter(setter(Builder::eventDataStoreArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventDataStoreArn").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(EventDataStore::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<Boolean> TERMINATION_PROTECTION_ENABLED_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("TerminationProtectionEnabled")
            .getter(getter(EventDataStore::terminationProtectionEnabled))
            .setter(setter(Builder::terminationProtectionEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TerminationProtectionEnabled")
                    .build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(EventDataStore::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<List<AdvancedEventSelector>> ADVANCED_EVENT_SELECTORS_FIELD = SdkField
            .<List<AdvancedEventSelector>> builder(MarshallingType.LIST)
            .memberName("AdvancedEventSelectors")
            .getter(getter(EventDataStore::advancedEventSelectors))
            .setter(setter(Builder::advancedEventSelectors))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdvancedEventSelectors").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AdvancedEventSelector> builder(MarshallingType.SDK_POJO)
                                            .constructor(AdvancedEventSelector::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> MULTI_REGION_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("MultiRegionEnabled").getter(getter(EventDataStore::multiRegionEnabled))
            .setter(setter(Builder::multiRegionEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MultiRegionEnabled").build())
            .build();

    private static final SdkField<Boolean> ORGANIZATION_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("OrganizationEnabled").getter(getter(EventDataStore::organizationEnabled))
            .setter(setter(Builder::organizationEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OrganizationEnabled").build())
            .build();

    private static final SdkField<Integer> RETENTION_PERIOD_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("RetentionPeriod").getter(getter(EventDataStore::retentionPeriod))
            .setter(setter(Builder::retentionPeriod))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RetentionPeriod").build()).build();

    private static final SdkField<Instant> CREATED_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedTimestamp").getter(getter(EventDataStore::createdTimestamp))
            .setter(setter(Builder::createdTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedTimestamp").build()).build();

    private static final SdkField<Instant> UPDATED_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("UpdatedTimestamp").getter(getter(EventDataStore::updatedTimestamp))
            .setter(setter(Builder::updatedTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UpdatedTimestamp").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EVENT_DATA_STORE_ARN_FIELD,
            NAME_FIELD, TERMINATION_PROTECTION_ENABLED_FIELD, STATUS_FIELD, ADVANCED_EVENT_SELECTORS_FIELD,
            MULTI_REGION_ENABLED_FIELD, ORGANIZATION_ENABLED_FIELD, RETENTION_PERIOD_FIELD, CREATED_TIMESTAMP_FIELD,
            UPDATED_TIMESTAMP_FIELD));

    private static final long serialVersionUID = 1L;

    private final String eventDataStoreArn;

    private final String name;

    private final Boolean terminationProtectionEnabled;

    private final String status;

    private final List<AdvancedEventSelector> advancedEventSelectors;

    private final Boolean multiRegionEnabled;

    private final Boolean organizationEnabled;

    private final Integer retentionPeriod;

    private final Instant createdTimestamp;

    private final Instant updatedTimestamp;

    private EventDataStore(BuilderImpl builder) {
        this.eventDataStoreArn = builder.eventDataStoreArn;
        this.name = builder.name;
        this.terminationProtectionEnabled = builder.terminationProtectionEnabled;
        this.status = builder.status;
        this.advancedEventSelectors = builder.advancedEventSelectors;
        this.multiRegionEnabled = builder.multiRegionEnabled;
        this.organizationEnabled = builder.organizationEnabled;
        this.retentionPeriod = builder.retentionPeriod;
        this.createdTimestamp = builder.createdTimestamp;
        this.updatedTimestamp = builder.updatedTimestamp;
    }

    /**
     * <p>
     * The ARN of the event data store.
     * </p>
     * 
     * @return The ARN of the event data store.
     */
    public final String eventDataStoreArn() {
        return eventDataStoreArn;
    }

    /**
     * <p>
     * The name of the event data store.
     * </p>
     * 
     * @return The name of the event data store.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * Indicates whether the event data store is protected from termination.
     * </p>
     * 
     * @return Indicates whether the event data store is protected from termination.
     * @deprecated TerminationProtectionEnabled is no longer returned by ListEventDataStores
     */
    @Deprecated
    public final Boolean terminationProtectionEnabled() {
        return terminationProtectionEnabled;
    }

    /**
     * <p>
     * The status of an event data store.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link EventDataStoreStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of an event data store.
     * @see EventDataStoreStatus
     * @deprecated Status is no longer returned by ListEventDataStores
     */
    @Deprecated
    public final EventDataStoreStatus status() {
        return EventDataStoreStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of an event data store.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link EventDataStoreStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of an event data store.
     * @see EventDataStoreStatus
     * @deprecated Status is no longer returned by ListEventDataStores
     */
    @Deprecated
    public final String statusAsString() {
        return status;
    }

    /**
     * For responses, this returns true if the service returned a value for the AdvancedEventSelectors property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     *
     * @deprecated AdvancedEventSelectors is no longer returned by ListEventDataStores
     */
    @Deprecated
    public final boolean hasAdvancedEventSelectors() {
        return advancedEventSelectors != null && !(advancedEventSelectors instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The advanced event selectors that were used to select events for the data store.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAdvancedEventSelectors} method.
     * </p>
     * 
     * @return The advanced event selectors that were used to select events for the data store.
     * @deprecated AdvancedEventSelectors is no longer returned by ListEventDataStores
     */
    @Deprecated
    public final List<AdvancedEventSelector> advancedEventSelectors() {
        return advancedEventSelectors;
    }

    /**
     * <p>
     * Indicates whether the event data store includes events from all Regions, or only from the Region in which it was
     * created.
     * </p>
     * 
     * @return Indicates whether the event data store includes events from all Regions, or only from the Region in which
     *         it was created.
     * @deprecated MultiRegionEnabled is no longer returned by ListEventDataStores
     */
    @Deprecated
    public final Boolean multiRegionEnabled() {
        return multiRegionEnabled;
    }

    /**
     * <p>
     * Indicates that an event data store is collecting logged events for an organization.
     * </p>
     * 
     * @return Indicates that an event data store is collecting logged events for an organization.
     * @deprecated OrganizationEnabled is no longer returned by ListEventDataStores
     */
    @Deprecated
    public final Boolean organizationEnabled() {
        return organizationEnabled;
    }

    /**
     * <p>
     * The retention period, in days.
     * </p>
     * 
     * @return The retention period, in days.
     * @deprecated RetentionPeriod is no longer returned by ListEventDataStores
     */
    @Deprecated
    public final Integer retentionPeriod() {
        return retentionPeriod;
    }

    /**
     * <p>
     * The timestamp of the event data store's creation.
     * </p>
     * 
     * @return The timestamp of the event data store's creation.
     * @deprecated CreatedTimestamp is no longer returned by ListEventDataStores
     */
    @Deprecated
    public final Instant createdTimestamp() {
        return createdTimestamp;
    }

    /**
     * <p>
     * The timestamp showing when an event data store was updated, if applicable. <code>UpdatedTimestamp</code> is
     * always either the same or newer than the time shown in <code>CreatedTimestamp</code>.
     * </p>
     * 
     * @return The timestamp showing when an event data store was updated, if applicable. <code>UpdatedTimestamp</code>
     *         is always either the same or newer than the time shown in <code>CreatedTimestamp</code>.
     * @deprecated UpdatedTimestamp is no longer returned by ListEventDataStores
     */
    @Deprecated
    public final Instant updatedTimestamp() {
        return updatedTimestamp;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(eventDataStoreArn());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(terminationProtectionEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasAdvancedEventSelectors() ? advancedEventSelectors() : null);
        hashCode = 31 * hashCode + Objects.hashCode(multiRegionEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(organizationEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(retentionPeriod());
        hashCode = 31 * hashCode + Objects.hashCode(createdTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(updatedTimestamp());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EventDataStore)) {
            return false;
        }
        EventDataStore other = (EventDataStore) obj;
        return Objects.equals(eventDataStoreArn(), other.eventDataStoreArn()) && Objects.equals(name(), other.name())
                && Objects.equals(terminationProtectionEnabled(), other.terminationProtectionEnabled())
                && Objects.equals(statusAsString(), other.statusAsString())
                && hasAdvancedEventSelectors() == other.hasAdvancedEventSelectors()
                && Objects.equals(advancedEventSelectors(), other.advancedEventSelectors())
                && Objects.equals(multiRegionEnabled(), other.multiRegionEnabled())
                && Objects.equals(organizationEnabled(), other.organizationEnabled())
                && Objects.equals(retentionPeriod(), other.retentionPeriod())
                && Objects.equals(createdTimestamp(), other.createdTimestamp())
                && Objects.equals(updatedTimestamp(), other.updatedTimestamp());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EventDataStore").add("EventDataStoreArn", eventDataStoreArn()).add("Name", name())
                .add("TerminationProtectionEnabled", terminationProtectionEnabled()).add("Status", statusAsString())
                .add("AdvancedEventSelectors", hasAdvancedEventSelectors() ? advancedEventSelectors() : null)
                .add("MultiRegionEnabled", multiRegionEnabled()).add("OrganizationEnabled", organizationEnabled())
                .add("RetentionPeriod", retentionPeriod()).add("CreatedTimestamp", createdTimestamp())
                .add("UpdatedTimestamp", updatedTimestamp()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "EventDataStoreArn":
            return Optional.ofNullable(clazz.cast(eventDataStoreArn()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "TerminationProtectionEnabled":
            return Optional.ofNullable(clazz.cast(terminationProtectionEnabled()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "AdvancedEventSelectors":
            return Optional.ofNullable(clazz.cast(advancedEventSelectors()));
        case "MultiRegionEnabled":
            return Optional.ofNullable(clazz.cast(multiRegionEnabled()));
        case "OrganizationEnabled":
            return Optional.ofNullable(clazz.cast(organizationEnabled()));
        case "RetentionPeriod":
            return Optional.ofNullable(clazz.cast(retentionPeriod()));
        case "CreatedTimestamp":
            return Optional.ofNullable(clazz.cast(createdTimestamp()));
        case "UpdatedTimestamp":
            return Optional.ofNullable(clazz.cast(updatedTimestamp()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<EventDataStore, T> g) {
        return obj -> g.apply((EventDataStore) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EventDataStore> {
        /**
         * <p>
         * The ARN of the event data store.
         * </p>
         * 
         * @param eventDataStoreArn
         *        The ARN of the event data store.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventDataStoreArn(String eventDataStoreArn);

        /**
         * <p>
         * The name of the event data store.
         * </p>
         * 
         * @param name
         *        The name of the event data store.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Indicates whether the event data store is protected from termination.
         * </p>
         * 
         * @param terminationProtectionEnabled
         *        Indicates whether the event data store is protected from termination.
         * @return Returns a reference to this object so that method calls can be chained together.
         * @deprecated TerminationProtectionEnabled is no longer returned by ListEventDataStores
         */
        @Deprecated
        Builder terminationProtectionEnabled(Boolean terminationProtectionEnabled);

        /**
         * <p>
         * The status of an event data store.
         * </p>
         * 
         * @param status
         *        The status of an event data store.
         * @see EventDataStoreStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventDataStoreStatus
         * @deprecated Status is no longer returned by ListEventDataStores
         */
        @Deprecated
        Builder status(String status);

        /**
         * <p>
         * The status of an event data store.
         * </p>
         * 
         * @param status
         *        The status of an event data store.
         * @see EventDataStoreStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventDataStoreStatus
         * @deprecated Status is no longer returned by ListEventDataStores
         */
        @Deprecated
        Builder status(EventDataStoreStatus status);

        /**
         * <p>
         * The advanced event selectors that were used to select events for the data store.
         * </p>
         * 
         * @param advancedEventSelectors
         *        The advanced event selectors that were used to select events for the data store.
         * @return Returns a reference to this object so that method calls can be chained together.
         * @deprecated AdvancedEventSelectors is no longer returned by ListEventDataStores
         */
        @Deprecated
        Builder advancedEventSelectors(Collection<AdvancedEventSelector> advancedEventSelectors);

        /**
         * <p>
         * The advanced event selectors that were used to select events for the data store.
         * </p>
         * 
         * @param advancedEventSelectors
         *        The advanced event selectors that were used to select events for the data store.
         * @return Returns a reference to this object so that method calls can be chained together.
         * @deprecated AdvancedEventSelectors is no longer returned by ListEventDataStores
         */
        @Deprecated
        Builder advancedEventSelectors(AdvancedEventSelector... advancedEventSelectors);

        /**
         * <p>
         * The advanced event selectors that were used to select events for the data store.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.cloudtrail.model.AdvancedEventSelector.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.cloudtrail.model.AdvancedEventSelector#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.cloudtrail.model.AdvancedEventSelector.Builder#build()} is called
         * immediately and its result is passed to {@link #advancedEventSelectors(List<AdvancedEventSelector>)}.
         * 
         * @param advancedEventSelectors
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.cloudtrail.model.AdvancedEventSelector.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #advancedEventSelectors(java.util.Collection<AdvancedEventSelector>)
         * @deprecated AdvancedEventSelectors is no longer returned by ListEventDataStores
         */
        @Deprecated
        Builder advancedEventSelectors(Consumer<AdvancedEventSelector.Builder>... advancedEventSelectors);

        /**
         * <p>
         * Indicates whether the event data store includes events from all Regions, or only from the Region in which it
         * was created.
         * </p>
         * 
         * @param multiRegionEnabled
         *        Indicates whether the event data store includes events from all Regions, or only from the Region in
         *        which it was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         * @deprecated MultiRegionEnabled is no longer returned by ListEventDataStores
         */
        @Deprecated
        Builder multiRegionEnabled(Boolean multiRegionEnabled);

        /**
         * <p>
         * Indicates that an event data store is collecting logged events for an organization.
         * </p>
         * 
         * @param organizationEnabled
         *        Indicates that an event data store is collecting logged events for an organization.
         * @return Returns a reference to this object so that method calls can be chained together.
         * @deprecated OrganizationEnabled is no longer returned by ListEventDataStores
         */
        @Deprecated
        Builder organizationEnabled(Boolean organizationEnabled);

        /**
         * <p>
         * The retention period, in days.
         * </p>
         * 
         * @param retentionPeriod
         *        The retention period, in days.
         * @return Returns a reference to this object so that method calls can be chained together.
         * @deprecated RetentionPeriod is no longer returned by ListEventDataStores
         */
        @Deprecated
        Builder retentionPeriod(Integer retentionPeriod);

        /**
         * <p>
         * The timestamp of the event data store's creation.
         * </p>
         * 
         * @param createdTimestamp
         *        The timestamp of the event data store's creation.
         * @return Returns a reference to this object so that method calls can be chained together.
         * @deprecated CreatedTimestamp is no longer returned by ListEventDataStores
         */
        @Deprecated
        Builder createdTimestamp(Instant createdTimestamp);

        /**
         * <p>
         * The timestamp showing when an event data store was updated, if applicable. <code>UpdatedTimestamp</code> is
         * always either the same or newer than the time shown in <code>CreatedTimestamp</code>.
         * </p>
         * 
         * @param updatedTimestamp
         *        The timestamp showing when an event data store was updated, if applicable.
         *        <code>UpdatedTimestamp</code> is always either the same or newer than the time shown in
         *        <code>CreatedTimestamp</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         * @deprecated UpdatedTimestamp is no longer returned by ListEventDataStores
         */
        @Deprecated
        Builder updatedTimestamp(Instant updatedTimestamp);
    }

    static final class BuilderImpl implements Builder {
        private String eventDataStoreArn;

        private String name;

        private Boolean terminationProtectionEnabled;

        private String status;

        private List<AdvancedEventSelector> advancedEventSelectors = DefaultSdkAutoConstructList.getInstance();

        private Boolean multiRegionEnabled;

        private Boolean organizationEnabled;

        private Integer retentionPeriod;

        private Instant createdTimestamp;

        private Instant updatedTimestamp;

        private BuilderImpl() {
        }

        private BuilderImpl(EventDataStore model) {
            eventDataStoreArn(model.eventDataStoreArn);
            name(model.name);
            terminationProtectionEnabled(model.terminationProtectionEnabled);
            status(model.status);
            advancedEventSelectors(model.advancedEventSelectors);
            multiRegionEnabled(model.multiRegionEnabled);
            organizationEnabled(model.organizationEnabled);
            retentionPeriod(model.retentionPeriod);
            createdTimestamp(model.createdTimestamp);
            updatedTimestamp(model.updatedTimestamp);
        }

        public final String getEventDataStoreArn() {
            return eventDataStoreArn;
        }

        public final void setEventDataStoreArn(String eventDataStoreArn) {
            this.eventDataStoreArn = eventDataStoreArn;
        }

        @Override
        public final Builder eventDataStoreArn(String eventDataStoreArn) {
            this.eventDataStoreArn = eventDataStoreArn;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        @Deprecated
        public final Boolean getTerminationProtectionEnabled() {
            return terminationProtectionEnabled;
        }

        @Deprecated
        public final void setTerminationProtectionEnabled(Boolean terminationProtectionEnabled) {
            this.terminationProtectionEnabled = terminationProtectionEnabled;
        }

        @Override
        @Deprecated
        public final Builder terminationProtectionEnabled(Boolean terminationProtectionEnabled) {
            this.terminationProtectionEnabled = terminationProtectionEnabled;
            return this;
        }

        @Deprecated
        public final String getStatus() {
            return status;
        }

        @Deprecated
        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        @Deprecated
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        @Deprecated
        public final Builder status(EventDataStoreStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        @Deprecated
        public final List<AdvancedEventSelector.Builder> getAdvancedEventSelectors() {
            List<AdvancedEventSelector.Builder> result = AdvancedEventSelectorsCopier.copyToBuilder(this.advancedEventSelectors);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Deprecated
        public final void setAdvancedEventSelectors(Collection<AdvancedEventSelector.BuilderImpl> advancedEventSelectors) {
            this.advancedEventSelectors = AdvancedEventSelectorsCopier.copyFromBuilder(advancedEventSelectors);
        }

        @Override
        @Deprecated
        public final Builder advancedEventSelectors(Collection<AdvancedEventSelector> advancedEventSelectors) {
            this.advancedEventSelectors = AdvancedEventSelectorsCopier.copy(advancedEventSelectors);
            return this;
        }

        @Override
        @SafeVarargs
        @Deprecated
        public final Builder advancedEventSelectors(AdvancedEventSelector... advancedEventSelectors) {
            advancedEventSelectors(Arrays.asList(advancedEventSelectors));
            return this;
        }

        @Override
        @SafeVarargs
        @Deprecated
        public final Builder advancedEventSelectors(Consumer<AdvancedEventSelector.Builder>... advancedEventSelectors) {
            advancedEventSelectors(Stream.of(advancedEventSelectors)
                    .map(c -> AdvancedEventSelector.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Deprecated
        public final Boolean getMultiRegionEnabled() {
            return multiRegionEnabled;
        }

        @Deprecated
        public final void setMultiRegionEnabled(Boolean multiRegionEnabled) {
            this.multiRegionEnabled = multiRegionEnabled;
        }

        @Override
        @Deprecated
        public final Builder multiRegionEnabled(Boolean multiRegionEnabled) {
            this.multiRegionEnabled = multiRegionEnabled;
            return this;
        }

        @Deprecated
        public final Boolean getOrganizationEnabled() {
            return organizationEnabled;
        }

        @Deprecated
        public final void setOrganizationEnabled(Boolean organizationEnabled) {
            this.organizationEnabled = organizationEnabled;
        }

        @Override
        @Deprecated
        public final Builder organizationEnabled(Boolean organizationEnabled) {
            this.organizationEnabled = organizationEnabled;
            return this;
        }

        @Deprecated
        public final Integer getRetentionPeriod() {
            return retentionPeriod;
        }

        @Deprecated
        public final void setRetentionPeriod(Integer retentionPeriod) {
            this.retentionPeriod = retentionPeriod;
        }

        @Override
        @Deprecated
        public final Builder retentionPeriod(Integer retentionPeriod) {
            this.retentionPeriod = retentionPeriod;
            return this;
        }

        @Deprecated
        public final Instant getCreatedTimestamp() {
            return createdTimestamp;
        }

        @Deprecated
        public final void setCreatedTimestamp(Instant createdTimestamp) {
            this.createdTimestamp = createdTimestamp;
        }

        @Override
        @Deprecated
        public final Builder createdTimestamp(Instant createdTimestamp) {
            this.createdTimestamp = createdTimestamp;
            return this;
        }

        @Deprecated
        public final Instant getUpdatedTimestamp() {
            return updatedTimestamp;
        }

        @Deprecated
        public final void setUpdatedTimestamp(Instant updatedTimestamp) {
            this.updatedTimestamp = updatedTimestamp;
        }

        @Override
        @Deprecated
        public final Builder updatedTimestamp(Instant updatedTimestamp) {
            this.updatedTimestamp = updatedTimestamp;
            return this;
        }

        @Override
        public EventDataStore build() {
            return new EventDataStore(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
