/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudtrail.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Use event selectors to further specify the management and data event settings for your trail. By default, trails
 * created without specific event selectors will be configured to log all read and write management events, and no data
 * events. When an event occurs in your account, CloudTrail evaluates the event selector for all trails. For each trail,
 * if the event matches any event selector, the trail processes and logs the event. If the event doesn't match any event
 * selector, the trail doesn't log the event.
 * </p>
 * <p>
 * You can configure up to five event selectors for a trail.
 * </p>
 * <p>
 * You cannot apply both event selectors and advanced event selectors to a trail.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EventSelector implements SdkPojo, Serializable, ToCopyableBuilder<EventSelector.Builder, EventSelector> {
    private static final SdkField<String> READ_WRITE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ReadWriteType").getter(getter(EventSelector::readWriteTypeAsString))
            .setter(setter(Builder::readWriteType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReadWriteType").build()).build();

    private static final SdkField<Boolean> INCLUDE_MANAGEMENT_EVENTS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("IncludeManagementEvents").getter(getter(EventSelector::includeManagementEvents))
            .setter(setter(Builder::includeManagementEvents))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IncludeManagementEvents").build())
            .build();

    private static final SdkField<List<DataResource>> DATA_RESOURCES_FIELD = SdkField
            .<List<DataResource>> builder(MarshallingType.LIST)
            .memberName("DataResources")
            .getter(getter(EventSelector::dataResources))
            .setter(setter(Builder::dataResources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataResources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DataResource> builder(MarshallingType.SDK_POJO)
                                            .constructor(DataResource::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> EXCLUDE_MANAGEMENT_EVENT_SOURCES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ExcludeManagementEventSources")
            .getter(getter(EventSelector::excludeManagementEventSources))
            .setter(setter(Builder::excludeManagementEventSources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExcludeManagementEventSources")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(READ_WRITE_TYPE_FIELD,
            INCLUDE_MANAGEMENT_EVENTS_FIELD, DATA_RESOURCES_FIELD, EXCLUDE_MANAGEMENT_EVENT_SOURCES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String readWriteType;

    private final Boolean includeManagementEvents;

    private final List<DataResource> dataResources;

    private final List<String> excludeManagementEventSources;

    private EventSelector(BuilderImpl builder) {
        this.readWriteType = builder.readWriteType;
        this.includeManagementEvents = builder.includeManagementEvents;
        this.dataResources = builder.dataResources;
        this.excludeManagementEventSources = builder.excludeManagementEventSources;
    }

    /**
     * <p>
     * Specify if you want your trail to log read-only events, write-only events, or all. For example, the EC2
     * <code>GetConsoleOutput</code> is a read-only API operation and <code>RunInstances</code> is a write-only API
     * operation.
     * </p>
     * <p>
     * By default, the value is <code>All</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #readWriteType}
     * will return {@link ReadWriteType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #readWriteTypeAsString}.
     * </p>
     * 
     * @return Specify if you want your trail to log read-only events, write-only events, or all. For example, the EC2
     *         <code>GetConsoleOutput</code> is a read-only API operation and <code>RunInstances</code> is a write-only
     *         API operation.</p>
     *         <p>
     *         By default, the value is <code>All</code>.
     * @see ReadWriteType
     */
    public final ReadWriteType readWriteType() {
        return ReadWriteType.fromValue(readWriteType);
    }

    /**
     * <p>
     * Specify if you want your trail to log read-only events, write-only events, or all. For example, the EC2
     * <code>GetConsoleOutput</code> is a read-only API operation and <code>RunInstances</code> is a write-only API
     * operation.
     * </p>
     * <p>
     * By default, the value is <code>All</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #readWriteType}
     * will return {@link ReadWriteType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #readWriteTypeAsString}.
     * </p>
     * 
     * @return Specify if you want your trail to log read-only events, write-only events, or all. For example, the EC2
     *         <code>GetConsoleOutput</code> is a read-only API operation and <code>RunInstances</code> is a write-only
     *         API operation.</p>
     *         <p>
     *         By default, the value is <code>All</code>.
     * @see ReadWriteType
     */
    public final String readWriteTypeAsString() {
        return readWriteType;
    }

    /**
     * <p>
     * Specify if you want your event selector to include management events for your trail.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-management-events-with-cloudtrail.html"
     * >Management Events</a> in the <i>CloudTrail User Guide</i>.
     * </p>
     * <p>
     * By default, the value is <code>true</code>.
     * </p>
     * <p>
     * The first copy of management events is free. You are charged for additional copies of management events that you
     * are logging on any subsequent trail in the same Region. For more information about CloudTrail pricing, see <a
     * href="http://aws.amazon.com/cloudtrail/pricing/">CloudTrail Pricing</a>.
     * </p>
     * 
     * @return Specify if you want your event selector to include management events for your trail.</p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-management-events-with-cloudtrail.html"
     *         >Management Events</a> in the <i>CloudTrail User Guide</i>.
     *         </p>
     *         <p>
     *         By default, the value is <code>true</code>.
     *         </p>
     *         <p>
     *         The first copy of management events is free. You are charged for additional copies of management events
     *         that you are logging on any subsequent trail in the same Region. For more information about CloudTrail
     *         pricing, see <a href="http://aws.amazon.com/cloudtrail/pricing/">CloudTrail Pricing</a>.
     */
    public final Boolean includeManagementEvents() {
        return includeManagementEvents;
    }

    /**
     * For responses, this returns true if the service returned a value for the DataResources property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDataResources() {
        return dataResources != null && !(dataResources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * CloudTrail supports data event logging for Amazon S3 objects, Lambda functions, and Amazon DynamoDB tables with
     * basic event selectors. You can specify up to 250 resources for an individual event selector, but the total number
     * of data resources cannot exceed 250 across all event selectors in a trail. This limit does not apply if you
     * configure resource logging for all data events.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-data-events-with-cloudtrail.html">Data
     * Events</a> and <a
     * href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/WhatIsCloudTrail-Limits.html">Limits in
     * CloudTrail</a> in the <i>CloudTrail User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDataResources} method.
     * </p>
     * 
     * @return CloudTrail supports data event logging for Amazon S3 objects, Lambda functions, and Amazon DynamoDB
     *         tables with basic event selectors. You can specify up to 250 resources for an individual event selector,
     *         but the total number of data resources cannot exceed 250 across all event selectors in a trail. This
     *         limit does not apply if you configure resource logging for all data events.</p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-data-events-with-cloudtrail.html"
     *         >Data Events</a> and <a
     *         href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/WhatIsCloudTrail-Limits.html">Limits in
     *         CloudTrail</a> in the <i>CloudTrail User Guide</i>.
     */
    public final List<DataResource> dataResources() {
        return dataResources;
    }

    /**
     * For responses, this returns true if the service returned a value for the ExcludeManagementEventSources property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasExcludeManagementEventSources() {
        return excludeManagementEventSources != null && !(excludeManagementEventSources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An optional list of service event sources from which you do not want management events to be logged on your
     * trail. In this release, the list can be empty (disables the filter), or it can filter out Key Management Service
     * or Amazon RDS Data API events by containing <code>kms.amazonaws.com</code> or <code>rdsdata.amazonaws.com</code>.
     * By default, <code>ExcludeManagementEventSources</code> is empty, and KMS and Amazon RDS Data API events are
     * logged to your trail. You can exclude management event sources only in Regions that support the event source.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasExcludeManagementEventSources} method.
     * </p>
     * 
     * @return An optional list of service event sources from which you do not want management events to be logged on
     *         your trail. In this release, the list can be empty (disables the filter), or it can filter out Key
     *         Management Service or Amazon RDS Data API events by containing <code>kms.amazonaws.com</code> or
     *         <code>rdsdata.amazonaws.com</code>. By default, <code>ExcludeManagementEventSources</code> is empty, and
     *         KMS and Amazon RDS Data API events are logged to your trail. You can exclude management event sources
     *         only in Regions that support the event source.
     */
    public final List<String> excludeManagementEventSources() {
        return excludeManagementEventSources;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(readWriteTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(includeManagementEvents());
        hashCode = 31 * hashCode + Objects.hashCode(hasDataResources() ? dataResources() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasExcludeManagementEventSources() ? excludeManagementEventSources() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EventSelector)) {
            return false;
        }
        EventSelector other = (EventSelector) obj;
        return Objects.equals(readWriteTypeAsString(), other.readWriteTypeAsString())
                && Objects.equals(includeManagementEvents(), other.includeManagementEvents())
                && hasDataResources() == other.hasDataResources() && Objects.equals(dataResources(), other.dataResources())
                && hasExcludeManagementEventSources() == other.hasExcludeManagementEventSources()
                && Objects.equals(excludeManagementEventSources(), other.excludeManagementEventSources());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("EventSelector")
                .add("ReadWriteType", readWriteTypeAsString())
                .add("IncludeManagementEvents", includeManagementEvents())
                .add("DataResources", hasDataResources() ? dataResources() : null)
                .add("ExcludeManagementEventSources", hasExcludeManagementEventSources() ? excludeManagementEventSources() : null)
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ReadWriteType":
            return Optional.ofNullable(clazz.cast(readWriteTypeAsString()));
        case "IncludeManagementEvents":
            return Optional.ofNullable(clazz.cast(includeManagementEvents()));
        case "DataResources":
            return Optional.ofNullable(clazz.cast(dataResources()));
        case "ExcludeManagementEventSources":
            return Optional.ofNullable(clazz.cast(excludeManagementEventSources()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<EventSelector, T> g) {
        return obj -> g.apply((EventSelector) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EventSelector> {
        /**
         * <p>
         * Specify if you want your trail to log read-only events, write-only events, or all. For example, the EC2
         * <code>GetConsoleOutput</code> is a read-only API operation and <code>RunInstances</code> is a write-only API
         * operation.
         * </p>
         * <p>
         * By default, the value is <code>All</code>.
         * </p>
         * 
         * @param readWriteType
         *        Specify if you want your trail to log read-only events, write-only events, or all. For example, the
         *        EC2 <code>GetConsoleOutput</code> is a read-only API operation and <code>RunInstances</code> is a
         *        write-only API operation.</p>
         *        <p>
         *        By default, the value is <code>All</code>.
         * @see ReadWriteType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReadWriteType
         */
        Builder readWriteType(String readWriteType);

        /**
         * <p>
         * Specify if you want your trail to log read-only events, write-only events, or all. For example, the EC2
         * <code>GetConsoleOutput</code> is a read-only API operation and <code>RunInstances</code> is a write-only API
         * operation.
         * </p>
         * <p>
         * By default, the value is <code>All</code>.
         * </p>
         * 
         * @param readWriteType
         *        Specify if you want your trail to log read-only events, write-only events, or all. For example, the
         *        EC2 <code>GetConsoleOutput</code> is a read-only API operation and <code>RunInstances</code> is a
         *        write-only API operation.</p>
         *        <p>
         *        By default, the value is <code>All</code>.
         * @see ReadWriteType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReadWriteType
         */
        Builder readWriteType(ReadWriteType readWriteType);

        /**
         * <p>
         * Specify if you want your event selector to include management events for your trail.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-management-events-with-cloudtrail.html"
         * >Management Events</a> in the <i>CloudTrail User Guide</i>.
         * </p>
         * <p>
         * By default, the value is <code>true</code>.
         * </p>
         * <p>
         * The first copy of management events is free. You are charged for additional copies of management events that
         * you are logging on any subsequent trail in the same Region. For more information about CloudTrail pricing,
         * see <a href="http://aws.amazon.com/cloudtrail/pricing/">CloudTrail Pricing</a>.
         * </p>
         * 
         * @param includeManagementEvents
         *        Specify if you want your event selector to include management events for your trail.</p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-management-events-with-cloudtrail.html"
         *        >Management Events</a> in the <i>CloudTrail User Guide</i>.
         *        </p>
         *        <p>
         *        By default, the value is <code>true</code>.
         *        </p>
         *        <p>
         *        The first copy of management events is free. You are charged for additional copies of management
         *        events that you are logging on any subsequent trail in the same Region. For more information about
         *        CloudTrail pricing, see <a href="http://aws.amazon.com/cloudtrail/pricing/">CloudTrail Pricing</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includeManagementEvents(Boolean includeManagementEvents);

        /**
         * <p>
         * CloudTrail supports data event logging for Amazon S3 objects, Lambda functions, and Amazon DynamoDB tables
         * with basic event selectors. You can specify up to 250 resources for an individual event selector, but the
         * total number of data resources cannot exceed 250 across all event selectors in a trail. This limit does not
         * apply if you configure resource logging for all data events.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-data-events-with-cloudtrail.html"
         * >Data Events</a> and <a
         * href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/WhatIsCloudTrail-Limits.html">Limits in
         * CloudTrail</a> in the <i>CloudTrail User Guide</i>.
         * </p>
         * 
         * @param dataResources
         *        CloudTrail supports data event logging for Amazon S3 objects, Lambda functions, and Amazon DynamoDB
         *        tables with basic event selectors. You can specify up to 250 resources for an individual event
         *        selector, but the total number of data resources cannot exceed 250 across all event selectors in a
         *        trail. This limit does not apply if you configure resource logging for all data events.</p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-data-events-with-cloudtrail.html"
         *        >Data Events</a> and <a
         *        href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/WhatIsCloudTrail-Limits.html">Limits
         *        in CloudTrail</a> in the <i>CloudTrail User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataResources(Collection<DataResource> dataResources);

        /**
         * <p>
         * CloudTrail supports data event logging for Amazon S3 objects, Lambda functions, and Amazon DynamoDB tables
         * with basic event selectors. You can specify up to 250 resources for an individual event selector, but the
         * total number of data resources cannot exceed 250 across all event selectors in a trail. This limit does not
         * apply if you configure resource logging for all data events.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-data-events-with-cloudtrail.html"
         * >Data Events</a> and <a
         * href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/WhatIsCloudTrail-Limits.html">Limits in
         * CloudTrail</a> in the <i>CloudTrail User Guide</i>.
         * </p>
         * 
         * @param dataResources
         *        CloudTrail supports data event logging for Amazon S3 objects, Lambda functions, and Amazon DynamoDB
         *        tables with basic event selectors. You can specify up to 250 resources for an individual event
         *        selector, but the total number of data resources cannot exceed 250 across all event selectors in a
         *        trail. This limit does not apply if you configure resource logging for all data events.</p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-data-events-with-cloudtrail.html"
         *        >Data Events</a> and <a
         *        href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/WhatIsCloudTrail-Limits.html">Limits
         *        in CloudTrail</a> in the <i>CloudTrail User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataResources(DataResource... dataResources);

        /**
         * <p>
         * CloudTrail supports data event logging for Amazon S3 objects, Lambda functions, and Amazon DynamoDB tables
         * with basic event selectors. You can specify up to 250 resources for an individual event selector, but the
         * total number of data resources cannot exceed 250 across all event selectors in a trail. This limit does not
         * apply if you configure resource logging for all data events.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-data-events-with-cloudtrail.html"
         * >Data Events</a> and <a
         * href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/WhatIsCloudTrail-Limits.html">Limits in
         * CloudTrail</a> in the <i>CloudTrail User Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.cloudtrail.model.DataResource.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.cloudtrail.model.DataResource#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.cloudtrail.model.DataResource.Builder#build()} is called immediately
         * and its result is passed to {@link #dataResources(List<DataResource>)}.
         * 
         * @param dataResources
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.cloudtrail.model.DataResource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dataResources(java.util.Collection<DataResource>)
         */
        Builder dataResources(Consumer<DataResource.Builder>... dataResources);

        /**
         * <p>
         * An optional list of service event sources from which you do not want management events to be logged on your
         * trail. In this release, the list can be empty (disables the filter), or it can filter out Key Management
         * Service or Amazon RDS Data API events by containing <code>kms.amazonaws.com</code> or
         * <code>rdsdata.amazonaws.com</code>. By default, <code>ExcludeManagementEventSources</code> is empty, and KMS
         * and Amazon RDS Data API events are logged to your trail. You can exclude management event sources only in
         * Regions that support the event source.
         * </p>
         * 
         * @param excludeManagementEventSources
         *        An optional list of service event sources from which you do not want management events to be logged on
         *        your trail. In this release, the list can be empty (disables the filter), or it can filter out Key
         *        Management Service or Amazon RDS Data API events by containing <code>kms.amazonaws.com</code> or
         *        <code>rdsdata.amazonaws.com</code>. By default, <code>ExcludeManagementEventSources</code> is empty,
         *        and KMS and Amazon RDS Data API events are logged to your trail. You can exclude management event
         *        sources only in Regions that support the event source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludeManagementEventSources(Collection<String> excludeManagementEventSources);

        /**
         * <p>
         * An optional list of service event sources from which you do not want management events to be logged on your
         * trail. In this release, the list can be empty (disables the filter), or it can filter out Key Management
         * Service or Amazon RDS Data API events by containing <code>kms.amazonaws.com</code> or
         * <code>rdsdata.amazonaws.com</code>. By default, <code>ExcludeManagementEventSources</code> is empty, and KMS
         * and Amazon RDS Data API events are logged to your trail. You can exclude management event sources only in
         * Regions that support the event source.
         * </p>
         * 
         * @param excludeManagementEventSources
         *        An optional list of service event sources from which you do not want management events to be logged on
         *        your trail. In this release, the list can be empty (disables the filter), or it can filter out Key
         *        Management Service or Amazon RDS Data API events by containing <code>kms.amazonaws.com</code> or
         *        <code>rdsdata.amazonaws.com</code>. By default, <code>ExcludeManagementEventSources</code> is empty,
         *        and KMS and Amazon RDS Data API events are logged to your trail. You can exclude management event
         *        sources only in Regions that support the event source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludeManagementEventSources(String... excludeManagementEventSources);
    }

    static final class BuilderImpl implements Builder {
        private String readWriteType;

        private Boolean includeManagementEvents;

        private List<DataResource> dataResources = DefaultSdkAutoConstructList.getInstance();

        private List<String> excludeManagementEventSources = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(EventSelector model) {
            readWriteType(model.readWriteType);
            includeManagementEvents(model.includeManagementEvents);
            dataResources(model.dataResources);
            excludeManagementEventSources(model.excludeManagementEventSources);
        }

        public final String getReadWriteType() {
            return readWriteType;
        }

        public final void setReadWriteType(String readWriteType) {
            this.readWriteType = readWriteType;
        }

        @Override
        public final Builder readWriteType(String readWriteType) {
            this.readWriteType = readWriteType;
            return this;
        }

        @Override
        public final Builder readWriteType(ReadWriteType readWriteType) {
            this.readWriteType(readWriteType == null ? null : readWriteType.toString());
            return this;
        }

        public final Boolean getIncludeManagementEvents() {
            return includeManagementEvents;
        }

        public final void setIncludeManagementEvents(Boolean includeManagementEvents) {
            this.includeManagementEvents = includeManagementEvents;
        }

        @Override
        public final Builder includeManagementEvents(Boolean includeManagementEvents) {
            this.includeManagementEvents = includeManagementEvents;
            return this;
        }

        public final List<DataResource.Builder> getDataResources() {
            List<DataResource.Builder> result = DataResourcesCopier.copyToBuilder(this.dataResources);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDataResources(Collection<DataResource.BuilderImpl> dataResources) {
            this.dataResources = DataResourcesCopier.copyFromBuilder(dataResources);
        }

        @Override
        public final Builder dataResources(Collection<DataResource> dataResources) {
            this.dataResources = DataResourcesCopier.copy(dataResources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dataResources(DataResource... dataResources) {
            dataResources(Arrays.asList(dataResources));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dataResources(Consumer<DataResource.Builder>... dataResources) {
            dataResources(Stream.of(dataResources).map(c -> DataResource.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Collection<String> getExcludeManagementEventSources() {
            if (excludeManagementEventSources instanceof SdkAutoConstructList) {
                return null;
            }
            return excludeManagementEventSources;
        }

        public final void setExcludeManagementEventSources(Collection<String> excludeManagementEventSources) {
            this.excludeManagementEventSources = ExcludeManagementEventSourcesCopier.copy(excludeManagementEventSources);
        }

        @Override
        public final Builder excludeManagementEventSources(Collection<String> excludeManagementEventSources) {
            this.excludeManagementEventSources = ExcludeManagementEventSourcesCopier.copy(excludeManagementEventSources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder excludeManagementEventSources(String... excludeManagementEventSources) {
            excludeManagementEventSources(Arrays.asList(excludeManagementEventSources));
            return this;
        }

        @Override
        public EventSelector build() {
            return new EventSelector(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
