/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the result of testing a CloudFront function with <code>TestFunction</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TestResult implements SdkPojo, Serializable, ToCopyableBuilder<TestResult.Builder, TestResult> {
    private static final SdkField<FunctionSummary> FUNCTION_SUMMARY_FIELD = SdkField
            .<FunctionSummary> builder(MarshallingType.SDK_POJO)
            .memberName("FunctionSummary")
            .getter(getter(TestResult::functionSummary))
            .setter(setter(Builder::functionSummary))
            .constructor(FunctionSummary::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FunctionSummary")
                    .unmarshallLocationName("FunctionSummary").build()).build();

    private static final SdkField<String> COMPUTE_UTILIZATION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ComputeUtilization")
            .getter(getter(TestResult::computeUtilization))
            .setter(setter(Builder::computeUtilization))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ComputeUtilization")
                    .unmarshallLocationName("ComputeUtilization").build()).build();

    private static final SdkField<List<String>> FUNCTION_EXECUTION_LOGS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("FunctionExecutionLogs")
            .getter(getter(TestResult::functionExecutionLogs))
            .setter(setter(Builder::functionExecutionLogs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FunctionExecutionLogs")
                    .unmarshallLocationName("FunctionExecutionLogs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").unmarshallLocationName("member").build()).build())
                            .build()).build();

    private static final SdkField<String> FUNCTION_ERROR_MESSAGE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("FunctionErrorMessage")
            .getter(getter(TestResult::functionErrorMessage))
            .setter(setter(Builder::functionErrorMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FunctionErrorMessage")
                    .unmarshallLocationName("FunctionErrorMessage").build()).build();

    private static final SdkField<String> FUNCTION_OUTPUT_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("FunctionOutput")
            .getter(getter(TestResult::functionOutput))
            .setter(setter(Builder::functionOutput))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FunctionOutput")
                    .unmarshallLocationName("FunctionOutput").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FUNCTION_SUMMARY_FIELD,
            COMPUTE_UTILIZATION_FIELD, FUNCTION_EXECUTION_LOGS_FIELD, FUNCTION_ERROR_MESSAGE_FIELD, FUNCTION_OUTPUT_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final FunctionSummary functionSummary;

    private final String computeUtilization;

    private final List<String> functionExecutionLogs;

    private final String functionErrorMessage;

    private final String functionOutput;

    private TestResult(BuilderImpl builder) {
        this.functionSummary = builder.functionSummary;
        this.computeUtilization = builder.computeUtilization;
        this.functionExecutionLogs = builder.functionExecutionLogs;
        this.functionErrorMessage = builder.functionErrorMessage;
        this.functionOutput = builder.functionOutput;
    }

    /**
     * <p>
     * Contains configuration information and metadata about the CloudFront function that was tested.
     * </p>
     * 
     * @return Contains configuration information and metadata about the CloudFront function that was tested.
     */
    public final FunctionSummary functionSummary() {
        return functionSummary;
    }

    /**
     * <p>
     * The amount of time that the function took to run as a percentage of the maximum allowed time. For example, a
     * compute utilization of 35 means that the function completed in 35% of the maximum allowed time.
     * </p>
     * 
     * @return The amount of time that the function took to run as a percentage of the maximum allowed time. For
     *         example, a compute utilization of 35 means that the function completed in 35% of the maximum allowed
     *         time.
     */
    public final String computeUtilization() {
        return computeUtilization;
    }

    /**
     * For responses, this returns true if the service returned a value for the FunctionExecutionLogs property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasFunctionExecutionLogs() {
        return functionExecutionLogs != null && !(functionExecutionLogs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Contains the log lines that the function wrote (if any) when running the test.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFunctionExecutionLogs} method.
     * </p>
     * 
     * @return Contains the log lines that the function wrote (if any) when running the test.
     */
    public final List<String> functionExecutionLogs() {
        return functionExecutionLogs;
    }

    /**
     * <p>
     * If the result of testing the function was an error, this field contains the error message.
     * </p>
     * 
     * @return If the result of testing the function was an error, this field contains the error message.
     */
    public final String functionErrorMessage() {
        return functionErrorMessage;
    }

    /**
     * <p>
     * The event object returned by the function. For more information about the structure of the event object, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/functions-event-structure.html">Event
     * object structure</a> in the <i>Amazon CloudFront Developer Guide</i>.
     * </p>
     * 
     * @return The event object returned by the function. For more information about the structure of the event object,
     *         see <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/functions-event-structure.html">Event
     *         object structure</a> in the <i>Amazon CloudFront Developer Guide</i>.
     */
    public final String functionOutput() {
        return functionOutput;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(functionSummary());
        hashCode = 31 * hashCode + Objects.hashCode(computeUtilization());
        hashCode = 31 * hashCode + Objects.hashCode(hasFunctionExecutionLogs() ? functionExecutionLogs() : null);
        hashCode = 31 * hashCode + Objects.hashCode(functionErrorMessage());
        hashCode = 31 * hashCode + Objects.hashCode(functionOutput());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TestResult)) {
            return false;
        }
        TestResult other = (TestResult) obj;
        return Objects.equals(functionSummary(), other.functionSummary())
                && Objects.equals(computeUtilization(), other.computeUtilization())
                && hasFunctionExecutionLogs() == other.hasFunctionExecutionLogs()
                && Objects.equals(functionExecutionLogs(), other.functionExecutionLogs())
                && Objects.equals(functionErrorMessage(), other.functionErrorMessage())
                && Objects.equals(functionOutput(), other.functionOutput());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TestResult").add("FunctionSummary", functionSummary())
                .add("ComputeUtilization", computeUtilization())
                .add("FunctionExecutionLogs", functionExecutionLogs() == null ? null : "*** Sensitive Data Redacted ***")
                .add("FunctionErrorMessage", functionErrorMessage() == null ? null : "*** Sensitive Data Redacted ***")
                .add("FunctionOutput", functionOutput() == null ? null : "*** Sensitive Data Redacted ***").build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "FunctionSummary":
            return Optional.ofNullable(clazz.cast(functionSummary()));
        case "ComputeUtilization":
            return Optional.ofNullable(clazz.cast(computeUtilization()));
        case "FunctionExecutionLogs":
            return Optional.ofNullable(clazz.cast(functionExecutionLogs()));
        case "FunctionErrorMessage":
            return Optional.ofNullable(clazz.cast(functionErrorMessage()));
        case "FunctionOutput":
            return Optional.ofNullable(clazz.cast(functionOutput()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("FunctionSummary", FUNCTION_SUMMARY_FIELD);
        map.put("ComputeUtilization", COMPUTE_UTILIZATION_FIELD);
        map.put("FunctionExecutionLogs", FUNCTION_EXECUTION_LOGS_FIELD);
        map.put("FunctionErrorMessage", FUNCTION_ERROR_MESSAGE_FIELD);
        map.put("FunctionOutput", FUNCTION_OUTPUT_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<TestResult, T> g) {
        return obj -> g.apply((TestResult) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TestResult> {
        /**
         * <p>
         * Contains configuration information and metadata about the CloudFront function that was tested.
         * </p>
         * 
         * @param functionSummary
         *        Contains configuration information and metadata about the CloudFront function that was tested.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder functionSummary(FunctionSummary functionSummary);

        /**
         * <p>
         * Contains configuration information and metadata about the CloudFront function that was tested.
         * </p>
         * This is a convenience method that creates an instance of the {@link FunctionSummary.Builder} avoiding the
         * need to create one manually via {@link FunctionSummary#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link FunctionSummary.Builder#build()} is called immediately and its
         * result is passed to {@link #functionSummary(FunctionSummary)}.
         * 
         * @param functionSummary
         *        a consumer that will call methods on {@link FunctionSummary.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #functionSummary(FunctionSummary)
         */
        default Builder functionSummary(Consumer<FunctionSummary.Builder> functionSummary) {
            return functionSummary(FunctionSummary.builder().applyMutation(functionSummary).build());
        }

        /**
         * <p>
         * The amount of time that the function took to run as a percentage of the maximum allowed time. For example, a
         * compute utilization of 35 means that the function completed in 35% of the maximum allowed time.
         * </p>
         * 
         * @param computeUtilization
         *        The amount of time that the function took to run as a percentage of the maximum allowed time. For
         *        example, a compute utilization of 35 means that the function completed in 35% of the maximum allowed
         *        time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder computeUtilization(String computeUtilization);

        /**
         * <p>
         * Contains the log lines that the function wrote (if any) when running the test.
         * </p>
         * 
         * @param functionExecutionLogs
         *        Contains the log lines that the function wrote (if any) when running the test.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder functionExecutionLogs(Collection<String> functionExecutionLogs);

        /**
         * <p>
         * Contains the log lines that the function wrote (if any) when running the test.
         * </p>
         * 
         * @param functionExecutionLogs
         *        Contains the log lines that the function wrote (if any) when running the test.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder functionExecutionLogs(String... functionExecutionLogs);

        /**
         * <p>
         * If the result of testing the function was an error, this field contains the error message.
         * </p>
         * 
         * @param functionErrorMessage
         *        If the result of testing the function was an error, this field contains the error message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder functionErrorMessage(String functionErrorMessage);

        /**
         * <p>
         * The event object returned by the function. For more information about the structure of the event object, see
         * <a href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/functions-event-structure.html">
         * Event object structure</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * </p>
         * 
         * @param functionOutput
         *        The event object returned by the function. For more information about the structure of the event
         *        object, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/functions-event-structure.html"
         *        >Event object structure</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder functionOutput(String functionOutput);
    }

    static final class BuilderImpl implements Builder {
        private FunctionSummary functionSummary;

        private String computeUtilization;

        private List<String> functionExecutionLogs = DefaultSdkAutoConstructList.getInstance();

        private String functionErrorMessage;

        private String functionOutput;

        private BuilderImpl() {
        }

        private BuilderImpl(TestResult model) {
            functionSummary(model.functionSummary);
            computeUtilization(model.computeUtilization);
            functionExecutionLogs(model.functionExecutionLogs);
            functionErrorMessage(model.functionErrorMessage);
            functionOutput(model.functionOutput);
        }

        public final FunctionSummary.Builder getFunctionSummary() {
            return functionSummary != null ? functionSummary.toBuilder() : null;
        }

        public final void setFunctionSummary(FunctionSummary.BuilderImpl functionSummary) {
            this.functionSummary = functionSummary != null ? functionSummary.build() : null;
        }

        @Override
        public final Builder functionSummary(FunctionSummary functionSummary) {
            this.functionSummary = functionSummary;
            return this;
        }

        public final String getComputeUtilization() {
            return computeUtilization;
        }

        public final void setComputeUtilization(String computeUtilization) {
            this.computeUtilization = computeUtilization;
        }

        @Override
        public final Builder computeUtilization(String computeUtilization) {
            this.computeUtilization = computeUtilization;
            return this;
        }

        public final Collection<String> getFunctionExecutionLogs() {
            if (functionExecutionLogs instanceof SdkAutoConstructList) {
                return null;
            }
            return functionExecutionLogs;
        }

        public final void setFunctionExecutionLogs(Collection<String> functionExecutionLogs) {
            this.functionExecutionLogs = FunctionExecutionLogListCopier.copy(functionExecutionLogs);
        }

        @Override
        public final Builder functionExecutionLogs(Collection<String> functionExecutionLogs) {
            this.functionExecutionLogs = FunctionExecutionLogListCopier.copy(functionExecutionLogs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder functionExecutionLogs(String... functionExecutionLogs) {
            functionExecutionLogs(Arrays.asList(functionExecutionLogs));
            return this;
        }

        public final String getFunctionErrorMessage() {
            return functionErrorMessage;
        }

        public final void setFunctionErrorMessage(String functionErrorMessage) {
            this.functionErrorMessage = functionErrorMessage;
        }

        @Override
        public final Builder functionErrorMessage(String functionErrorMessage) {
            this.functionErrorMessage = functionErrorMessage;
            return this;
        }

        public final String getFunctionOutput() {
            return functionOutput;
        }

        public final void setFunctionOutput(String functionOutput) {
            this.functionOutput = functionOutput;
        }

        @Override
        public final Builder functionOutput(String functionOutput) {
            this.functionOutput = functionOutput;
            return this;
        }

        @Override
        public TestResult build() {
            return new TestResult(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
