/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudformation.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies the Organizations accounts where you want to create, update, or delete stack instances. You can target
 * either your entire organization or specific accounts using organizational units (OUs) and account filter options.
 * </p>
 * <p>
 * CloudFormation doesn't deploy stack instances to the organization management account, even if the organization
 * management account is in your organization or in an OU in your organization.
 * </p>
 * <p>
 * When performing create operations, if you specify both <code>OrganizationalUnitIds</code> and <code>Accounts</code>,
 * you must also specify the <code>AccountFilterType</code> property.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DeploymentTargets implements SdkPojo, Serializable,
        ToCopyableBuilder<DeploymentTargets.Builder, DeploymentTargets> {
    private static final SdkField<List<String>> ACCOUNTS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Accounts")
            .getter(getter(DeploymentTargets::accounts))
            .setter(setter(Builder::accounts))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Accounts").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> ACCOUNTS_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AccountsUrl").getter(getter(DeploymentTargets::accountsUrl)).setter(setter(Builder::accountsUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccountsUrl").build()).build();

    private static final SdkField<List<String>> ORGANIZATIONAL_UNIT_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("OrganizationalUnitIds")
            .getter(getter(DeploymentTargets::organizationalUnitIds))
            .setter(setter(Builder::organizationalUnitIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OrganizationalUnitIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> ACCOUNT_FILTER_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AccountFilterType").getter(getter(DeploymentTargets::accountFilterTypeAsString))
            .setter(setter(Builder::accountFilterType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccountFilterType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCOUNTS_FIELD,
            ACCOUNTS_URL_FIELD, ORGANIZATIONAL_UNIT_IDS_FIELD, ACCOUNT_FILTER_TYPE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<String> accounts;

    private final String accountsUrl;

    private final List<String> organizationalUnitIds;

    private final String accountFilterType;

    private DeploymentTargets(BuilderImpl builder) {
        this.accounts = builder.accounts;
        this.accountsUrl = builder.accountsUrl;
        this.organizationalUnitIds = builder.organizationalUnitIds;
        this.accountFilterType = builder.accountFilterType;
    }

    /**
     * For responses, this returns true if the service returned a value for the Accounts property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAccounts() {
        return accounts != null && !(accounts instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon Web Services account IDs where you want to perform stack operations. How these accounts are used
     * depends on the <code>AccountFilterType</code> property.
     * </p>
     * <p>
     * If you have many account numbers, you can provide those accounts using the <code>AccountsUrl</code> property
     * instead.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAccounts} method.
     * </p>
     * 
     * @return The Amazon Web Services account IDs where you want to perform stack operations. How these accounts are
     *         used depends on the <code>AccountFilterType</code> property.</p>
     *         <p>
     *         If you have many account numbers, you can provide those accounts using the <code>AccountsUrl</code>
     *         property instead.
     */
    public final List<String> accounts() {
        return accounts;
    }

    /**
     * <p>
     * The Amazon S3 URL path to a file that contains a list of Amazon Web Services account IDs. The file format must be
     * either <code>.csv</code> or <code>.txt</code>, and the data can be comma-separated or new-line-separated. There
     * is currently a 10MB limit for the data (approximately 800,000 accounts).
     * </p>
     * <p>
     * This property serves the same purpose as <code>Accounts</code> but allows you to specify a large number of
     * accounts.
     * </p>
     * 
     * @return The Amazon S3 URL path to a file that contains a list of Amazon Web Services account IDs. The file format
     *         must be either <code>.csv</code> or <code>.txt</code>, and the data can be comma-separated or
     *         new-line-separated. There is currently a 10MB limit for the data (approximately 800,000 accounts).</p>
     *         <p>
     *         This property serves the same purpose as <code>Accounts</code> but allows you to specify a large number
     *         of accounts.
     */
    public final String accountsUrl() {
        return accountsUrl;
    }

    /**
     * For responses, this returns true if the service returned a value for the OrganizationalUnitIds property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasOrganizationalUnitIds() {
        return organizationalUnitIds != null && !(organizationalUnitIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The organization root ID or organizational unit (OU) IDs where you want to perform stack operations.
     * CloudFormation will perform operations on accounts within these OUs and their child OUs.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasOrganizationalUnitIds} method.
     * </p>
     * 
     * @return The organization root ID or organizational unit (OU) IDs where you want to perform stack operations.
     *         CloudFormation will perform operations on accounts within these OUs and their child OUs.
     */
    public final List<String> organizationalUnitIds() {
        return organizationalUnitIds;
    }

    /**
     * <p>
     * Refines which accounts will have stack operations performed on them by specifying how to use the
     * <code>Accounts</code> and <code>OrganizationalUnitIds</code> properties together.
     * </p>
     * <p>
     * The following values determine how CloudFormation selects target accounts:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>INTERSECTION</code>: Performs stack operations only on specific individual accounts within the selected
     * OUs. Only accounts that are both specified in the <code>Accounts</code> property and belong to the specified OUs
     * will be targeted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DIFFERENCE</code>: Performs stack operations on all accounts in the selected OUs except for specific
     * accounts listed in the <code>Accounts</code> property. This enables you to exclude certain accounts within an OU,
     * such as suspended accounts.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UNION</code>: Performs stack operations on the specified OUs plus additional individual accounts listed in
     * the <code>Accounts</code> property. This is the default value if <code>AccountFilterType</code> is not provided.
     * This lets you target an entire OU and individual accounts from a different OU in one request. Note that
     * <code>UNION</code> is not supported for <code>CreateStackInstances</code> operations.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NONE</code>: Performs stack operations on all accounts in the specified organizational units (OUs).
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #accountFilterType}
     * will return {@link AccountFilterType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #accountFilterTypeAsString}.
     * </p>
     * 
     * @return Refines which accounts will have stack operations performed on them by specifying how to use the
     *         <code>Accounts</code> and <code>OrganizationalUnitIds</code> properties together.</p>
     *         <p>
     *         The following values determine how CloudFormation selects target accounts:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>INTERSECTION</code>: Performs stack operations only on specific individual accounts within the
     *         selected OUs. Only accounts that are both specified in the <code>Accounts</code> property and belong to
     *         the specified OUs will be targeted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DIFFERENCE</code>: Performs stack operations on all accounts in the selected OUs except for
     *         specific accounts listed in the <code>Accounts</code> property. This enables you to exclude certain
     *         accounts within an OU, such as suspended accounts.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UNION</code>: Performs stack operations on the specified OUs plus additional individual accounts
     *         listed in the <code>Accounts</code> property. This is the default value if <code>AccountFilterType</code>
     *         is not provided. This lets you target an entire OU and individual accounts from a different OU in one
     *         request. Note that <code>UNION</code> is not supported for <code>CreateStackInstances</code> operations.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NONE</code>: Performs stack operations on all accounts in the specified organizational units (OUs).
     *         </p>
     *         </li>
     * @see AccountFilterType
     */
    public final AccountFilterType accountFilterType() {
        return AccountFilterType.fromValue(accountFilterType);
    }

    /**
     * <p>
     * Refines which accounts will have stack operations performed on them by specifying how to use the
     * <code>Accounts</code> and <code>OrganizationalUnitIds</code> properties together.
     * </p>
     * <p>
     * The following values determine how CloudFormation selects target accounts:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>INTERSECTION</code>: Performs stack operations only on specific individual accounts within the selected
     * OUs. Only accounts that are both specified in the <code>Accounts</code> property and belong to the specified OUs
     * will be targeted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DIFFERENCE</code>: Performs stack operations on all accounts in the selected OUs except for specific
     * accounts listed in the <code>Accounts</code> property. This enables you to exclude certain accounts within an OU,
     * such as suspended accounts.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UNION</code>: Performs stack operations on the specified OUs plus additional individual accounts listed in
     * the <code>Accounts</code> property. This is the default value if <code>AccountFilterType</code> is not provided.
     * This lets you target an entire OU and individual accounts from a different OU in one request. Note that
     * <code>UNION</code> is not supported for <code>CreateStackInstances</code> operations.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NONE</code>: Performs stack operations on all accounts in the specified organizational units (OUs).
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #accountFilterType}
     * will return {@link AccountFilterType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #accountFilterTypeAsString}.
     * </p>
     * 
     * @return Refines which accounts will have stack operations performed on them by specifying how to use the
     *         <code>Accounts</code> and <code>OrganizationalUnitIds</code> properties together.</p>
     *         <p>
     *         The following values determine how CloudFormation selects target accounts:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>INTERSECTION</code>: Performs stack operations only on specific individual accounts within the
     *         selected OUs. Only accounts that are both specified in the <code>Accounts</code> property and belong to
     *         the specified OUs will be targeted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DIFFERENCE</code>: Performs stack operations on all accounts in the selected OUs except for
     *         specific accounts listed in the <code>Accounts</code> property. This enables you to exclude certain
     *         accounts within an OU, such as suspended accounts.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UNION</code>: Performs stack operations on the specified OUs plus additional individual accounts
     *         listed in the <code>Accounts</code> property. This is the default value if <code>AccountFilterType</code>
     *         is not provided. This lets you target an entire OU and individual accounts from a different OU in one
     *         request. Note that <code>UNION</code> is not supported for <code>CreateStackInstances</code> operations.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NONE</code>: Performs stack operations on all accounts in the specified organizational units (OUs).
     *         </p>
     *         </li>
     * @see AccountFilterType
     */
    public final String accountFilterTypeAsString() {
        return accountFilterType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasAccounts() ? accounts() : null);
        hashCode = 31 * hashCode + Objects.hashCode(accountsUrl());
        hashCode = 31 * hashCode + Objects.hashCode(hasOrganizationalUnitIds() ? organizationalUnitIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(accountFilterTypeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DeploymentTargets)) {
            return false;
        }
        DeploymentTargets other = (DeploymentTargets) obj;
        return hasAccounts() == other.hasAccounts() && Objects.equals(accounts(), other.accounts())
                && Objects.equals(accountsUrl(), other.accountsUrl())
                && hasOrganizationalUnitIds() == other.hasOrganizationalUnitIds()
                && Objects.equals(organizationalUnitIds(), other.organizationalUnitIds())
                && Objects.equals(accountFilterTypeAsString(), other.accountFilterTypeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DeploymentTargets").add("Accounts", hasAccounts() ? accounts() : null)
                .add("AccountsUrl", accountsUrl())
                .add("OrganizationalUnitIds", hasOrganizationalUnitIds() ? organizationalUnitIds() : null)
                .add("AccountFilterType", accountFilterTypeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Accounts":
            return Optional.ofNullable(clazz.cast(accounts()));
        case "AccountsUrl":
            return Optional.ofNullable(clazz.cast(accountsUrl()));
        case "OrganizationalUnitIds":
            return Optional.ofNullable(clazz.cast(organizationalUnitIds()));
        case "AccountFilterType":
            return Optional.ofNullable(clazz.cast(accountFilterTypeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Accounts", ACCOUNTS_FIELD);
        map.put("AccountsUrl", ACCOUNTS_URL_FIELD);
        map.put("OrganizationalUnitIds", ORGANIZATIONAL_UNIT_IDS_FIELD);
        map.put("AccountFilterType", ACCOUNT_FILTER_TYPE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<DeploymentTargets, T> g) {
        return obj -> g.apply((DeploymentTargets) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DeploymentTargets> {
        /**
         * <p>
         * The Amazon Web Services account IDs where you want to perform stack operations. How these accounts are used
         * depends on the <code>AccountFilterType</code> property.
         * </p>
         * <p>
         * If you have many account numbers, you can provide those accounts using the <code>AccountsUrl</code> property
         * instead.
         * </p>
         * 
         * @param accounts
         *        The Amazon Web Services account IDs where you want to perform stack operations. How these accounts are
         *        used depends on the <code>AccountFilterType</code> property.</p>
         *        <p>
         *        If you have many account numbers, you can provide those accounts using the <code>AccountsUrl</code>
         *        property instead.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accounts(Collection<String> accounts);

        /**
         * <p>
         * The Amazon Web Services account IDs where you want to perform stack operations. How these accounts are used
         * depends on the <code>AccountFilterType</code> property.
         * </p>
         * <p>
         * If you have many account numbers, you can provide those accounts using the <code>AccountsUrl</code> property
         * instead.
         * </p>
         * 
         * @param accounts
         *        The Amazon Web Services account IDs where you want to perform stack operations. How these accounts are
         *        used depends on the <code>AccountFilterType</code> property.</p>
         *        <p>
         *        If you have many account numbers, you can provide those accounts using the <code>AccountsUrl</code>
         *        property instead.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accounts(String... accounts);

        /**
         * <p>
         * The Amazon S3 URL path to a file that contains a list of Amazon Web Services account IDs. The file format
         * must be either <code>.csv</code> or <code>.txt</code>, and the data can be comma-separated or
         * new-line-separated. There is currently a 10MB limit for the data (approximately 800,000 accounts).
         * </p>
         * <p>
         * This property serves the same purpose as <code>Accounts</code> but allows you to specify a large number of
         * accounts.
         * </p>
         * 
         * @param accountsUrl
         *        The Amazon S3 URL path to a file that contains a list of Amazon Web Services account IDs. The file
         *        format must be either <code>.csv</code> or <code>.txt</code>, and the data can be comma-separated or
         *        new-line-separated. There is currently a 10MB limit for the data (approximately 800,000 accounts).</p>
         *        <p>
         *        This property serves the same purpose as <code>Accounts</code> but allows you to specify a large
         *        number of accounts.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountsUrl(String accountsUrl);

        /**
         * <p>
         * The organization root ID or organizational unit (OU) IDs where you want to perform stack operations.
         * CloudFormation will perform operations on accounts within these OUs and their child OUs.
         * </p>
         * 
         * @param organizationalUnitIds
         *        The organization root ID or organizational unit (OU) IDs where you want to perform stack operations.
         *        CloudFormation will perform operations on accounts within these OUs and their child OUs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder organizationalUnitIds(Collection<String> organizationalUnitIds);

        /**
         * <p>
         * The organization root ID or organizational unit (OU) IDs where you want to perform stack operations.
         * CloudFormation will perform operations on accounts within these OUs and their child OUs.
         * </p>
         * 
         * @param organizationalUnitIds
         *        The organization root ID or organizational unit (OU) IDs where you want to perform stack operations.
         *        CloudFormation will perform operations on accounts within these OUs and their child OUs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder organizationalUnitIds(String... organizationalUnitIds);

        /**
         * <p>
         * Refines which accounts will have stack operations performed on them by specifying how to use the
         * <code>Accounts</code> and <code>OrganizationalUnitIds</code> properties together.
         * </p>
         * <p>
         * The following values determine how CloudFormation selects target accounts:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>INTERSECTION</code>: Performs stack operations only on specific individual accounts within the selected
         * OUs. Only accounts that are both specified in the <code>Accounts</code> property and belong to the specified
         * OUs will be targeted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DIFFERENCE</code>: Performs stack operations on all accounts in the selected OUs except for specific
         * accounts listed in the <code>Accounts</code> property. This enables you to exclude certain accounts within an
         * OU, such as suspended accounts.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UNION</code>: Performs stack operations on the specified OUs plus additional individual accounts listed
         * in the <code>Accounts</code> property. This is the default value if <code>AccountFilterType</code> is not
         * provided. This lets you target an entire OU and individual accounts from a different OU in one request. Note
         * that <code>UNION</code> is not supported for <code>CreateStackInstances</code> operations.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NONE</code>: Performs stack operations on all accounts in the specified organizational units (OUs).
         * </p>
         * </li>
         * </ul>
         * 
         * @param accountFilterType
         *        Refines which accounts will have stack operations performed on them by specifying how to use the
         *        <code>Accounts</code> and <code>OrganizationalUnitIds</code> properties together.</p>
         *        <p>
         *        The following values determine how CloudFormation selects target accounts:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>INTERSECTION</code>: Performs stack operations only on specific individual accounts within the
         *        selected OUs. Only accounts that are both specified in the <code>Accounts</code> property and belong
         *        to the specified OUs will be targeted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DIFFERENCE</code>: Performs stack operations on all accounts in the selected OUs except for
         *        specific accounts listed in the <code>Accounts</code> property. This enables you to exclude certain
         *        accounts within an OU, such as suspended accounts.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UNION</code>: Performs stack operations on the specified OUs plus additional individual accounts
         *        listed in the <code>Accounts</code> property. This is the default value if
         *        <code>AccountFilterType</code> is not provided. This lets you target an entire OU and individual
         *        accounts from a different OU in one request. Note that <code>UNION</code> is not supported for
         *        <code>CreateStackInstances</code> operations.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NONE</code>: Performs stack operations on all accounts in the specified organizational units
         *        (OUs).
         *        </p>
         *        </li>
         * @see AccountFilterType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AccountFilterType
         */
        Builder accountFilterType(String accountFilterType);

        /**
         * <p>
         * Refines which accounts will have stack operations performed on them by specifying how to use the
         * <code>Accounts</code> and <code>OrganizationalUnitIds</code> properties together.
         * </p>
         * <p>
         * The following values determine how CloudFormation selects target accounts:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>INTERSECTION</code>: Performs stack operations only on specific individual accounts within the selected
         * OUs. Only accounts that are both specified in the <code>Accounts</code> property and belong to the specified
         * OUs will be targeted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DIFFERENCE</code>: Performs stack operations on all accounts in the selected OUs except for specific
         * accounts listed in the <code>Accounts</code> property. This enables you to exclude certain accounts within an
         * OU, such as suspended accounts.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UNION</code>: Performs stack operations on the specified OUs plus additional individual accounts listed
         * in the <code>Accounts</code> property. This is the default value if <code>AccountFilterType</code> is not
         * provided. This lets you target an entire OU and individual accounts from a different OU in one request. Note
         * that <code>UNION</code> is not supported for <code>CreateStackInstances</code> operations.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NONE</code>: Performs stack operations on all accounts in the specified organizational units (OUs).
         * </p>
         * </li>
         * </ul>
         * 
         * @param accountFilterType
         *        Refines which accounts will have stack operations performed on them by specifying how to use the
         *        <code>Accounts</code> and <code>OrganizationalUnitIds</code> properties together.</p>
         *        <p>
         *        The following values determine how CloudFormation selects target accounts:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>INTERSECTION</code>: Performs stack operations only on specific individual accounts within the
         *        selected OUs. Only accounts that are both specified in the <code>Accounts</code> property and belong
         *        to the specified OUs will be targeted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DIFFERENCE</code>: Performs stack operations on all accounts in the selected OUs except for
         *        specific accounts listed in the <code>Accounts</code> property. This enables you to exclude certain
         *        accounts within an OU, such as suspended accounts.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UNION</code>: Performs stack operations on the specified OUs plus additional individual accounts
         *        listed in the <code>Accounts</code> property. This is the default value if
         *        <code>AccountFilterType</code> is not provided. This lets you target an entire OU and individual
         *        accounts from a different OU in one request. Note that <code>UNION</code> is not supported for
         *        <code>CreateStackInstances</code> operations.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NONE</code>: Performs stack operations on all accounts in the specified organizational units
         *        (OUs).
         *        </p>
         *        </li>
         * @see AccountFilterType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AccountFilterType
         */
        Builder accountFilterType(AccountFilterType accountFilterType);
    }

    static final class BuilderImpl implements Builder {
        private List<String> accounts = DefaultSdkAutoConstructList.getInstance();

        private String accountsUrl;

        private List<String> organizationalUnitIds = DefaultSdkAutoConstructList.getInstance();

        private String accountFilterType;

        private BuilderImpl() {
        }

        private BuilderImpl(DeploymentTargets model) {
            accounts(model.accounts);
            accountsUrl(model.accountsUrl);
            organizationalUnitIds(model.organizationalUnitIds);
            accountFilterType(model.accountFilterType);
        }

        public final Collection<String> getAccounts() {
            if (accounts instanceof SdkAutoConstructList) {
                return null;
            }
            return accounts;
        }

        public final void setAccounts(Collection<String> accounts) {
            this.accounts = AccountListCopier.copy(accounts);
        }

        @Override
        public final Builder accounts(Collection<String> accounts) {
            this.accounts = AccountListCopier.copy(accounts);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder accounts(String... accounts) {
            accounts(Arrays.asList(accounts));
            return this;
        }

        public final String getAccountsUrl() {
            return accountsUrl;
        }

        public final void setAccountsUrl(String accountsUrl) {
            this.accountsUrl = accountsUrl;
        }

        @Override
        public final Builder accountsUrl(String accountsUrl) {
            this.accountsUrl = accountsUrl;
            return this;
        }

        public final Collection<String> getOrganizationalUnitIds() {
            if (organizationalUnitIds instanceof SdkAutoConstructList) {
                return null;
            }
            return organizationalUnitIds;
        }

        public final void setOrganizationalUnitIds(Collection<String> organizationalUnitIds) {
            this.organizationalUnitIds = OrganizationalUnitIdListCopier.copy(organizationalUnitIds);
        }

        @Override
        public final Builder organizationalUnitIds(Collection<String> organizationalUnitIds) {
            this.organizationalUnitIds = OrganizationalUnitIdListCopier.copy(organizationalUnitIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder organizationalUnitIds(String... organizationalUnitIds) {
            organizationalUnitIds(Arrays.asList(organizationalUnitIds));
            return this;
        }

        public final String getAccountFilterType() {
            return accountFilterType;
        }

        public final void setAccountFilterType(String accountFilterType) {
            this.accountFilterType = accountFilterType;
        }

        @Override
        public final Builder accountFilterType(String accountFilterType) {
            this.accountFilterType = accountFilterType;
            return this;
        }

        @Override
        public final Builder accountFilterType(AccountFilterType accountFilterType) {
            this.accountFilterType(accountFilterType == null ? null : accountFilterType.toString());
            return this;
        }

        @Override
        public DeploymentTargets build() {
            return new DeploymentTargets(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
