/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockagentruntime.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about the function that was called from the action group and the response that was returned.
 * </p>
 * <p>
 * This data type is used in the following API operations:
 * </p>
 * <ul>
 * <li>
 * <p>
 * In the <code>returnControlInvocationResults</code> of the <a href=
 * "https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_Retrieve.html#API_agent-runtime_Retrieve_RequestSyntax"
 * >Retrieve request</a>
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class FunctionResult implements SdkPojo, Serializable, ToCopyableBuilder<FunctionResult.Builder, FunctionResult> {
    private static final SdkField<String> ACTION_GROUP_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("actionGroup").getter(getter(FunctionResult::actionGroup)).setter(setter(Builder::actionGroup))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("actionGroup").build()).build();

    private static final SdkField<String> FUNCTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("function").getter(getter(FunctionResult::function)).setter(setter(Builder::function))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("function").build()).build();

    private static final SdkField<Map<String, ContentBody>> RESPONSE_BODY_FIELD = SdkField
            .<Map<String, ContentBody>> builder(MarshallingType.MAP)
            .memberName("responseBody")
            .getter(getter(FunctionResult::responseBody))
            .setter(setter(Builder::responseBody))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("responseBody").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<ContentBody> builder(MarshallingType.SDK_POJO)
                                            .constructor(ContentBody::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> RESPONSE_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("responseState").getter(getter(FunctionResult::responseStateAsString))
            .setter(setter(Builder::responseState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("responseState").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACTION_GROUP_FIELD,
            FUNCTION_FIELD, RESPONSE_BODY_FIELD, RESPONSE_STATE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String actionGroup;

    private final String function;

    private final Map<String, ContentBody> responseBody;

    private final String responseState;

    private FunctionResult(BuilderImpl builder) {
        this.actionGroup = builder.actionGroup;
        this.function = builder.function;
        this.responseBody = builder.responseBody;
        this.responseState = builder.responseState;
    }

    /**
     * <p>
     * The action group that the function belongs to.
     * </p>
     * 
     * @return The action group that the function belongs to.
     */
    public final String actionGroup() {
        return actionGroup;
    }

    /**
     * <p>
     * The name of the function that was called.
     * </p>
     * 
     * @return The name of the function that was called.
     */
    public final String function() {
        return function;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResponseBody property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResponseBody() {
        return responseBody != null && !(responseBody instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The response from the function call using the parameters. The response may be returned directly or from the
     * Lambda function.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResponseBody} method.
     * </p>
     * 
     * @return The response from the function call using the parameters. The response may be returned directly or from
     *         the Lambda function.
     */
    public final Map<String, ContentBody> responseBody() {
        return responseBody;
    }

    /**
     * <p>
     * Controls the final response state returned to end user when API/Function execution failed. When this state is
     * FAILURE, the request would fail with dependency failure exception. When this state is REPROMPT, the API/function
     * response will be sent to model for re-prompt
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #responseState}
     * will return {@link ResponseState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #responseStateAsString}.
     * </p>
     * 
     * @return Controls the final response state returned to end user when API/Function execution failed. When this
     *         state is FAILURE, the request would fail with dependency failure exception. When this state is REPROMPT,
     *         the API/function response will be sent to model for re-prompt
     * @see ResponseState
     */
    public final ResponseState responseState() {
        return ResponseState.fromValue(responseState);
    }

    /**
     * <p>
     * Controls the final response state returned to end user when API/Function execution failed. When this state is
     * FAILURE, the request would fail with dependency failure exception. When this state is REPROMPT, the API/function
     * response will be sent to model for re-prompt
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #responseState}
     * will return {@link ResponseState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #responseStateAsString}.
     * </p>
     * 
     * @return Controls the final response state returned to end user when API/Function execution failed. When this
     *         state is FAILURE, the request would fail with dependency failure exception. When this state is REPROMPT,
     *         the API/function response will be sent to model for re-prompt
     * @see ResponseState
     */
    public final String responseStateAsString() {
        return responseState;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(actionGroup());
        hashCode = 31 * hashCode + Objects.hashCode(function());
        hashCode = 31 * hashCode + Objects.hashCode(hasResponseBody() ? responseBody() : null);
        hashCode = 31 * hashCode + Objects.hashCode(responseStateAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof FunctionResult)) {
            return false;
        }
        FunctionResult other = (FunctionResult) obj;
        return Objects.equals(actionGroup(), other.actionGroup()) && Objects.equals(function(), other.function())
                && hasResponseBody() == other.hasResponseBody() && Objects.equals(responseBody(), other.responseBody())
                && Objects.equals(responseStateAsString(), other.responseStateAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("FunctionResult").add("ActionGroup", actionGroup()).add("Function", function())
                .add("ResponseBody", hasResponseBody() ? responseBody() : null).add("ResponseState", responseStateAsString())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "actionGroup":
            return Optional.ofNullable(clazz.cast(actionGroup()));
        case "function":
            return Optional.ofNullable(clazz.cast(function()));
        case "responseBody":
            return Optional.ofNullable(clazz.cast(responseBody()));
        case "responseState":
            return Optional.ofNullable(clazz.cast(responseStateAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<FunctionResult, T> g) {
        return obj -> g.apply((FunctionResult) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, FunctionResult> {
        /**
         * <p>
         * The action group that the function belongs to.
         * </p>
         * 
         * @param actionGroup
         *        The action group that the function belongs to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actionGroup(String actionGroup);

        /**
         * <p>
         * The name of the function that was called.
         * </p>
         * 
         * @param function
         *        The name of the function that was called.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder function(String function);

        /**
         * <p>
         * The response from the function call using the parameters. The response may be returned directly or from the
         * Lambda function.
         * </p>
         * 
         * @param responseBody
         *        The response from the function call using the parameters. The response may be returned directly or
         *        from the Lambda function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder responseBody(Map<String, ContentBody> responseBody);

        /**
         * <p>
         * Controls the final response state returned to end user when API/Function execution failed. When this state is
         * FAILURE, the request would fail with dependency failure exception. When this state is REPROMPT, the
         * API/function response will be sent to model for re-prompt
         * </p>
         * 
         * @param responseState
         *        Controls the final response state returned to end user when API/Function execution failed. When this
         *        state is FAILURE, the request would fail with dependency failure exception. When this state is
         *        REPROMPT, the API/function response will be sent to model for re-prompt
         * @see ResponseState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResponseState
         */
        Builder responseState(String responseState);

        /**
         * <p>
         * Controls the final response state returned to end user when API/Function execution failed. When this state is
         * FAILURE, the request would fail with dependency failure exception. When this state is REPROMPT, the
         * API/function response will be sent to model for re-prompt
         * </p>
         * 
         * @param responseState
         *        Controls the final response state returned to end user when API/Function execution failed. When this
         *        state is FAILURE, the request would fail with dependency failure exception. When this state is
         *        REPROMPT, the API/function response will be sent to model for re-prompt
         * @see ResponseState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResponseState
         */
        Builder responseState(ResponseState responseState);
    }

    static final class BuilderImpl implements Builder {
        private String actionGroup;

        private String function;

        private Map<String, ContentBody> responseBody = DefaultSdkAutoConstructMap.getInstance();

        private String responseState;

        private BuilderImpl() {
        }

        private BuilderImpl(FunctionResult model) {
            actionGroup(model.actionGroup);
            function(model.function);
            responseBody(model.responseBody);
            responseState(model.responseState);
        }

        public final String getActionGroup() {
            return actionGroup;
        }

        public final void setActionGroup(String actionGroup) {
            this.actionGroup = actionGroup;
        }

        @Override
        public final Builder actionGroup(String actionGroup) {
            this.actionGroup = actionGroup;
            return this;
        }

        public final String getFunction() {
            return function;
        }

        public final void setFunction(String function) {
            this.function = function;
        }

        @Override
        public final Builder function(String function) {
            this.function = function;
            return this;
        }

        public final Map<String, ContentBody.Builder> getResponseBody() {
            Map<String, ContentBody.Builder> result = ResponseBodyCopier.copyToBuilder(this.responseBody);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setResponseBody(Map<String, ContentBody.BuilderImpl> responseBody) {
            this.responseBody = ResponseBodyCopier.copyFromBuilder(responseBody);
        }

        @Override
        public final Builder responseBody(Map<String, ContentBody> responseBody) {
            this.responseBody = ResponseBodyCopier.copy(responseBody);
            return this;
        }

        public final String getResponseState() {
            return responseState;
        }

        public final void setResponseState(String responseState) {
            this.responseState = responseState;
        }

        @Override
        public final Builder responseState(String responseState) {
            this.responseState = responseState;
            return this;
        }

        @Override
        public final Builder responseState(ResponseState responseState) {
            this.responseState(responseState == null ? null : responseState.toString());
            return this;
        }

        @Override
        public FunctionResult build() {
            return new FunctionResult(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
