/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrock.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the ARN of the Amazon Bedrock models specified in your model evaluation job. Each Amazon Bedrock model
 * supports different <code>inferenceParams</code>. To learn more about supported inference parameters for Amazon
 * Bedrock models, see <a href="https://docs.aws.amazon.com/bedrock/latest/userguide/model-parameters.html">Inference
 * parameters for foundation models</a>.
 * </p>
 * <p>
 * The <code>inferenceParams</code> are specified using JSON. To successfully insert JSON as string make sure that all
 * quotations are properly escaped. For example, <code>"temperature":"0.25"</code> key value pair would need to be
 * formatted as <code>\"temperature\":\"0.25\"</code> to successfully accepted in the request.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EvaluationBedrockModel implements SdkPojo, Serializable,
        ToCopyableBuilder<EvaluationBedrockModel.Builder, EvaluationBedrockModel> {
    private static final SdkField<String> MODEL_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("modelIdentifier").getter(getter(EvaluationBedrockModel::modelIdentifier))
            .setter(setter(Builder::modelIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("modelIdentifier").build()).build();

    private static final SdkField<String> INFERENCE_PARAMS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("inferenceParams").getter(getter(EvaluationBedrockModel::inferenceParams))
            .setter(setter(Builder::inferenceParams))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("inferenceParams").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MODEL_IDENTIFIER_FIELD,
            INFERENCE_PARAMS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String modelIdentifier;

    private final String inferenceParams;

    private EvaluationBedrockModel(BuilderImpl builder) {
        this.modelIdentifier = builder.modelIdentifier;
        this.inferenceParams = builder.inferenceParams;
    }

    /**
     * <p>
     * The ARN of the Amazon Bedrock model specified.
     * </p>
     * 
     * @return The ARN of the Amazon Bedrock model specified.
     */
    public final String modelIdentifier() {
        return modelIdentifier;
    }

    /**
     * <p>
     * Each Amazon Bedrock support different inference parameters that change how the model behaves during inference.
     * </p>
     * 
     * @return Each Amazon Bedrock support different inference parameters that change how the model behaves during
     *         inference.
     */
    public final String inferenceParams() {
        return inferenceParams;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(modelIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(inferenceParams());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EvaluationBedrockModel)) {
            return false;
        }
        EvaluationBedrockModel other = (EvaluationBedrockModel) obj;
        return Objects.equals(modelIdentifier(), other.modelIdentifier())
                && Objects.equals(inferenceParams(), other.inferenceParams());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EvaluationBedrockModel").add("ModelIdentifier", modelIdentifier())
                .add("InferenceParams", inferenceParams() == null ? null : "*** Sensitive Data Redacted ***").build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "modelIdentifier":
            return Optional.ofNullable(clazz.cast(modelIdentifier()));
        case "inferenceParams":
            return Optional.ofNullable(clazz.cast(inferenceParams()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<EvaluationBedrockModel, T> g) {
        return obj -> g.apply((EvaluationBedrockModel) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EvaluationBedrockModel> {
        /**
         * <p>
         * The ARN of the Amazon Bedrock model specified.
         * </p>
         * 
         * @param modelIdentifier
         *        The ARN of the Amazon Bedrock model specified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modelIdentifier(String modelIdentifier);

        /**
         * <p>
         * Each Amazon Bedrock support different inference parameters that change how the model behaves during
         * inference.
         * </p>
         * 
         * @param inferenceParams
         *        Each Amazon Bedrock support different inference parameters that change how the model behaves during
         *        inference.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inferenceParams(String inferenceParams);
    }

    static final class BuilderImpl implements Builder {
        private String modelIdentifier;

        private String inferenceParams;

        private BuilderImpl() {
        }

        private BuilderImpl(EvaluationBedrockModel model) {
            modelIdentifier(model.modelIdentifier);
            inferenceParams(model.inferenceParams);
        }

        public final String getModelIdentifier() {
            return modelIdentifier;
        }

        public final void setModelIdentifier(String modelIdentifier) {
            this.modelIdentifier = modelIdentifier;
        }

        @Override
        public final Builder modelIdentifier(String modelIdentifier) {
            this.modelIdentifier = modelIdentifier;
            return this;
        }

        public final String getInferenceParams() {
            return inferenceParams;
        }

        public final void setInferenceParams(String inferenceParams) {
            this.inferenceParams = inferenceParams;
        }

        @Override
        public final Builder inferenceParams(String inferenceParams) {
            this.inferenceParams = inferenceParams;
            return this;
        }

        @Override
        public EvaluationBedrockModel build() {
            return new EvaluationBedrockModel(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
