/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.athena.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A workgroup, which contains a name, description, creation time, state, and other configuration, listed under
 * <a>WorkGroup$Configuration</a>. Each workgroup enables you to isolate queries for you or your group of users from
 * other queries in the same account, to configure the query results location and the encryption configuration (known as
 * workgroup settings), to enable sending query metrics to Amazon CloudWatch, and to establish per-query data usage
 * control limits for all queries in a workgroup. The workgroup settings override is specified in
 * EnforceWorkGroupConfiguration (true/false) in the WorkGroupConfiguration. See
 * <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class WorkGroup implements SdkPojo, Serializable, ToCopyableBuilder<WorkGroup.Builder, WorkGroup> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(WorkGroup::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(WorkGroup::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State").build()).build();

    private static final SdkField<WorkGroupConfiguration> CONFIGURATION_FIELD = SdkField
            .<WorkGroupConfiguration> builder(MarshallingType.SDK_POJO).getter(getter(WorkGroup::configuration))
            .setter(setter(Builder::configuration)).constructor(WorkGroupConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Configuration").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(WorkGroup::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(WorkGroup::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, STATE_FIELD,
            CONFIGURATION_FIELD, DESCRIPTION_FIELD, CREATION_TIME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String state;

    private final WorkGroupConfiguration configuration;

    private final String description;

    private final Instant creationTime;

    private WorkGroup(BuilderImpl builder) {
        this.name = builder.name;
        this.state = builder.state;
        this.configuration = builder.configuration;
        this.description = builder.description;
        this.creationTime = builder.creationTime;
    }

    /**
     * <p>
     * The workgroup name.
     * </p>
     * 
     * @return The workgroup name.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The state of the workgroup: ENABLED or DISABLED.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link WorkGroupState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the workgroup: ENABLED or DISABLED.
     * @see WorkGroupState
     */
    public WorkGroupState state() {
        return WorkGroupState.fromValue(state);
    }

    /**
     * <p>
     * The state of the workgroup: ENABLED or DISABLED.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link WorkGroupState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the workgroup: ENABLED or DISABLED.
     * @see WorkGroupState
     */
    public String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The configuration of the workgroup, which includes the location in Amazon S3 where query results are stored, the
     * encryption configuration, if any, used for query results; whether the Amazon CloudWatch Metrics are enabled for
     * the workgroup; whether workgroup settings override client-side settings; and the data usage limits for the amount
     * of data scanned per query or per workgroup. The workgroup settings override is specified in
     * EnforceWorkGroupConfiguration (true/false) in the WorkGroupConfiguration. See
     * <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a>.
     * </p>
     * 
     * @return The configuration of the workgroup, which includes the location in Amazon S3 where query results are
     *         stored, the encryption configuration, if any, used for query results; whether the Amazon CloudWatch
     *         Metrics are enabled for the workgroup; whether workgroup settings override client-side settings; and the
     *         data usage limits for the amount of data scanned per query or per workgroup. The workgroup settings
     *         override is specified in EnforceWorkGroupConfiguration (true/false) in the WorkGroupConfiguration. See
     *         <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a>.
     */
    public WorkGroupConfiguration configuration() {
        return configuration;
    }

    /**
     * <p>
     * The workgroup description.
     * </p>
     * 
     * @return The workgroup description.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * The date and time the workgroup was created.
     * </p>
     * 
     * @return The date and time the workgroup was created.
     */
    public Instant creationTime() {
        return creationTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(configuration());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof WorkGroup)) {
            return false;
        }
        WorkGroup other = (WorkGroup) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(configuration(), other.configuration()) && Objects.equals(description(), other.description())
                && Objects.equals(creationTime(), other.creationTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("WorkGroup").add("Name", name()).add("State", stateAsString())
                .add("Configuration", configuration()).add("Description", description()).add("CreationTime", creationTime())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "Configuration":
            return Optional.ofNullable(clazz.cast(configuration()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<WorkGroup, T> g) {
        return obj -> g.apply((WorkGroup) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, WorkGroup> {
        /**
         * <p>
         * The workgroup name.
         * </p>
         * 
         * @param name
         *        The workgroup name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The state of the workgroup: ENABLED or DISABLED.
         * </p>
         * 
         * @param state
         *        The state of the workgroup: ENABLED or DISABLED.
         * @see WorkGroupState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WorkGroupState
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the workgroup: ENABLED or DISABLED.
         * </p>
         * 
         * @param state
         *        The state of the workgroup: ENABLED or DISABLED.
         * @see WorkGroupState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WorkGroupState
         */
        Builder state(WorkGroupState state);

        /**
         * <p>
         * The configuration of the workgroup, which includes the location in Amazon S3 where query results are stored,
         * the encryption configuration, if any, used for query results; whether the Amazon CloudWatch Metrics are
         * enabled for the workgroup; whether workgroup settings override client-side settings; and the data usage
         * limits for the amount of data scanned per query or per workgroup. The workgroup settings override is
         * specified in EnforceWorkGroupConfiguration (true/false) in the WorkGroupConfiguration. See
         * <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a>.
         * </p>
         * 
         * @param configuration
         *        The configuration of the workgroup, which includes the location in Amazon S3 where query results are
         *        stored, the encryption configuration, if any, used for query results; whether the Amazon CloudWatch
         *        Metrics are enabled for the workgroup; whether workgroup settings override client-side settings; and
         *        the data usage limits for the amount of data scanned per query or per workgroup. The workgroup
         *        settings override is specified in EnforceWorkGroupConfiguration (true/false) in the
         *        WorkGroupConfiguration. See <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configuration(WorkGroupConfiguration configuration);

        /**
         * <p>
         * The configuration of the workgroup, which includes the location in Amazon S3 where query results are stored,
         * the encryption configuration, if any, used for query results; whether the Amazon CloudWatch Metrics are
         * enabled for the workgroup; whether workgroup settings override client-side settings; and the data usage
         * limits for the amount of data scanned per query or per workgroup. The workgroup settings override is
         * specified in EnforceWorkGroupConfiguration (true/false) in the WorkGroupConfiguration. See
         * <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link WorkGroupConfiguration.Builder} avoiding the
         * need to create one manually via {@link WorkGroupConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link WorkGroupConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #configuration(WorkGroupConfiguration)}.
         * 
         * @param configuration
         *        a consumer that will call methods on {@link WorkGroupConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #configuration(WorkGroupConfiguration)
         */
        default Builder configuration(Consumer<WorkGroupConfiguration.Builder> configuration) {
            return configuration(WorkGroupConfiguration.builder().applyMutation(configuration).build());
        }

        /**
         * <p>
         * The workgroup description.
         * </p>
         * 
         * @param description
         *        The workgroup description.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The date and time the workgroup was created.
         * </p>
         * 
         * @param creationTime
         *        The date and time the workgroup was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String state;

        private WorkGroupConfiguration configuration;

        private String description;

        private Instant creationTime;

        private BuilderImpl() {
        }

        private BuilderImpl(WorkGroup model) {
            name(model.name);
            state(model.state);
            configuration(model.configuration);
            description(model.description);
            creationTime(model.creationTime);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getStateAsString() {
            return state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(WorkGroupState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final void setState(String state) {
            this.state = state;
        }

        public final WorkGroupConfiguration.Builder getConfiguration() {
            return configuration != null ? configuration.toBuilder() : null;
        }

        @Override
        public final Builder configuration(WorkGroupConfiguration configuration) {
            this.configuration = configuration;
            return this;
        }

        public final void setConfiguration(WorkGroupConfiguration.BuilderImpl configuration) {
            this.configuration = configuration != null ? configuration.build() : null;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public WorkGroup build() {
            return new WorkGroup(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
