/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.accessanalyzer.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Proposed access control configuration for an Amazon S3 bucket. You can propose a configuration for a new Amazon S3
 * bucket or an existing Amazon S3 bucket that you own by specifying the Amazon S3 bucket policy, bucket ACLs, bucket
 * BPA settings, Amazon S3 access points, and multi-region access points attached to the bucket. If the configuration is
 * for an existing Amazon S3 bucket and you do not specify the Amazon S3 bucket policy, the access preview uses the
 * existing policy attached to the bucket. If the access preview is for a new resource and you do not specify the Amazon
 * S3 bucket policy, the access preview assumes a bucket without a policy. To propose deletion of an existing bucket
 * policy, you can specify an empty string. For more information about bucket policy limits, see <a
 * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/example-bucket-policies.html">Bucket Policy Examples</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3BucketConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<S3BucketConfiguration.Builder, S3BucketConfiguration> {
    private static final SdkField<Map<String, S3AccessPointConfiguration>> ACCESS_POINTS_FIELD = SdkField
            .<Map<String, S3AccessPointConfiguration>> builder(MarshallingType.MAP)
            .memberName("accessPoints")
            .getter(getter(S3BucketConfiguration::accessPoints))
            .setter(setter(Builder::accessPoints))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("accessPoints").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<S3AccessPointConfiguration> builder(MarshallingType.SDK_POJO)
                                            .constructor(S3AccessPointConfiguration::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<S3BucketAclGrantConfiguration>> BUCKET_ACL_GRANTS_FIELD = SdkField
            .<List<S3BucketAclGrantConfiguration>> builder(MarshallingType.LIST)
            .memberName("bucketAclGrants")
            .getter(getter(S3BucketConfiguration::bucketAclGrants))
            .setter(setter(Builder::bucketAclGrants))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("bucketAclGrants").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<S3BucketAclGrantConfiguration> builder(MarshallingType.SDK_POJO)
                                            .constructor(S3BucketAclGrantConfiguration::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> BUCKET_POLICY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("bucketPolicy").getter(getter(S3BucketConfiguration::bucketPolicy)).setter(setter(Builder::bucketPolicy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("bucketPolicy").build()).build();

    private static final SdkField<S3PublicAccessBlockConfiguration> BUCKET_PUBLIC_ACCESS_BLOCK_FIELD = SdkField
            .<S3PublicAccessBlockConfiguration> builder(MarshallingType.SDK_POJO).memberName("bucketPublicAccessBlock")
            .getter(getter(S3BucketConfiguration::bucketPublicAccessBlock)).setter(setter(Builder::bucketPublicAccessBlock))
            .constructor(S3PublicAccessBlockConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("bucketPublicAccessBlock").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCESS_POINTS_FIELD,
            BUCKET_ACL_GRANTS_FIELD, BUCKET_POLICY_FIELD, BUCKET_PUBLIC_ACCESS_BLOCK_FIELD));

    private static final long serialVersionUID = 1L;

    private final Map<String, S3AccessPointConfiguration> accessPoints;

    private final List<S3BucketAclGrantConfiguration> bucketAclGrants;

    private final String bucketPolicy;

    private final S3PublicAccessBlockConfiguration bucketPublicAccessBlock;

    private S3BucketConfiguration(BuilderImpl builder) {
        this.accessPoints = builder.accessPoints;
        this.bucketAclGrants = builder.bucketAclGrants;
        this.bucketPolicy = builder.bucketPolicy;
        this.bucketPublicAccessBlock = builder.bucketPublicAccessBlock;
    }

    /**
     * For responses, this returns true if the service returned a value for the AccessPoints property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAccessPoints() {
        return accessPoints != null && !(accessPoints instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The configuration of Amazon S3 access points or multi-region access points for the bucket. You can propose up to
     * 10 new access points per bucket.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAccessPoints} method.
     * </p>
     * 
     * @return The configuration of Amazon S3 access points or multi-region access points for the bucket. You can
     *         propose up to 10 new access points per bucket.
     */
    public final Map<String, S3AccessPointConfiguration> accessPoints() {
        return accessPoints;
    }

    /**
     * For responses, this returns true if the service returned a value for the BucketAclGrants property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasBucketAclGrants() {
        return bucketAclGrants != null && !(bucketAclGrants instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The proposed list of ACL grants for the Amazon S3 bucket. You can propose up to 100 ACL grants per bucket. If the
     * proposed grant configuration is for an existing bucket, the access preview uses the proposed list of grant
     * configurations in place of the existing grants. Otherwise, the access preview uses the existing grants for the
     * bucket.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasBucketAclGrants} method.
     * </p>
     * 
     * @return The proposed list of ACL grants for the Amazon S3 bucket. You can propose up to 100 ACL grants per
     *         bucket. If the proposed grant configuration is for an existing bucket, the access preview uses the
     *         proposed list of grant configurations in place of the existing grants. Otherwise, the access preview uses
     *         the existing grants for the bucket.
     */
    public final List<S3BucketAclGrantConfiguration> bucketAclGrants() {
        return bucketAclGrants;
    }

    /**
     * <p>
     * The proposed bucket policy for the Amazon S3 bucket.
     * </p>
     * 
     * @return The proposed bucket policy for the Amazon S3 bucket.
     */
    public final String bucketPolicy() {
        return bucketPolicy;
    }

    /**
     * <p>
     * The proposed block public access configuration for the Amazon S3 bucket.
     * </p>
     * 
     * @return The proposed block public access configuration for the Amazon S3 bucket.
     */
    public final S3PublicAccessBlockConfiguration bucketPublicAccessBlock() {
        return bucketPublicAccessBlock;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasAccessPoints() ? accessPoints() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasBucketAclGrants() ? bucketAclGrants() : null);
        hashCode = 31 * hashCode + Objects.hashCode(bucketPolicy());
        hashCode = 31 * hashCode + Objects.hashCode(bucketPublicAccessBlock());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3BucketConfiguration)) {
            return false;
        }
        S3BucketConfiguration other = (S3BucketConfiguration) obj;
        return hasAccessPoints() == other.hasAccessPoints() && Objects.equals(accessPoints(), other.accessPoints())
                && hasBucketAclGrants() == other.hasBucketAclGrants()
                && Objects.equals(bucketAclGrants(), other.bucketAclGrants())
                && Objects.equals(bucketPolicy(), other.bucketPolicy())
                && Objects.equals(bucketPublicAccessBlock(), other.bucketPublicAccessBlock());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("S3BucketConfiguration").add("AccessPoints", hasAccessPoints() ? accessPoints() : null)
                .add("BucketAclGrants", hasBucketAclGrants() ? bucketAclGrants() : null).add("BucketPolicy", bucketPolicy())
                .add("BucketPublicAccessBlock", bucketPublicAccessBlock()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "accessPoints":
            return Optional.ofNullable(clazz.cast(accessPoints()));
        case "bucketAclGrants":
            return Optional.ofNullable(clazz.cast(bucketAclGrants()));
        case "bucketPolicy":
            return Optional.ofNullable(clazz.cast(bucketPolicy()));
        case "bucketPublicAccessBlock":
            return Optional.ofNullable(clazz.cast(bucketPublicAccessBlock()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<S3BucketConfiguration, T> g) {
        return obj -> g.apply((S3BucketConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3BucketConfiguration> {
        /**
         * <p>
         * The configuration of Amazon S3 access points or multi-region access points for the bucket. You can propose up
         * to 10 new access points per bucket.
         * </p>
         * 
         * @param accessPoints
         *        The configuration of Amazon S3 access points or multi-region access points for the bucket. You can
         *        propose up to 10 new access points per bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessPoints(Map<String, S3AccessPointConfiguration> accessPoints);

        /**
         * <p>
         * The proposed list of ACL grants for the Amazon S3 bucket. You can propose up to 100 ACL grants per bucket. If
         * the proposed grant configuration is for an existing bucket, the access preview uses the proposed list of
         * grant configurations in place of the existing grants. Otherwise, the access preview uses the existing grants
         * for the bucket.
         * </p>
         * 
         * @param bucketAclGrants
         *        The proposed list of ACL grants for the Amazon S3 bucket. You can propose up to 100 ACL grants per
         *        bucket. If the proposed grant configuration is for an existing bucket, the access preview uses the
         *        proposed list of grant configurations in place of the existing grants. Otherwise, the access preview
         *        uses the existing grants for the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucketAclGrants(Collection<S3BucketAclGrantConfiguration> bucketAclGrants);

        /**
         * <p>
         * The proposed list of ACL grants for the Amazon S3 bucket. You can propose up to 100 ACL grants per bucket. If
         * the proposed grant configuration is for an existing bucket, the access preview uses the proposed list of
         * grant configurations in place of the existing grants. Otherwise, the access preview uses the existing grants
         * for the bucket.
         * </p>
         * 
         * @param bucketAclGrants
         *        The proposed list of ACL grants for the Amazon S3 bucket. You can propose up to 100 ACL grants per
         *        bucket. If the proposed grant configuration is for an existing bucket, the access preview uses the
         *        proposed list of grant configurations in place of the existing grants. Otherwise, the access preview
         *        uses the existing grants for the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucketAclGrants(S3BucketAclGrantConfiguration... bucketAclGrants);

        /**
         * <p>
         * The proposed list of ACL grants for the Amazon S3 bucket. You can propose up to 100 ACL grants per bucket. If
         * the proposed grant configuration is for an existing bucket, the access preview uses the proposed list of
         * grant configurations in place of the existing grants. Otherwise, the access preview uses the existing grants
         * for the bucket.
         * </p>
         * This is a convenience that creates an instance of the {@link List<S3BucketAclGrantConfiguration>.Builder}
         * avoiding the need to create one manually via {@link List<S3BucketAclGrantConfiguration>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<S3BucketAclGrantConfiguration>.Builder#build()} is called
         * immediately and its result is passed to {@link #bucketAclGrants(List<S3BucketAclGrantConfiguration>)}.
         * 
         * @param bucketAclGrants
         *        a consumer that will call methods on {@link List<S3BucketAclGrantConfiguration>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #bucketAclGrants(List<S3BucketAclGrantConfiguration>)
         */
        Builder bucketAclGrants(Consumer<S3BucketAclGrantConfiguration.Builder>... bucketAclGrants);

        /**
         * <p>
         * The proposed bucket policy for the Amazon S3 bucket.
         * </p>
         * 
         * @param bucketPolicy
         *        The proposed bucket policy for the Amazon S3 bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucketPolicy(String bucketPolicy);

        /**
         * <p>
         * The proposed block public access configuration for the Amazon S3 bucket.
         * </p>
         * 
         * @param bucketPublicAccessBlock
         *        The proposed block public access configuration for the Amazon S3 bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucketPublicAccessBlock(S3PublicAccessBlockConfiguration bucketPublicAccessBlock);

        /**
         * <p>
         * The proposed block public access configuration for the Amazon S3 bucket.
         * </p>
         * This is a convenience that creates an instance of the {@link S3PublicAccessBlockConfiguration.Builder}
         * avoiding the need to create one manually via {@link S3PublicAccessBlockConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link S3PublicAccessBlockConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #bucketPublicAccessBlock(S3PublicAccessBlockConfiguration)}.
         * 
         * @param bucketPublicAccessBlock
         *        a consumer that will call methods on {@link S3PublicAccessBlockConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #bucketPublicAccessBlock(S3PublicAccessBlockConfiguration)
         */
        default Builder bucketPublicAccessBlock(Consumer<S3PublicAccessBlockConfiguration.Builder> bucketPublicAccessBlock) {
            return bucketPublicAccessBlock(S3PublicAccessBlockConfiguration.builder().applyMutation(bucketPublicAccessBlock)
                    .build());
        }
    }

    static final class BuilderImpl implements Builder {
        private Map<String, S3AccessPointConfiguration> accessPoints = DefaultSdkAutoConstructMap.getInstance();

        private List<S3BucketAclGrantConfiguration> bucketAclGrants = DefaultSdkAutoConstructList.getInstance();

        private String bucketPolicy;

        private S3PublicAccessBlockConfiguration bucketPublicAccessBlock;

        private BuilderImpl() {
        }

        private BuilderImpl(S3BucketConfiguration model) {
            accessPoints(model.accessPoints);
            bucketAclGrants(model.bucketAclGrants);
            bucketPolicy(model.bucketPolicy);
            bucketPublicAccessBlock(model.bucketPublicAccessBlock);
        }

        public final Map<String, S3AccessPointConfiguration.Builder> getAccessPoints() {
            Map<String, S3AccessPointConfiguration.Builder> result = S3AccessPointConfigurationsMapCopier
                    .copyToBuilder(this.accessPoints);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setAccessPoints(Map<String, S3AccessPointConfiguration.BuilderImpl> accessPoints) {
            this.accessPoints = S3AccessPointConfigurationsMapCopier.copyFromBuilder(accessPoints);
        }

        @Override
        @Transient
        public final Builder accessPoints(Map<String, S3AccessPointConfiguration> accessPoints) {
            this.accessPoints = S3AccessPointConfigurationsMapCopier.copy(accessPoints);
            return this;
        }

        public final List<S3BucketAclGrantConfiguration.Builder> getBucketAclGrants() {
            List<S3BucketAclGrantConfiguration.Builder> result = S3BucketAclGrantConfigurationsListCopier
                    .copyToBuilder(this.bucketAclGrants);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setBucketAclGrants(Collection<S3BucketAclGrantConfiguration.BuilderImpl> bucketAclGrants) {
            this.bucketAclGrants = S3BucketAclGrantConfigurationsListCopier.copyFromBuilder(bucketAclGrants);
        }

        @Override
        @Transient
        public final Builder bucketAclGrants(Collection<S3BucketAclGrantConfiguration> bucketAclGrants) {
            this.bucketAclGrants = S3BucketAclGrantConfigurationsListCopier.copy(bucketAclGrants);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder bucketAclGrants(S3BucketAclGrantConfiguration... bucketAclGrants) {
            bucketAclGrants(Arrays.asList(bucketAclGrants));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder bucketAclGrants(Consumer<S3BucketAclGrantConfiguration.Builder>... bucketAclGrants) {
            bucketAclGrants(Stream.of(bucketAclGrants).map(c -> S3BucketAclGrantConfiguration.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getBucketPolicy() {
            return bucketPolicy;
        }

        public final void setBucketPolicy(String bucketPolicy) {
            this.bucketPolicy = bucketPolicy;
        }

        @Override
        @Transient
        public final Builder bucketPolicy(String bucketPolicy) {
            this.bucketPolicy = bucketPolicy;
            return this;
        }

        public final S3PublicAccessBlockConfiguration.Builder getBucketPublicAccessBlock() {
            return bucketPublicAccessBlock != null ? bucketPublicAccessBlock.toBuilder() : null;
        }

        public final void setBucketPublicAccessBlock(S3PublicAccessBlockConfiguration.BuilderImpl bucketPublicAccessBlock) {
            this.bucketPublicAccessBlock = bucketPublicAccessBlock != null ? bucketPublicAccessBlock.build() : null;
        }

        @Override
        @Transient
        public final Builder bucketPublicAccessBlock(S3PublicAccessBlockConfiguration bucketPublicAccessBlock) {
            this.bucketPublicAccessBlock = bucketPublicAccessBlock;
            return this;
        }

        @Override
        public S3BucketConfiguration build() {
            return new S3BucketConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
