/*
 * Decompiled with CFR 0.152.
 */
package pl.allegro.tech.embeddedelasticsearch;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.SystemUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import pl.allegro.tech.embeddedelasticsearch.EmbeddedElasticsearchStartupException;
import pl.allegro.tech.embeddedelasticsearch.JavaHomeOption;

class ElasticServer {
    private static final Logger logger = LoggerFactory.getLogger(ElasticServer.class);
    private final String esJavaOpts;
    private final File installationDirectory;
    private final File executableFile;
    private final long startTimeoutInMs;
    private final boolean cleanInstallationDirectoryOnStop;
    private boolean started;
    private final Object startedLock = new Object();
    private Process elastic;
    private Thread ownerThread;
    private volatile int pid = -1;
    private volatile int httpPort = -1;
    private volatile int transportTcpPort = -1;
    private JavaHomeOption javaHome;

    ElasticServer(String esJavaOpts, File installationDirectory, File executableFile, long startTimeoutInMs, boolean cleanInstallationDirectoryOnStop, JavaHomeOption javaHome) {
        this.esJavaOpts = esJavaOpts;
        this.installationDirectory = installationDirectory;
        this.executableFile = executableFile;
        this.startTimeoutInMs = startTimeoutInMs;
        this.cleanInstallationDirectoryOnStop = cleanInstallationDirectoryOnStop;
        this.javaHome = javaHome;
    }

    void start() throws InterruptedException {
        this.startElasticProcess();
        this.installExitHook();
        this.waitForElasticToStart();
    }

    void stop() {
        try {
            this.stopElasticServer();
            this.finalizeClose();
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    boolean isStarted() {
        return this.started;
    }

    private void deleteInstallationDirectory() {
        try {
            FileUtils.deleteDirectory((File)this.installationDirectory);
        }
        catch (IOException e) {
            throw new EmbeddedElasticsearchStartupException("Could not delete data directory of embedded elasticsearch server. Possibly an instance is running.", e);
        }
    }

    private void startElasticProcess() {
        this.ownerThread = new Thread(() -> {
            try {
                String line;
                ElasticServer elasticServer = this;
                synchronized (elasticServer) {
                    ProcessBuilder builder = new ProcessBuilder(new String[0]);
                    builder.environment().put("ES_JAVA_OPTS", this.esJavaOpts);
                    this.javaHome.ifNeedBeSet(javaHomeValue -> builder.environment().put("JAVA_HOME", (String)javaHomeValue));
                    builder.redirectErrorStream(true);
                    builder.command(this.elasticExecutable());
                    this.elastic = builder.start();
                }
                BufferedReader outputStream = new BufferedReader(new InputStreamReader(this.elastic.getInputStream(), StandardCharsets.UTF_8));
                while ((line = this.readLine(outputStream)) != null) {
                    logger.info(line);
                    this.parseElasticLogLine(line);
                }
            }
            catch (Exception e) {
                throw new EmbeddedElasticsearchStartupException(e);
            }
        }, "EmbeddedElsHandler");
        this.ownerThread.start();
    }

    private String readLine(BufferedReader outputStream) {
        try {
            return outputStream.readLine();
        }
        catch (IOException e) {
            return null;
        }
    }

    private void installExitHook() {
        Runtime.getRuntime().addShutdownHook(new Thread(this::stop, "ElsInstanceCleaner"));
    }

    private String elasticExecutable() {
        return this.executableFile.getAbsolutePath();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void waitForElasticToStart() throws InterruptedException {
        logger.info("Waiting for ElasticSearch to start...");
        long waitUntil = System.currentTimeMillis() + this.startTimeoutInMs;
        Object object = this.startedLock;
        synchronized (object) {
            boolean timedOut = false;
            while (!(this.started || timedOut || this.elastic != null && !this.elastic.isAlive())) {
                this.startedLock.wait(100L);
                timedOut = System.currentTimeMillis() > waitUntil;
            }
            if (!this.started) {
                String message = timedOut ? "Failed to start elasticsearch within time-out" : "Failed to start elasticsearch. Check previous logs for details";
                throw new EmbeddedElasticsearchStartupException(message);
            }
        }
        logger.info("ElasticSearch started...");
    }

    private void parseElasticLogLine(String line) {
        if (this.started) {
            return;
        }
        if (line.contains("] started")) {
            this.signalElasticStarted();
        } else if (line.contains(", pid[")) {
            this.tryExtractPid(line);
        } else if (line.contains("publish_address") && (line.contains("[http") || line.contains("HttpServer"))) {
            this.tryExtractHttpPort(line);
        } else if (line.contains("publish_address") && (line.contains("[transport") || line.contains("TransportService"))) {
            this.tryExtractTransportTcpPort(line);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void signalElasticStarted() {
        Object object = this.startedLock;
        synchronized (object) {
            this.started = true;
            this.startedLock.notifyAll();
        }
    }

    private void tryExtractPid(String line) {
        Matcher matcher = Pattern.compile("pid\\[(\\d+)]").matcher(line);
        Validate.isTrue((boolean)matcher.find());
        this.pid = Integer.parseInt(matcher.group(1));
        logger.info("Detected Elasticsearch PID : " + this.pid);
    }

    private void tryExtractHttpPort(String line) {
        Matcher matcher = Pattern.compile("publish_address \\{.*?:(\\d+).?}").matcher(line);
        Validate.isTrue((boolean)matcher.find());
        this.httpPort = Integer.parseInt(matcher.group(1));
        logger.info("Detected Elasticsearch http port : " + this.httpPort);
    }

    private void tryExtractTransportTcpPort(String line) {
        Matcher matcher = Pattern.compile("publish_address \\{.*?:(\\d+).?}").matcher(line);
        Validate.isTrue((boolean)matcher.find());
        this.transportTcpPort = Integer.parseInt(matcher.group(1));
        logger.info("Detected Elasticsearch transport tcp port : " + this.transportTcpPort);
    }

    private void stopElasticServer() throws IOException, InterruptedException {
        logger.info("Stopping elasticsearch server...");
        if (this.pid > -1) {
            this.stopElasticGracefully();
        }
        this.pid = -1;
        if (this.elastic != null) {
            int rc = this.elastic.waitFor();
            logger.info("Elasticsearch exited with RC " + rc);
        }
        this.elastic = null;
        if (this.ownerThread != null) {
            this.ownerThread.join();
        }
        this.ownerThread = null;
    }

    private void stopElasticGracefully() throws IOException {
        if (SystemUtils.IS_OS_WINDOWS) {
            this.stopElasticOnWindows();
        } else {
            this.elastic.destroy();
        }
    }

    private void stopElasticOnWindows() throws IOException {
        Runtime.getRuntime().exec("taskkill /f /pid " + this.pid);
    }

    private void finalizeClose() {
        if (this.cleanInstallationDirectoryOnStop) {
            logger.info("Removing installation directory...");
            this.deleteInstallationDirectory();
        }
        logger.info("Finishing...");
        this.started = false;
    }

    int getHttpPort() {
        return this.httpPort;
    }

    int getTransportTcpPort() {
        return this.transportTcpPort;
    }
}

