/*
 * Decompiled with CFR 0.152.
 */
package org.sonarsource.scanner.lib.internal.util;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermission;
import java.util.EnumSet;
import java.util.Enumeration;
import java.util.List;
import java.util.Set;
import java.util.function.Predicate;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import org.apache.commons.compress.archivers.ArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.compress.compressors.gzip.GzipCompressorInputStream;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.SystemUtils;

public final class CompressionUtils {
    private static final String ERROR_CREATING_DIRECTORY = "Error creating directory: ";
    private static final List<PosixFilePermission> POSIX_PERMISSIONS = List.of(PosixFilePermission.OTHERS_EXECUTE, PosixFilePermission.OTHERS_WRITE, PosixFilePermission.OTHERS_READ, PosixFilePermission.GROUP_EXECUTE, PosixFilePermission.GROUP_WRITE, PosixFilePermission.GROUP_READ, PosixFilePermission.OWNER_EXECUTE, PosixFilePermission.OWNER_WRITE, PosixFilePermission.OWNER_READ);
    private static final int MAX_MODE = (1 << POSIX_PERMISSIONS.size()) - 1;

    private CompressionUtils() {
    }

    public static Path unzip(Path zip, Path toDir) throws IOException {
        return CompressionUtils.unzip(zip, toDir, ze -> true);
    }

    public static Path unzip(Path zip, Path toDir, Predicate<ZipEntry> filter) throws IOException {
        Path targetDirNormalizedPath = toDir.normalize();
        try (ZipFile zipFile = new ZipFile(zip.toFile());){
            Enumeration<? extends ZipEntry> entries = zipFile.entries();
            while (entries.hasMoreElements()) {
                ZipEntry entry = entries.nextElement();
                if (!filter.test(entry)) continue;
                Path target = toDir.resolve(entry.getName());
                CompressionUtils.verifyInsideTargetDirectory(entry.getName(), target, targetDirNormalizedPath);
                if (entry.isDirectory()) {
                    CompressionUtils.throwExceptionIfDirectoryIsNotCreatable(target);
                    continue;
                }
                Path parent = target.getParent();
                CompressionUtils.throwExceptionIfDirectoryIsNotCreatable(parent);
                CompressionUtils.copy(zipFile, entry, target);
            }
            Path path = toDir;
            return path;
        }
    }

    private static void verifyInsideTargetDirectory(String entryName, Path entryPath, Path targetDirNormalizedPath) {
        if (!entryPath.normalize().startsWith(targetDirNormalizedPath)) {
            throw new IllegalStateException("Extracting an entry outside the target directory is not allowed: " + entryName);
        }
    }

    private static void throwExceptionIfDirectoryIsNotCreatable(Path to) throws IOException {
        try {
            Files.createDirectories(to, new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new IOException(ERROR_CREATING_DIRECTORY + String.valueOf(to), e);
        }
    }

    private static void copy(ZipFile zipFile, ZipEntry entry, Path to) throws IOException {
        try (InputStream input = zipFile.getInputStream(entry);
             OutputStream fos = Files.newOutputStream(to, new OpenOption[0]);){
            IOUtils.copy((InputStream)input, (OutputStream)fos);
        }
    }

    public static void extractTarGz(Path compressedFile, Path targetDir) throws IOException {
        Path targetDirNormalizedPath = targetDir.normalize();
        try (InputStream fis = Files.newInputStream(compressedFile, new OpenOption[0]);
             BufferedInputStream bis = new BufferedInputStream(fis);
             GzipCompressorInputStream gzis = new GzipCompressorInputStream((InputStream)bis);
             TarArchiveInputStream tarArchiveInputStream = new TarArchiveInputStream((InputStream)gzis);){
            TarArchiveEntry targzEntry;
            while ((targzEntry = tarArchiveInputStream.getNextEntry()) != null) {
                if (!tarArchiveInputStream.canReadEntryData((ArchiveEntry)targzEntry)) continue;
                Path target = targetDir.resolve(targzEntry.getName());
                CompressionUtils.verifyInsideTargetDirectory(targzEntry.getName(), target, targetDirNormalizedPath);
                if (targzEntry.isDirectory()) {
                    Files.createDirectories(target, new FileAttribute[0]);
                    continue;
                }
                if (!Files.isDirectory(target.getParent(), new LinkOption[0])) {
                    Files.createDirectories(target.getParent(), new FileAttribute[0]);
                }
                Files.copy((InputStream)tarArchiveInputStream, target, StandardCopyOption.REPLACE_EXISTING);
                int mode = targzEntry.getMode();
                if (mode == 0 || SystemUtils.IS_OS_WINDOWS) continue;
                Set<PosixFilePermission> permissions = CompressionUtils.fromFileMode(mode);
                Files.setPosixFilePermissions(target, permissions);
            }
        }
    }

    static Set<PosixFilePermission> fromFileMode(int fileMode) {
        if ((fileMode & MAX_MODE) != fileMode) {
            throw new IllegalStateException("Invalid file mode '" + Integer.toOctalString(fileMode) + "'. File mode must be between 0 and " + MAX_MODE + " (" + Integer.toOctalString(MAX_MODE) + " in octal)");
        }
        EnumSet<PosixFilePermission> ret = EnumSet.noneOf(PosixFilePermission.class);
        for (int i = 0; i < POSIX_PERMISSIONS.size(); ++i) {
            if ((fileMode & 1 << i) == 0) continue;
            ret.add(POSIX_PERMISSIONS.get(i));
        }
        return ret;
    }
}

