/*
 * Decompiled with CFR 0.152.
 */
package org.sonarsource.scanner.lib.internal.cache;

import java.io.IOException;
import java.nio.file.AtomicMoveNotSupportedException;
import java.nio.file.CopyOption;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import javax.annotation.CheckForNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.sonarsource.scanner.lib.internal.cache.CachedFile;
import org.sonarsource.scanner.lib.internal.cache.FileHashes;
import org.sonarsource.scanner.lib.internal.cache.HashMismatchException;

public class FileCache {
    private static final Logger LOG = LoggerFactory.getLogger(FileCache.class);
    private final Path dir;
    private final Path tmpDir;
    private final FileHashes hashes;

    FileCache(Path dir, FileHashes fileHashes) {
        this.hashes = fileHashes;
        this.dir = FileCache.createDir(dir, "user cache: ");
        LOG.info("User cache: {}", (Object)dir);
        this.tmpDir = FileCache.createDir(dir.resolve("_tmp"), "temp dir");
    }

    public static FileCache create(Path sonarUserHome) {
        Path dir = sonarUserHome.resolve("cache");
        return new FileCache(dir, new FileHashes());
    }

    public Path getDir() {
        return this.dir;
    }

    @CheckForNull
    public Path get(String filename, String hash) {
        Path cachedFile = this.dir.resolve(hash).resolve(filename);
        if (Files.exists(cachedFile, new LinkOption[0])) {
            return cachedFile;
        }
        LOG.debug("No file found in the cache with name {} and hash {}", (Object)filename, (Object)hash);
        return null;
    }

    public CachedFile getOrDownload(String filename, String hash, String hashAlgorithm, Downloader downloader) {
        Path hashDir = this.hashDir(hash);
        Path targetFile = hashDir.resolve(filename);
        if (Files.exists(targetFile, new LinkOption[0])) {
            return new CachedFile(targetFile, true);
        }
        Path tempFile = this.newTempFile();
        FileCache.download(downloader, filename, tempFile);
        String downloadedHash = this.hashes.of(tempFile.toFile(), hashAlgorithm);
        if (!hash.equals(downloadedHash)) {
            throw new HashMismatchException("INVALID HASH: File " + String.valueOf(tempFile.toAbsolutePath()) + " was expected to have hash " + hash + " but was downloaded with hash " + downloadedHash);
        }
        FileCache.mkdirQuietly(hashDir);
        FileCache.renameQuietly(tempFile, targetFile);
        return new CachedFile(targetFile, false);
    }

    private static void download(Downloader downloader, String filename, Path tempFile) {
        try {
            downloader.download(filename, tempFile);
        }
        catch (IOException e) {
            throw new IllegalStateException("Fail to download " + filename + " to " + String.valueOf(tempFile), e);
        }
    }

    private static void renameQuietly(Path sourceFile, Path targetFile) {
        try {
            Files.move(sourceFile, targetFile, StandardCopyOption.ATOMIC_MOVE);
        }
        catch (AtomicMoveNotSupportedException ex) {
            LOG.warn("Unable to rename {} to {}", (Object)sourceFile.toAbsolutePath(), (Object)targetFile.toAbsolutePath());
            LOG.warn("A copy/delete will be tempted but with no guarantee of atomicity");
            try {
                Files.move(sourceFile, targetFile, new CopyOption[0]);
            }
            catch (IOException e) {
                throw new IllegalStateException("Fail to move " + String.valueOf(sourceFile.toAbsolutePath()) + " to " + String.valueOf(targetFile), e);
            }
        }
        catch (FileAlreadyExistsException ex) {
        }
        catch (IOException e) {
            throw new IllegalStateException("Fail to move " + String.valueOf(sourceFile.toAbsolutePath()) + " to " + String.valueOf(targetFile), e);
        }
    }

    private Path hashDir(String hash) {
        return this.dir.resolve(hash);
    }

    private static void mkdirQuietly(Path hashDir) {
        try {
            Files.createDirectories(hashDir, new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new IllegalStateException("Fail to create cache directory: " + String.valueOf(hashDir), e);
        }
    }

    private Path newTempFile() {
        try {
            return Files.createTempFile(this.tmpDir, "fileCache", null, new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new IllegalStateException("Fail to create temp file in " + String.valueOf(this.tmpDir), e);
        }
    }

    private static Path createDir(Path dir, String debugTitle) {
        LOG.debug("Create: {}", (Object)dir);
        try {
            Files.createDirectories(dir, new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new IllegalStateException("Unable to create " + debugTitle + dir.toString(), e);
        }
        return dir;
    }

    @FunctionalInterface
    public static interface Downloader {
        public void download(String var1, Path var2) throws IOException;
    }
}

