package breeze.numerics.constants

import scala.collection.immutable.HashMap
import scala.util.matching.Regex

/**Object encapsulating CODATA2010 constants.
 *
 * @author ktakagaki
 * @date 3/12/14.
 */
object Database {

  /** Return a vector containing all database keys.
   */
  def list() = databaseHM.keys.toVector

  /** Look up the value of a specific entry.
   * Name must be an exact match with the entry.
   */
  def value(string: String) = databaseHM(string)._1

  /** Look up the value of entries with name matching the given regex expression.
   *
   * @param regex enter a regex object (i.e. """regex expression""".r)
   * @return returns a vector of (String, Double) objects, which contain the full entry name (String) and the value (Double)
   */
  def value(regex: Regex) = databaseHM.collect {
    case (key: String, (value: Double, _, _)) if regex.findFirstIn(key).nonEmpty => (key, value)
  }

  /** Look up the uncertainty of a specific entry.
   * Name must be an exact match with the entry.
   */
  def uncertainty(string: String) = databaseHM(string)._2

  /** Look up the uncertainty of entries with name matching the given regex expression.
   *
   * @param regex enter a regex object (i.e. """regex expression""".r)
   * @return returns a vector of (String, Double) objects, which contain the full entry name (String) and the uncertainty (Double)
   */
  def uncertainty(regex: Regex) = databaseHM.collect {
    case (key: String, (_, uncert: Double, _)) if regex.findFirstIn(key).nonEmpty => (key, uncert)
  }

  /** Look up the unit of a specific entry.
   * Name must be an exact match with the entry.
   */
  def unit(string: String) = databaseHM(string)._3

  /** Look up the unit of entries with name matching the given regex expression.
   *
   * @param regex enter a regex object (i.e. """regex expression""".r)
   * @return returns a vector of (String, Double) objects, which contain the full entry name (String) and the uncertainty (Double)
   */
  def unit(regex: Regex) = databaseHM.collect {
    case (key: String, (_, _, unit: String)) if regex.findFirstIn(key).nonEmpty => (key, unit)
  }

  // <editor-fold defaultstate="collapsed" desc=" CODATA2010 Database HashMap ">

  private lazy val databaseHM = HashMap[String, (Double, Double, String)](
    ("""{220} lattice spacing of silicon""", (1.920155714E-10, 3.2E-18, """m""")),
    ("""alpha particle-electron mass ratio""", (7294.29953610, 2.9E-6, """""")),
    ("""alpha particle mass""", (6.64465675E-27, 2.9E-34, """kg""")),
    ("""alpha particle mass energy equivalent""", (5.97191967E-10, 2.6E-17, """J""")),
    ("""alpha particle mass energy equivalent in MeV""", (3727.379240, 0.000082, """MeV""")),
    ("""alpha particle mass in u""", (4.0015061791250, 6.2E-11, """u""")),
    ("""alpha particle molar mass""", (0.0040015061791250, 6.2E-14, """kg mol^-1""")),
    ("""alpha particle-proton mass ratio""", (3.97259968933, 3.6E-10, """""")),
    ("""Angstrom star""", (1.00001495E-10, 9.0E-17, """m""")),
    ("""atomic mass constant""", (1.660538921E-27, 7.3E-35, """kg""")),
    ("""atomic mass constant energy equivalent""", (1.492417954E-10, 6.6E-18, """J""")),
    ("""atomic mass constant energy equivalent in MeV""", (931.4940610, 0.000021, """MeV""")),
    ("""atomic mass unit-electron volt relationship""", (9.31494061E8, 21.0, """eV""")),
    ("""atomic mass unit-hartree relationship""", (3.4231776845E7, 0.024, """E_h""")),
    ("""atomic mass unit-hertz relationship""", (2.2523427168E23, 1.6E14, """Hz""")),
    ("""atomic mass unit-inverse meter relationship""", (7.5130066042E14, 530000.0, """m^-1""")),
    ("""atomic mass unit-joule relationship""", (1.492417954E-10, 6.6E-18, """J""")),
    ("""atomic mass unit-kelvin relationship""", (1.08095408E13, 9.8E6, """K""")),
    ("""atomic mass unit-kilogram relationship""", (1.660538921E-27, 7.3E-35, """kg""")),
    ("""atomic unit of 1st hyperpolarizability""", (3.206361449E-53, 7.1E-61, """C^3 m^3 J^-2""")),
    ("""atomic unit of 2nd hyperpolarizability""", (6.23538054E-65, 2.8E-72, """C^4 m^4 J^-3""")),
    ("""atomic unit of action""", (1.054571726E-34, 4.7E-42, """J s""")),
    ("""atomic unit of charge""", (1.602176565E-19, 3.5E-27, """C""")),
    ("""atomic unit of charge density""", (1.081202338E12, 24000.0, """C m^-3""")),
    ("""atomic unit of current""", (0.006623617950, 1.5E-10, """A""")),
    ("""atomic unit of electric dipole mom.""", (8.47835326E-30, 1.9E-37, """C m""")),
    ("""atomic unit of electric field""", (5.14220652E11, 11000.0, """V m^-1""")),
    ("""atomic unit of electric field gradient""", (9.717362E21, 2.1E14, """V m^-2""")),
    ("""atomic unit of electric polarizability""", (1.6487772754E-41, 1.6E-50, """C^2 m^2 J^-1""")),
    ("""atomic unit of electric potential""", (27.21138505, 6.0E-7, """V""")),
    ("""atomic unit of electric quadrupole mom.""", (4.486551331E-40, 9.9E-48, """C m^2""")),
    ("""atomic unit of energy""", (4.35974434E-18, 1.9E-25, """J""")),
    ("""atomic unit of force""", (8.23872278E-8, 3.6E-15, """N""")),
    ("""atomic unit of length""", (5.2917721092E-11, 1.7E-20, """m""")),
    ("""atomic unit of mag. dipole mom.""", (1.854801936E-23, 4.1E-31, """J T^-1""")),
    ("""atomic unit of mag. flux density""", (235051.7464, 0.0052, """T""")),
    ("""atomic unit of magnetizability""", (7.891036607E-29, 1.3E-37, """J T^-2""")),
    ("""atomic unit of mass""", (9.10938291E-31, 4.0E-38, """kg""")),
    ("""atomic unit of mom.um""", (1.99285174E-24, 8.8E-32, """kg m s^-1""")),
    ("""atomic unit of permittivity""", (1.112650056E-10, 0d, """F m^-1""")),
    ("""atomic unit of time""", (2.418884326502E-17, 1.2E-28, """s""")),
    ("""atomic unit of velocity""", (2.18769126379E6, 0.00071, """m s^-1""")),
    ("""Avogadro constant""", (6.02214129E23, 2.7E16, """mol^-1""")),
    ("""Bohr magneton""", (9.27400968E-24, 2.0E-31, """J T^-1""")),
    ("""Bohr magneton in eV/T""", (0.0000578838180660, 3.8E-14, """eV T^-1""")),
    ("""Bohr magneton in Hz/T""", (1.399624555E10, 310.0, """Hz T^-1""")),
    ("""Bohr magneton in inverse meters per tesla""", (46.68644980, 1.0E-6, """m^-1 T^-1""")),
    ("""Bohr magneton in K/T""", (0.67171388, 6.1E-7, """K T^-1""")),
    ("""Bohr radius""", (5.2917721092E-11, 1.7E-20, """m""")),
    ("""Boltzmann constant""", (1.3806488E-23, 1.3E-29, """J K^-1""")),
    ("""Boltzmann constant in eV/K""", (0.000086173324, 7.8E-11, """eV K^-1""")),
    ("""Boltzmann constant in Hz/K""", (2.0836618E10, 19000.0, """Hz K^-1""")),
    ("""Boltzmann constant in inverse meters per kelvin""", (69.503476, 0.000063, """m^-1 K^-1""")),
    ("""characteristic impedance of vacuum""", (376.730313461, 0d, """ohm""")),
    ("""classical electron radius""", (2.8179403267E-15, 2.7E-24, """m""")),
    ("""Compton wavelength""", (2.4263102389E-12, 1.6E-21, """m""")),
    ("""Compton wavelength over 2 pi""", (3.86159268E-13, 2.5E-22, """m""")),
    ("""conductance quantum""", (0.0000774809173460, 2.5E-14, """S""")),
    ("""conventional value of Josephson constant""", (4.835979E14, 0d, """Hz V^-1""")),
    ("""conventional value of von Klitzing constant""", (25812.807, 0d, """ohm""")),
    ("""Cu x unit""", (1.00207697E-13, 2.8E-20, """m""")),
    ("""deuteron-electron mag. mom. ratio""", (-0.0004664345537, 3.9E-12, """""")),
    ("""deuteron-electron mass ratio""", (3670.48296520, 1.5E-6, """""")),
    ("""deuteron g factor""", (0.8574382308, 7.2E-9, """""")),
    ("""deuteron mag. mom.""", (4.33073489E-27, 1.0E-34, """J T^-1""")),
    ("""deuteron mag. mom. to Bohr magneton ratio""", (0.0004669754556, 3.9E-12, """""")),
    ("""deuteron mag. mom. to nuclear magneton ratio""", (0.8574382308, 7.2E-9, """""")),
    ("""deuteron mass""", (3.34358348E-27, 1.5E-34, """kg""")),
    ("""deuteron mass energy equivalent""", (3.00506297E-10, 1.3E-17, """J""")),
    ("""deuteron mass energy equivalent in MeV""", (1875.612859, 0.000041, """MeV""")),
    ("""deuteron mass in u""", (2.0135532127120, 7.7E-11, """u""")),
    ("""deuteron molar mass""", (0.0020135532127120, 7.7E-14, """kg mol^-1""")),
    ("""deuteron-neutron mag. mom. ratio""", (-0.44820652, 1.1E-7, """""")),
    ("""deuteron-proton mag. mom. ratio""", (0.3070122070, 2.4E-9, """""")),
    ("""deuteron-proton mass ratio""", (1.99900750097, 1.8E-10, """""")),
    ("""deuteron rms charge radius""", (2.1424E-15, 2.1E-18, """m""")),
    ("""electric constant""", (8.854187817E-12, 0d, """F m^-1""")),
    ("""electron charge to mass quotient""", (-1.758820088E11, 3900.0, """C kg^-1""")),
    ("""electron-deuteron mag. mom. ratio""", (-2143.923498, 0.000018, """""")),
    ("""electron-deuteron mass ratio""", (0.000272443710950, 1.1E-13, """""")),
    ("""electron g factor""", (-2.002319304361530, 5.3E-13, """""")),
    ("""electron gyromag. ratio""", (1.760859708E11, 3900.0, """s^-1 T^-1""")),
    ("""electron gyromag. ratio over 2 pi""", (28024.95266, 0.00062, """MHz T^-1""")),
    ("""electron-helion mass ratio""", (0.000181954307610, 1.7E-13, """""")),
    ("""electron mag. mom.""", (-9.2847643E-24, 2.1E-31, """J T^-1""")),
    ("""electron mag. mom. anomaly""", (0.00115965218076, 2.7E-13, """""")),
    ("""electron mag. mom. to Bohr magneton ratio""", (-1.001159652180760, 2.7E-13, """""")),
    ("""electron mag. mom. to nuclear magneton ratio""", (-1838.28197090, 7.5E-7, """""")),
    ("""electron mass""", (9.10938291E-31, 4.0E-38, """kg""")),
    ("""electron mass energy equivalent""", (8.18710506E-14, 3.6E-21, """J""")),
    ("""electron mass energy equivalent in MeV""", (0.5109989280, 1.1E-8, """MeV""")),
    ("""electron mass in u""", (0.000548579909460, 2.2E-13, """u""")),
    ("""electron molar mass""", (5.4857990946E-7, 2.2E-16, """kg mol^-1""")),
    ("""electron-muon mag. mom. ratio""", (206.7669896, 5.2E-6, """""")),
    ("""electron-muon mass ratio""", (0.004836331660, 1.2E-10, """""")),
    ("""electron-neutron mag. mom. ratio""", (960.92050, 0.00023, """""")),
    ("""electron-neutron mass ratio""", (0.000543867344610, 3.2E-13, """""")),
    ("""electron-proton mag. mom. ratio""", (-658.2106848, 5.4E-6, """""")),
    ("""electron-proton mass ratio""", (0.000544617021780, 2.2E-13, """""")),
    ("""electron-tau mass ratio""", (0.000287592, 2.6E-8, """""")),
    ("""electron to alpha particle mass ratio""", (0.000137093355578, 5.5E-14, """""")),
    ("""electron to shielded helion mag. mom. ratio""", (864.0582570, 1.0E-5, """""")),
    ("""electron to shielded proton mag. mom. ratio""", (-658.2275971, 7.2E-6, """""")),
    ("""electron-triton mass ratio""", (0.000181920006530, 1.7E-13, """""")),
    ("""electron volt""", (1.602176565E-19, 3.5E-27, """J""")),
    ("""electron volt-atomic mass unit relationship""", (1.07354415E-9, 2.4E-17, """u""")),
    ("""electron volt-hartree relationship""", (0.03674932379, 8.1E-10, """E_h""")),
    ("""electron volt-hertz relationship""", (2.417989348E14, 5.3E6, """Hz""")),
    ("""electron volt-inverse meter relationship""", (806554.4290, 0.018, """m^-1""")),
    ("""electron volt-joule relationship""", (1.602176565E-19, 3.5E-27, """J""")),
    ("""electron volt-kelvin relationship""", (11604.519, 0.011, """K""")),
    ("""electron volt-kilogram relationship""", (1.782661845E-36, 3.9E-44, """kg""")),
    ("""elementary charge""", (1.602176565E-19, 3.5E-27, """C""")),
    ("""elementary charge over h""", (2.417989348E14, 5.3E6, """A J^-1""")),
    ("""Faraday constant""", (96485.33650, 0.0021, """C mol^-1""")),
    ("""Faraday constant for conventional electric current""", (96485.33210, 0.0043, """C_90 mol^-1""")),
    ("""Fermi coupling constant""", (0.000011663640, 5.0E-11, """GeV^-2""")),
    ("""fine-structure constant""", (0.00729735256980, 2.4E-12, """""")),
    ("""first radiation constant""", (3.74177153E-16, 1.7E-23, """W m^2""")),
    ("""first radiation constant for spectral radiance""", (1.191042869E-16, 5.3E-24, """W m^2 sr^-1""")),
    ("""hartree-atomic mass unit relationship""", (2.9212623246E-8, 2.1E-17, """u""")),
    ("""hartree-electron volt relationship""", (27.21138505, 6.0E-7, """eV""")),
    ("""Hartree energy""", (4.35974434E-18, 1.9E-25, """J""")),
    ("""Hartree energy in eV""", (27.21138505, 6.0E-7, """eV""")),
    ("""hartree-hertz relationship""", (6.579683920729E15, 33000.0, """Hz""")),
    ("""hartree-inverse meter relationship""", (2.194746313708E7, 0.00011, """m^-1""")),
    ("""hartree-joule relationship""", (4.35974434E-18, 1.9E-25, """J""")),
    ("""hartree-kelvin relationship""", (315775.04, 0.29, """K""")),
    ("""hartree-kilogram relationship""", (4.85086979E-35, 2.1E-42, """kg""")),
    ("""helion-electron mass ratio""", (5495.88527540, 5.0E-6, """""")),
    ("""helion g factor""", (-4.255250613, 5.0E-8, """""")),
    ("""helion mag. mom.""", (-1.074617486E-26, 2.7E-34, """J T^-1""")),
    ("""helion mag. mom. to Bohr magneton ratio""", (-0.001158740958, 1.4E-11, """""")),
    ("""helion mag. mom. to nuclear magneton ratio""", (-2.127625306, 2.5E-8, """""")),
    ("""helion mass""", (5.00641234E-27, 2.2E-34, """kg""")),
    ("""helion mass energy equivalent""", (4.49953902E-10, 2.0E-17, """J""")),
    ("""helion mass energy equivalent in MeV""", (2808.391482, 0.000062, """MeV""")),
    ("""helion mass in u""", (3.01493224680, 2.5E-9, """u""")),
    ("""helion molar mass""", (0.00301493224680, 2.5E-12, """kg mol^-1""")),
    ("""helion-proton mass ratio""", (2.99315267070, 2.5E-9, """""")),
    ("""hertz-atomic mass unit relationship""", (4.4398216689E-24, 3.1E-33, """u""")),
    ("""hertz-electron volt relationship""", (4.135667516E-15, 9.1E-23, """eV""")),
    ("""hertz-hartree relationship""", (1.5198298460045E-16, 7.6E-28, """E_h""")),
    ("""hertz-inverse meter relationship""", (3.335640951E-9, 0d, """m^-1""")),
    ("""hertz-joule relationship""", (6.62606957E-34, 2.9E-41, """J""")),
    ("""hertz-kelvin relationship""", (4.7992434E-11, 4.4E-17, """K""")),
    ("""hertz-kilogram relationship""", (7.37249668E-51, 3.3E-58, """kg""")),
    ("""inverse fine-structure constant""", (137.035999074, 4.4E-8, """""")),
    ("""inverse meter-atomic mass unit relationship""", (1.3310250512E-15, 9.4E-25, """u""")),
    ("""inverse meter-electron volt relationship""", (1.23984193E-6, 2.7E-14, """eV""")),
    ("""inverse meter-hartree relationship""", (4.556335252755E-8, 2.3E-19, """E_h""")),
    ("""inverse meter-hertz relationship""", (2.99792458E8, 0d, """Hz""")),
    ("""inverse meter-joule relationship""", (1.986445684E-25, 8.8E-33, """J""")),
    ("""inverse meter-kelvin relationship""", (0.014387770, 1.3E-8, """K""")),
    ("""inverse meter-kilogram relationship""", (2.210218902E-42, 9.8E-50, """kg""")),
    ("""inverse of conductance quantum""", (12906.4037217, 4.2E-6, """ohm""")),
    ("""Josephson constant""", (4.8359787E14, 1.1E7, """Hz V^-1""")),
    ("""joule-atomic mass unit relationship""", (6.70053585E9, 300.0, """u""")),
    ("""joule-electron volt relationship""", (6.24150934E18, 1.4E11, """eV""")),
    ("""joule-hartree relationship""", (2.29371248E17, 1.0E10, """E_h""")),
    ("""joule-hertz relationship""", (1.509190311E33, 6.7E25, """Hz""")),
    ("""joule-inverse meter relationship""", (5.03411701E24, 2.2E17, """m^-1""")),
    ("""joule-kelvin relationship""", (7.2429716E22, 6.6E16, """K""")),
    ("""joule-kilogram relationship""", (1.112650056E-17, 0d, """kg""")),
    ("""kelvin-atomic mass unit relationship""", (9.2510868E-14, 8.4E-20, """u""")),
    ("""kelvin-electron volt relationship""", (0.000086173324, 7.8E-11, """eV""")),
    ("""kelvin-hartree relationship""", (3.1668114E-6, 2.9E-12, """E_h""")),
    ("""kelvin-hertz relationship""", (2.0836618E10, 19000.0, """Hz""")),
    ("""kelvin-inverse meter relationship""", (69.503476, 0.000063, """m^-1""")),
    ("""kelvin-joule relationship""", (1.3806488E-23, 1.3E-29, """J""")),
    ("""kelvin-kilogram relationship""", (1.536179E-40, 1.4E-46, """kg""")),
    ("""kilogram-atomic mass unit relationship""", (6.02214129E26, 2.7E19, """u""")),
    ("""kilogram-electron volt relationship""", (5.60958885E35, 1.2E28, """eV""")),
    ("""kilogram-hartree relationship""", (2.061485968E34, 9.1E26, """E_h""")),
    ("""kilogram-hertz relationship""", (1.356392608E50, 6.0E42, """Hz""")),
    ("""kilogram-inverse meter relationship""", (4.52443873E41, 2.0E34, """m^-1""")),
    ("""kilogram-joule relationship""", (8.987551787E16, 0d, """J""")),
    ("""kilogram-kelvin relationship""", (6.5096582E39, 5.9E33, """K""")),
    ("""lattice parameter of silicon""", (5.431020504E-10, 8.9E-18, """m""")),
    ("""Loschmidt constant (273.15 K, 100 kPa)""", (2.6516462E25, 2.4E19, """m^-3""")),
    ("""Loschmidt constant (273.15 K, 101.325 kPa)""", (2.6867805E25, 2.4E19, """m^-3""")),
    ("""mag. constant""", (1.2566370614E-6, 0d, """N A^-2""")),
    ("""mag. flux quantum""", (2.067833758E-15, 4.6E-23, """Wb""")),
    ("""molar gas constant""", (8.3144621, 7.5E-6, """J mol^-1 K^-1""")),
    ("""molar mass constant""", (0.0010, 0d, """kg mol^-1""")),
    ("""molar mass of carbon-12""", (0.012, 0d, """kg mol^-1""")),
    ("""molar Planck constant""", (3.9903127176E-10, 2.8E-19, """J s mol^-1""")),
    ("""molar Planck constant times c""", (0.119626565779, 8.4E-11, """J m mol^-1""")),
    ("""molar volume of ideal gas (273.15 K, 100 kPa)""", (0.022710953, 2.1E-8, """m^3 mol^-1""")),
    ("""molar volume of ideal gas (273.15 K, 101.325 kPa)""", (0.022413968, 2.0E-8, """m^3 mol^-1""")),
    ("""molar volume of silicon""", (0.00001205883301, 8.0E-13, """m^3 mol^-1""")),
    ("""Mo x unit""", (1.00209952E-13, 5.3E-20, """m""")),
    ("""muon Compton wavelength""", (1.173444103E-14, 3.0E-22, """m""")),
    ("""muon Compton wavelength over 2 pi""", (1.867594294E-15, 4.7E-23, """m""")),
    ("""muon-electron mass ratio""", (206.7682843, 5.2E-6, """""")),
    ("""muon g factor""", (-2.00233184180, 1.3E-9, """""")),
    ("""muon mag. mom.""", (-4.49044807E-26, 1.5E-33, """J T^-1""")),
    ("""muon mag. mom. anomaly""", (0.001165920910, 6.3E-10, """""")),
    ("""muon mag. mom. to Bohr magneton ratio""", (-0.004841970440, 1.2E-10, """""")),
    ("""muon mag. mom. to nuclear magneton ratio""", (-8.890596970, 2.2E-7, """""")),
    ("""muon mass""", (1.883531475E-28, 9.6E-36, """kg""")),
    ("""muon mass energy equivalent""", (1.692833667E-11, 8.6E-19, """J""")),
    ("""muon mass energy equivalent in MeV""", (105.6583715, 3.5E-6, """MeV""")),
    ("""muon mass in u""", (0.1134289267, 2.9E-9, """u""")),
    ("""muon molar mass""", (0.0001134289267, 2.9E-12, """kg mol^-1""")),
    ("""muon-neutron mass ratio""", (0.1124545177, 2.8E-9, """""")),
    ("""muon-proton mag. mom. ratio""", (-3.183345107, 8.4E-8, """""")),
    ("""muon-proton mass ratio""", (0.1126095272, 2.8E-9, """""")),
    ("""muon-tau mass ratio""", (0.0594649, 5.4E-6, """""")),
    ("""natural unit of action""", (1.054571726E-34, 4.7E-42, """J s""")),
    ("""natural unit of action in eV s""", (6.58211928E-16, 1.5E-23, """eV s""")),
    ("""natural unit of energy""", (8.18710506E-14, 3.6E-21, """J""")),
    ("""natural unit of energy in MeV""", (0.5109989280, 1.1E-8, """MeV""")),
    ("""natural unit of length""", (3.86159268E-13, 2.5E-22, """m""")),
    ("""natural unit of mass""", (9.10938291E-31, 4.0E-38, """kg""")),
    ("""natural unit of mom.um""", (2.73092429E-22, 1.2E-29, """kg m s^-1""")),
    ("""natural unit of mom.um in MeV/c""", (0.5109989280, 1.1E-8, """MeV/c""")),
    ("""natural unit of time""", (1.28808866833E-21, 8.3E-31, """s""")),
    ("""natural unit of velocity""", (2.99792458E8, 0d, """m s^-1""")),
    ("""neutron Compton wavelength""", (1.3195909068E-15, 1.1E-24, """m""")),
    ("""neutron Compton wavelength over 2 pi""", (2.1001941568E-16, 1.7E-25, """m""")),
    ("""neutron-electron mag. mom. ratio""", (0.001040668820, 2.5E-10, """""")),
    ("""neutron-electron mass ratio""", (1838.68366050, 1.1E-6, """""")),
    ("""neutron g factor""", (-3.826085450, 9.0E-7, """""")),
    ("""neutron gyromag. ratio""", (1.83247179E8, 43.0, """s^-1 T^-1""")),
    ("""neutron gyromag. ratio over 2 pi""", (29.16469430, 6.9E-6, """MHz T^-1""")),
    ("""neutron mag. mom.""", (-9.6623647E-27, 2.3E-33, """J T^-1""")),
    ("""neutron mag. mom. to Bohr magneton ratio""", (-0.001041875630, 2.5E-10, """""")),
    ("""neutron mag. mom. to nuclear magneton ratio""", (-1.913042720, 4.5E-7, """""")),
    ("""neutron mass""", (1.674927351E-27, 7.4E-35, """kg""")),
    ("""neutron mass energy equivalent""", (1.505349631E-10, 6.6E-18, """J""")),
    ("""neutron mass energy equivalent in MeV""", (939.5653790, 0.000021, """MeV""")),
    ("""neutron mass in u""", (1.008664916, 4.3E-10, """u""")),
    ("""neutron molar mass""", (0.001008664916, 4.3E-13, """kg mol^-1""")),
    ("""neutron-muon mass ratio""", (8.8924840, 2.2E-7, """""")),
    ("""neutron-proton mag. mom. ratio""", (-0.68497934, 1.6E-7, """""")),
    ("""neutron-proton mass difference""", (2.30557392E-30, 7.6E-37, """""")),
    ("""neutron-proton mass difference energy equivalent""", (2.0721465E-13, 6.8E-20, """""")),
    ("""neutron-proton mass difference energy equivalent in MeV""", (1.293332170, 4.2E-7, """""")),
    ("""neutron-proton mass difference in u""", (0.001388449190, 4.5E-10, """""")),
    ("""neutron-proton mass ratio""", (1.00137841917, 4.5E-10, """""")),
    ("""neutron-tau mass ratio""", (0.528790, 0.000048, """""")),
    ("""neutron to shielded proton mag. mom. ratio""", (-0.68499694, 1.6E-7, """""")),
    ("""Newtonian constant of gravitation""", (6.67384E-11, 8.0E-15, """m^3 kg^-1 s^-2""")),
    ("""Newtonian constant of gravitation over h-bar c""", (6.70837E-39, 8.0E-43, """(GeV/c^2)^-2""")),
    ("""nuclear magneton""", (5.05078353E-27, 1.1E-34, """J T^-1""")),
    ("""nuclear magneton in eV/T""", (3.1524512605E-8, 2.2E-17, """eV T^-1""")),
    ("""nuclear magneton in inverse meters per tesla""", (0.02542623527, 5.6E-10, """m^-1 T^-1""")),
    ("""nuclear magneton in K/T""", (0.00036582682, 3.3E-10, """K T^-1""")),
    ("""nuclear magneton in MHz/T""", (7.622593570, 1.7E-7, """MHz T^-1""")),
    ("""Planck constant""", (6.62606957E-34, 2.9E-41, """J s""")),
    ("""Planck constant in eV s""", (4.135667516E-15, 9.1E-23, """eV s""")),
    ("""Planck constant over 2 pi""", (1.054571726E-34, 4.7E-42, """J s""")),
    ("""Planck constant over 2 pi in eV s""", (6.58211928E-16, 1.5E-23, """eV s""")),
    ("""Planck constant over 2 pi times c in MeV fm""", (197.3269718, 4.4E-6, """MeV fm""")),
    ("""Planck length""", (1.616199E-35, 9.7E-40, """m""")),
    ("""Planck mass""", (2.17651E-8, 1.3E-12, """kg""")),
    ("""Planck mass energy equivalent in GeV""", (1.220932E19, 7.3E14, """GeV""")),
    ("""Planck temperature""", (1.416833E32, 8.5E27, """K""")),
    ("""Planck time""", (5.39106E-44, 3.2E-48, """s""")),
    ("""proton charge to mass quotient""", (9.57883358E7, 2.1, """C kg^-1""")),
    ("""proton Compton wavelength""", (1.32140985623E-15, 9.4E-25, """m""")),
    ("""proton Compton wavelength over 2 pi""", (2.1030891047E-16, 1.5E-25, """m""")),
    ("""proton-electron mass ratio""", (1836.15267245, 7.5E-7, """""")),
    ("""proton g factor""", (5.585694713, 4.6E-8, """""")),
    ("""proton gyromag. ratio""", (2.675222005E8, 6.3, """s^-1 T^-1""")),
    ("""proton gyromag. ratio over 2 pi""", (42.57748060, 1.0E-6, """MHz T^-1""")),
    ("""proton mag. mom.""", (1.410606743E-26, 3.3E-34, """J T^-1""")),
    ("""proton mag. mom. to Bohr magneton ratio""", (0.001521032210, 1.2E-11, """""")),
    ("""proton mag. mom. to nuclear magneton ratio""", (2.792847356, 2.3E-8, """""")),
    ("""proton mag. shielding correction""", (0.0000256940, 1.4E-8, """""")),
    ("""proton mass""", (1.672621777E-27, 7.4E-35, """kg""")),
    ("""proton mass energy equivalent""", (1.503277484E-10, 6.6E-18, """J""")),
    ("""proton mass energy equivalent in MeV""", (938.2720460, 0.000021, """MeV""")),
    ("""proton mass in u""", (1.0072764668120, 9.0E-11, """u""")),
    ("""proton molar mass""", (0.0010072764668120, 9.0E-14, """kg mol^-1""")),
    ("""proton-muon mass ratio""", (8.880243310, 2.2E-7, """""")),
    ("""proton-neutron mag. mom. ratio""", (-1.459898060, 3.4E-7, """""")),
    ("""proton-neutron mass ratio""", (0.998623478260, 4.5E-10, """""")),
    ("""proton rms charge radius""", (8.7750E-16, 5.1E-18, """m""")),
    ("""proton-tau mass ratio""", (0.528063, 0.000048, """""")),
    ("""quantum of circulation""", (0.0003636947552, 2.4E-13, """m^2 s^-1""")),
    ("""quantum of circulation times 2""", (0.0007273895104, 4.7E-13, """m^2 s^-1""")),
    ("""Rydberg constant""", (1.0973731568539E7, 0.000055, """m^-1""")),
    ("""Rydberg constant times c in Hz""", (3.289841960364E15, 17000.0, """Hz""")),
    ("""Rydberg constant times hc in eV""", (13.60569253, 3.0E-7, """eV""")),
    ("""Rydberg constant times hc in J""", (2.179872171E-18, 9.6E-26, """J""")),
    ("""Sackur-Tetrode constant (1 K, 100 kPa)""", (-1.1517078, 2.3E-6, """""")),
    ("""Sackur-Tetrode constant (1 K, 101.325 kPa)""", (-1.1648708, 2.3E-6, """""")),
    ("""second radiation constant""", (0.014387770, 1.3E-8, """m K""")),
    ("""shielded helion gyromag. ratio""", (2.037894659E8, 5.1, """s^-1 T^-1""")),
    ("""shielded helion gyromag. ratio over 2 pi""", (32.43410084, 8.1E-7, """MHz T^-1""")),
    ("""shielded helion mag. mom.""", (-1.074553044E-26, 2.7E-34, """J T^-1""")),
    ("""shielded helion mag. mom. to Bohr magneton ratio""", (-0.001158671471, 1.4E-11, """""")),
    ("""shielded helion mag. mom. to nuclear magneton ratio""", (-2.127497718, 2.5E-8, """""")),
    ("""shielded helion to proton mag. mom. ratio""", (-0.7617665580, 1.1E-8, """""")),
    ("""shielded helion to shielded proton mag. mom. ratio""", (-0.7617861313, 3.3E-9, """""")),
    ("""shielded proton gyromag. ratio""", (2.675153268E8, 6.6, """s^-1 T^-1""")),
    ("""shielded proton gyromag. ratio over 2 pi""", (42.57638660, 1.0E-6, """MHz T^-1""")),
    ("""shielded proton mag. mom.""", (1.410570499E-26, 3.5E-34, """J T^-1""")),
    ("""shielded proton mag. mom. to Bohr magneton ratio""", (0.001520993128, 1.7E-11, """""")),
    ("""shielded proton mag. mom. to nuclear magneton ratio""", (2.792775598, 3.0E-8, """""")),
    ("""speed of light in vacuum""", (2.99792458E8, 0d, """m s^-1""")),
    ("""standard acceleration of gravity""", (9.80665, 0d, """m s^-2""")),
    ("""standard atmosphere""", (101325.0, 0d, """Pa""")),
    ("""standard-state pressure""", (100000.0, 0d, """Pa""")),
    ("""Stefan-Boltzmann constant""", (5.670373E-8, 2.1E-13, """W m^-2 K^-4""")),
    ("""tau Compton wavelength""", (6.97787E-16, 6.3E-20, """m""")),
    ("""tau Compton wavelength over 2 pi""", (1.11056E-16, 1.0E-20, """m""")),
    ("""tau-electron mass ratio""", (3477.15, 0.31, """""")),
    ("""tau mass""", (3.16747E-27, 2.9E-31, """kg""")),
    ("""tau mass energy equivalent""", (2.84678E-10, 2.6E-14, """J""")),
    ("""tau mass energy equivalent in MeV""", (1776.82, 0.16, """MeV""")),
    ("""tau mass in u""", (1.90749, 0.00017, """u""")),
    ("""tau molar mass""", (0.00190749, 1.7E-7, """kg mol^-1""")),
    ("""tau-muon mass ratio""", (16.8167, 0.0015, """""")),
    ("""tau-neutron mass ratio""", (1.89111, 0.00017, """""")),
    ("""tau-proton mass ratio""", (1.89372, 0.00017, """""")),
    ("""Thomson cross section""", (6.652458734E-29, 1.3E-37, """m^2""")),
    ("""triton-electron mass ratio""", (5496.92152670, 5.0E-6, """""")),
    ("""triton g factor""", (5.957924896, 7.6E-8, """""")),
    ("""triton mag. mom.""", (1.504609447E-26, 3.8E-34, """J T^-1""")),
    ("""triton mag. mom. to Bohr magneton ratio""", (0.001622393657, 2.1E-11, """""")),
    ("""triton mag. mom. to nuclear magneton ratio""", (2.978962448, 3.8E-8, """""")),
    ("""triton mass""", (5.0073563E-27, 2.2E-34, """kg""")),
    ("""triton mass energy equivalent""", (4.50038741E-10, 2.0E-17, """J""")),
    ("""triton mass energy equivalent in MeV""", (2808.921005, 0.000062, """MeV""")),
    ("""triton mass in u""", (3.01550071340, 2.5E-9, """u""")),
    ("""triton molar mass""", (0.00301550071340, 2.5E-12, """kg mol^-1""")),
    ("""triton-proton mass ratio""", (2.99371703080, 2.5E-9, """""")),
    ("""unified atomic mass unit""", (1.660538921E-27, 7.3E-35, """kg""")),
    ("""von Klitzing constant""", (25812.8074434, 8.4E-6, """ohm""")),
    ("""weak mixing angle""", (0.2223, 0.0021, """""")),
    ("""Wien frequency displacement law constant""", (5.8789254E10, 53000.0, """Hz K^-1""")),
    ("""Wien wavelength displacement law constant""", (0.0028977721, 2.6E-9, """m K"""))
  )

  // </editor-fold>

}
