/*
 * Copyright (C) 2006 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.internal.telephony;

/**
 * Call fail causes from TS 24.008 .
 * These are mostly the cause codes we need to distinguish for the UI.
 * See 22.001 Annex F.4 for mapping of cause codes to local tones.
 *
 * CDMA call failure reasons are derived from the possible call failure scenarios described
 * in "CDMA IS2000 - Release A (C.S0005-A v6.0)" standard.
 *
 * The detailed fail causes are defined in ITU Recommendation Q.850.
 *
 * {@hide}
 *
 */
public interface CallFailCause {
    // The disconnect cause is not valid (Not received a disconnect cause)
    int NOT_VALID = -1;

    // Unassigned/Unobtainable number
    int UNOBTAINABLE_NUMBER = 1;

    int NO_ROUTE_TO_DEST = 3;
    int CHANNEL_UNACCEPTABLE = 6;
    int OPERATOR_DETERMINED_BARRING = 8;
    int NORMAL_CLEARING     = 16;
    int USER_BUSY           = 17;
    int NO_USER_RESPONDING = 18;

    /**
     * This cause is used when the called party has been alerted but does not respond with a connect
     * indication within a prescribed period of time. Note - This cause is not necessarily generated
     * by Q.931 procedures but may be generated by internal network timers.
     */
    int USER_ALERTING_NO_ANSWER = 19;

    /**
     * The equipment sending this cause does not wish to accept this call, although it could have
     * accepted the call because the equipment sending this cause is neither busy nor incompatible.
     * The network may also generate this cause, indicating that the call was cleared due to a
     * supplementary service constraint. The diagnostic field may contain additional information
     * about the supplementary service and reason for rejection.
     */
    int CALL_REJECTED = 21;

    int NUMBER_CHANGED      = 22;
    int PRE_EMPTION = 25;

    // The user has not been awarded the incoming call.
    int NON_SELECTED_USER_CLEARING = 26;

    int DESTINATION_OUT_OF_ORDER = 27;

    // Incomplete number
    int INVALID_NUMBER_FORMAT = 28;

    // Supplementary service requested by the user cannot be provide by the network.
    int FACILITY_REJECTED = 29;

    int STATUS_ENQUIRY      = 30;
    int NORMAL_UNSPECIFIED  = 31;
    int NO_CIRCUIT_AVAIL    = 34;

    // Resource unavailable
    int NETWORK_OUT_OF_ORDER = 38;
    int TEMPORARY_FAILURE   = 41;
    int SWITCHING_CONGESTION    = 42;
    int ACCESS_INFORMATION_DISCARDED = 43;
    int CHANNEL_NOT_AVAIL   = 44;
    int RESOURCES_UNAVAILABLE_UNSPECIFIED = 47;
    int QOS_NOT_AVAIL       = 49;

    // Service or option unavailable
    /**
     * The user has requested a supplementary service, which is available, but the user is not
     * authorized to use.
     */
    int REQUESTED_FACILITY_NOT_SUBSCRIBED = 50;
    /**
     * Although the called party is a member of the CUG (Closed User Group) for the incoming CUG
     * call, incoming calls are not allowed to this member of the CUG.
     */
    int INCOMING_CALL_BARRED_WITHIN_CUG = 55;
    int BEARER_CAPABILITY_NOT_AUTHORISED = 57;
    int BEARER_NOT_AVAIL    = 58;
    /**
     * This cause is used to report a service or option not available event only when no other cause
     * between 49-62 (where a service or option is unavailable) applies.
     */
    int SERVICE_OR_OPTION_NOT_AVAILABLE = 63;
    int BEARER_SERVICE_NOT_IMPLEMENTED = 65;

    // Service or option not implemented
    int ACM_LIMIT_EXCEEDED = 68;
    int REQUESTED_FACILITY_NOT_IMPLEMENTED = 69;
    /**
     * The calling party has requested an unrestricted bearer service but that the equipment sending
     * this cause only supports the restricted version of the requested bearer capability.
     */
    int ONLY_RESTRICTED_DIGITAL_INFO_BC_AVAILABLE = 70;
    int SERVICE_OR_OPTION_NOT_IMPLEMENTED = 79;
    int INVALID_TRANSACTION_ID_VALUE = 81;

    // Invalid message
    int USER_NOT_MEMBER_OF_CUG = 87;
    int INCOMPATIBLE_DESTINATION = 88;
    int INVALID_TRANSIT_NETWORK_SELECTION = 91;
    int SEMANTICALLY_INCORRECT_MESSAGE = 95;
    int INVALID_MANDATORY_INFORMATION = 96;

    // Protocol error
    int MESSAGE_TYPE_NON_EXISTENT = 97;
    int MESSAGE_TYPE_NOT_COMPATIBLE_WITH_PROT_STATE = 98;
    int IE_NON_EXISTENT_OR_NOT_IMPLEMENTED = 99;
    /**
     * The equipment sending this cause has received an information element which it has
     * implemented; however, one or more fields in the information element are coded in such a way
     * which has not been implemented by the equipment sending this cause.
     */
    int CONDITIONAL_IE_ERROR = 100;
    int MESSAGE_NOT_COMPATIBLE_WITH_PROTOCOL_STATE = 101;
    int RECOVERY_ON_TIMER_EXPIRY = 102;
    int PROTOCOL_ERROR_UNSPECIFIED = 111;
    int INTERWORKING_UNSPECIFIED = 127;

    // Others
    int CALL_BARRED        = 240;
    int FDN_BLOCKED        = 241;
    int IMEI_NOT_ACCEPTED  = 243;

    // Stk Call Control
    int DIAL_MODIFIED_TO_USSD = 244;
    int DIAL_MODIFIED_TO_SS   = 245;
    int DIAL_MODIFIED_TO_DIAL = 246;

    //Access class blocked - TS 31.121 5.2.1
    int ACCESS_CLASS_BLOCKED = 260;

    //Emergency Redial
    int EMERGENCY_TEMP_FAILURE = 325;
    int EMERGENCY_PERM_FAILURE = 326;

    int CDMA_LOCKED_UNTIL_POWER_CYCLE  = 1000;
    int CDMA_DROP                      = 1001;
    int CDMA_INTERCEPT                 = 1002;
    int CDMA_REORDER                   = 1003;
    int CDMA_SO_REJECT                 = 1004;
    int CDMA_RETRY_ORDER               = 1005;
    int CDMA_ACCESS_FAILURE            = 1006;
    int CDMA_PREEMPTED                 = 1007;

    // For non-emergency number dialed while in emergency callback mode.
    int CDMA_NOT_EMERGENCY             = 1008;

    // Access Blocked by CDMA Network.
    int CDMA_ACCESS_BLOCKED            = 1009;

    int ERROR_UNSPECIFIED = 0xffff;

}
