//===----------------------------------------------------------------------===//
// Copyright © 2024 Apple Inc. and the Pkl project authors. All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//===----------------------------------------------------------------------===//

/// Common settings for Pkl's own evaluator.
@ModuleInfo { minPklVersion = "0.26.0" }
@Since { version = "0.26.0" }
module pkl.EvaluatorSettings

/// The external properties available to Pkl, read using the `prop:` scheme.
externalProperties: Mapping<String, String>?

/// The environment variables available to Pkl, read using the `env:` scheme.
///
/// Example:
/// ```
/// env {
///   ["IS_PROD"] = "true"
/// }
/// ```
env: Mapping<String, String>?

/// The set of module URI patterns that can be imported.
///
/// Each element is a regular expression pattern that is tested against a module import.
///
/// Modules are imported either through an amends or extends clause, an import clause, or an
/// import expression.
///
/// Example:
/// ```
/// allowedModules {
///   "https:"
///   "file:"
///   "package:"
///   "projectpackage:"
/// }
/// ```
allowedModules: Listing<String(isRegex)>?

/// The set of resource URI patterns that can be imported.
///
/// Each element is a regular expression pattern that is tested against a resource read.
///
/// Example:
/// ```
/// allowedResources {
///   "https:"
///   "file:"
///   "package:"
///   "projectpackage:"
///   "env:"
///   "prop:"
/// }
/// ```
allowedResources: Listing<String(isRegex)>?

/// Disables the file system cache for `package:` modules.
///
/// When caching is disabled, packages are loaded over the network and stored in memory.
noCache: Boolean?

/// A collection of jars, zips, or directories to be placed into the module path.
///
/// Module path modules and resources may be read and imported using the `modulepath:` scheme.
modulePath: Listing<String>?

/// The duration after which evaluation of a source module will be timed out.
///
/// Note that a timeout is treated the same as a program error in that any subsequent source modules will not be evaluated.
timeout: Duration?

/// The directory where `package:` modules are cached.
moduleCacheDir: String?

/// Restricts access to file-based modules and resources to those located under this directory.
rootDir: String?

/// Configuration of outgoing HTTP requests.
http: Http?

/// Settings that control how Pkl talks to HTTP(S) servers.
class Http {
  /// Configuration of the HTTP proxy to use.
  proxy: Proxy?
}

/// Settings that control how Pkl talks to HTTP proxies.
class Proxy {
  /// The proxy to use for HTTP(S) connections.
  ///
  /// Only HTTP proxies are supported.
  /// The proxy address must start with `http://`, and can only contain a host and a port.
  /// If a port is omitted, it defaults to port `80`.
  ///
  /// Proxy authentication is not supported.
  ///
  /// Example:
  /// ```
  /// address = "http://my.proxy.example.com:5080"
  /// ```
  address: Uri(startsWith("http://"))?

  /// Hosts to which all connections should bypass a proxy.
  ///
  /// Values can be either hostnames, or IP addresses.
  /// IP addresses can optionally be provided using
  /// [CIDR notation](https://en.wikipedia.org/wiki/Classless_Inter-Domain_Routing#CIDR_notation).
  ///
  /// The value `"*"` is a wildcard that disables proxying for all hosts.
  ///
  /// A hostname matches all subdomains.
  /// For example, `example.com` matches `foo.example.com`, but not `fooexample.com`.
  /// A hostname that is prefixed with a dot matches the hostname itself,
  /// so `.example.com` matches `example.com`.
  ///
  /// Hostnames do not match their resolved IP addresses.
  /// For example, the hostname `localhost` will not match `127.0.0.1`.
  ///
  /// Optionally, a port can be specified.
  /// If a port is omitted, all ports are matched.
  ///
  /// Example:
  ///
  /// ```
  /// noProxy {
  ///   "127.0.0.1"
  ///   "169.254.0.0/16"
  ///   "example.com"
  ///   "localhost:5050"
  /// }
  /// ```
  noProxy: Listing<String>(isDistinct)
}
