/*
 * Copyright 2012 JBoss Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.overlord.sramp.shell.commands.core;

import java.io.File;
import java.io.InputStream;
import java.util.List;

import javax.xml.namespace.QName;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.oasis_open.docs.s_ramp.ns.s_ramp_v1.BaseArtifactType;
import org.overlord.sramp.atom.archive.SrampArchive;
import org.overlord.sramp.atom.archive.expand.DefaultMetaDataFactory;
import org.overlord.sramp.atom.archive.expand.ZipToSrampArchive;
import org.overlord.sramp.atom.archive.expand.registry.ZipToSrampArchiveRegistry;
import org.overlord.sramp.client.SrampAtomApiClient;
import org.overlord.sramp.common.ArtifactType;
import org.overlord.sramp.common.ArtifactTypeEnum;
import org.overlord.sramp.common.visitors.ArtifactVisitorHelper;
import org.overlord.sramp.shell.BuiltInShellCommand;
import org.overlord.sramp.shell.i18n.Messages;
import org.overlord.sramp.shell.util.FileNameCompleter;
import org.overlord.sramp.shell.util.PrintArtifactMetaDataVisitor;

/**
 * Uploads an artifact to the s-ramp repository.
 *
 * @author eric.wittmann@redhat.com
 */
public class UploadArtifactCommand extends BuiltInShellCommand {

	/**
	 * Constructor.
	 */
	public UploadArtifactCommand() {
	}

	/**
	 * @see org.overlord.sramp.shell.api.shell.ShellCommand#execute()
	 */
	@Override
	public boolean execute() throws Exception {
		String filePathArg = this.requiredArgument(0, Messages.i18n.format("Upload.InvalidArgMsg.LocalFile")); //$NON-NLS-1$
		String artifactTypeArg = this.optionalArgument(1);

		QName clientVarName = new QName("s-ramp", "client"); //$NON-NLS-1$ //$NON-NLS-2$
		SrampAtomApiClient client = (SrampAtomApiClient) getContext().getVariable(clientVarName);
		if (client == null) {
			print(Messages.i18n.format("MissingSRAMPConnection")); //$NON-NLS-1$
			return false;
		}
		InputStream content = null;
        ZipToSrampArchive expander = null;
        SrampArchive archive = null;
		try {
			File file = new File(filePathArg);
			ArtifactType artifactType = null;
			if (artifactTypeArg != null) {
				artifactType = ArtifactType.valueOf(artifactTypeArg);
				if (artifactType.isExtendedType()) {
				    artifactType = ArtifactType.ExtendedDocument(artifactType.getExtendedType());
				}
			} else {
				artifactType = determineArtifactType(file);
			}
			content = FileUtils.openInputStream(file);
			BaseArtifactType artifact = client.uploadArtifact(artifactType, content, file.getName());
            IOUtils.closeQuietly(content);

            // Now also add "expanded" content to the s-ramp repository
            expander = ZipToSrampArchiveRegistry.createExpander(artifactType, file);
            if (expander != null) {
                expander.setContextParam(DefaultMetaDataFactory.PARENT_UUID, artifact.getUuid());
                archive = expander.createSrampArchive();
                client.uploadBatch(archive);
            }

			// Put the artifact in the session as the active artifact
			QName artifactVarName = new QName("s-ramp", "artifact"); //$NON-NLS-1$ //$NON-NLS-2$
			getContext().setVariable(artifactVarName, artifact);
			print(Messages.i18n.format("Upload.Success")); //$NON-NLS-1$
			PrintArtifactMetaDataVisitor visitor = new PrintArtifactMetaDataVisitor();
			ArtifactVisitorHelper.visitArtifact(visitor, artifact);
		} catch (Exception e) {
			print(Messages.i18n.format("Upload.Failure")); //$NON-NLS-1$
			print("\t" + e.getMessage()); //$NON-NLS-1$
			IOUtils.closeQuietly(content);
            return false;
		}
        return true;
	}

	/**
	 * Try to figure out what kind of artifact we're dealing with.
	 * @param file
	 */
	private ArtifactType determineArtifactType(File file) {
		ArtifactType type = null;
		String extension = FilenameUtils.getExtension(file.getName());
		if ("wsdl".equals(extension)) { //$NON-NLS-1$
			type = ArtifactType.WsdlDocument();
		} else if ("xsd".equals(extension)) { //$NON-NLS-1$
			type = ArtifactType.XsdDocument();
		} else if ("wspolicy".equals(extension)) { //$NON-NLS-1$
			type = ArtifactType.PolicyDocument();
		} else if ("xml".equals(extension)) { //$NON-NLS-1$
			type = ArtifactType.XmlDocument();
		} else {
			type = ArtifactType.Document();
		}
		return type;
	}

	/**
	 * @see org.overlord.sramp.shell.api.shell.AbstractShellCommand#tabCompletion(java.lang.String, java.util.List)
	 */
	@Override
	public int tabCompletion(String lastArgument, List<CharSequence> candidates) {
		if (getArguments().isEmpty()) {
			if (lastArgument == null)
				lastArgument = ""; //$NON-NLS-1$
			FileNameCompleter delegate = new FileNameCompleter();
			return delegate.complete(lastArgument, lastArgument.length(), candidates);
		} else if (getArguments().size() == 1) {
			for (ArtifactTypeEnum t : ArtifactTypeEnum.values()) {
				String candidate = t.getType();
				if (lastArgument == null || candidate.startsWith(lastArgument)) {
					candidates.add(candidate);
				}
			}
			return 0;
		}
		return -1;
	}

}
