/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 */

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

/*
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

package org.opensearch.client.opensearch.indices.rollover;

import jakarta.json.stream.JsonGenerator;
import java.util.function.Function;
import javax.annotation.Nullable;
import org.opensearch.client.json.JsonpDeserializable;
import org.opensearch.client.json.JsonpDeserializer;
import org.opensearch.client.json.JsonpMapper;
import org.opensearch.client.json.JsonpSerializable;
import org.opensearch.client.json.ObjectBuilderDeserializer;
import org.opensearch.client.json.ObjectDeserializer;
import org.opensearch.client.opensearch._types.Time;
import org.opensearch.client.util.ObjectBuilder;
import org.opensearch.client.util.ObjectBuilderBase;

// typedef: indices.rollover.RolloverConditions

@JsonpDeserializable
public class RolloverConditions implements JsonpSerializable {
    @Nullable
    private final Time maxAge;

    @Nullable
    private final Long maxDocs;

    @Nullable
    private final String maxSize;

    @Nullable
    private final String maxPrimaryShardSize;

    // ---------------------------------------------------------------------------------------------

    private RolloverConditions(Builder builder) {

        this.maxAge = builder.maxAge;
        this.maxDocs = builder.maxDocs;
        this.maxSize = builder.maxSize;
        this.maxPrimaryShardSize = builder.maxPrimaryShardSize;

    }

    public static RolloverConditions of(Function<Builder, ObjectBuilder<RolloverConditions>> fn) {
        return fn.apply(new Builder()).build();
    }

    /**
     * API name: {@code max_age}
     */
    @Nullable
    public final Time maxAge() {
        return this.maxAge;
    }

    /**
     * API name: {@code max_docs}
     */
    @Nullable
    public final Long maxDocs() {
        return this.maxDocs;
    }

    /**
     * API name: {@code max_size}
     */
    @Nullable
    public final String maxSize() {
        return this.maxSize;
    }

    /**
     * API name: {@code max_primary_shard_size}
     */
    @Nullable
    public final String maxPrimaryShardSize() {
        return this.maxPrimaryShardSize;
    }

    /**
     * Serialize this object to JSON.
     */
    public void serialize(JsonGenerator generator, JsonpMapper mapper) {
        generator.writeStartObject();
        serializeInternal(generator, mapper);
        generator.writeEnd();
    }

    protected void serializeInternal(JsonGenerator generator, JsonpMapper mapper) {

        if (this.maxAge != null) {
            generator.writeKey("max_age");
            this.maxAge.serialize(generator, mapper);

        }
        if (this.maxDocs != null) {
            generator.writeKey("max_docs");
            generator.write(this.maxDocs);

        }
        if (this.maxSize != null) {
            generator.writeKey("max_size");
            generator.write(this.maxSize);

        }
        if (this.maxPrimaryShardSize != null) {
            generator.writeKey("max_primary_shard_size");
            generator.write(this.maxPrimaryShardSize);

        }

    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Builder for {@link RolloverConditions}.
     */

    public static class Builder extends ObjectBuilderBase implements ObjectBuilder<RolloverConditions> {
        @Nullable
        private Time maxAge;

        @Nullable
        private Long maxDocs;

        @Nullable
        private String maxSize;

        @Nullable
        private String maxPrimaryShardSize;

        /**
         * API name: {@code max_age}
         */
        public final Builder maxAge(@Nullable Time value) {
            this.maxAge = value;
            return this;
        }

        /**
         * API name: {@code max_age}
         */
        public final Builder maxAge(Function<Time.Builder, ObjectBuilder<Time>> fn) {
            return this.maxAge(fn.apply(new Time.Builder()).build());
        }

        /**
         * API name: {@code max_docs}
         */
        public final Builder maxDocs(@Nullable Long value) {
            this.maxDocs = value;
            return this;
        }

        /**
         * API name: {@code max_size}
         */
        public final Builder maxSize(@Nullable String value) {
            this.maxSize = value;
            return this;
        }

        /**
         * API name: {@code max_primary_shard_size}
         */
        public final Builder maxPrimaryShardSize(@Nullable String value) {
            this.maxPrimaryShardSize = value;
            return this;
        }

        /**
         * Builds a {@link RolloverConditions}.
         *
         * @throws NullPointerException
         *             if some of the required fields are null.
         */
        public RolloverConditions build() {
            _checkSingleUse();

            return new RolloverConditions(this);
        }
    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Json deserializer for {@link RolloverConditions}
     */
    public static final JsonpDeserializer<RolloverConditions> _DESERIALIZER = ObjectBuilderDeserializer.lazy(
        Builder::new,
        RolloverConditions::setupRolloverConditionsDeserializer
    );

    protected static void setupRolloverConditionsDeserializer(ObjectDeserializer<RolloverConditions.Builder> op) {

        op.add(Builder::maxAge, Time._DESERIALIZER, "max_age");
        op.add(Builder::maxDocs, JsonpDeserializer.longDeserializer(), "max_docs");
        op.add(Builder::maxSize, JsonpDeserializer.stringDeserializer(), "max_size");
        op.add(Builder::maxPrimaryShardSize, JsonpDeserializer.stringDeserializer(), "max_primary_shard_size");

    }

}
