/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 */

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

/*
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

package org.opensearch.client.opensearch.ingest;

import jakarta.json.stream.JsonGenerator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import javax.annotation.Nullable;
import org.opensearch.client.json.JsonData;
import org.opensearch.client.json.JsonpDeserializable;
import org.opensearch.client.json.JsonpDeserializer;
import org.opensearch.client.json.JsonpMapper;
import org.opensearch.client.json.JsonpSerializable;
import org.opensearch.client.json.ObjectBuilderDeserializer;
import org.opensearch.client.json.ObjectDeserializer;
import org.opensearch.client.opensearch._types.ErrorResponse;
import org.opensearch.client.opensearch._types.RequestBase;
import org.opensearch.client.opensearch._types.Time;
import org.opensearch.client.transport.Endpoint;
import org.opensearch.client.transport.endpoints.SimpleEndpoint;
import org.opensearch.client.util.ApiTypeHelper;
import org.opensearch.client.util.ObjectBuilder;
import org.opensearch.client.util.ObjectBuilderBase;

// typedef: ingest.put_pipeline.Request

/**
 * Creates or updates a pipeline.
 *
 */
@JsonpDeserializable
public class PutPipelineRequest extends RequestBase implements JsonpSerializable {
    private final Map<String, JsonData> meta;

    @Nullable
    private final String description;

    private final String id;

    @Deprecated
    @Nullable
    private final Time masterTimeout;

    @Nullable
    private final Time clusterManagerTimeout;

    private final List<Processor> onFailure;

    private final List<Processor> processors;

    @Nullable
    private final Time timeout;

    @Nullable
    private final Long version;

    // ---------------------------------------------------------------------------------------------

    private PutPipelineRequest(Builder builder) {

        this.meta = ApiTypeHelper.unmodifiable(builder.meta);
        this.description = builder.description;
        this.id = ApiTypeHelper.requireNonNull(builder.id, this, "id");
        this.masterTimeout = builder.masterTimeout;
        this.clusterManagerTimeout = builder.clusterManagerTimeout;
        this.onFailure = ApiTypeHelper.unmodifiable(builder.onFailure);
        this.processors = ApiTypeHelper.unmodifiable(builder.processors);
        this.timeout = builder.timeout;
        this.version = builder.version;

    }

    public static PutPipelineRequest of(Function<Builder, ObjectBuilder<PutPipelineRequest>> fn) {
        return fn.apply(new Builder()).build();
    }

    /**
     * Optional metadata about the ingest pipeline. May have any contents. This map
     * is not automatically generated by Elasticsearch.
     * <p>
     * API name: {@code _meta}
     */
    public final Map<String, JsonData> meta() {
        return this.meta;
    }

    /**
     * Description of the ingest pipeline.
     * <p>
     * API name: {@code description}
     */
    @Nullable
    public final String description() {
        return this.description;
    }

    /**
     * Required - Pipeline ID
     * <p>
     * API name: {@code id}
     */
    public final String id() {
        return this.id;
    }

    /**
     * Period to wait for a connection to the master node. If no response is
     * received before the timeout expires, the request fails and returns an error.
     * <p>
     * API name: {@code master_timeout}
     */
    @Deprecated
    @Nullable
    public final Time masterTimeout() {
        return this.masterTimeout;
    }

    /**
     * Period to wait for a connection to the cluster-manager node. If no response is
     * received before the timeout expires, the request fails and returns an error.
     * <p>
     * API name: {@code cluster_manager_timeout}
     */
    @Nullable
    public final Time clusterManagerTimeout() {
        return this.clusterManagerTimeout;
    }

    /**
     * API name: {@code on_failure}
     */
    public final List<Processor> onFailure() {
        return this.onFailure;
    }

    /**
     * API name: {@code processors}
     */
    public final List<Processor> processors() {
        return this.processors;
    }

    /**
     * Explicit operation timeout
     * <p>
     * API name: {@code timeout}
     */
    @Nullable
    public final Time timeout() {
        return this.timeout;
    }

    /**
     * API name: {@code version}
     */
    @Nullable
    public final Long version() {
        return this.version;
    }

    /**
     * Serialize this object to JSON.
     */
    public void serialize(JsonGenerator generator, JsonpMapper mapper) {
        generator.writeStartObject();
        serializeInternal(generator, mapper);
        generator.writeEnd();
    }

    protected void serializeInternal(JsonGenerator generator, JsonpMapper mapper) {

        if (ApiTypeHelper.isDefined(this.meta)) {
            generator.writeKey("_meta");
            generator.writeStartObject();
            for (Map.Entry<String, JsonData> item0 : this.meta.entrySet()) {
                generator.writeKey(item0.getKey());
                item0.getValue().serialize(generator, mapper);

            }
            generator.writeEnd();

        }
        if (this.description != null) {
            generator.writeKey("description");
            generator.write(this.description);

        }
        if (ApiTypeHelper.isDefined(this.onFailure)) {
            generator.writeKey("on_failure");
            generator.writeStartArray();
            for (Processor item0 : this.onFailure) {
                item0.serialize(generator, mapper);

            }
            generator.writeEnd();

        }
        if (ApiTypeHelper.isDefined(this.processors)) {
            generator.writeKey("processors");
            generator.writeStartArray();
            for (Processor item0 : this.processors) {
                item0.serialize(generator, mapper);

            }
            generator.writeEnd();

        }
        if (this.version != null) {
            generator.writeKey("version");
            generator.write(this.version);

        }

    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Builder for {@link PutPipelineRequest}.
     */

    public static class Builder extends ObjectBuilderBase implements ObjectBuilder<PutPipelineRequest> {
        @Nullable
        private Map<String, JsonData> meta;

        @Nullable
        private String description;

        private String id;

        @Deprecated
        @Nullable
        private Time masterTimeout;

        @Nullable
        private Time clusterManagerTimeout;

        @Nullable
        private List<Processor> onFailure;

        @Nullable
        private List<Processor> processors;

        @Nullable
        private Time timeout;

        @Nullable
        private Long version;

        /**
         * Optional metadata about the ingest pipeline. May have any contents. This map
         * is not automatically generated by Elasticsearch.
         * <p>
         * API name: {@code _meta}
         * <p>
         * Adds all entries of <code>map</code> to <code>meta</code>.
         */
        public final Builder meta(Map<String, JsonData> map) {
            this.meta = _mapPutAll(this.meta, map);
            return this;
        }

        /**
         * Optional metadata about the ingest pipeline. May have any contents. This map
         * is not automatically generated by Elasticsearch.
         * <p>
         * API name: {@code _meta}
         * <p>
         * Adds an entry to <code>meta</code>.
         */
        public final Builder meta(String key, JsonData value) {
            this.meta = _mapPut(this.meta, key, value);
            return this;
        }

        /**
         * Description of the ingest pipeline.
         * <p>
         * API name: {@code description}
         */
        public final Builder description(@Nullable String value) {
            this.description = value;
            return this;
        }

        /**
         * Required - Pipeline ID
         * <p>
         * API name: {@code id}
         */
        public final Builder id(String value) {
            this.id = value;
            return this;
        }

        /**
         * Period to wait for a connection to the master node. If no response is
         * received before the timeout expires, the request fails and returns an error.
         * <p>
         * API name: {@code master_timeout}
         */
        @Deprecated
        public final Builder masterTimeout(@Nullable Time value) {
            this.masterTimeout = value;
            return this;
        }

        /**
         * Period to wait for a connection to the master node. If no response is
         * received before the timeout expires, the request fails and returns an error.
         * <p>
         * API name: {@code master_timeout}
         */
        @Deprecated
        public final Builder masterTimeout(Function<Time.Builder, ObjectBuilder<Time>> fn) {
            return this.masterTimeout(fn.apply(new Time.Builder()).build());
        }

        /**
         * Period to wait for a connection to the cluster-manager node. If no response is
         * received before the timeout expires, the request fails and returns an error.
         * <p>
         * API name: {@code cluster_manager_timeout}
         */
        public final Builder clusterManagerTimeout(@Nullable Time value) {
            this.clusterManagerTimeout = value;
            return this;
        }

        /**
         * Period to wait for a connection to the cluster-manager node. If no response is
         * received before the timeout expires, the request fails and returns an error.
         * <p>
         * API name: {@code cluster_manager_timeout}
         */
        public final Builder clusterManagerTimeout(Function<Time.Builder, ObjectBuilder<Time>> fn) {
            return this.clusterManagerTimeout(fn.apply(new Time.Builder()).build());
        }

        /**
         * API name: {@code on_failure}
         * <p>
         * Adds all elements of <code>list</code> to <code>onFailure</code>.
         */
        public final Builder onFailure(List<Processor> list) {
            this.onFailure = _listAddAll(this.onFailure, list);
            return this;
        }

        /**
         * API name: {@code on_failure}
         * <p>
         * Adds one or more values to <code>onFailure</code>.
         */
        public final Builder onFailure(Processor value, Processor... values) {
            this.onFailure = _listAdd(this.onFailure, value, values);
            return this;
        }

        /**
         * API name: {@code on_failure}
         * <p>
         * Adds a value to <code>onFailure</code> using a builder lambda.
         */
        public final Builder onFailure(Function<Processor.Builder, ObjectBuilder<Processor>> fn) {
            return onFailure(fn.apply(new Processor.Builder()).build());
        }

        /**
         * API name: {@code processors}
         * <p>
         * Adds all elements of <code>list</code> to <code>processors</code>.
         */
        public final Builder processors(List<Processor> list) {
            this.processors = _listAddAll(this.processors, list);
            return this;
        }

        /**
         * API name: {@code processors}
         * <p>
         * Adds one or more values to <code>processors</code>.
         */
        public final Builder processors(Processor value, Processor... values) {
            this.processors = _listAdd(this.processors, value, values);
            return this;
        }

        /**
         * API name: {@code processors}
         * <p>
         * Adds a value to <code>processors</code> using a builder lambda.
         */
        public final Builder processors(Function<Processor.Builder, ObjectBuilder<Processor>> fn) {
            return processors(fn.apply(new Processor.Builder()).build());
        }

        /**
         * Explicit operation timeout
         * <p>
         * API name: {@code timeout}
         */
        public final Builder timeout(@Nullable Time value) {
            this.timeout = value;
            return this;
        }

        /**
         * Explicit operation timeout
         * <p>
         * API name: {@code timeout}
         */
        public final Builder timeout(Function<Time.Builder, ObjectBuilder<Time>> fn) {
            return this.timeout(fn.apply(new Time.Builder()).build());
        }

        /**
         * API name: {@code version}
         */
        public final Builder version(@Nullable Long value) {
            this.version = value;
            return this;
        }

        /**
         * Builds a {@link PutPipelineRequest}.
         *
         * @throws NullPointerException
         *             if some of the required fields are null.
         */
        public PutPipelineRequest build() {
            _checkSingleUse();

            return new PutPipelineRequest(this);
        }
    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Json deserializer for {@link PutPipelineRequest}
     */
    public static final JsonpDeserializer<PutPipelineRequest> _DESERIALIZER = ObjectBuilderDeserializer.lazy(
        Builder::new,
        PutPipelineRequest::setupPutPipelineRequestDeserializer
    );

    protected static void setupPutPipelineRequestDeserializer(ObjectDeserializer<PutPipelineRequest.Builder> op) {

        op.add(Builder::meta, JsonpDeserializer.stringMapDeserializer(JsonData._DESERIALIZER), "_meta");
        op.add(Builder::description, JsonpDeserializer.stringDeserializer(), "description");
        op.add(Builder::onFailure, JsonpDeserializer.arrayDeserializer(Processor._DESERIALIZER), "on_failure");
        op.add(Builder::processors, JsonpDeserializer.arrayDeserializer(Processor._DESERIALIZER), "processors");
        op.add(Builder::version, JsonpDeserializer.longDeserializer(), "version");

    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Endpoint "{@code ingest.put_pipeline}".
     */
    public static final Endpoint<PutPipelineRequest, PutPipelineResponse, ErrorResponse> _ENDPOINT = new SimpleEndpoint<>(

        // Request method
        request -> {
            return "PUT";

        },

        // Request path
        request -> {
            final int _id = 1 << 0;

            int propsSet = 0;

            propsSet |= _id;

            if (propsSet == (_id)) {
                StringBuilder buf = new StringBuilder();
                buf.append("/_ingest");
                buf.append("/pipeline");
                buf.append("/");
                SimpleEndpoint.pathEncode(request.id, buf);
                return buf.toString();
            }
            throw SimpleEndpoint.noPathTemplateFound("path");

        },

        // Request parameters
        request -> {
            Map<String, String> params = new HashMap<>();
            if (request.masterTimeout != null) {
                params.put("master_timeout", request.masterTimeout._toJsonString());
            }
            if (request.clusterManagerTimeout != null) {
                params.put("cluster_manager_timeout", request.clusterManagerTimeout._toJsonString());
            }
            if (request.timeout != null) {
                params.put("timeout", request.timeout._toJsonString());
            }
            return params;

        },
        SimpleEndpoint.emptyMap(),
        true,
        PutPipelineResponse._DESERIALIZER
    );
}
