/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 */

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

/*
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

package org.opensearch.client.opensearch.cluster;

import jakarta.json.stream.JsonGenerator;
import java.util.HashMap;
import java.util.Map;
import java.util.function.Function;
import javax.annotation.Nullable;
import org.opensearch.client.json.JsonData;
import org.opensearch.client.json.JsonpDeserializable;
import org.opensearch.client.json.JsonpDeserializer;
import org.opensearch.client.json.JsonpMapper;
import org.opensearch.client.json.JsonpSerializable;
import org.opensearch.client.json.ObjectBuilderDeserializer;
import org.opensearch.client.json.ObjectDeserializer;
import org.opensearch.client.opensearch._types.ErrorResponse;
import org.opensearch.client.opensearch._types.RequestBase;
import org.opensearch.client.opensearch._types.Time;
import org.opensearch.client.opensearch._types.mapping.TypeMapping;
import org.opensearch.client.opensearch.indices.AliasDefinition;
import org.opensearch.client.opensearch.indices.IndexSettings;
import org.opensearch.client.opensearch.indices.IndexState;
import org.opensearch.client.transport.Endpoint;
import org.opensearch.client.transport.endpoints.SimpleEndpoint;
import org.opensearch.client.util.ApiTypeHelper;
import org.opensearch.client.util.ObjectBuilder;
import org.opensearch.client.util.ObjectBuilderBase;

// typedef: cluster.put_component_template.Request

/**
 * Creates or updates a component template
 *
 */
@JsonpDeserializable
public class PutComponentTemplateRequest extends RequestBase implements JsonpSerializable {
    private final Map<String, JsonData> meta;

    private final Map<String, AliasDefinition> aliases;

    @Nullable
    private final Boolean create;

    @Nullable
    private final TypeMapping mappings;

    @Deprecated
    @Nullable
    private final Time masterTimeout;

    @Nullable
    private final Time clusterManagerTimeout;

    private final String name;

    @Nullable
    private final IndexSettings settings;

    private final IndexState template;

    @Nullable
    private final Long version;

    // ---------------------------------------------------------------------------------------------

    private PutComponentTemplateRequest(Builder builder) {

        this.meta = ApiTypeHelper.unmodifiable(builder.meta);
        this.aliases = ApiTypeHelper.unmodifiable(builder.aliases);
        this.create = builder.create;
        this.mappings = builder.mappings;
        this.masterTimeout = builder.masterTimeout;
        this.clusterManagerTimeout = builder.clusterManagerTimeout;
        this.name = ApiTypeHelper.requireNonNull(builder.name, this, "name");
        this.settings = builder.settings;
        this.template = ApiTypeHelper.requireNonNull(builder.template, this, "template");
        this.version = builder.version;

    }

    public static PutComponentTemplateRequest of(Function<Builder, ObjectBuilder<PutComponentTemplateRequest>> fn) {
        return fn.apply(new Builder()).build();
    }

    /**
     * API name: {@code _meta}
     */
    public final Map<String, JsonData> meta() {
        return this.meta;
    }

    /**
     * API name: {@code aliases}
     */
    public final Map<String, AliasDefinition> aliases() {
        return this.aliases;
    }

    /**
     * Whether the index template should only be added if new or can also replace an
     * existing one
     * <p>
     * API name: {@code create}
     */
    @Nullable
    public final Boolean create() {
        return this.create;
    }

    /**
     * API name: {@code mappings}
     */
    @Nullable
    public final TypeMapping mappings() {
        return this.mappings;
    }

    /**
     * Specify timeout for connection to master
     * <p>
     * API name: {@code master_timeout}
     */
    @Deprecated
    @Nullable
    public final Time masterTimeout() {
        return this.masterTimeout;
    }

    /**
     * Specify timeout for connection to cluster-manager
     * <p>
     * API name: {@code cluster_manager_timeout}
     */
    @Nullable
    public final Time clusterManagerTimeout() {
        return this.clusterManagerTimeout;
    }

    /**
     * Required - The name of the template
     * <p>
     * API name: {@code name}
     */
    public final String name() {
        return this.name;
    }

    /**
     * API name: {@code settings}
     */
    @Nullable
    public final IndexSettings settings() {
        return this.settings;
    }

    /**
     * Required - API name: {@code template}
     */
    public final IndexState template() {
        return this.template;
    }

    /**
     * API name: {@code version}
     */
    @Nullable
    public final Long version() {
        return this.version;
    }

    /**
     * Serialize this object to JSON.
     */
    public void serialize(JsonGenerator generator, JsonpMapper mapper) {
        generator.writeStartObject();
        serializeInternal(generator, mapper);
        generator.writeEnd();
    }

    protected void serializeInternal(JsonGenerator generator, JsonpMapper mapper) {

        if (ApiTypeHelper.isDefined(this.meta)) {
            generator.writeKey("_meta");
            generator.writeStartObject();
            for (Map.Entry<String, JsonData> item0 : this.meta.entrySet()) {
                generator.writeKey(item0.getKey());
                item0.getValue().serialize(generator, mapper);

            }
            generator.writeEnd();

        }
        if (ApiTypeHelper.isDefined(this.aliases)) {
            generator.writeKey("aliases");
            generator.writeStartObject();
            for (Map.Entry<String, AliasDefinition> item0 : this.aliases.entrySet()) {
                generator.writeKey(item0.getKey());
                item0.getValue().serialize(generator, mapper);

            }
            generator.writeEnd();

        }
        if (this.mappings != null) {
            generator.writeKey("mappings");
            this.mappings.serialize(generator, mapper);

        }
        if (this.settings != null) {
            generator.writeKey("settings");
            this.settings.serialize(generator, mapper);

        }
        generator.writeKey("template");
        this.template.serialize(generator, mapper);

        if (this.version != null) {
            generator.writeKey("version");
            generator.write(this.version);

        }

    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Builder for {@link PutComponentTemplateRequest}.
     */

    public static class Builder extends ObjectBuilderBase implements ObjectBuilder<PutComponentTemplateRequest> {
        @Nullable
        private Map<String, JsonData> meta;

        @Nullable
        private Map<String, AliasDefinition> aliases;

        @Nullable
        private Boolean create;

        @Nullable
        private TypeMapping mappings;

        @Deprecated
        @Nullable
        private Time masterTimeout;

        @Nullable
        private Time clusterManagerTimeout;

        private String name;

        @Nullable
        private IndexSettings settings;

        private IndexState template;

        @Nullable
        private Long version;

        /**
         * API name: {@code _meta}
         * <p>
         * Adds all entries of <code>map</code> to <code>meta</code>.
         */
        public final Builder meta(Map<String, JsonData> map) {
            this.meta = _mapPutAll(this.meta, map);
            return this;
        }

        /**
         * API name: {@code _meta}
         * <p>
         * Adds an entry to <code>meta</code>.
         */
        public final Builder meta(String key, JsonData value) {
            this.meta = _mapPut(this.meta, key, value);
            return this;
        }

        /**
         * API name: {@code aliases}
         * <p>
         * Adds all entries of <code>map</code> to <code>aliases</code>.
         */
        public final Builder aliases(Map<String, AliasDefinition> map) {
            this.aliases = _mapPutAll(this.aliases, map);
            return this;
        }

        /**
         * API name: {@code aliases}
         * <p>
         * Adds an entry to <code>aliases</code>.
         */
        public final Builder aliases(String key, AliasDefinition value) {
            this.aliases = _mapPut(this.aliases, key, value);
            return this;
        }

        /**
         * API name: {@code aliases}
         * <p>
         * Adds an entry to <code>aliases</code> using a builder lambda.
         */
        public final Builder aliases(String key, Function<AliasDefinition.Builder, ObjectBuilder<AliasDefinition>> fn) {
            return aliases(key, fn.apply(new AliasDefinition.Builder()).build());
        }

        /**
         * Whether the index template should only be added if new or can also replace an
         * existing one
         * <p>
         * API name: {@code create}
         */
        public final Builder create(@Nullable Boolean value) {
            this.create = value;
            return this;
        }

        /**
         * API name: {@code mappings}
         */
        public final Builder mappings(@Nullable TypeMapping value) {
            this.mappings = value;
            return this;
        }

        /**
         * API name: {@code mappings}
         */
        public final Builder mappings(Function<TypeMapping.Builder, ObjectBuilder<TypeMapping>> fn) {
            return this.mappings(fn.apply(new TypeMapping.Builder()).build());
        }

        /**
         * Specify timeout for connection to master
         * <p>
         * API name: {@code master_timeout}
         */
        @Deprecated
        public final Builder masterTimeout(@Nullable Time value) {
            this.masterTimeout = value;
            return this;
        }

        /**
         * Specify timeout for connection to master
         * <p>
         * API name: {@code master_timeout}
         */
        @Deprecated
        public final Builder masterTimeout(Function<Time.Builder, ObjectBuilder<Time>> fn) {
            return this.masterTimeout(fn.apply(new Time.Builder()).build());
        }

        /**
         * Specify timeout for connection to cluster-manager
         * <p>
         * API name: {@code cluster_manager_timeout}
         */
        public final Builder clusterManagerTimeout(@Nullable Time value) {
            this.clusterManagerTimeout = value;
            return this;
        }

        /**
         * Specify timeout for connection to cluster-manager
         * <p>
         * API name: {@code cluster_manager_timeout}
         */
        public final Builder clusterManagerTimeout(Function<Time.Builder, ObjectBuilder<Time>> fn) {
            return this.clusterManagerTimeout(fn.apply(new Time.Builder()).build());
        }

        /**
         * Required - The name of the template
         * <p>
         * API name: {@code name}
         */
        public final Builder name(String value) {
            this.name = value;
            return this;
        }

        /**
         * API name: {@code settings}
         */
        public final Builder settings(@Nullable IndexSettings value) {
            this.settings = value;
            return this;
        }

        /**
         * API name: {@code settings}
         */
        public final Builder settings(Function<IndexSettings.Builder, ObjectBuilder<IndexSettings>> fn) {
            return this.settings(fn.apply(new IndexSettings.Builder()).build());
        }

        /**
         * Required - API name: {@code template}
         */
        public final Builder template(IndexState value) {
            this.template = value;
            return this;
        }

        /**
         * Required - API name: {@code template}
         */
        public final Builder template(Function<IndexState.Builder, ObjectBuilder<IndexState>> fn) {
            return this.template(fn.apply(new IndexState.Builder()).build());
        }

        /**
         * API name: {@code version}
         */
        public final Builder version(@Nullable Long value) {
            this.version = value;
            return this;
        }

        /**
         * Builds a {@link PutComponentTemplateRequest}.
         *
         * @throws NullPointerException
         *             if some of the required fields are null.
         */
        public PutComponentTemplateRequest build() {
            _checkSingleUse();

            return new PutComponentTemplateRequest(this);
        }
    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Json deserializer for {@link PutComponentTemplateRequest}
     */
    public static final JsonpDeserializer<PutComponentTemplateRequest> _DESERIALIZER = ObjectBuilderDeserializer.lazy(
        Builder::new,
        PutComponentTemplateRequest::setupPutComponentTemplateRequestDeserializer
    );

    protected static void setupPutComponentTemplateRequestDeserializer(ObjectDeserializer<PutComponentTemplateRequest.Builder> op) {

        op.add(Builder::meta, JsonpDeserializer.stringMapDeserializer(JsonData._DESERIALIZER), "_meta");
        op.add(Builder::aliases, JsonpDeserializer.stringMapDeserializer(AliasDefinition._DESERIALIZER), "aliases");
        op.add(Builder::mappings, TypeMapping._DESERIALIZER, "mappings");
        op.add(Builder::settings, IndexSettings._DESERIALIZER, "settings");
        op.add(Builder::template, IndexState._DESERIALIZER, "template");
        op.add(Builder::version, JsonpDeserializer.longDeserializer(), "version");

    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Endpoint "{@code cluster.put_component_template}".
     */
    public static final Endpoint<PutComponentTemplateRequest, PutComponentTemplateResponse, ErrorResponse> _ENDPOINT = new SimpleEndpoint<>(

        // Request method
        request -> {
            return "PUT";

        },

        // Request path
        request -> {
            final int _name = 1 << 0;

            int propsSet = 0;

            propsSet |= _name;

            if (propsSet == (_name)) {
                StringBuilder buf = new StringBuilder();
                buf.append("/_component_template");
                buf.append("/");
                SimpleEndpoint.pathEncode(request.name, buf);
                return buf.toString();
            }
            throw SimpleEndpoint.noPathTemplateFound("path");

        },

        // Request parameters
        request -> {
            Map<String, String> params = new HashMap<>();
            if (request.masterTimeout != null) {
                params.put("master_timeout", request.masterTimeout._toJsonString());
            }
            if (request.clusterManagerTimeout != null) {
                params.put("cluster_manager", request.clusterManagerTimeout._toJsonString());
            }
            if (request.create != null) {
                params.put("create", String.valueOf(request.create));
            }
            return params;

        },
        SimpleEndpoint.emptyMap(),
        true,
        PutComponentTemplateResponse._DESERIALIZER
    );
}
