/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opensaml.security.credential.impl;

import java.util.ArrayList;
import java.util.Collection;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import net.shibboleth.utilities.java.support.resolver.CriteriaSet;
import net.shibboleth.utilities.java.support.resolver.ResolverException;

import org.opensaml.security.credential.Credential;

/**
 * An implementation of {@link org.opensaml.security.credential.CredentialResolver} that
 * uses a {@link Collection} as the underlying credential source.
 * 
 * <p>
 * The credentials returned are filtered based on any
 * {@link org.opensaml.security.credential.criteria.impl.EvaluableCredentialCriterion} that may
 * have been present in the specified criteria set, or that are resolved by lookup in the
 * {@link org.opensaml.security.credential.criteria.impl.EvaluableCredentialCriteriaRegistry}.
 * </p>
 */
public class CollectionCredentialResolver extends AbstractCriteriaFilteringCredentialResolver {
    
    /** The collection of credentials which is the underlying store for the resolver. */
    private final Collection<Credential> collection;
    
    /**
     * Constructor.
     * 
     * An {@link ArrayList} is used as the underlying collection implementation.
     *
     */
    public CollectionCredentialResolver() {
        super();
        collection = new ArrayList<>();
    }
    
    /**
     * Constructor.
     *
     * @param credentials the credential collection which is the backing store for the resolver
     */
    public CollectionCredentialResolver(@Nonnull final Collection<Credential> credentials) {
        super();
        collection = credentials;
    }
    
    /**
     * Get the (modifiable) credential collection which is the backing store for the resolver.
     * 
     * @return the credential collection backing store
     */
    @Nonnull public Collection<Credential> getCollection() {
        return collection;
    }

    /** {@inheritDoc} */
    @Nonnull protected Iterable<Credential> resolveFromSource(@Nullable final CriteriaSet criteriaSet)
            throws ResolverException {
        return collection;
    }

}